#include "baseline.h"

#include "base.h"
#include "color.h"

#define  MAX_DEGREE_ORDER		3

#define  SMALL_NUMBER			1.0e-4

static int baseline_type;
static int base_fit_type;
static int half_width;
static int degree_order;
static int npoints;
static int npoints_orig;
static int first = 0;
static int step = 1;
static float *data;

static int nbaseline;
static int *baseline;
static Bool have_enough_baseline;

static int order;
static int width;

static int nbase_points = 0;
static int nbase_points_alloc = 0;
static int *base_points = (int *) NULL;

static int nbase_points_orig = 0;
static int nbase_points_orig_alloc = 0;
static int *base_points_orig = (int *) NULL;

static void free_base_points()
{
    FREE(base_points, int);

    nbase_points_alloc = 0;
}

static Status alloc_base_points(String error_msg)
{
    if (npoints > nbase_points_alloc)
    {
	free_base_points();

	sprintf(error_msg, "allocating base point memory");
	MALLOC(base_points, int, npoints);

	nbase_points_alloc = npoints;
    }

    return  OK;
}

static void free_base_points_orig()
{
    FREE(base_points_orig, int);

    nbase_points_orig_alloc = 0;
}

static Status alloc_base_points_orig(String error_msg)
{
    if (nbase_points_orig > nbase_points_orig_alloc)
    {
	free_base_points_orig();

	sprintf(error_msg, "allocating base point memory");
	MALLOC(base_points_orig, int, nbase_points_orig);

	nbase_points_orig_alloc = nbase_points_orig;
    }

    return  OK;
}

static int nearest_point(float point)  /* assumes points start at 1 */
{
    int p;

    p = NEAREST_INTEGER(point);

    return  p;
}

Status insert_baseline_point(float point, int n, String error_msg)
{
    int i, j, p;

    npoints = n;

    p = nearest_point(point);

    if ((p < 1) || (p > npoints))
	return  OK;

    CHECK_STATUS(alloc_base_points(error_msg));

    for (i = 0; i < nbase_points; i++)
    {
	if (p <= base_points[i])
	    break;
    }

    if ((i < nbase_points) && (p == base_points[i]))
	return  OK;

    for (j = nbase_points; j > i; j--)
	base_points[j] = base_points[j-1];

    base_points[i] = p;

    nbase_points++;

/*
    printf("insert (%d):", nbase_points);
    for (i = 0; i < nbase_points; i++)
	printf(" %d", base_points[i]);
    printf("\n");
*/

    return  OK;
}

Status delete_baseline_point(float point, int n, String error_msg)
{
    int i, j, p, p1, p2;

    if (nbase_points == 0)
	return  OK;

    npoints = n;

    p = nearest_point(point);

    p = MAX(p, 1);
    p = MIN(p, npoints);

    for (i = 0; i < nbase_points; i++)
    {
	if (p <= base_points[i])
	    break;
    }

    if (i == nbase_points)
    {
	i--;
    }
    else if (i > 0)
    {
	p1 = base_points[i-1];
	p2 = base_points[i];

	if ((p-p1) < (p2-p))
	    i--;
    }


    for (j = i; j < (nbase_points-1); j++)
	base_points[j] = base_points[j+1];

    nbase_points--;

/*
    printf("delete (%d):", nbase_points);
    for (i = 0; i < nbase_points; i++)
	printf(" %d", base_points[i]);
    printf("\n");
*/

    return  OK;
}

static Status find_auto_baseline(String error_msg)
{
    int nchisq = 0;
    float avg_min_chisq = 0;

    if (half_width < 1)
	RETURN_ERROR_MSG("baseline half width must be >= 1");

    width = 2*half_width + 1;

    if (npoints < width)
	RETURN_ERROR_MSG("baseline width too large for given #points");

    have_enough_baseline = find_baseline(npoints, width, &avg_min_chisq,
					&nchisq, data, &nbaseline, &baseline);

    return  OK;
}

static Status transform_base_points(String error_msg)
{
    int i;
    float p;

    if (step == 0) /* should never happen */
	RETURN_ERROR_MSG("panic: illegal step size in transform_base_points");

    nbase_points = 0;

    for (i = 0; i < nbase_points_orig; i++)
    {
	p = ((float) (base_points_orig[i] - first)) / ((float) step);
	CHECK_STATUS(insert_baseline_point(p, npoints, error_msg));
    }

    nbase_points_orig = 0;

    return  OK;
}

static Status find_pick_baseline(String error_msg)
{
    if (nbase_points_orig > 0)
	CHECK_STATUS(transform_base_points(error_msg));

    if (nbase_points == 0) 
    {
	have_enough_baseline = FALSE;
    }
    else
    {
	nbaseline = nbase_points;
	baseline = base_points;
	have_enough_baseline = TRUE;
    }

    return  OK;
}

static Status init_baseline(String error_msg)
{
    if (base_fit_type == CONST_BASELINE)
	order = 1;
    else if (base_fit_type == POLY_BASELINE)
	order = degree_order + 1;
    else /* (base_fit_type == TRIG_BASELINE) */
	order = 2*degree_order + 1;

    if ((base_fit_type != CONST_BASELINE) && (degree_order > MAX_DEGREE_ORDER))
	RETURN_ERROR_MSG("degree/order not allowed to be that large");

    if (order < 1)
	RETURN_ERROR_MSG("order must be >= 1");

    if (order > npoints)
	RETURN_ERROR_MSG("order too large for given #points");

    if (alloc_baseline(npoints, order) == ERROR)
	RETURN_ERROR_MSG("allocating baseine memory");

    return  OK;
}

static void fit_baseline()
{
    if ((base_fit_type == CONST_BASELINE) || (order == 1))
	fit_const_baseline(npoints, data, nbaseline, baseline);
    else if (base_fit_type == POLY_BASELINE)
	fit_poly_baseline(npoints, order, data, nbaseline, baseline);
    else /* (base_fit_type == TRIG_BASELINE) */
	fit_trig_baseline(npoints, npoints_orig, order, data,
							nbaseline, baseline);
}

Status do_baseline(Baseline_info *info, String error_msg)
{
    baseline_type = info->baseline_type;
    base_fit_type = info->base_fit_type;
    half_width = info->half_width;
    degree_order = info->degree_order;
    npoints = info->npoints;
    npoints_orig = info->npoints_orig;
    first = info->first;
    step = info->step;
    data = info->data;

    if (baseline_type == NO_BASELINE)
	return  OK;

    CHECK_STATUS(init_baseline(error_msg));

    if (baseline_type == AUTO_BASELINE)
    {
	CHECK_STATUS(find_auto_baseline(error_msg));
    }
    else /* (baseline_type == PICK_BASELINE) */
    {
	CHECK_STATUS(find_pick_baseline(error_msg));
    }

    if (have_enough_baseline)
	fit_baseline();

    return  OK;
}

void draw_baseline(Draw_funcs *funcs, int n, float *d,
						float *lower, float *upper)
{
    int i, j, begin, end;
    float a0, b0, a1, b1, tick_size;

    (*(funcs->new_draw_range))(lower[0], lower[1], upper[0], upper[1], TRUE);

    begin = lower[0] - 1;
		/* floor, taking into account that points start at 1 */
    begin = MAX(0, begin);

    end = upper[0] - SMALL_NUMBER;
		/* ceiling, taking into account that points start at 1 */
    end = MIN(n-1, end);

#define  TICK_SIZE_SCALE	0.05

    tick_size = TICK_SIZE_SCALE * (upper[1] - lower[1]);

/*
    printf("draw: begin = %d, end = %d\n", begin, end);
*/

    for (i = 0; i < nbase_points; i++)
    {
	j = base_points[i] - 1;

	if ((j < begin) || (j > end))
	    continue;

	a0 = a1 = j + 1;
	b0 = d[j] - tick_size;
	b1 = d[j] + tick_size;

	(*(funcs->draw_line))(a0, b0, a1, b1);
    }
}

Status load_baseline_points(String file, String error_msg)
{
    int n;
    Line line;
    FILE *fp;

    sprintf(error_msg, "baseline file \"%s\": ", file);
    error_msg += strlen(error_msg);

    if (OPEN_FOR_READING(fp, file) == ERROR)
	RETURN_ERROR_MSG("opening for reading");

    nbase_points_orig = 0;
    while (fgets(line, LINE_SIZE, fp))
    {
	if (fscanf(fp, "%d", &n) == 1)
	    nbase_points_orig = MAX(n, nbase_points_orig);
    }

    fclose(fp);

    if (nbase_points_orig == 0)
	RETURN_ERROR_MSG("no baseline points");

    CHECK_STATUS(alloc_base_points_orig(error_msg));

    return  read_base_points(file, nbase_points_orig, &nbase_points_orig,
	                                        base_points_orig, error_msg);
}

Status save_baseline_points(String file, String error_msg)
{
    int i, p;
    FILE *fp;

    sprintf(error_msg, "baseline file \"%s\": ", file);
    error_msg += strlen(error_msg);

    if (OPEN_FOR_WRITING(fp, file) == ERROR)
	RETURN_ERROR_MSG("opening for writing");

    for (i = 0; i < nbase_points; i++)
    {
	p = first + step*base_points[i];
	fprintf(fp, "%d\n", p);
    }

    fclose(fp);

    return  OK;
}

void clear_baseline_points()
{
    nbase_points = 0;
}
