#include "weight.h"

#include "input.h"
#include "parser.h"
#include "script.h"

#define  SMALL_NUMBER  (1.0e-6)

#define  SINEBELL_ANGLE		2

#define  GAUSSIAN_A		2
#define  GAUSSIAN_B		3

#define  GAUSSIAN_LB		2
#define  GAUSSIAN_S		3
#define  GAUSSIAN_SW		4

#define  INV_COSINE_FREQ	2
#define  INV_COSINE_SW		3

#define  DECAY_D		2

#define  DECAY_LB		2
#define  DECAY_SW		3

#define  WEIGHT_FILE		2

static double exp_x; /* for below only */
#define  MAX_EXP  30.0
#define  MIN_EXP  -MAX_EXP
#define  EXP(x) \
    ((exp_x = x) > MAX_EXP ? LARGE_FLOAT : (exp_x < MIN_EXP) ? 0 : exp(exp_x))

static Status do_sinebell(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float angle, delta, weight;
    float *data_in, *data_out;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_int_to_float(store[SINEBELL_ANGLE]);
    angle = *((float *) (store[SINEBELL_ANGLE]->data));

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    angle *= RADIAN_SCALE;
    delta = (PI - angle) / ((float) n);

    for (i = 0; i < n; i++)
    {
	weight = sin((double) (i*delta + angle));

	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    return  OK;
}

static Status do_sinebell2(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float angle, delta, weight;
    float *data_in, *data_out;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_int_to_float(store[SINEBELL_ANGLE]);
    angle = *((float *) (store[SINEBELL_ANGLE]->data));

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    angle *= RADIAN_SCALE;
    delta = (PI - angle) / ((float) n);

    for (i = 0; i < n; i++)
    {
	weight = sin((double) (i*delta + angle));
	weight *= weight;

	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    return  OK;
}

static Status do_gaussian(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float a, b, c, weight;
    float *data_in, *data_out;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_int_to_float(store[GAUSSIAN_A]);
    store_int_to_float(store[GAUSSIAN_B]);
    a = *((float *) (store[GAUSSIAN_A]->data));
    b = *((float *) (store[GAUSSIAN_B]->data));

    if ((a < SMALL_NUMBER) || (a > (1-SMALL_NUMBER)))
    {
	sprintf(error_msg, "one fraction = %2.1e, must be between 0 and 1", a);
	return  ERROR;
    }

    if (b < SMALL_NUMBER)
    {
	sprintf(error_msg, "end value = %2.1e, must be > 0", b);
	return  ERROR;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    b = log((double) b) / ((a-1)*(a-1));

    for (i = 0; i < n; i++)
    {
	c = ((float) i) / ((float) (n-1));
	weight = EXP(b * (c - a) * (c - a));

	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    return  OK;
}

/*  gaussian_sw suggested by Rasmus Fogh  */
static Status do_gaussian_sw(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float lb, s, sw, a, b, c, weight;
    float *data_in, *data_out;
    Ref_info *ref;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_int_to_float(store[GAUSSIAN_LB]);
    store_int_to_float(store[GAUSSIAN_S]);
    store_int_to_float(store[GAUSSIAN_SW]);
    lb = *((float *) (store[GAUSSIAN_LB]->data));
    s = *((float *) (store[GAUSSIAN_S]->data));
    sw = *((float *) (store[GAUSSIAN_SW]->data));

    if (sw < SMALL_NUMBER)
    {
	ref = get_input_ref();

	if (!ref)
	    RETURN_ERROR_MSG("null reference");

	sw = ref->sw;
    }

    if (sw < SMALL_NUMBER)
    {
	sprintf(error_msg, "SW = %2.1e, must be > 0", sw);
	return  ERROR;
    }

    if (s < SMALL_NUMBER)
    {
	sprintf(error_msg, "sharpening factor = %2.1e, must be > 0", s);
	return  ERROR;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    a = - LN2 / (s * s);
    b = PI * lb * step / (2 * sw);
    c = - b * b * s * s / (4 * LN2);

    for (i = 0; i < n; i++)
    {
	weight = EXP(a + b*i + c*i*i);

	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    return  OK;
}

static Status do_decay(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float c, d, weight;
    float *data_in, *data_out;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_int_to_float(store[DECAY_D]);
    d = *((float *) (store[DECAY_D]->data));

    if (d < SMALL_NUMBER)
    {
	sprintf(error_msg, "end value = %2.1e, must be > 0", d);
	return  ERROR;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    if (n < 2)  /* unusual */
    {
	COPY_VECTOR(data_out, data_in, n*step);
	return  OK;
    }

    d = log((double) d);

    for (i = 0; i < n; i++)
    {
	c = ((float) i) / ((float) (n-1));
	weight = EXP(c * d);

	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    return  OK;
}

/*  decay_sw suggested by Rasmus Fogh  */
static Status do_decay_sw(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float lb, sw, r, weight;
    float *data_in, *data_out;
    Ref_info *ref;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_int_to_float(store[DECAY_LB]);
    store_int_to_float(store[DECAY_SW]);
    lb = *((float *) (store[DECAY_LB]->data));
    sw = *((float *) (store[DECAY_SW]->data));

    if (sw < SMALL_NUMBER)
    {
	ref = get_input_ref();

	if (!ref)
	    RETURN_ERROR_MSG("null reference");

	sw = ref->sw;
    }

    if (sw < SMALL_NUMBER)
    {
	sprintf(error_msg, "SW = %2.1e, must be > 0", sw);
	return  ERROR;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    if (n < 2)  /* unusual */
    {
	COPY_VECTOR(data_out, data_in, n*step);
	return  OK;
    }

    r = - PI * lb * step / (2 * sw);

    for (i = 0; i < n; i++)
    {
	weight = EXP(r * i);

	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    return  OK;
}

/*  inv_cosine suggested by David Agard  */
static Status do_inv_cosine(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float freq, sw, c, weight;
    float *data_in, *data_out;
    double del;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_int_to_float(store[INV_COSINE_FREQ]);
    store_int_to_float(store[INV_COSINE_SW]);
    freq = *((float *) (store[INV_COSINE_FREQ]->data));
    sw = *((float *) (store[INV_COSINE_SW]->data));

    if (freq < SMALL_NUMBER)
    {
	sprintf(error_msg, "frequency = %2.1e, must be > 0", freq);
	return  ERROR;
    }

    if (sw < SMALL_NUMBER)
    {
	sprintf(error_msg, "spectral width = %2.1e, must be > 0", sw);
	return  ERROR;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    del = HALF * PI * freq / sw;

    for (i = 0; i < n; i++)
    {
	c = cos(i * del);
	weight = c / (c*c + SMALL_NUMBER);

	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    return  OK;
}

static Status do_weight_file(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, j, n, step;
    float weight;
    float *data_in, *data_out;
    String file = ((String) (store[WEIGHT_FILE]->data));
    FILE *fp;

    n = store[INPUT_X]->ndata;
    store_int_to_float(store[INPUT_X]);
    data_in = (float *) (store[INPUT_X]->data);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));
    data_out = (float *) (store[OUTPUT_X]->data);

    if (store[INPUT_X]->data_type & PARSER_REAL)
	step = 1;
    else /* (store[INPUT_X]->data_type & PARSER_COMPLEX) */
	step = 2;

    if (OPEN_FOR_READING(fp, file))
    {
        sprintf(error_msg, "opening '%s' for reading", file);
        return  ERROR;
    }

    for (i = 0; (i < n) && (fscanf(fp, "%f", &weight) == 1); i++)
    {
	for (j = 0; j < step; j++)
	    data_out[i*step+j] = weight * data_in[i*step+j];
    }

    fclose(fp);

    if (i < n)
    {
	sprintf(error_msg, "only found %d values, expecting %d", i, n);
	return  ERROR;
    }

    return  OK;
}

Status init_sinebell(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "sinebell", do_sinebell,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_sinebell2(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "sinebell2", do_sinebell2,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_gaussian(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "gaussian", do_gaussian,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_gaussian_sw(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "gaussian_sw", do_gaussian_sw,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_decay(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "decay", do_decay,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_decay_sw(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "decay_sw", do_decay_sw,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_inv_cosine(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "inv_cosine", do_inv_cosine,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_weight_file(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "weight_file", do_weight_file,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}
