#include "slider_popup.h"

#include "slider.hlp"

#include "callback.h"
#include "display_popup.h"
#include "param.h"
#include "slider.h"
#include "window.h"

static Widget slider_popup = (Widget) NULL;
static Widget slider_form;

static int nsliders;
static int ntexts = 2;
static Textfield_item **slider_text;
static Widget **slider_field;
static Param_value **slider_value;
static Widget *slider_scale;
static Scale_numbers *numbers;
static Bool *slider_active;

#define  VALUE_IN_RANGE(numbers) \
	 {   if ((numbers.value < numbers.minimum) || \
		 (numbers.value > numbers.maximum)) \
		 numbers.value = 0.5*(numbers.minimum + numbers.maximum);   }

#define  SET_VALUE(string, value) \
	 {   sprintf(string, "%5.2f", value);   }

static void set_slider_fields()
{
    int i;

    for (i = 0; i < nsliders; i++)
    {
	set_textfields(slider_text[i], 1);
	set_textfields(slider_text[i]+1, 1);
	set_scale_numbers(slider_scale[i], &numbers[i]);
    }
}

static void set_slider_defaults()
{
    int i;
    Slider_info slider_info;

    for (i = 0; i < nsliders; i++)
    {
	slider_numbers(i, &slider_info);

	numbers[i].minimum = slider_info.minimum;
	numbers[i].maximum = slider_info.maximum;
	numbers[i].value = slider_info.value;

	SET_VALUE(slider_value[i][0], numbers[i].minimum);
	SET_VALUE(slider_value[i][1], numbers[i].maximum);
    }
}

static void set_slider_numbers()
{
    int i;

    for (i = 0; i < nsliders; i++)
    {
	numbers[i].minimum = atof(slider_value[i][0]);
	numbers[i].maximum = atof(slider_value[i][1]);
    }
}

void update_slider_params()
{
    int i;

    if (slider_popup)
    {
	for (i = 0; i < nsliders; i++)
	{
	    get_textfields(slider_text[i], 1);
	    get_textfields(slider_text[i]+1, 1);
	}
    }
}

static Status valid_slider_numbers(String error_msg)
{
    int i;

    for (i = 0; i < nsliders; i++)
    {
	if (numbers[i].minimum >= numbers[i].maximum)
	{
	    sprintf(error_msg, "slider %d: minimum >= maximum", i+1);
	    return  ERROR;
	}

	VALUE_IN_RANGE(numbers[i]);

	if (slider_popup)
	    set_scale_numbers(slider_scale[i], &numbers[i]);
    }

    return  OK;
}

static void make_sliders_inactive()
{
    int i;

    for (i = 0; i < nsliders; i++)
	slider_active[i] = FALSE;
}

static void scale_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    int n, command;
    Status status;
    static int previous_status = OK, previous_slider = -1;
    Line error_msg;

    n = ((Widget *) data) - slider_scale;

    if (!slider_active[n])
    {
	make_sliders_inactive();
	slider_active[n] = TRUE;
	set_scale_numbers(slider_scale[n], &numbers[n]);

	return;
    }

    get_scale_value(cbs, &numbers[n]);

    command = change_slider(n, numbers[n].value);
 
    if (command >= 0)
    {
	status = do_display(command, error_msg);

/*  only report error message if have not just had one for same slider  */
/*  this may not work perfectly, but avoids irritating multiple messages  */

	if ((status == ERROR) &&
			((previous_status == OK) || (previous_slider != n)))
	{
	    previous_status = status;
	    previous_slider = n;
	    ERROR_AND_RETURN(error_msg);
	}

	previous_status = status;
	previous_slider = n;
    }
}

static void apply_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    update_slider_params();
    set_slider_numbers();

    if (valid_slider_numbers(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);

    if (do_display_drawing(0, error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

#define  SCROLLED_WIDTH		600
#define  SCROLLED_HEIGHT	500
#define  SCROLLED_MARGIN	35
#define  SCROLL_SLIDERS		7

static void create_slider_popup(Widget parent)
{
    int i, width, height;
    Widget buttons, separator, slider, sw, sw_form;
    Scale_info scale_info;
    Text_scale_info text_scale_info;
    Scrolled_window_info scrolled_info;
    Bool scrolls;
    Apply_dismiss_help_info adh_info;
    char apply_label[] = "apply";

    update_slider_params();

    slider_popup = create_popup(parent, "Sliders");
    CHECK_WIDGET_WARNING(slider_popup);

    slider_form = create_form(slider_popup);
    CHECK_WIDGET_DESTROY_WARNING(slider_form, slider_popup);

    scrolls = (nsliders > SCROLL_SLIDERS)  ?  TRUE  :  FALSE;

    if (scrolls)
    {
	scrolled_info.width = SCROLLED_WIDTH + SCROLLED_MARGIN;
	scrolled_info.height = SCROLLED_HEIGHT;
	sw = create_scrolled_window(slider_form, &scrolled_info);
	CHECK_WIDGET_DESTROY_WARNING(sw, slider_popup);
	attachments(sw, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

	sw_form = create_form(sw);
	CHECK_WIDGET_DESTROY_WARNING(sw_form, slider_popup);
	attachments(sw_form, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH,
								FORM_ATTACH);
    }
    else
    {
	sw_form = slider_form;
    }

/*
    XtVaSetValues(sw_form,
			XmNwidth,	SCROLLED_WIDTH,
			NULL);
*/
    get_widget_size(sw_form, &width, &height);
    width = SCROLLED_WIDTH;
    set_widget_size(sw_form, width, height);

    scale_info.orientation = HORIZONTAL;
    scale_info.show_value = TRUE;
    scale_info.value_changed = scale_callback;
    scale_info.dragged = scale_callback;
    scale_info.data = (Generic_ptr) NULL; /* get rid of later */

    text_scale_info.scale_info = &scale_info;

    separator = FORM_ATTACH;

    set_slider_defaults();

    for (i = 0; i < nsliders; i++)
    {
	scale_info.numbers = numbers + i;
	scale_info.title = slider_label(i);
	scale_info.data = (Generic_ptr) (slider_scale + i);

	text_scale_info.scale = slider_scale + i;
	text_scale_info.limits = slider_text[i];

	slider = create_text_scale(sw_form, &text_scale_info);
	CHECK_WIDGET_DESTROY_WARNING(slider, slider_popup);
	attachments(slider, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

	separator = create_separator(sw_form);
	CHECK_WIDGET_DESTROY_WARNING(separator, slider_popup);
	attachments(separator, slider, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);
    }

    adh_info.apply_label = apply_label;
    adh_info.apply_callback = apply_callback;
    adh_info.dismiss_form = slider_form;
    adh_info.help_message = slider_help;

    buttons = create_apply_dismiss_help(slider_form, &adh_info);
    CHECK_WIDGET_DESTROY_WARNING(buttons, slider_popup);

    if (scrolls)
	attachments(buttons, sw, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);
    else
	attachments(buttons, separator, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);

    offsets(buttons, UNIT_OFFSET, UNIT_OFFSET, NO_OFFSET, NO_OFFSET);

    set_slider_fields();

    if (scrolls)
    {
	manage_widget(sw_form);
	manage_widget(sw);
    }

    manage_widget(slider_form);
}

static Status alloc_slider_popup_memory()
{
    int i, j;

    if (nsliders == 0)
	return  OK;

    MALLOC(slider_active, Bool, nsliders);
    make_sliders_inactive();

    MALLOC(slider_text, Textfield_item *, nsliders);
    MALLOC(slider_field, Widget *, nsliders);
    MALLOC(slider_value, Param_value *, nsliders);

    MALLOC(slider_scale, Widget, nsliders);
    MALLOC(numbers, Scale_numbers, nsliders);

    for (i = 0; i < nsliders; i++)
    {
	MALLOC(slider_text[i], Textfield_item, ntexts);
	MALLOC(slider_field[i], Widget, ntexts);
	MALLOC(slider_value[i], Param_value, ntexts);

	for (j = 0; j < ntexts; j++)
	{
	    slider_text[i][j].label = (char *) NULL;
	    slider_text[i][j].value = slider_value[i][j];
	    slider_text[i][j].callback = (Callback_proc) NULL;
	    slider_text[i][j].data = (Callback_ptr) NULL;
	    slider_text[i][j].field = &slider_field[i][j];
	}
    }

    return  OK;
}

void slider_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!slider_popup)
    {
	nsliders = number_sliders();
	if (alloc_slider_popup_memory() == ERROR)
	    ERROR_AND_EXIT("allocating slider popup memory");

	create_slider_popup(parent);
    }


    if (slider_popup)
    {
	make_sliders_inactive();
	popup(slider_form);
    }
}
