#include "complex.h"

#include "parser.h"
#include "script.h"

static Status do_complex(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int m, n;
    float *data_in, *data_out;
    Parser_store *s;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    if (s->data_type & PARSER_REAL)
    {
	if (n % 2)
	{
	    sprintf(error_msg, "number of points = %d, must be even", n);
	    return  ERROR;
	}

	m = n;
	n /= 2;
    }
    else if (s->data_type & PARSER_COMPLEX)
    {
	m = 2 * n;
    }

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    COPY_VECTOR(data_out, data_in, m);

    return  OK;
}

static Status do_complexify(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in, *data_out;
    Parser_store *s;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    for (i = 0; i < n; i++)
    {
	data_out[2*i] = data_in[i];
	data_out[2*i+1] = 0;
    }

    return  OK;
}

static Status do_conjugate(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in, *data_out;
    Parser_store *s;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    n *= 2;
    for (i = 0; i < n; i += 2)
    {
	data_out[i] = data_in[i];
	data_out[i+1] = - data_in[i+1];
    }

    return  OK;
}

static Status do_exchange(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in, *data_out;
    Parser_store *s;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    n *= 2;
    for (i = 0; i < n; i += 2)
    {
	data_out[i] = data_in[i+1];
	data_out[i+1] = data_in[i];
    }

    return  OK;
}

static Status do_real(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int n;
    float *data_in, *data_out;
    Parser_store *s;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    if (s->data_type & PARSER_COMPLEX)
	n *= 2;

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    COPY_VECTOR(data_out, data_in, n);

    return  OK;
}

static Status do_reduce(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in, *data_out;
    Parser_store *s;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    for (i = 0; i < n; i++)
	data_out[i] = data_in[2*i];

    return  OK;
}

static Status do_magnitude(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in, *data_out;
    Parser_store *s;
    double d;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    if (s->data_type & PARSER_COMPLEX)
    {
	for (i = 0; i < n; i++)
	{
	    d = data_in[2*i]*data_in[2*i] + data_in[2*i+1]*data_in[2*i+1];
	    data_out[i] = sqrt(d);
	}
    }
    else /* (s->data_type & PARSER_REAL) */
    {
	for (i = 0; i < n; i++)
	    data_out[i] = ABS(data_in[i]);
    }

    return  OK;
}

static Status do_magnitude2(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    int i, n;
    float *data_in, *data_out;
    Parser_store *s;

    s = store[INPUT_X];
    n = s->ndata;
    store_int_to_float(s);

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], n, error_msg));

    data_in = (float *) (s->data);
    data_out = (float *) (store[OUTPUT_X]->data);

    if (s->data_type & PARSER_COMPLEX)
    {
	for (i = 0; i < n; i++)
	    data_out[i] = data_in[2*i]*data_in[2*i] + data_in[2*i+1]*data_in[2*i+1];
    }
    else /* (s->data_type & PARSER_REAL) */
    {
	for (i = 0; i < n; i++)
	    data_out[i] = data_in[i] * data_in[i];
    }

    return  OK;
}

Status init_complex(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "complex", do_complex,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_REAL;
    store[OUTPUT_X]->data_type |= PARSER_COMPLEX;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_complexify(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "complexify", do_complexify,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_REAL;
    store[OUTPUT_X]->data_type |= PARSER_COMPLEX;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_conjugate(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "conjugate", do_conjugate,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_exchange(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "exchange", do_exchange,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_real(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "real", do_real,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_COMPLEX;
    store[OUTPUT_X]->data_type |= PARSER_REAL;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_reduce(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "reduce", do_reduce, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_COMPLEX;
    store[OUTPUT_X]->data_type |= PARSER_REAL;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_magnitude(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "magnitude", do_magnitude,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_COMPLEX;
    store[OUTPUT_X]->data_type |= PARSER_REAL;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}

Status init_magnitude2(int nstore, Parser_store **store, String error_msg)
{
    if (setup_command(nstore, store, "magnitude2", do_magnitude2,
							error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = store[INPUT_X]->data_type & ~PARSER_COMPLEX;
    store[OUTPUT_X]->data_type |= PARSER_REAL;
    store[OUTPUT_X]->data_type |= PARSER_INT | PARSER_FLOAT;

    return  OK;
}
