#include "parse.h"

#define  MAX_NARGS		200

#define  PARSE_SEPARATOR	" \t"

static Generic_ptr args[MAX_NARGS];
static int int_args[MAX_NARGS];
static float float_args[MAX_NARGS];

static FILE *parse_fp;

Status no_parse_func(Generic_ptr *args, String error_msg)
{
    return  OK;
}

/*  find_key_word is terribly inefficient, but should not matter  */

static Parse_line *find_key_word(String key, Parse_line *table)
{
    Parse_line *p;

    for (p = table; p->pattern; p++)
	if (!strcmp(p->pattern, key))
	    return  p;

    return  (Parse_line *) NULL;
}

static Status read_token(String ptr, int n, int type, String error_msg)
{
    if (n >= MAX_NARGS)
	RETURN_ERROR_MSG("too many arguments");

    if(type & PARSE_INT)
    {
	if (sscanf(ptr, "%d", int_args+n) != 1)
	{
	    RETURN_ERROR_MSG("not an integer");
	}

	args[n] = (Generic_ptr) (int_args+n);
    }
    else if(type & PARSE_FLOAT)
    {
	if (sscanf(ptr, "%f", float_args+n) != 1)
	{
	    RETURN_ERROR_MSG("not a floating point number");
	}

	args[n] = (Generic_ptr) (float_args+n);
    }
    else if(type & PARSE_STRING)
    {
	args[n] = (Generic_ptr) ptr;
    }
    else if(type & PARSE_FILE)
    {
	args[n] = (Generic_ptr) parse_fp;
    }
    else if(type & PARSE_REST)
    {
	args[n] = (Generic_ptr) ptr;
    }
    else /* should never get here! */
    {
	sprintf(error_msg, "unknown type %d: probably a bug", type);
	return  ERROR;
    }

    return  OK;
}

static Status free_parse(String line, int n, int type, String error_msg)
{
    int i;
    String ptr, msg;

    for (i = n+1; ptr = strtok(NULL, PARSE_SEPARATOR); i++)
    {
	sprintf(error_msg, "argument #%d: ", i);
	msg = error_msg + strlen(error_msg);
	CHECK_STATUS(read_token(ptr, i, type, msg));
    }

    int_args[n] = i - n - 1;  /* this is how many tokens have been found */
    args[n] = (Generic_ptr) (int_args+n);

    return  OK;
}

Status parse_line(String line, Parse_line *table, Bool all_keys,
							String error_msg)
{
    int i, type, length;
    String ptr, msg;
    Parse_line *key;
    Bool have_rest = FALSE, have_args = TRUE;

    if (ptr = strchr(line, PARSE_COMMENT))
	*ptr = 0;

    if (ptr = strchr(line, NEW_LINE))
	*ptr = 0;

    length = strlen(line);

    if (!(ptr = strtok(line, PARSE_SEPARATOR)))  /* blank line */
	return  OK;

    if ((line+length) == (ptr+strlen(ptr)))
	have_args = FALSE;

    if (!(key = find_key_word(ptr, table)))
    {
	if (all_keys)
	{
	    sprintf(error_msg, "unknown key word '%s'", ptr);
	    return  ERROR;
	}
	else
	{
	    return  OK;
	}
    }

    sprintf(error_msg, "'%s': ", ptr);
    msg = error_msg + strlen(error_msg);

    for (i = 0; i < key->nargs; i++)
    {
	type = key->arg_types[i];

	if (type & PARSE_FREE)
	{
	    if (i != (key->nargs - 1))
	    {
		sprintf(msg, "not allowed any arguments after PARSE_FREE: bug");
		return  ERROR;
	    }

	    CHECK_STATUS(free_parse(ptr, i, type, msg));
	    break;
	}

	if (type & PARSE_FILE)
	{
	    if (i != (key->nargs - 1))
	    {
		sprintf(msg, "not allowed any arguments after PARSE_FILE: bug");
		return  ERROR;
	    }
	}
	else if (type & PARSE_REST)
	{
	    if (i != (key->nargs - 1))
	    {
		sprintf(msg, "not allowed any arguments after PARSE_REST: bug");
		return  ERROR;
	    }

	    have_rest = TRUE;

	    ptr += strlen(key->pattern);

	    if (have_args)
		ptr++;  /* skip past 0 put down by strtok */
	}
	else
	{
	    if (!(ptr = strtok(NULL, PARSE_SEPARATOR)))
	    {
		sprintf(msg, "missing argument #%d", i+1);
		return  ERROR;
	    }
	}

	sprintf(msg, "argument #%d: ", i+1);
	CHECK_STATUS(read_token(ptr, i, type, msg+strlen(msg)));
    }

    if (!have_rest && (ptr = strtok(NULL, PARSE_SEPARATOR)))
    {
	sprintf(msg, "extra argument #%d", i+1);
	return  ERROR;
    }

    return  ((*(key->func))(args, error_msg));
}

Status parse_subfile(FILE *fp, Parse_line *table, Bool all_keys,
					String end_msg, String error_msg)
{
    Long_line start, line;

    while (fgets(line, LONG_LINE_SIZE, fp))
    {
	parse_fp = fp;

	sscanf(line, "%s", start);
	if (!strcmp(start, end_msg))
	    return  OK;

	if (parse_line(line, table, all_keys, error_msg) == ERROR)
	    return  ERROR;
    }

    sprintf(error_msg, "missing '%s'", end_msg);

    return  ERROR;
}

Status parse_file(String file, Parse_line *table, Bool all_keys,
							String error_msg)
{
    Long_line line;
    FILE *fp;

    if (OPEN_FOR_READING(fp, file))
    {
	sprintf(error_msg, "opening file \"%s\"", file);
	return  ERROR;
    }

    while (fgets(line, LONG_LINE_SIZE, fp))
    {
	parse_fp = fp;

	if(parse_line(line, table, all_keys, error_msg) == ERROR)
	{
	    fclose(fp);
	    return  ERROR;
	}
    }

    fclose(fp);

    return  OK;
}

Status skip_parse_subfile(FILE *fp, String end_msg, String error_msg)
{
    Long_line start, line;

    while (fgets(line, LONG_LINE_SIZE, fp))
    {
	sscanf(line, "%s", start);
	if (!strcmp(start, end_msg))
	    return  OK;
    }

    sprintf(error_msg, "missing '%s'", end_msg);

    return  ERROR;
}
