#include "output.h"

#include "block_io.h"
#include "data.h"
#include "fitter.h"
#include "par.h"
#include "ref.h"
#include "utility.h"

#define  SIZE_OF_BLOCK		4096

static int ndim;
static int *npoints;
static int nfitted;
static int *dim_fitted;
static int *first;
static int *last;

static String output_ideal_file;
static String output_rest_file;
static String par_ideal_file;
static String par_rest_file;
static FILE *file_ideal_out;
static FILE *file_rest_out;

static int size_of_block = SIZE_OF_BLOCK;
static int total_blocks;
static int total_points;

static int npts[MAX_NDIM];
static int block_size[MAX_NDIM];
static int nblocks[MAX_NDIM];
static int cum_points[MAX_NDIM];
static int cum_blocks[MAX_NDIM];
static int cum_block_size[MAX_NDIM];
static int array[MAX_NDIM];
static int point[MAX_NDIM];
static int block[MAX_NDIM];
static int base_point[MAX_NDIM];

static float data[SIZE_OF_BLOCK];
static int *directory;

static float level;
static Bool have_deflation;

static Block_IO block_io;

static void init_arrays()
{
    int i, j, n;

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];
	npts[i] = last[j] - first[j];
    }

    find_block_sizes(nfitted, size_of_block, npts, block_size, FALSE);

    BLOCKS(nblocks, npts, block_size, nfitted);
    CUMULATIVE(cum_blocks, nblocks, total_blocks, nfitted);
    CUMULATIVE(cum_block_size, block_size, n, nfitted);
}

static void init_block()
{
    int i, j;

    ZERO_VECTOR(data, size_of_block);

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];
	base_point[i] = block[i] * block_size[i] + first[j];
    }

    for (i = 0; i < nfitted; i++)
    {
	if (block[i] == (nblocks[i]-1))
	    array[i] = 1 + (npts[i]-1) % block_size[i];
	else
	    array[i] = block_size[i];
    }

    CUMULATIVE(cum_points, array, total_points, nfitted);
}

static Status write_block(FILE *fp)
{
/*
    if (FWRITE(data, BYTES_PER_WORD, size_of_block, fp))
	return  ERROR;
*/
    CHECK_STATUS(endian_fwrite((char *) data, size_of_block, fp));

    return  OK;
}

static float calc_peak_contribs(int nfit, Fit_info **fit_info)
{
    int i;
    float v;

    v = 0;
    for (i = 0; i < nfit; i++)
	v += peak_contribution(nfitted, point, fit_info[i]->peak);

    return  v;
}

static int find_index(int ind)
{
    int i, j;

    if (total_points == size_of_block)
	return  ind;

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];
	array[i] = (point[i] - first[j]) % block_size[i];
    }

    INDEX_OF_ARRAY(ind, array, cum_block_size, nfitted);

    return  ind;
}

static Status block_process(int nfit, Fit_info **fit_info,
							String error_msg)
{
    int i, j, k, p;
    float v;

    p = total_blocks / 20;
    p = MAX(p, 1);

    for (i = 0; i < total_blocks; i++)
    {
	if (!(i % p))
	    printf("\t...outputting data (%1.0f%% done)\n",
						(100.0*i)/total_blocks);

	ARRAY_OF_INDEX(block, i, cum_blocks, nfitted);

	init_block();

	for (j = 0; j < total_points; j++)
	{
	    find_point(nfitted, j, point, cum_points, base_point, array, FALSE);
	    k = find_index(j);
	    data[k] = calc_peak_contribs(nfit, fit_info);
	}

	if (output_ideal_file)
	{
	    if (have_deflation)
	    {
		CHECK_STATUS(write_file_block(&block_io, i, data, error_msg));
	    }
	    else
	    {
		sprintf(error_msg, "file \"%s\": writing block %d: ",
							output_ideal_file, i);

		CHECK_STATUS(write_block(file_ideal_out));
	    }
	}

	if (output_rest_file)
	{
	    for (j = 0; j < total_points; j++)
	    {
		find_point(nfitted, j, point, cum_points, base_point,
								array, FALSE);
		CHECK_STATUS(data_value(&v, point, error_msg));
		k = find_index(j);
		data[k] = v - data[k];
	    }

	    sprintf(error_msg, "file \"%s\": writing block %d: ",
							output_rest_file, i);
	    CHECK_STATUS(write_block(file_rest_out));
	}
    }

    return  OK;
}

static void write_par_output(String par_file, String output_file,
							Ref_info *ref_info)
{
    int i, j;
    Line error_msg;
    Par_info par_info;
    Ref_info ref[MAX_NDIM];

    par_info.file = output_file;
    par_info.ndim = nfitted;
    par_info.npoints = npts;
    par_info.block_size = block_size;
    par_info.ref = ref;
    par_info.swapped = FALSE;
    par_info.integer = FALSE;
    par_info.blocked = TRUE;
    par_info.header = 0;
    par_info.param_dim = -1;

    if (have_deflation && (par_file == par_ideal_file))
    {
	par_info.deflated = TRUE;
	par_info.level = level;
    }
    else
    {
	par_info.deflated = FALSE;
	par_info.level = 0;
    }

    for (i = 0; i < nfitted; i++)
    {
	j = dim_fitted[i];
	find_sub_ref(ref+i, ref_info+j, first[j], last[j], 1, npoints[j]);
    }

    if (write_par_file(par_file, &par_info, error_msg) == ERROR)
	printf("Warning: %s\n", error_msg);
}

static Status init_deflation(String error_msg)
{
    int dir_size;

    dir_size = size_of_block * BLOCK(total_blocks, size_of_block);

    sprintf(error_msg, "allocating directory for deflation");
    MALLOC(directory, int, dir_size);

    block_io.name = output_ideal_file;
    block_io.file = file_ideal_out;
    block_io.deflated = TRUE;
    block_io.dir_size = dir_size;
    block_io.directory = directory;
    block_io.block_size = size_of_block;
    block_io.level = level;

    CHECK_STATUS(init_block_write(&block_io, error_msg));

    return  OK;
}

static Status end_deflation(String error_msg)
{
    CHECK_STATUS(end_block_write(&block_io, error_msg));

    FREE(directory, int);

    return  OK;
}

Status output_spectra(Size_info *size_info, Fit_param *fit_param,
			File_info *file_info, Ref_info *ref_info,
			int nfit, Fit_info **fit_info,
			String error_msg)
{
    ndim = size_info->ndim;
    npoints = size_info->npoints;

    level = fit_param->level;
    nfitted = fit_param->nfitted;
    dim_fitted = fit_param->dim_fitted;
    first = fit_param->first;
    last = fit_param->last;

    output_ideal_file = file_info->output_ideal_file;
    output_rest_file = file_info->output_rest_file;
    par_ideal_file = file_info->par_ideal_file;
    par_rest_file = file_info->par_rest_file;
    file_ideal_out = file_info->file_ideal_out;
    file_rest_out = file_info->file_rest_out;

    if (output_rest_file && (nfitted != ndim))
        RETURN_ERROR_MSG("to output rest, must fit peaks in all dimensions");

    init_arrays();

    if (output_ideal_file && (level > 0))
	have_deflation = TRUE;
    else
	have_deflation = FALSE;

    if (have_deflation)
	CHECK_STATUS(init_deflation(error_msg));

    if (block_process(nfit, fit_info, error_msg) == ERROR)
    {
	if (have_deflation)
	    FREE(directory, int);

	return  ERROR;
    }

    if (output_ideal_file)
	write_par_output(par_ideal_file, output_ideal_file, ref_info);

    if (output_rest_file)
	write_par_output(par_rest_file, output_rest_file, ref_info);

    if (have_deflation)
    {
	if (end_deflation(error_msg) == ERROR)
	{
	    FREE(directory, int);
	    return  ERROR;
	}
    }

    return  OK;
}
