#include "group.h"

#include "sorts.h"

static int ngroups;
static int group_max;
static int *upper;
static int *group;

static Status alloc_group_memory(int n, String error_msg)
{
    sprintf(error_msg, "allocating group memory");

    MALLOC(group, int, n);
    MALLOC(upper, int, n);  /* n is upper bound on what actually need */

    return  OK;
}

static Bool within_range(int ndim, int *width, int *posn1, int *posn2)
{
    int i, d;

    for (i = 0; i < ndim; i++)
    {
	d = posn1[i] - posn2[i];
	d = ABS(d);

	if (d > (2*width[i]))
	    return  FALSE;
    }

    return  TRUE;
}

static int cmp_group(Generic_ptr p1, Generic_ptr p2)
{
    Fit_info *info1 = (Fit_info *) p1;
    Fit_info *info2 = (Fit_info *) p2;
    int n1 = info1->n;
    int n2 = info2->n;

    if (group[n1] < group[n2])
	return  -1;
    else if (group[n1] > group[n2])
	return  1;
    else if (n1 < n2)
	return  -1;
    else /* (n1 > n2) */
	return  1;
}

static void group_peaks(int ndim, int *width, int nfit, Fit_info **fit_info)
{
    int i, j, n1, n2;

    for (i = 0; i < nfit; i++)
	group[i] = i;

    for (i = 0; i < nfit; i++)  /* O(N*N), so potentially very slow! */
    {
	for (j = i+1; j < nfit; j++)
	{
	    if (within_range(ndim, width, fit_info[i]->before.position,
					fit_info[j]->before.position))
	    {
		if (group[i] < group[j])
		    group[j] = group[i];
		else
		    group[i] = group[j];
	    }
	}
    }

    for (i = 0; i < nfit; i++)
	group[i] = group[group[i]];

    heap_sort((Generic_ptr *) fit_info, nfit, TRUE, cmp_group);

    ngroups = 0;
    group_max = 0;
    for (i = 0; i < nfit; i = j)
    {
	n1 = fit_info[i]->n;

	for (j = i+1; j < nfit; j++)
	{
	    n2 = fit_info[j]->n;

	    if (group[n1] != group[n2])
		break;
	}

	upper[ngroups++] = j;
	group_max = MAX(group_max, j-i);
    }
}

static void print_groups(Fit_info **fit_info)
{
    int i, j, l;

    printf("#groups = %d, group_max = %d\n", ngroups, group_max);

    for (i = 0; i < ngroups; i++)
    {
	l = (i > 0)  ?  upper[i-1]  :  0;
	printf("group %d peak%s:", i, (upper[i] == (l+1)) ? "" : "s");

	for (j = l; j < upper[i]; j++)
	    printf(" %d", fit_info[j]->n);

	printf("\n");
    }
}

Status determine_fit_groups(int ndim, int *width, Bool have_grouping, int nfit,
		Fit_info **fit_info, Group_info *group_info, String error_msg)
{
    int i;
/*
    group_info->ngroups = 1;
    group_info->group_max = nfit;
    group_info->upper = upper;
    group_info->upper[0] = nfit;
*/
/*
    group_info->ngroups = nfit;
    group_info->group_max = 1;
    group_info->upper = upper;

    for (i = 0; i < nfit; i++)
	group_info->upper[i] = i+1;
*/

    CHECK_STATUS(alloc_group_memory(nfit, error_msg));

    if (have_grouping)
    {
	group_peaks(ndim, width, nfit, fit_info);
    }
    else
    {
	ngroups = nfit;
	group_max = 1;

	for (i = 0; i < nfit; i++)
	    upper[i] = i+1;
    }

    FREE(group, int);

    group_info->ngroups = ngroups;
    group_info->group_max = group_max;
    group_info->upper = upper;

    print_groups(fit_info);

    return  OK;
}

static int cmp_place(Generic_ptr p1, Generic_ptr p2)
{
    Fit_info *info1 = (Fit_info *) p1;
    Fit_info *info2 = (Fit_info *) p2;
    int n1 = info1->n;
    int n2 = info2->n;

    if (n1 < n2)
	return  -1;
    else /* (n1 > n2) */
	return  1;
}

void ungroup_peaks(int nfit, Fit_info **fit_info)
{
    heap_sort((Generic_ptr *) fit_info, nfit, TRUE, cmp_place);
}
