#include "poly_roots.h"

#include "complex.h"

#define  MT  10
#define  MR  8
#define  MAX_ITER  (MT * MR)
#define  EPS  (4.0e-6)
#define  EPSS  (2.0e-7)

static float frac[] = { 0.5, 0.25, 0.75, 0.13, 0.38, 0.62, 0.88, 1.0 };

static Bool laguerre(Complex *a, int n, Complex *x, Bool polish)
{
    int i, j;
    float err, abx, abp, abm;
    Complex b, d, f, g, h, g2, s, gp, gm, dx, x1, t;

    for (i = 0; i < MAX_ITER; i++)
    {
	COMPLEX_SET(b, COMPLEX_ELEMENT(a, n));
	err = complex_abs(b);

	COMPLEX_ZERO(d);
	COMPLEX_ZERO(f);

	abx = complex_abs(*x);

	for (j = n-1; j >= 0; j--)
	{
	    COMPLEX_MULTIPLY(t, *x, f);
	    COMPLEX_ADD(f, t, d);

            COMPLEX_MULTIPLY(t, *x, d);
	    COMPLEX_ADD(d, t, b);

	    COMPLEX_MULTIPLY(t, *x, b);
	    COMPLEX_ADD(b, t, COMPLEX_ELEMENT(a, j));

	    err = complex_abs(b) + abx * err;
	}

	err *= EPSS;

	if (complex_abs(b) <= err)
	    return  TRUE;

	complex_divide(&g, d, b);
	COMPLEX_MULTIPLY(g2, g, g);
	COMPLEX_SCALE(f, 2);
	complex_divide(&t, f, b);
	COMPLEX_SUBTRACT(h, g2, t);
	COMPLEX_SCALE(h, n);
	COMPLEX_SUBTRACT(t, h, g2);
	COMPLEX_SCALE(t, n-1);
	complex_sqrt(t, &s);
	COMPLEX_ADD(gp, g, s);
	COMPLEX_SUBTRACT(gm, g, s);
	abp = complex_abs(gp);
	abm = complex_abs(gm);

	if (abp < abm)
	    COMPLEX_SET(gp, gm);

	if (MAX(abp, abm) > 0)
	{
            COMPLEX_NEW(t, n, 0);
	    complex_divide(&dx, t, gp);
	}
	else
	{
	    dx.r = (1 + abx) * cos((double) (i+1));
	    dx.i = (1 + abx) * sin((double) (i+1));
	}

	COMPLEX_SUBTRACT(x1, *x, dx);

	if (COMPLEX_EQUAL(x1, *x))
	    return  TRUE;

	if ((i+1) % MT)
	{
	    COMPLEX_SET(*x, x1);
	}
	else
	{
	    j = i / MT;
	    COMPLEX_SCALE(dx, frac[j]);
	    COMPLEX_SUBTRACT(*x, *x, dx);
	}
}

    /* not converged */
    return  FALSE;
}

Bool poly_roots(Complex *a, int n, Complex *roots, Bool polish, Complex *w)
{
    int i, j;
    Complex x, b, c, d;

    COMPLEX_COPY(w, a, n+1);

    for (i = n-1; i >= 0; i--)
    {
	COMPLEX_ZERO(x);

	if (!laguerre(w, i+1, &x, FALSE))
        {
            COMPLEX_NEW(x, 1, 0);
            if (!laguerre(w, i+1, &x, FALSE))
            {
                COMPLEX_NEW(x, 0, 1);
                if (!laguerre(w, i+1, &x, FALSE))
	            return  FALSE;
            }
        }

	if (ABS(COMPLEX_IMAG(x)) <= (EPS*ABS(COMPLEX_REAL(x))))
	    COMPLEX_IMAG(x) = 0;

	COMPLEX_SET(COMPLEX_ELEMENT(roots, i), x);

	COMPLEX_SET(b, COMPLEX_ELEMENT(w, i+1));

	for (j = i; j >= 0; j--)
	{
	    COMPLEX_SET(c, COMPLEX_ELEMENT(w, j));
	    COMPLEX_SET(COMPLEX_ELEMENT(w, j), b);
	    COMPLEX_MULTIPLY(d, x, b);
	    COMPLEX_ADD(b, d, c);
	}
    }

    if (polish)
    {
	for (i = 0; i < n; i++)
	{
	    if (!laguerre(a, n, &COMPLEX_ELEMENT(roots, i), TRUE))
		return  FALSE;
	}
    }

    for (i = 1; i < n; i++)
    {
	COMPLEX_SET(x, COMPLEX_ELEMENT(roots, i));

	for (j = i-1; j >= 0; j--)
	{
	    if (COMPLEX_REAL(COMPLEX_ELEMENT(roots, j)) <= COMPLEX_REAL(x))
		break;

	    COMPLEX_SET(COMPLEX_ELEMENT(roots, j+1), COMPLEX_ELEMENT(roots, j));
	}

	COMPLEX_SET(COMPLEX_ELEMENT(roots, j+1), x);
    }

    return  TRUE;
}
