#include "parser.h"

#define  MAX_NARGS		20

#define  PARSER_SEPARATOR	" \t"

#define  PARSER_COMMENT		'!'

#define  PARSER_ASSIGNMENT	'='

#define  BEGIN_STRING		'"'
#define  END_STRING		'"'

#define  NVARIABLES_ALLOC	20
#define  NSTORE_ALLOC		100

#define  DEFAULT_ARG		"x"

typedef struct
{
    String name;
    Parser_store *store;
}   Parser_variable;

static char *tokens[MAX_NARGS];
static Parser_store *stores[MAX_NARGS];

static Bool ok_string;

static int nvariables_alloc = 0;
static int nvariables;
static Parser_variable *variables;

static int nstore_alloc = 0;
static int nstore_list;
static Parser_store **store_list;
static Parser_store dummy_store;

static Line default_arg;

Status no_parser_func(int nstore, Parser_store **store, String error_msg)
{
    return  OK;
}

static void free_parser_memory()
{
    int i;

    if (nvariables_alloc > 0)
    {
	for (i = 0; i < nvariables; i++)
	    FREE(variables[i].name, char);

	FREE(variables, Parser_variable);

	nvariables_alloc = 0;
    }

    if (nstore_alloc > 0)
    {
	for (i = 0; i < nstore_list; i++)
	    FREE(store_list[i], Parser_store);

	FREE(store_list, Parser_store *);

	nstore_alloc = 0;
    }
}

static Status alloc_parser_memory()
{
    int n;

    free_parser_memory();

    n = NVARIABLES_ALLOC;
    MALLOC(variables, Parser_variable, n);
    nvariables_alloc = n;

    n = NSTORE_ALLOC;
    MALLOC(store_list, Parser_store *, n);
    nstore_alloc = n;

    return  OK;
}

static Status realloc_variable_memory()
{
    int n;

    n = nvariables_alloc + NVARIABLES_ALLOC;
    REALLOC(variables, Parser_variable, n);
    nvariables_alloc = n;

    return  OK;
}

static Status realloc_store_memory()
{
    int n;

    n = nstore_alloc + NSTORE_ALLOC;
    REALLOC(store_list, Parser_store *, n);
    nstore_alloc = n;

    return  OK;
}

Status init_parser(String error_msg)
{
    sprintf(error_msg, "allocating memory for parser");
    CHECK_STATUS(alloc_parser_memory());

    nvariables = nstore_list = 0;

    dummy_store.ndata = dummy_store.nalloc = 0;
    dummy_store.data_type = PARSER_NONE;
    dummy_store.data = NULL;

    sprintf(default_arg, "%s", DEFAULT_ARG);

    return  OK;
}

static Status new_store(Parser_store **p_store, String error_msg)
{
    Parser_store *store;

    sprintf(error_msg, "allocating memory for store list");
    if (nstore_list == nstore_alloc)
	CHECK_STATUS(realloc_store_memory());

    sprintf(error_msg, "allocating memory for store");
    MALLOC(store, Parser_store, 1);

    store->nalloc = store->ndata = 0;
    store->data_type = PARSER_NONE;
    store->data = (Generic_ptr) NULL;

    *p_store = store;

    store_list[nstore_list++] = store;

    return  OK;
}

static Status new_variable(String token, Parser_store **p_store,
							String error_msg)
{
    int n;
    String name;
    Parser_store *store;

    sprintf(error_msg, "allocating memory for variable");
    if (nvariables == nvariables_alloc)
	CHECK_STATUS(realloc_variable_memory());

    n = strlen(token) + 1;
    MALLOC(name, char, n);
    strcpy(name, token);

    CHECK_STATUS(new_store(&store, error_msg));

    variables[nvariables].name = name;
    variables[nvariables].store = store;

    nvariables++;

    *p_store = store;

    return  OK;
}

static Parser_variable *known_variable(String token)
{
    int i;

    for (i = 0; i < nvariables; i++)
	if (!strcmp(token, variables[i].name))
	    return  (variables + i);

    return  (Parser_variable *) NULL;
}

static Parser_store *lookup_variable(String token, String error_msg)
{
    Parser_variable *var;

    if (!(var = known_variable(token)))
    {
	sprintf(error_msg, "unknown variable '%s'", token);
	return  (Parser_store *) NULL;
    }

    return  var->store;
}

static Status read_int(String token, int n, String error_msg)
{
    int x, *p;
    Parser_store *store;

    if(stores[n] = lookup_variable(token, error_msg))
	return  OK;

    if (sscanf(token, "%d", &x) != 1)
	return  ERROR;
 
    CHECK_STATUS(new_store(&store, error_msg));
    sprintf(error_msg, "allocating memory for integer");
    MALLOC(p, int, 1);

    *p = x;
    store->nalloc = store->ndata = 1;
    store->data_type = PARSER_INT | PARSER_REAL | PARSER_SCALAR;
    store->data = (Generic_ptr) p;

    stores[n] = store;

    return  OK;
}

static Status read_float(String token, int n, String error_msg)
{
    float x, *p;
    Parser_store *store;

    if(stores[n] = lookup_variable(token, error_msg))
	return  OK;

    if (sscanf(token, "%f", &x) != 1)
	return  ERROR;
 
    CHECK_STATUS(new_store(&store, error_msg));
    sprintf(error_msg, "allocating memory for floating point");
    MALLOC(p, float, 1);

    *p = x;
    store->nalloc = store->ndata = 1;
    store->data_type = PARSER_FLOAT | PARSER_REAL | PARSER_SCALAR;
    store->data = (Generic_ptr) p;

    stores[n] = store;

    return  OK;
}

static Status read_string(String token, int n, String error_msg)
{
    int m;
    char *p;
    Parser_store *store;

    if (store = lookup_variable(token, error_msg))
	return  OK;

    if (*token == BEGIN_STRING)
    {
	token[strlen(token)-1] = 0;  /* END_STRING */
	token++;
    }

    m = strlen(token) + 1;

    CHECK_STATUS(new_store(&store, error_msg));
    sprintf(error_msg, "allocating memory for string");
    MALLOC(p, char, m);

    strcpy(p, token);
    store->nalloc = store->ndata = m;
    store->data_type = PARSER_STRING | PARSER_SCALAR;
    store->data = (Generic_ptr) p;

    stores[n] = store;

    return  OK;
}

#define  INT_MESSAGE		"integer"
#define  FLOAT_MESSAGE		"floating point"
#define  STRING_MESSAGE		"string"

#define  REAL_MESSAGE		"real"
#define  COMPLEX_MESSAGE	"complex"

#define  SCALAR_MESSAGE		"scalar"
#define  ARRAY_MESSAGE		"array"

#define  CHECK_HAVE_TYPE(t, string) \
	 {   if (type & t) \
	     {   if (count++ > 0)  strcat(msg, " or "); \
		 strcat(msg, string);   }   }

#define  CHECK_TYPE(mask, routine) \
	 {   f = found_type & mask;  n = need_type & mask;  \
	     if (f && !(f & n)) \
	     {   routine(f, msg1);  routine(n, msg2); \
		 sprintf(error_msg, "found %s type, need %s type", \
			     msg1, msg2);  return  ERROR;   }   }

#define  CHECK_NO_TYPE \
	 {   if (count == 0)  sprintf(msg, "no");   }

static void ifs_message(int type, String msg)
{
    int count = 0;

    *msg = 0;
    CHECK_HAVE_TYPE(PARSER_INT, INT_MESSAGE);
    CHECK_HAVE_TYPE(PARSER_FLOAT, FLOAT_MESSAGE);
    CHECK_HAVE_TYPE(PARSER_STRING, STRING_MESSAGE);
    CHECK_NO_TYPE;
}

static void rc_message(int type, String msg)
{
    int count = 0;

    *msg = 0;
    CHECK_HAVE_TYPE(PARSER_REAL, REAL_MESSAGE);
    CHECK_HAVE_TYPE(PARSER_COMPLEX, COMPLEX_MESSAGE);
    CHECK_NO_TYPE;
}

static void sa_message(int type, String msg)
{
    int count = 0;

    *msg = 0;
    CHECK_HAVE_TYPE(PARSER_SCALAR, SCALAR_MESSAGE);
    CHECK_HAVE_TYPE(PARSER_ARRAY, ARRAY_MESSAGE);
    CHECK_NO_TYPE;
}

static Status check_type(int found_type, int need_type, String error_msg)
{
    int n, f;
    Line msg1, msg2;

    CHECK_TYPE(PARSER_IFS_MASK, ifs_message);
    CHECK_TYPE(PARSER_RC_MASK, rc_message);
    CHECK_TYPE(PARSER_SA_MASK, sa_message);

    return  OK;
}

static Status read_token(String token, int n, int type, String error_msg)
{
    if(!(stores[n] = lookup_variable(token, error_msg)))
    {
	if(type & PARSER_FLOAT)
	{
	    CHECK_STATUS(read_float(token, n, error_msg));
	}
	else if(type & PARSER_INT)
	{
	    CHECK_STATUS(read_int(token, n, error_msg));
	}
	else if(type & PARSER_STRING)
	{
	    CHECK_STATUS(read_string(token, n, error_msg));
	}
	else /* should never get here! */
	{
	    sprintf(error_msg, "unknown type %d: probably a bug", type);
	    return  ERROR;
	}
    }

    CHECK_STATUS(check_type(stores[n]->data_type, type, error_msg));

    return  OK;
}

static Status free_parser(int nargs, Parser_line *key, String error_msg)
{
    int i, n, type;
    String msg;

    n = key->nargs - 1;
    for (i = 1; i < n; i++)
    {
	type = key->arg_types[i];
	sprintf(error_msg, "argument #%d: ", i);
	msg = error_msg + strlen(error_msg);
	CHECK_STATUS(read_token(tokens[i], i, type, msg));
    }

    type = key->arg_types[n];
    for (i = n ; i < nargs; i++)
    {
	sprintf(error_msg, "argument #%d: ", i);
	msg = error_msg + strlen(error_msg);
	CHECK_STATUS(read_token(tokens[i], i, type, msg));
    }

    return  OK;
}

static Status fixed_parser(Parser_line *key, String error_msg)
{
    int i, type;
    String msg;

    for (i = 1; i < key->nargs; i++)
    {
	type = key->arg_types[i];
	sprintf(error_msg, "argument #%d: ", i);
	msg = error_msg + strlen(error_msg);
	CHECK_STATUS(read_token(tokens[i], i, type, msg));
    }

    return  OK;
}

static String next_non_separator(String ptr)
{
    String s;

    for ( ; *ptr; ptr++) 
    {
	for (s = PARSER_SEPARATOR; *s; s++)
	    if (*s == *ptr)
		break;

	if (!(*s))
	    break;
    }

    return  ptr;
}

static String next_separator(String ptr)
{
    String s;

    for ( ; *ptr; ptr++) 
    {
	for (s = PARSER_SEPARATOR; *s; s++)
	    if (*s == *ptr)
		return  ptr;
    }

    return  ptr;
}

/*  next_token is own version of strtok  */
/*  need it because of complication of constant strings with space in them  */

static String next_token(String line)
{
    String token, ptr, limit;
    static String next;

    if (line)
	next = line;

    ok_string = TRUE;
    token = next_non_separator(next);

    if (!(*token))
	return  (String) NULL;

    if (*token == BEGIN_STRING)
    {
	limit = token + strlen(token);

	for (ptr = token+1; ptr < limit; ptr++)
	{
	    if (*ptr == END_STRING)
		break;
	}

	if (ptr == limit)
	{
	    ok_string = FALSE;
	}
	else
	{
	    next = ptr + 1;
	    *next++ = 0;
	}
    }
    else
    {
	next = next_separator(token);
	*next++ = 0;
    }

    return  token;
}

static Status assigned_variable(String token, String error_msg)
{
    Parser_variable *var;

    if (var = known_variable(token))
    {
	CHECK_STATUS(new_store(&stores[OUTPUT_X], error_msg));
	var->store = stores[OUTPUT_X];
    }
    else
    {
	CHECK_STATUS(new_variable(token, &stores[OUTPUT_X], error_msg));
    }

    return  OK;
}

static Status parser_key(Parser_line *key, String line, String error_msg)
{
    int i, n, nargs;
    String ptr, msg;

    sprintf(error_msg, "'%s': ", key->pattern);
    msg = error_msg + strlen(error_msg);

    ptr = line;
    for (nargs = 1; (tokens[nargs] = next_token(ptr)) && ok_string; nargs++)
	ptr = (String) NULL;

    if (!ok_string)
    {
	sprintf(msg, "unfinished string '%s'", tokens[nargs]);
	return  ERROR;
    }

    n = key->nargs - 1;
    if (key->arg_types[n] & PARSER_FREE)
    {
	CHECK_STATUS(free_parser(nargs, key, msg));
    }
    else
    {
	if (n == nargs)
	{
	    for (i = nargs; i > 0; i--)
		tokens[i+1] = tokens[i];

	    tokens[INPUT_X] = default_arg;
	    nargs++;
	}
	else if (n != (nargs-1))
	{
	    if (n > 0)
		sprintf(msg, "have %d arguments, should have %d or %d",
							nargs-1, n-1, n);
	    else
		sprintf(msg, "have %d arguments, should have 0", nargs-1);

	    return  ERROR;
	}

	CHECK_STATUS(fixed_parser(key, msg));
    }

    if (key->arg_types[OUTPUT_X] == PARSER_NONE)
	stores[OUTPUT_X] = &dummy_store;  /* protection */
    else
	CHECK_STATUS(assigned_variable(tokens[OUTPUT_X], error_msg));

    return  ((*(key->func))(nargs, stores, error_msg));
}

/*  find_key_word is terribly inefficient, but works for now  */

static Parser_line *find_key_word(String key, Parser_line *table)
{
    Parser_line *p;

    for (p = table; p->pattern; p++)
	if (!strcmp(p->pattern, key))
	    return  p;

    return  (Parser_line *) NULL;
}

Status parser_line(String line, Parser_line *table, Bool all_keys,
							String error_msg)
{
    String ptr, msg;
    Parser_line *key;
    Bool have_output;

    if (ptr = strchr(line, PARSER_COMMENT))
	*ptr = 0;

    if (ptr = strchr(line, NEW_LINE))
	*ptr = 0;

    msg = error_msg;
    sprintf(error_msg, "'%s': ", line);
    error_msg += strlen(error_msg);

    if (ptr = strchr(line, PARSER_ASSIGNMENT))
    {
	*ptr++ = 0;

	if (!(tokens[OUTPUT_X] = strtok(line, PARSER_SEPARATOR)))
	    RETURN_ERROR_MSG("missing variable to left of assignment");

	if (strtok(NULL, PARSER_SEPARATOR))
	    RETURN_ERROR_MSG("more than one variable to left of assignment");

	if (!(ptr = strtok(ptr, PARSER_SEPARATOR)))
	    RETURN_ERROR_MSG("nothing to right of assignment");

	have_output = TRUE;
    }
    else
    {
	if (!(ptr = strtok(line, PARSER_SEPARATOR)))
	    return  OK;

	tokens[OUTPUT_X] = default_arg;

	have_output = FALSE;
    }

    if (!(key = find_key_word(ptr, table)))
    {
	if (all_keys)
	{
	    sprintf(error_msg, "unknown key word '%s'", ptr);
	    return  ERROR;
	}
	else
	{
	    return  OK;
	}
    }

    if (have_output && (key->arg_types[OUTPUT_X] == PARSER_NONE))
    {
	sprintf(error_msg, "key word '%s' cannot have assignment", ptr);
	return  ERROR;
    }

    return  parser_key(key, ptr+strlen(ptr)+1, msg);
}

Status parser_file(String file, Parser_line *table, Bool all_keys,
							String error_msg)
{
    int n;
    char *msg;
    Line line;
    FILE *fp;

    if (OPEN_FOR_READING(fp, file))
    {
	sprintf(error_msg, "opening file \"%s\"", file);
	return  ERROR;
    }

    sprintf(error_msg, "%s: ", file);
    error_msg += strlen(error_msg);

    for (n = 0; fgets(line, LINE_SIZE, fp); n++)
    {
	sprintf(error_msg, "line number %d: ", n+1);
	msg = error_msg + strlen(error_msg);

	if (parser_line(line, table, all_keys, msg) == ERROR)
	{
	    fclose(fp);
	    return  ERROR;
	}
    }

    fclose(fp);

    return  OK;
}

Status check_parser_alloc(Parser_store *store, int n, String error_msg)
{
    int m, nalloc, type, *x;
    float *y;
    char *z;

    nalloc = store->nalloc;

    if (n <= nalloc)
    {
	store->ndata = n;
	return  OK;
    }

    sprintf(error_msg, "allocating memory for data");

    type = store->data_type;

    if (type & PARSER_COMPLEX)
	m = 2 * n;
    else
	m = n;

    store->nalloc = 0;

    if (type & PARSER_INT)
    {
	x = (int *) (store->data);

	if (nalloc > 0)
	    FREE(x, int);

	store->data = (Generic_ptr) NULL;

	MALLOC(x, int, m);

	store->data = (Generic_ptr) x;
    }
    else if (type & PARSER_FLOAT)
    {
	y = (float *) (store->data);

	if (nalloc > 0)
	    FREE(y, float);

	store->data = (Generic_ptr) NULL;

	MALLOC(y, float, m);

	store->data = (Generic_ptr) y;
    }
    else if (type & PARSER_STRING)
    {
	z = (char *) (store->data);

	if (nalloc > 0)
	    FREE(z, char);

	store->data = (Generic_ptr) NULL;

	MALLOC(z, char, m);

	store->data = (Generic_ptr) z;
    }
    else
    {
	sprintf(error_msg, "unknown type %d: probably a bug", type);
	return  ERROR;
    }

    store->ndata = store->nalloc = n;

    return  OK;
}

void store_type_float(Parser_store *store)
{
    store->data_type &= ~PARSER_INT;
    store->data_type |= PARSER_FLOAT;
}

void store_type_int(Parser_store *store)
{
    store->data_type &= ~PARSER_FLOAT;
    store->data_type |= PARSER_INT;
}

void store_int_to_float(Parser_store *store)
{
    if (store->data_type & PARSER_FLOAT)
	return;

    float_words((float *) (store->data), store->ndata);

    store_type_float(store);
}

void store_float_to_int(Parser_store *store)
{
    if (store->data_type & PARSER_INT)
	return;

    int_words((int *) (store->data), store->ndata);

    store_type_int(store);
}
