#include "crosspeak.h"

#include "atom.h"
#include "table.h"

#define  PARSE_SEPARATOR	"\t"

#define  CROSSPEAK_ALLOC	1000

static int ncrosspeaks = 0;
static int ncrosspeaks_alloc = 0;
static Crosspeak **crosspeaks;

static Status alloc_crosspeak_memory(int ndim, String error_msg)
{
    int i, n;

    if (ncrosspeaks >= ncrosspeaks_alloc)
    {
        n = ncrosspeaks_alloc + CROSSPEAK_ALLOC;

        if (ncrosspeaks_alloc == 0)
        {
            sprintf(error_msg, "allocating memory");
            MALLOC(crosspeaks, Crosspeak *, n);
        }
        else
        {
            sprintf(error_msg, "reallocating memory");
            REALLOC(crosspeaks, Crosspeak *, n);
        }

        sprintf(error_msg, "allocating memory");

        for (i = ncrosspeaks_alloc; i < n; i++)
	{
            MALLOC(crosspeaks[i], Crosspeak, 1);

	    MALLOC(crosspeaks[i]->atom, Atom, ndim);
/*  this was done when thought all crosspeak files had dim 4
	    MALLOC(crosspeaks[i]->atom, Atom, CROSSPEAK_DIM);
*/
	}

        ncrosspeaks_alloc = n;
    }

    return  OK;
}

static void find_distances(Crosspeak_data *data, int nintensity_dists,
			Intensity_dist *intensity_dists, float intensity)
{
    int i;

    if (nintensity_dists == 0)
    {
	data->distance = data->distance_minus = ABS(intensity);
	data->distance_plus = 0;
    }

    for (i = 0; i < nintensity_dists-1; i++)
    {
	if (ABS(intensity) >= intensity_dists[i].intensity)
	    break;
    }

    data->distance = intensity_dists[i].distance;
    data->distance_minus = intensity_dists[i].distance_minus;
    data->distance_plus = intensity_dists[i].distance_plus;
}

static Status parse_crosspeak_dim(Atom *atom, String line,
							String error_msg)
{
    String atom_type;
    int residue;
    Line chain;
    String residue_chain;

    CHECK_STATUS(get_table_string(&line, &(atom->res_name), error_msg));
/*  added chain check 13 Sep 00
    CHECK_STATUS(get_table_integer(&line, &(atom->residue), error_msg));
*/
    CHECK_STATUS(get_table_string(&line, &residue_chain, error_msg));
    CHECK_STATUS(get_table_string(&line, &(atom->atom_name), error_msg));
    CHECK_STATUS(get_table_string(&line, &atom_type, error_msg));
    CHECK_STATUS(get_table_float(&line, &(atom->shift), error_msg));
 
    atom->atom_type = atom_type[0];

    FREE(atom_type, char);

    if (sscanf(residue_chain, "%d%s", &residue, chain) != 2)
    {
        if (sscanf(residue_chain, "%d", &residue) != 1)
	    residue = 0;

        *chain = 0;
    }

    atom->residue = residue;
    STRING_MALLOC_COPY(atom->chain, chain);
    FREE(residue_chain, char);

    return  OK;
}

static Status parse_crosspeak(int ndim, Crosspeak *crosspeak,
			Connect_info *info, String line, String error_msg)
{
    int i;
    Crosspeak_data *data;

/*  this was done when thought all crosspeak files had dim 4
    for (i = 0; i < CROSSPEAK_DIM; i++)
*/
    for (i = 0; i < ndim; i++)
    {
	CHECK_STATUS(parse_crosspeak_dim(crosspeak->atom+i, line, error_msg));
	line = (String) NULL;
    }

    data = &(crosspeak->data);

    CHECK_STATUS(get_table_float(&line, &(data->intensity_unscaled),error_msg));
    CHECK_STATUS(get_table_string(&line, &(data->spectrum), error_msg));
    CHECK_STATUS(get_table_integer(&line, &(data->number), error_msg));
    CHECK_STATUS(get_table_float(&line, &(data->intensity), error_msg));

    find_distances(data, info->nintensity_dists,
				info->intensity_dists, data->intensity);

    return  OK;
}

#define  DIM_COLUMNS		5 /* number of dim-dependent columns per dim */
#define  NON_DIM_COLUMNS	4 /* number of non-dim-dependent columns */
#define  EXTRA_COLUMNS		2 /* extra columns for xpk output */

static Status get_number_dimensions(int *ndim, String line, String error_msg)
{
    int n;
    Line entry;

    for (n = 0; get_table_entry(&line, entry, error_msg) == OK; n++)
	;

    if (((n % DIM_COLUMNS) != NON_DIM_COLUMNS) &&
	((n % DIM_COLUMNS) != ((NON_DIM_COLUMNS+EXTRA_COLUMNS) % DIM_COLUMNS)))
    {
	sprintf(error_msg, "number of columns must be = %d or %d modulo %d",
		NON_DIM_COLUMNS,
		(NON_DIM_COLUMNS+EXTRA_COLUMNS) % DIM_COLUMNS, DIM_COLUMNS);
	return  ERROR;
    }

/*  below assumes that EXTRA_COLUMNS < DIM_COLUMNS  */
    *ndim = (n - NON_DIM_COLUMNS) / DIM_COLUMNS;

    return  OK;
}

Status read_crosspeak_file(FILE *fp, Connect_info *info, int *p_ncrosspeaks,
				Crosspeak ***p_crosspeaks, String error_msg)
{
    int ndim;
    Long_line line;
    String msg;

    printf("reading crosspeak file\n");

    sprintf(error_msg, "crosspeak file: ");
    error_msg += strlen(error_msg);

    fgets(line, LONG_LINE_SIZE, fp);  /* header1 */
    CHECK_STATUS(get_number_dimensions(&ndim, line, error_msg));

    fgets(line, LONG_LINE_SIZE, fp);  /* header2 */

    for (ncrosspeaks = 0; fgets(line, LONG_LINE_SIZE, fp)
				&& !empty_string(line) ; ncrosspeaks++)
    {
	if (!(ncrosspeaks % 100))
	    printf("\t...record %d\n", ncrosspeaks);

	sprintf(error_msg, "record %d: ", ncrosspeaks);
	msg = error_msg + strlen(error_msg);

	CHECK_STATUS(alloc_crosspeak_memory(ndim, msg));

	CHECK_STATUS(parse_crosspeak(ndim, crosspeaks[ncrosspeaks], info, line, msg));
    }

    *p_ncrosspeaks = ncrosspeaks;
    *p_crosspeaks = crosspeaks;

    info->crosspeak_ndim = ndim;

    printf("\tfound %d crosspeaks\n", ncrosspeaks);

    return  OK;
}
