/*
**  Copyright (c) 2010-2012, 2017, The Trusted Domain Project.
**    All rights reserved.
*/

/* for Solaris */
#ifndef _REENTRANT
#define _REENTRANT
#endif /* ! REENTRANT */

/* system includes */
#include <arpa/nameser.h>
#include <assert.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <resolv.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>

/* libopenarc includes */
#include "arc-dns.h"
#include "arc-internal.h"
#include "arc.h"

/* OpenARC includes */
#include "build-config.h"

/* macros, limits, etc. */
#ifndef MAXPACKET
#define MAXPACKET 8192
#endif /* ! MAXPACKET */

/*
**  Standard UNIX resolver stub functions
*/

struct arc_res_qh
{
    int    rq_error;
    int    rq_dnssec;
    size_t rq_buflen;
};

/*
**  ARC_RES_INIT -- initialize the resolver
**
**  Parameters:
**  	srv -- service handle (returned)
**
**  Return value
**  	0 on success, !0 on failure
*/

int
arc_res_init(void **srv)
{
#ifdef HAVE_RES_NINIT
    struct __res_state *res;

    res = ARC_CALLOC(1, sizeof(struct __res_state));
    if (res == NULL)
    {
        return -1;
    }

    if (res_ninit(res) != 0)
    {
        ARC_FREE(res);
        return -1;
    }

    *srv = res;

    return 0;
#else  /* HAVE_RES_NINIT */
    if (res_init() == 0)
    {
        *srv = (void *) 0x01;
        return 0;
    }
    else
    {
        return -1;
    }
#endif /* HAVE_RES_NINIT */
}

/*
**  ARC_RES_CLOSE -- shut down the resolver
**
**  Parameters:
**  	srv -- service handle
**
**  Return value:
**  	None.
*/

void
arc_res_close(void *srv)
{
#ifdef HAVE_RES_NINIT
    struct __res_state *res;

    res = srv;

    if (res != NULL)
    {
        res_nclose(res);
        ARC_FREE(res);
    }
#endif /* HAVE_RES_NINIT */
}

/*
**  ARC_RES_CANCEL -- cancel a pending resolver query
**
**  Parameters:
**  	srv -- query service handle (ignored)
**  	qh -- query handle (ignored)
**
**  Return value:
**  	0 on success, !0 on error
**
**  Notes:
**  	The standard UNIX resolver is synchronous, so in theory this can
**  	never get called.  We have not yet got any use cases for one thread
**  	canceling another thread's pending queries, so for now just return 0.
*/

int
arc_res_cancel(void *srv, void *qh)
{
    if (qh != NULL)
    {
        ARC_FREE(qh);
    }

    return 0;
}

/*
**  ARC_RES_QUERY -- initiate a DNS query
**
**  Parameters:
**  	srv -- service handle (ignored)
**  	type -- RR type to query
**  	query -- the question to ask
**  	buf -- where to write the answer
**  	buflen -- bytes at "buf"
** 	qh -- query handle, used with arc_res_waitreply
**
**  Return value:
**  	0 on success, -1 on error
**
**  Notes:
**  	This is a stub for the stock UNIX resolver (res_) functions, which
**  	are synchronous so no handle needs to be created, so "qh" is set to
**  	"buf".  "buf" is actually populated before this returns (unless
**  	there's an error).
*/

int
arc_res_query(void          *srv,
              int            type,
              const char    *query,
              unsigned char *buf,
              size_t         buflen,
              void         **qh)
{
    int                n;
    int                ret;
    struct arc_res_qh *rq;
    unsigned char      qbuf[HFIXEDSZ + MAXPACKET];
#ifdef HAVE_RES_NINIT
    struct __res_state *statp;
#endif /* HAVE_RES_NINIT */

#ifdef HAVE_RES_NINIT
    statp = srv;
    n = res_nmkquery(statp, QUERY, query, C_IN, type, NULL, 0, NULL, qbuf,
                     sizeof qbuf);
#else  /* HAVE_RES_NINIT */
    n = res_mkquery(QUERY, query, C_IN, type, NULL, 0, NULL, qbuf, sizeof qbuf);
#endif /* HAVE_RES_NINIT */
    if (n == (size_t) -1)
    {
        return ARC_DNS_ERROR;
    }

#ifdef HAVE_RES_NINIT
    ret = res_nsend(statp, qbuf, n, buf, buflen);
#else  /* HAVE_RES_NINIT */
    ret = res_send(qbuf, n, buf, buflen);
#endif /* HAVE_RES_NINIT */
    if (ret == -1)
    {
        return ARC_DNS_ERROR;
    }

    rq = ARC_MALLOC(sizeof *rq);
    if (rq == NULL)
    {
        return ARC_DNS_ERROR;
    }

    rq->rq_dnssec = ARC_DNSSEC_UNKNOWN;
    if (ret == -1)
    {
        rq->rq_error = errno;
        rq->rq_buflen = 0;
    }
    else
    {
        rq->rq_error = 0;
        rq->rq_buflen = (size_t) ret;
    }

    *qh = (void *) rq;

    return ARC_DNS_SUCCESS;
}

/*
**  ARC_RES_WAITREPLY -- wait for a reply to a pending query
**
**  Parameters:
**  	srv -- service handle
**  	qh -- query handle
**  	to -- timeout
**  	bytes -- number of bytes in the reply (returned)
**  	error -- error code (returned)
**
**  Return value:
**  	A ARC_DNS_* code.
**
**  Notes:
**  	Since the stock UNIX resolver is synchronous, the reply was completed
** 	before arc_res_query() returned, and thus this is almost a no-op.
*/

int
arc_res_waitreply(void           *srv,
                  void           *qh,
                  struct timeval *to,
                  size_t         *bytes,
                  int            *error,
                  int            *dnssec)
{
    struct arc_res_qh *rq;

    assert(qh != NULL);

    rq = qh;

    if (bytes != NULL)
    {
        *bytes = rq->rq_buflen;
    }
    if (error != NULL)
    {
        *error = rq->rq_error;
    }
    if (dnssec != NULL)
    {
        *dnssec = rq->rq_dnssec;
    }

    return ARC_DNS_SUCCESS;
}
