/*

   Copyright 2001,2003  The Apache Software Foundation 

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

 */
package org.apache.batik.svggen;

import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.Paint;
import java.awt.TexturePaint;
import java.util.LinkedList;
import java.util.List;

import org.apache.batik.ext.awt.g2d.GraphicContext;

/**
 * Utility class that converts a Paint object into an
 * SVG element.
 *
 * @author <a href="mailto:cjolif@ilog.fr">Christophe Jolif</a>
 * @author <a href="mailto:vincent.hardy@eng.sun.com">Vincent Hardy</a>
 * @version $Id: SVGPaint.java,v 1.11 2004/08/18 07:15:08 vhardy Exp $
 * @see              org.apache.batik.svggen.SVGLinearGradient
 * @see              org.apache.batik.svggen.SVGTexturePaint
 */
public class SVGPaint implements SVGConverter {
    /**
     * All GradientPaint convertions are handed to svgLinearGradient
     */
    private SVGLinearGradient svgLinearGradient;

    /**
     * All TexturePaint convertions are handed to svgTextureGradient
     */
    private SVGTexturePaint svgTexturePaint;

    /**
     * All Color convertions are handed to svgColor
     */
    private SVGColor svgColor;

    /**
     * All custom Paint convetions are handed to svgCustomPaint
     */
    private SVGCustomPaint svgCustomPaint;

    /**
     * Used to generate DOM elements
     */
    private SVGGeneratorContext generatorContext;

    /**
     * @param generatorContext the context.
     */
    public SVGPaint(final SVGGeneratorContext generatorContext) {
        this.svgLinearGradient = new SVGLinearGradient(generatorContext);
        this.svgTexturePaint = new SVGTexturePaint(generatorContext);
        this.svgCustomPaint = new SVGCustomPaint(generatorContext);
        this.svgColor = new SVGColor(generatorContext);
        this.generatorContext = generatorContext;
    }

    /**
     * @return Set of Elements defining the Paints this
     *         converter has processed since it was created
     */
    public List getDefinitionSet(){
        final List paintDefs = new LinkedList(this.svgLinearGradient.getDefinitionSet());
        paintDefs.addAll(this.svgTexturePaint.getDefinitionSet());
        paintDefs.addAll(this.svgCustomPaint.getDefinitionSet());
        paintDefs.addAll(this.svgColor.getDefinitionSet());
        return paintDefs;
    }

    public SVGTexturePaint getTexturePaintConverter(){
        return this.svgTexturePaint;
    }

    public SVGLinearGradient getGradientPaintConverter(){
        return this.svgLinearGradient;
    }

    public SVGCustomPaint getCustomPaintConverter(){
        return this.svgCustomPaint;
    }

    public SVGColor getColorConverter(){
        return this.svgColor;
    }

    /**
     * Converts part or all of the input GraphicContext into
     * a set of attribute/value pairs and related definitions
     *
     * @param gc GraphicContext to be converted
     * @return descriptor of the attributes required to represent
     *         some or all of the GraphicContext state, along
     *         with the related definitions
     * @see org.apache.batik.svggen.SVGDescriptor
     */
    public SVGDescriptor toSVG(final GraphicContext gc){
        return this.toSVG(gc.getPaint());
    }

    /**
     * @param paint Paint to be converted to SVG
     * @return a descriptor of the corresponding SVG paint
     */
    public SVGPaintDescriptor toSVG(final Paint paint){
        // we first try the extension handler because we may
        // want to override the way a Paint is managed!
        SVGPaintDescriptor paintDesc = this.svgCustomPaint.toSVG(paint);

        if (paintDesc == null) {
            if (paint instanceof Color)
                paintDesc = SVGColor.toSVG((Color)paint, this.generatorContext);
            else if (paint instanceof GradientPaint)
                paintDesc = this.svgLinearGradient.toSVG((GradientPaint)paint);
            else if (paint instanceof TexturePaint)
                paintDesc = this.svgTexturePaint.toSVG((TexturePaint)paint);
        }

        return paintDesc;
    }
}
