package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * 松屋の住所を取得するクラスです。
 * @author Kumano Tatsuo
 * 2005/12/03
 */
public class Matsuya {
	/**
	 * 最初のURL
	 * この後に市区町村コードが来ます。
	 */
	private static final String URL = "http://sp.chizumaru.com/dbh/matsuyaf/list.aspx?account=matsuyaf&accmd=0&arg=&c1=&c2=&c3=&c4=&c5=&c6=&c7=&c8=&c9=&c10=&c11=&c12=&c13=&c14=&c15=&c16=&c17=&c18=&c19=&c20=&c21=&c22=&c23=&c24=&c25=&c26=&c27=&c28=&c29=&c30=&adr=";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "SJIS";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator
		+ "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "matsuya_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * 松屋の座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param cityID 市区町村コード
	 * @param cityName 市区町村名
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String cityID,
		final String cityName, final Map<String, Point2D> isj, final MapPanel panel)
		throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Matsuya.CACHE_DIR).exists()) {
			new File(Matsuya.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Matsuya.CACHE_DIR + File.separator
			+ Matsuya.PREFIX + cityID + Matsuya.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(
				cacheFile), "SJIS");
			ISJUtil.parseAddresses(Matsuya.getAddresses(cityID,
				cityName, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(
			new FileInputStream(new File(cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "松屋");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * 松屋の市区町村別ページを解析して住所の一覧を取得します。
	 * @param cityID 市区町村コード
	 * @param cityName 市区町村名
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(
		final String cityID, final String cityName, final MapPanel panel)
		throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL + cityID);
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(
				new InputStreamReader(url.openStream(), ENCODING));
			String caption = null;
			final Pattern pattern = Pattern
				.compile("<a href=[^<>]+>([^<>]+)</a>");
			final Pattern pattern2 = Pattern
				.compile("<td [^<>]+>([^<>]+)</td>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					caption = matcher.group(1);
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					if (caption != null) {
						final String address = matcher2.group(1);
						if (address.contains(cityName)) {
							ret.put(address, caption);
						}
					}
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
		return ret;
	}
}
