package client;

/**
 * プレイヤを扱うスレッドです。
 * @author Kumano Tatsuo
 * Created on 2005/01/19 10:56:43
 */
public class Player extends Thread {
    /**
     * 目的地のx座標を定数倍したもの
     */
    private int destX;

    /**
     * 目的地のy座標を定数倍したもの
     */
    private int destY;

    /**
     * ゲーム中かどうか
     */
    private boolean isPlaying;

    /**
     * 動いている向き
     */
    private ClientConstants.Direction direction;

    /**
     * 名前
     */
    private String playerName;

    /**
     * レート
     */
    private int rate;

    /**
     * 速度を定数倍したもの[px / ms]
     */
    private int speed;

    /**
     * プレイヤのx座標を定数倍したもの
     */
    private int x;

    /**
     * プレイヤのy座標を定数倍したもの
     */
    private int y;

    /**
     * @return プレイヤの名前
     */
    public String getPlayerName() {
        return this.playerName;
    }

    /**
     * @return レート
     */
    public int getRate() {
        return this.rate;
    }

    /**
     * @return x座標
     */
    public int getX() {
        return this.x / ClientConstants.RESOLUTION;
    }

    /**
     * @return y座標
     */
    public int getY() {
        return this.y / ClientConstants.RESOLUTION;
    }

    /**
     * @return プレイ中かどうか
     */
    public boolean isPlaying() {
        return this.isPlaying;
    }

    /**
     * @return 垂直方向に動いているかどうか
     */
    public boolean isVertical() {
        return this.direction == ClientConstants.Direction.UP
                || this.direction == ClientConstants.Direction.DOWN;
    }

    public void run() {
        long lastTime = System.nanoTime();
        while (true) {
            try {
                long nowTime = System.nanoTime();
                if (this.x / ClientConstants.RESOLUTION < this.destX / ClientConstants.RESOLUTION) {
                    if (this.x + (nowTime - lastTime) * this.speed / 1000000 < this.destX) {
                        this.x += (nowTime - lastTime) * this.speed / 1000000;
                        this.direction = ClientConstants.Direction.RIGHT;
                    } else {
                        this.x = this.destX;
                    }
                } else if (this.x / ClientConstants.RESOLUTION > this.destX
                        / ClientConstants.RESOLUTION) {
                    if (this.x - (nowTime - lastTime) * this.speed / 1000000 > this.destX) {
                        this.x -= (nowTime - lastTime) * this.speed / 1000000;
                        this.direction = ClientConstants.Direction.LEFT;
                    } else {
                        this.x = this.destX;
                    }
                }
                if (this.y / ClientConstants.RESOLUTION < this.destY / ClientConstants.RESOLUTION) {
                    if (this.y + (nowTime - lastTime) * this.speed / 1000000 < this.destY) {
                        this.y += (nowTime - lastTime) * this.speed / 1000000;
                        this.direction = ClientConstants.Direction.DOWN;
                    } else {
                        this.y = this.destY;
                    }
                } else if (this.y / ClientConstants.RESOLUTION > this.destY
                        / ClientConstants.RESOLUTION) {
                    if (this.y - (nowTime - lastTime) * this.speed / 1000000 > this.destY) {
                        this.y -= (nowTime - lastTime) * this.speed / 1000000;
                        this.direction = ClientConstants.Direction.UP;
                    } else {
                        this.y = this.destY;
                    }
                }
                if (this.x / ClientConstants.RESOLUTION == this.destX / ClientConstants.RESOLUTION
                        && this.y / ClientConstants.RESOLUTION == this.destY
                                / ClientConstants.RESOLUTION) {
                    this.speed = 0;
                }
                lastTime = System.nanoTime();
                Thread.sleep(ClientConstants.REPAINT_WAIT_TIME);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 目的地を設定します。
     * @param x x座標
     * @param y y座標
     */
    public void setDestination(int x, int y) {
        this.destX = x * ClientConstants.RESOLUTION;
        this.destY = y * ClientConstants.RESOLUTION;
    }

    /**
     * プレイヤの座標を設定します。
     * @param x x座標
     * @param y y座標
     */
    public void setLocation(int x, int y) {
        this.x = x * ClientConstants.RESOLUTION;
        this.destX = this.x;
        this.y = y * ClientConstants.RESOLUTION;
        this.destY = this.y;
        this.speed = 0;
    }

    /**
     * @param name プレイヤの名前
     */
    public void setPlayerName(String name) {
        this.playerName = name;
    }

    /**
     * @param isPlaying プレイ中かどうかを設定します。
     */
    public void setPlaying(boolean isPlaying) {
        this.isPlaying = isPlaying;
    }

    /**
     * @param rate レート
     */
    public void setRate(int rate) {
        this.rate = rate;
    }

    /**
     * @param speed 速度を定数倍したもの[px / ms]
     */
    public void setSpeed(int speed) {
        this.speed = speed;
    }

    public String toString() {
        return "[name = " + this.playerName + ", rate = " + this.rate + "]";
    }
    /**
     * @return プレイヤの向きを取得します。
     */
    public ClientConstants.Direction getDirection() {
        return this.direction;
    }
}
