package client;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.ConcurrentModificationException;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import javax.swing.JPanel;
import common.CommonConstants;

/**
 * 描画を行なうパネルです。
 * @author Kumano Tatsuo
 * Created on 2005/01/19 10:46:49
 */
public class MainPanel extends JPanel implements Runnable {
    /**
     * ゲームの情報
     */
    private Game game;

    /**
     * パネルを初期化します。
     * @param in1 1Pの入力ストリーム
     * @param out1 1Pの出力ストリーム
     * @param in2 2Pの入力ストリーム
     * @param out2 2Pの出力ストリーム
     * @param handler キー入力を扱うオブジェクト
     * @param game ゲーム
     */
    public MainPanel(DataInputStream in1, DataOutputStream out1, DataInputStream in2,
            DataOutputStream out2, KeyHandler handler, Game game) {
        this.game = game;
        this.setPreferredSize(new Dimension(ClientConstants.MAIN_PANEL_WIDTH,
                ClientConstants.SCREEN_HEIGHT));
        this.addKeyListener(handler);
        this.backgroundImage = null;
        this.startTime = System.currentTimeMillis();
    }

    public void run() {
        while (true) {
            try {
                this.repaint();
                Thread.sleep(ClientConstants.REPAINT_WAIT_TIME);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 背景画像
     */
    private Image backgroundImage;

    /**
     * 開始時刻
     */
    private final long startTime;

    public void paint(Graphics graphics) {
        if (this.backgroundImage == null) {
            this.backgroundImage = createImage(this.getWidth(), this.getHeight());
            Graphics g = this.backgroundImage.getGraphics();
            //            g.drawImage(ClientConstants.IMAGE_BACKGROUND, 0, 0, this);
            for (Image image : ClientConstants.IMAGE_BOMS_HORIZONTAL) {
                g.drawImage(image, 0, 0, this);
            }
            for (Image image : ClientConstants.IMAGE_BOMS_VERTICAL) {
                g.drawImage(image, 0, 0, this);
            }
            for (Image image : ClientConstants.IMAGE_HARD_BLOCKS) {
                g.drawImage(image, 0, 0, this);
            }
            for (Image image : ClientConstants.IMAGE_SOFT_BLOCKS) {
                g.drawImage(image, 0, 0, this);
            }
            for (Image image : ClientConstants.IMAGE_PLAYERS_DOWN) {
                g.drawImage(image, 0, 0, this);
            }
            for (Image image : ClientConstants.IMAGE_PLAYERS_UP) {
                g.drawImage(image, 0, 0, this);
            }
            for (Image image : ClientConstants.IMAGE_PLAYERS_LEFT) {
                g.drawImage(image, 0, 0, this);
            }
            for (Image image : ClientConstants.IMAGE_PLAYERS_RIGHT) {
                g.drawImage(image, 0, 0, this);
            }
            g.drawImage(ClientConstants.IMAGE_BLOW, 0, 0, this);
            //            g.setColor(Color.WHITE);
            //            g.fillRect(0, 0, this.getWidth(), this.getHeight());
        }
        Graphics2D g = (Graphics2D) this.backgroundImage.getGraphics();
        g.setColor(Color.BLACK);
        for (byte i = 0; i < CommonConstants.STAGE_HEIGHT; ++i) {
            for (byte j = 0; j < CommonConstants.STAGE_WIDTH; ++j) {
                if (this.game.isChanged(i, j)) {
                    g.setClip(j * ClientConstants.GRID_WIDTH + ClientConstants.LEFT_MARGIN, (i - 1)
                            * ClientConstants.GRID_HEIGHT + ClientConstants.TOP_MARGIN,
                            ClientConstants.GRID_WIDTH, ClientConstants.GRID_HEIGHT * 2);
                    if (i > 0) {
                        drawBlock(g, (byte) (i - 1), j);
                    }
                    drawBlock(g, i, j);
                    if (i < CommonConstants.STAGE_HEIGHT - 1) {
                        drawBlock(g, (byte) (i + 1), j);
                    }
                    g.setClip(0, 0, this.getWidth(), this.getHeight());
                    if (this.game.getPlayers().containsKey(this.game.getNumber1())
                            || this.game.getPlayers().containsKey(this.game.getNumber2())
                            || (System.currentTimeMillis() - this.startTime) > 3000) {
                        this.game.unsetChanged(i, j);
                    }
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, -1 * ClientConstants.GRID_WIDTH
                            + ClientConstants.LEFT_MARGIN, -1 * ClientConstants.GRID_HEIGHT
                            + ClientConstants.TOP_MARGIN, this);
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, ClientConstants.GRID_WIDTH
                            * CommonConstants.STAGE_WIDTH + ClientConstants.LEFT_MARGIN, -1
                            * ClientConstants.GRID_HEIGHT + ClientConstants.TOP_MARGIN, this);
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, -1 * ClientConstants.GRID_WIDTH
                            + ClientConstants.LEFT_MARGIN, CommonConstants.STAGE_HEIGHT
                            * ClientConstants.GRID_HEIGHT + ClientConstants.TOP_MARGIN, this);
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, ClientConstants.GRID_WIDTH
                            * CommonConstants.STAGE_WIDTH + ClientConstants.LEFT_MARGIN,
                            CommonConstants.STAGE_HEIGHT * ClientConstants.GRID_HEIGHT
                                    + ClientConstants.TOP_MARGIN, this);
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, -1 * ClientConstants.GRID_WIDTH
                            + ClientConstants.LEFT_MARGIN, i * ClientConstants.GRID_HEIGHT
                            + ClientConstants.TOP_MARGIN, this);
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, CommonConstants.STAGE_WIDTH
                            * ClientConstants.GRID_WIDTH + ClientConstants.LEFT_MARGIN, i
                            * ClientConstants.GRID_HEIGHT + ClientConstants.TOP_MARGIN, this);
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, i * ClientConstants.GRID_WIDTH
                            + ClientConstants.LEFT_MARGIN, CommonConstants.STAGE_HEIGHT
                            * ClientConstants.GRID_HEIGHT + ClientConstants.TOP_MARGIN, this);
                    g.setClip(ClientConstants.LEFT_MARGIN, ClientConstants.TOP_MARGIN
                            - ClientConstants.GRID_HEIGHT, ClientConstants.GRID_WIDTH
                            * CommonConstants.STAGE_WIDTH, ClientConstants.GRID_HEIGHT);
                    g.drawImage(ClientConstants.IMAGE_BACKGROUND, i * ClientConstants.GRID_WIDTH
                            + ClientConstants.LEFT_MARGIN, -1 * ClientConstants.GRID_HEIGHT
                            + ClientConstants.TOP_MARGIN, this);
                    drawBlock(g, (byte) 0, i);
                    g.setClip(0, 0, this.getWidth(), this.getHeight());
                }
            }
        }
        graphics.drawImage(this.backgroundImage, 0, 0, this);
        Graphics2D g2 = (Graphics2D) graphics;
        if (!this.game.getBlows().isEmpty()) {
            try {
                for (Rectangle blow : this.game.getBlows()) {
                    for (int i = 0; i < blow.height; ++i) {
                        for (int j = 0; j < blow.width; ++j) {
                            g2.drawImage(ClientConstants.IMAGE_BLOW, (blow.x + j)
                                    * ClientConstants.GRID_WIDTH + ClientConstants.OFFSET_BLOW.x
                                    + ClientConstants.LEFT_MARGIN, (blow.y + i)
                                    * ClientConstants.GRID_HEIGHT + ClientConstants.OFFSET_BLOW.y
                                    + ClientConstants.TOP_MARGIN, this);
                        }
                    }
                    g2.setClip(blow.x * ClientConstants.GRID_WIDTH + ClientConstants.LEFT_MARGIN,
                            (blow.y + blow.height - 1) * ClientConstants.GRID_HEIGHT
                                    + ClientConstants.TOP_MARGIN, blow.width
                                    * ClientConstants.GRID_WIDTH, ClientConstants.GRID_HEIGHT);
                    for (int i = 0; i < blow.width; ++i) {
                        drawBlock(g2, (byte) (blow.y + blow.height), (byte) (blow.x + i));
                    }
                    g2.setClip(0, 0, this.getWidth(), this.getHeight());
                }
                this.game.getBlows().poll();
            } catch (ConcurrentModificationException e) {
            }
        }
        try {
            for (Map.Entry<Integer, Player> entry : this.game.getBombs().entrySet()) {
                int number = entry.getKey();
                Player bomb = entry.getValue();
                if (bomb.isVertical()) {
                    g2.drawImage(ClientConstants.IMAGE_BOMS_VERTICAL[bomb.getY() / 2
                            % ClientConstants.IMAGE_BOMS_VERTICAL.length], bomb.getX()
                            + ClientConstants.OFFSET_BOMB.x + ClientConstants.LEFT_MARGIN, bomb
                            .getY()
                            + ClientConstants.OFFSET_BOMB.y + ClientConstants.TOP_MARGIN, this);
                } else {
                    g2.drawImage(ClientConstants.IMAGE_BOMS_HORIZONTAL[bomb.getX() / 2
                            % ClientConstants.IMAGE_BOMS_HORIZONTAL.length], bomb.getX()
                            + ClientConstants.OFFSET_BOMB.x + ClientConstants.LEFT_MARGIN, bomb
                            .getY()
                            + ClientConstants.OFFSET_BOMB.y + ClientConstants.TOP_MARGIN, this);
                }
                final byte i = (byte) ((bomb.getY() + ClientConstants.GRID_HEIGHT / 2)
                        / ClientConstants.GRID_HEIGHT + 1);
                final byte j = (byte) ((bomb.getX() + ClientConstants.GRID_WIDTH / 2) / ClientConstants.GRID_WIDTH);
                g2.setClip(j * ClientConstants.GRID_WIDTH + ClientConstants.LEFT_MARGIN, (i - 1)
                        * ClientConstants.GRID_HEIGHT + ClientConstants.TOP_MARGIN,
                        ClientConstants.GRID_WIDTH, ClientConstants.GRID_HEIGHT);
                drawBlock(g2, i, j);
                g2.setClip(0, 0, this.getWidth(), this.getHeight());
            }
        } catch (ConcurrentModificationException e) {
        }
        Map<Integer, Player> sortedPlayers = new TreeMap<Integer, Player>();
        for (Player player : this.game.getPlayers().values()) {
            sortedPlayers.put(player.getY() + player.getX() * ClientConstants.GRID_HEIGHT
                    / CommonConstants.STAGE_WIDTH, player);
        }
        for (Player player : sortedPlayers.values()) {
            if (player.isPlaying()) {
                g2.setColor(Color.BLACK);
                if (player.getDirection() == ClientConstants.Direction.LEFT) {
                    g2.drawImage(ClientConstants.IMAGE_PLAYERS_LEFT[player.getX() * 2
                            / ClientConstants.GRID_HEIGHT
                            % ClientConstants.IMAGE_PLAYERS_LEFT.length], player.getX()
                            + ClientConstants.OFFSET_PLAYER.x + ClientConstants.LEFT_MARGIN, player
                            .getY()
                            + ClientConstants.OFFSET_PLAYER.y + ClientConstants.TOP_MARGIN, this);
                } else if (player.getDirection() == ClientConstants.Direction.RIGHT) {
                    g2.drawImage(ClientConstants.IMAGE_PLAYERS_RIGHT[player.getX() * 2
                            / ClientConstants.GRID_HEIGHT
                            % ClientConstants.IMAGE_PLAYERS_RIGHT.length], player.getX()
                            + ClientConstants.OFFSET_PLAYER.x + ClientConstants.LEFT_MARGIN, player
                            .getY()
                            + ClientConstants.OFFSET_PLAYER.y + ClientConstants.TOP_MARGIN, this);
                } else if (player.getDirection() == ClientConstants.Direction.UP) {
                    g2.drawImage(
                            ClientConstants.IMAGE_PLAYERS_UP[player.getY() * 2
                                    / ClientConstants.GRID_HEIGHT
                                    % ClientConstants.IMAGE_PLAYERS_UP.length],
                            player.getX() + ClientConstants.OFFSET_PLAYER.x
                                    + ClientConstants.LEFT_MARGIN, player.getY()
                                    + ClientConstants.OFFSET_PLAYER.y + ClientConstants.TOP_MARGIN,
                            this);
                } else {
                    g2.drawImage(ClientConstants.IMAGE_PLAYERS_DOWN[player.getY() * 2
                            / ClientConstants.GRID_HEIGHT
                            % ClientConstants.IMAGE_PLAYERS_DOWN.length], player.getX()
                            + ClientConstants.OFFSET_PLAYER.x + ClientConstants.LEFT_MARGIN, player
                            .getY()
                            + ClientConstants.OFFSET_PLAYER.y + ClientConstants.TOP_MARGIN, this);
                }
                final byte i = (byte) ((player.getY() + ClientConstants.GRID_HEIGHT / 2)
                        / ClientConstants.GRID_HEIGHT + 1);
                final byte j = (byte) ((player.getX() + ClientConstants.GRID_WIDTH / 2) / ClientConstants.GRID_WIDTH);
                g2.setClip(j * ClientConstants.GRID_WIDTH + ClientConstants.LEFT_MARGIN, (i - 1)
                        * ClientConstants.GRID_HEIGHT + ClientConstants.TOP_MARGIN,
                        ClientConstants.GRID_WIDTH, ClientConstants.GRID_HEIGHT);
                drawBlock(g2, i, j);
                g2.setClip(0, 0, this.getWidth(), this.getHeight());
                g2.setColor(Color.BLACK);
                drawName(g2, player, -1, 0);
                drawName(g2, player, -1, 0);
                drawName(g2, player, 1, -1);
                drawName(g2, player, 1, 1);
                drawName(g2, player, -1, 0);
                drawName(g2, player, 1, 0);
                drawName(g2, player, 0, -1);
                drawName(g2, player, 0, 1);
                g2.setColor(Color.WHITE);
                drawName(g2, player, 0, 0);
            }
        }
        if (this.game.getPlayers().containsKey(this.game.getNumber1())) {
            Player player = this.game.getPlayers().get(this.game.getNumber1());
            if (player.isPlaying()) {
                drawTriangle(g2, player, Color.RED);
            }
        }
        if (this.game.getPlayers().containsKey(this.game.getNumber2())) {
            Player player = this.game.getPlayers().get(this.game.getNumber2());
            if (player.isPlaying()) {
                drawTriangle(g2, player, Color.YELLOW);
            }
        }
    }

    /**
     * ブロックを描画します。
     * @param g グラフィクスコンテキスト
     * @param i 行
     * @param j 列
     */
    private void drawBlock(Graphics2D g, byte i, byte j) {
        final int index = (i == 0 || j == 0 || i == CommonConstants.STAGE_HEIGHT - 1 || j == CommonConstants.STAGE_WIDTH - 1) ? new Random(
                i * CommonConstants.STAGE_WIDTH + j).nextInt(3)
                : new Random((i * CommonConstants.STAGE_WIDTH + j) * 1000)
                        .nextInt(ClientConstants.IMAGE_HARD_BLOCKS.length);
        switch (this.game.getData(i, j)) {
        case CommonConstants.DATA_HARD_BLOCK:
            g.drawImage(ClientConstants.IMAGE_BACKGROUND, j * ClientConstants.GRID_WIDTH
                    + ClientConstants.LEFT_MARGIN, i * ClientConstants.GRID_HEIGHT
                    + ClientConstants.TOP_MARGIN, this);
            g.drawImage(ClientConstants.IMAGE_HARD_BLOCKS[index], j * ClientConstants.GRID_WIDTH
                    + ClientConstants.OFFSET_HARD_BLOCK.x + ClientConstants.LEFT_MARGIN, i
                    * ClientConstants.GRID_HEIGHT + ClientConstants.OFFSET_HARD_BLOCK.y
                    + ClientConstants.TOP_MARGIN, this);
            break;
        case CommonConstants.DATA_SOFT_BLOCK:
            g.drawImage(ClientConstants.IMAGE_SOFT_BLOCKS[new Random(i
                    * CommonConstants.STAGE_WIDTH + j)
                    .nextInt(ClientConstants.IMAGE_SOFT_BLOCKS.length)], j
                    * ClientConstants.GRID_WIDTH + ClientConstants.OFFSET_SOFT_BLOCK.x
                    + ClientConstants.LEFT_MARGIN, i * ClientConstants.GRID_HEIGHT
                    + ClientConstants.OFFSET_SOFT_BLOCK.y + ClientConstants.TOP_MARGIN, this);
            break;
        default:
            g.drawImage(ClientConstants.IMAGE_BACKGROUND, j * ClientConstants.GRID_WIDTH
                    + ClientConstants.LEFT_MARGIN, i * ClientConstants.GRID_HEIGHT
                    + ClientConstants.TOP_MARGIN, this);
            break;
        }
    }

    /**
     * 三角形を描画します。
     * @param g グラフィクスコンテキスト
     * @param player プレイヤ
     * @param color 色
     */
    private void drawTriangle(Graphics2D g, Player player, Color color) {
        Polygon polygon = new Polygon(new int[] {
                player.getX() + (int) Math.round(ClientConstants.GRID_WIDTH * 1.0 / 6) + ClientConstants.LEFT_MARGIN,
                player.getX() + (int) Math.round(ClientConstants.GRID_WIDTH * 5.0 / 6) + ClientConstants.LEFT_MARGIN,
                player.getX() + (int) Math.round(ClientConstants.GRID_WIDTH * 3.0 / 6) + ClientConstants.LEFT_MARGIN },
                new int[] {
                        player.getY() - ClientConstants.GRID_HEIGHT * 3 / 2
                                + ClientConstants.TOP_MARGIN - 8,
                        player.getY() - ClientConstants.GRID_HEIGHT * 3 / 2
                                + ClientConstants.TOP_MARGIN - 8,
                        player.getY() - ClientConstants.GRID_HEIGHT * 3 / 4
                                + ClientConstants.TOP_MARGIN - 8 }, 3);
        g.setColor(color);
        g.fill(polygon);
        g.setColor(Color.BLACK);
        g.draw(polygon);
    }

    /**
     * 名前を描画します。
     * @param g グラフィクスコンテキスト
     * @param player プレイヤ
     * @param dx x方向にずらす距離
     * @param dy y方向にずらす距離
     */
    private void drawName(Graphics2D g, Player player, int dx, int dy) {
        FontMetrics metrics = this.getFontMetrics(this.getFont());
        g.drawString(player.getPlayerName(), player.getX() + ClientConstants.GRID_WIDTH / 2
                - metrics.stringWidth(player.getPlayerName()) / 2 + ClientConstants.LEFT_MARGIN
                + dx, player.getY() + ClientConstants.TOP_MARGIN + dy - 15);
    }
}
