package client;

import java.awt.Rectangle;
import java.io.DataInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Queue;
import common.CommonConstants;

/**
 * ゲームの情報を扱うスレッドです。
 * @author Kumano Tatsuo
 * Created on 2005/01/19 11:06:51
 */
public class Game extends Thread {
    /**
     * 爆弾の一覧
     */
    private HashMap<Integer, Player> bombs;

    /**
     * ブロックの状態
     */
    private byte[][] data;
    
    /**
     * 変更されたかどうか
     */
    private boolean[][] isChanged;

    /**
     * @param row 行
     * @param col 列
     * @return 変更されたかどうか
     */
    public boolean isChanged(byte row, byte col) {
        return this.isChanged[row][col];
    }
    
    /**
     * 変更されていない状態に戻します。
     * @param row 行
     * @param col 列
     */
    public void unsetChanged(byte row, byte col) {
        this.isChanged[row][col] = false;
    }

    /**
     * 変更されたことにします。
     * @param row 行
     * @param col 列
     */
    public void setChanged(byte row, byte col) {
        this.isChanged[row][col] = true;
    }
    
    /**
     * 1Pの入力ストリーム
     */
    private DataInputStream in1;

    /**
     * 2Pの入力ストリーム
     */
    private DataInputStream in2;

    /**
     * 名前の一覧
     */
    private HashMap<Integer, String> names;

    /**
     * 1Pの番号
     */
    private int number1;

    /**
     * 2Pの番号
     */
    private int number2;

    /**
     * プレイヤの一覧
     */
    private HashMap<Integer, Player> players;

    /**
     * 炎の一覧
     */
    public Queue<Rectangle> blows;

    /**
     * ゲームを初期化します。
     * @param in1 1Pの入力ストリーム
     * @param in2 2Pの入力ストリーム
     */
    public Game(DataInputStream in1, DataInputStream in2) {
        this.in1 = in1;
        this.in2 = in2;
        this.players = new HashMap<Integer, Player>();
        this.bombs = new HashMap<Integer, Player>();
        this.names = new HashMap<Integer, String>();
        this.data = new byte[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
        this.isChanged= new boolean[CommonConstants.STAGE_HEIGHT][CommonConstants.STAGE_WIDTH];
        for (int i = 0; i < CommonConstants.STAGE_HEIGHT; ++i) {
            for (int j = 0; j < CommonConstants.STAGE_WIDTH; ++j) {
                this.isChanged[i][j] = true;
            }
        }
        this.blows = new LinkedList<Rectangle>();
    }

    /**
     * @return 爆弾の一覧
     */
    public HashMap<Integer, Player> getBombs() {
        return this.bombs;
    }

    /**
     * @param row 行
     * @param col 列
     * @return ブロックの情報
     */
    public byte getData(byte row, byte col) {
        return this.data[row][col];
    }

    /**
     * @return 1Pの番号
     */
    public int getNumber1() {
        return this.number1;
    }

    /**
     * @return 2Pの番号
     */
    public int getNumber2() {
        return this.number2;
    }

    /**
     * @return プレイヤの一覧
     */
    public HashMap<Integer, Player> getPlayers() {
        return this.players;
    }

    public void run() {
        while (true) {
            try {
                if (this.in1 != null) {
                    if (this.in1.available() > 0) {
                        byte command = this.in1.readByte();
                        switch (command) {
                        case CommonConstants.COMMAND_YOUR_NUMBER:
                            this.number1 = this.in1.readInt();
                            break;
                        case CommonConstants.COMMAND_PLAYER_NAME: {
                            int number = this.in1.readInt();
                            String name = this.in1.readUTF();
                            this.names.put(number, name);
                            break;
                        }
                        case CommonConstants.COMMAND_PLAYER_LOCATION: {
                            int number = this.in1.readInt();
                            int time = this.in1.readInt();
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            if (!this.players.containsKey(number)) {
                                Player player = new Player();
                                player.setPlayerName(this.names.get(number));
                                player.setLocation(col * ClientConstants.GRID_WIDTH, row
                                        * ClientConstants.GRID_HEIGHT);
                                this.players.put(number, player);
                                player.setPlaying(true);
                                player.start();
                            } else {
                                Player player = this.players.get(number);
                                player.setSpeed(ClientConstants.GRID_WIDTH * ClientConstants.RESOLUTION
                                        / time);
                                player.setDestination(col * ClientConstants.GRID_WIDTH, row
                                        * ClientConstants.GRID_HEIGHT);
                                player.setPlaying(true);
                            }
                            break;
                        }
                        case CommonConstants.COMMAND_BOMB_LOCATION: {
                            int number = this.in1.readInt();
                            int time = this.in1.readInt();
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            if (!this.bombs.containsKey(number)) {
                                Player bomb = new Player();
                                bomb.setPlayerName(this.names.get(number));
                                bomb.setLocation(col * ClientConstants.GRID_WIDTH, row
                                        * ClientConstants.GRID_HEIGHT);
                                this.bombs.put(number, bomb);
                                bomb.start();
                            } else {
                                Player bomb = this.bombs.get(number);
                                bomb.setSpeed(ClientConstants.GRID_WIDTH * ClientConstants.RESOLUTION
                                        / time);
                                bomb.setDestination(col * ClientConstants.GRID_WIDTH, row
                                        * ClientConstants.GRID_HEIGHT);
                            }
                            break;
                        }
                        case CommonConstants.COMMAND_HORIZONTAL_BLOW: {
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            byte length = this.in1.readByte();
                            this.blows.add(new Rectangle(col, row, length, 1));
                            break;
                        }
                        case CommonConstants.COMMAND_VERTICAL_BLOW: {
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            byte length = this.in1.readByte();
                            this.blows.add(new Rectangle(col, row, 1, length));
                            break;
                        }
                        case CommonConstants.COMMAND_PLAYER_DEAD: {
                            int number = this.in1.readInt();
                            if (this.players.containsKey(number)) {
                                this.players.get(number).setPlaying(false);
                            }
                            break;
                        }
                        case CommonConstants.COMMAND_BOMB_BLOW: {
                            int number = this.in1.readInt();
                            if (this.bombs.containsKey(number)) {
                                this.bombs.remove(number);
                            }
                            break;
                        }
                        case CommonConstants.COMMAND_PLAYER_RATE: {
                            int number = this.in1.readInt();
                            int rate = this.in1.readInt();
                            if (this.players.containsKey(number)) {
                                this.players.get(number).setRate(rate);
                            }
                            break;
                        }
                        case CommonConstants.COMMAND_DATA_NO_BLOCK: {
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            this.data[row][col] = CommonConstants.DATA_NO_BLOCK;
                            this.isChanged[row][col] = true;
                            break;
                        }
                        case CommonConstants.COMMAND_DATA_HARD_BLOCK: {
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            this.data[row][col] = CommonConstants.DATA_HARD_BLOCK;
                            this.isChanged[row][col] = true;
                            break;
                        }
                        case CommonConstants.COMMAND_DATA_SOFT_BLOCK: {
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            this.data[row][col] = CommonConstants.DATA_SOFT_BLOCK;
                            this.isChanged[row][col] = true;
                            break;
                        }
                        case CommonConstants.COMMAND_DATA_ITEM:
                            byte row = this.in1.readByte();
                            byte col = this.in1.readByte();
                            this.data[row][col] = CommonConstants.DATA_ITEM;
                            this.isChanged[row][col] = true;
                            break;
                        default:
                            System.err.println("WARNING: UNKNOWN COMMAND " + command);
                            break;
                        }
                    } else {
                        Thread.sleep(ClientConstants.KEY_HANDLER_THREAD_WAIT_TIME);
                    }
                }
                if (this.in2 != null) {
                    if (this.in2.available() > 0) {
                        byte command = this.in2.readByte();
                        switch (command) {
                        case CommonConstants.COMMAND_YOUR_NUMBER:
                            this.number2 = this.in2.readInt();
                            break;
                        case CommonConstants.COMMAND_PLAYER_NAME:
                            this.in2.readInt();
                            this.in2.readUTF();
                            break;
                        case CommonConstants.COMMAND_PLAYER_LOCATION:
                        case CommonConstants.COMMAND_BOMB_LOCATION:
                            this.in2.readInt();
                            this.in2.readInt();
                            this.in2.readByte();
                            this.in2.readByte();
                            break;
                        case CommonConstants.COMMAND_PLAYER_DEAD:
                        case CommonConstants.COMMAND_BOMB_BLOW:
                            this.in2.readInt();
                            break;
                        case CommonConstants.COMMAND_PLAYER_RATE:
                            this.in2.readInt();
                            this.in2.readInt();
                            break;
                        case CommonConstants.COMMAND_DATA_NO_BLOCK:
                        case CommonConstants.COMMAND_DATA_HARD_BLOCK:
                        case CommonConstants.COMMAND_DATA_SOFT_BLOCK:
                        case CommonConstants.COMMAND_DATA_ITEM:
                            this.in2.readByte();
                            this.in2.readByte();
                            break;
                        case CommonConstants.COMMAND_HORIZONTAL_BLOW:
                        case CommonConstants.COMMAND_VERTICAL_BLOW:
                            this.in2.readByte();
                            this.in2.readByte();
                            this.in2.readByte();
                            break;
                        default:
                            System.err.println("WARNING: UNKNOWN COMMAND " + command);
                            break;
                        }
                    } else {
                        Thread.sleep(ClientConstants.KEY_HANDLER_THREAD_WAIT_TIME);
                    }
                }
            } catch (IOException e) {
                e.printStackTrace();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * @return 炎の一覧
     */
    public Queue<Rectangle> getBlows() {
        return this.blows;
    }
}
