<?php
/*
Plugin Name: Otenki
Plugin URI: http://wppluginsj.sourceforge.jp/wp-otenki/
Description: Add weather meta information by <a href="http://weather.livedoor.com/weather_hacks/webservice">LWWS</a>
Author: hiromasa
Version: 2.0.0
Author URI: http://another.maple4ever.net/
*/

/*  Copyright 2013 hiromasa  (email : h1romas4@gmail.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/******************************************************************************
 * THIS FILE IS CALLED ONLY.
 *****************************************************************************/

if(!(defined('ABSPATH') && defined('WPINC'))) return;

/******************************************************************************
 * Otenki Class
 * 
 * @author     hiromasa
 * @version    2.0.0
 *****************************************************************************/
class Otenki {

    /**
     * カスタムフィールド・メタキー.
     */
    const META_KEY = 'otenki';

    /**
     * オプション値・キー.
     */
    const OPTION_KEY = 'otenki';

    /**
     * API 取得スケジュールアクションフック名.
     */
    const HOOK_EXTRACT = 'extract_otenki';

    /**
     * API バージョン情報(V1/JSON).
     */
    const LWWS_V1_JSON = 'LWWS_V1_JSON';

    /**
     * シングルトンインスタンス格納.
     */
    static private $INSTANCE = null;

    /**
     * LWWS URL 引数（地域別値）.
     * 
     * @see http://weather.livedoor.com/forecast/rss/primary_area.xml
     */
    var $city;

    /**
     * 画像出力時の CSS クラス.
     */
    var $css_class;

    /**
     * WordPress 拡張処理用 Otenki インスタンスの取得.
     * 
     * @return Otenki
     */
    static function getInstance() {
        if(self::$INSTANCE == null) {
            self::$INSTANCE = new Otenki();
        }
        return self::$INSTANCE;
    }

    /**
     * コンストラクタ・プラグインデフォルト値設定及びオプション値の取得.
     */
    private function __construct($city = '016010', $css_class = 'otenki') {
        // LWWS URL 引数（地域別値）
        $this->city = $city;
        // 画像出力時の CSS クラス
        $this->css_class = $css_class;
        // 投稿スケジュール用アクションフック登録
        add_action(self::HOOK_EXTRACT, array ($this, 'extract'));
        // 管理画面アクションフック登録
        add_action('admin_init', array($this, 'setting'));
        // 管理画面地域ID設定を取得し存在すれば上書き
        $value = get_option(self::OPTION_KEY);
        if(!$value) {
            $this->city = $value;
        }
    }

    /**
     * 記事公開フックで天気情報取得スケジュールを作成する.
     * 
     * @param $post_ID
     * @param $post
     */
    function prepare($post_ID, $post) {
        // "今日"のポストの時だけ天気を取得
        if(date("Y/m/d", strtotime($post->post_date))
            != date("Y/m/d", current_time('timestamp'))) return;
        // 天気取得の非同期スケジュール作成
        wp_schedule_single_event(
            time()
            , self::HOOK_EXTRACT
            , array($post_ID));
    }

    /**
     * LWWS から天気情報を取得し記事のカスタムフィールドに格納する.
     * 
     * @param $post_ID
     */
    function extract($post_ID) {
        // LWWS(V1/JSON)より天気情報を取得
        $json = wp_remote_get(
            'http://weather.livedoor.com/forecast/webservice/json/v1?'
            . http_build_query(array('city' => $this->city)));
        // 取得できなかった場合は次回の公開に期待
        if(is_wp_error($json)) return;
        // リクエストから JSON データ取得し PHP 連想配列に変換
        $otenki = json_decode($json['body'], true);
        // 正常な JSON 形式ではなければ次回の公開に期待
        if(!$this->isOtenki($otenki)) return;
        // カスタムフィールドにバージョン情報付きで格納
        update_post_meta(
            $post_ID
            , self::META_KEY
            , array(self::LWWS_V1_JSON => $otenki));
    }

    /**
     * カスタムフィールドに取得したお天気アイコン用 HTML をアクションフックで出力.
     */
    function output() {
        // ループ中の記事IDを取得
        $post_ID = get_the_ID();
        // カスタムフォールドから天気情報取得
        $otenki = get_post_meta($post_ID, self::META_KEY, true);
        // カスタムフィールドが正しい形式でなければ何もしない
        if(!(isset($otenki[self::LWWS_V1_JSON])
            && $this->isOtenki($otenki[self::LWWS_V1_JSON]))) return;
        // お天気情報を元に HTML を構築して出力
        $today = $otenki[self::LWWS_V1_JSON]['forecasts'][0];
        $image = $today['image'];
        $telop = $today['telop'];
        echo "<img src=\"${image['url']}\""
            . " width=\"${image['width']}\" height=\"${image['height']}\""
            . " alt =\"${telop}\""
            . " class=\"$this->css_class\" />";
    }

    /**
     * お天気情報バリデーション.
     * 
     * @param $otenki
     * @return boolean
     */
    private function isOtenki($otenki) {
        if(!(is_array($otenki)
            && isset($otenki['forecasts'])
            && is_array($otenki['forecasts'])
            && isset($otenki['forecasts'][0]['telop'])
            && isset($otenki['forecasts'][0]['image']))) {
            return false;
        }
        return true;
    }

    /**
     * カスタムフィールドに取得したお天気アイコン用 HTML をアクションフックで出力.
     * (下位互換用)
     */
    function compat() {
        $otenki = get_post_custom_values('wpotenki');
        if(!is_array($otenki)) return;
        $otenki = unserialize($otenki[0]);
        if(!(isset($otenki['mark'])
            && isset($otenki['otenki']))) return;
        echo
            "<img src=\"${otenki['mark']}\""
            . " alt =\"${otenki['otenki']}\""
            . " class=\"$this->css_class\" />";
    }

    /**
     * 管理画面の投稿設定に地域ID設定を追加.
     */
    function setting() {
        // 投稿設定画面
        $writing = 'writing';
        // 投稿設定にセクション追加
        add_settings_section(
            'otenki_setting_section'
            , 'お天気情報取得サービス'
            , array($this, 'echoSettingSection')
            , $writing);
        // セクションに地域ID指定テキストボックス追加
        add_settings_field(
            self::OPTION_KEY
            , '地域ID'
            , array($this, 'echoSettingField')
            , $writing
            , 'otenki_setting_section');
        // 設定に登録
        register_setting($writing, self::OPTION_KEY);
    }

    /**
     * セクション文字列出力.
     */
    function echoSettingSection() {
        echo '<a href="http://weather.livedoor.com/forecast/rss/primary_area.xml"'
            . ' target="_blank">LWWS</a> のソースから'
            . '取得したい地域IDを指定してください。';
    }

    /**
     * 地域ID設定テキストボックス出力.
     */
    function echoSettingField() {
        echo '<input name="' . self::OPTION_KEY . '"'
            . ' id="' . self::OPTION_KEY . '"'
            . ' type="text" class="code"'
            . ' value="'
            . esc_attr(get_option(self::OPTION_KEY)) . '" />';
    }
}

/******************************************************************************
 * Otenki - WordPress Interface Define
 *****************************************************************************/

add_action('publish_post'
    , array(Otenki::getInstance(), 'prepare'), 10, 2);
add_action('the_otenki'
    , array(Otenki::getInstance(), 'output'));
add_action('the_otenki'
    , array(Otenki::getInstance(), 'compat'));
?>