<?php
/*
Plugin Name: JSeries Notifier
Plugin URI: http://wppluginsj.sourceforge.jp/jseries-notifier/
Version: 0.9.5
Description: Notify updates of plugins which are supplied as WordPress Plugins/JSeries.
Author: IKEDA Yuriko
Author URI: http://en.yuriko.net/
Text Domain: jseries-notifier
Domain Path: /
*/

/*  Copyright (c) 2007-2010 IKEDA Yuriko

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

if ( !defined('WP_INSTALLING') || !WP_INSTALLING ) :

// define ('JN_USE_LOCAL_FILE', true); // Debug use
define ('JSERIES_PROJECT_WEB', 'http://wppluginsj.sourceforge.jp/');
define ('JSERIES_RELEASE_HOST', 'sourceforge.jp');
define ('JSERIES_RELEASE_JS_PATH', '/projects/wppluginsj/releases/compact.js');
define ('JSERIES_RELEASE_JSON', 'w.set_data({');
define ('JN_RETRIEVE_INTERVAL', 'twicedaily');
define ('JN_BAD_VERSION', '0.0.0');
define ('JN_SFJP_NO_RESPONSE', -1);
define ('JN_JSON_PARSE_ERROR', -2);

/* ==================================================
 *   JSeriesNotifier class
   ================================================== */

class JSeriesNotifier {
	static $wp_vers = NULL;
	static $jseries_plugins = NULL;
	var $plugin_dir;
	var $text_domain = 'jseries-notifier';
	var $domain_path = '';
	var $textdomain_loaded = false;
	var $hook_name = 'jseries-update-check';
	var $intereval;
//	var $plugin_name;
	var $plugin_dist_uri;
	var $versions;

function JSeries_Notifier() {
	$this->__construct();
}

function __construct() {
	if ($this->check_wp_version('2.3', '<')) {
		return;
	}
	$this->plugin_dir = basename(dirname(__FILE__));
	$schedules = wp_get_schedules();
	$this->interval = $schedules[JN_RETRIEVE_INTERVAL]['interval'];

	add_action('plugins_loaded', array($this, 'load_textdomain'));
	add_action($this->hook_name, array($this, 'check_updates'));
	add_action('admin_init', array($this, 'maybe_check_updates'));
	if ($this->check_wp_version('2.9')) {
		add_filter('transient_update_plugins', array($this, 'add_update_plugins'));
		add_action('admin_init', array($this, 'replace_update_row'), 11);
	} elseif ($this->check_wp_version('2.6')) {
		add_action('after_plugin_row', array($this, 'plugin_update_row'), 10, 2);
	} else {
		add_action('load-plugins.php', array($this, 'check_updates'));
		add_action('after_plugin_row', array($this, 'plugin_update_row23'));
	}
	register_activation_hook(__FILE__, array($this, 'start_schedule'));
	register_activation_hook(__FILE__, array($this, 'delete_option'));
	register_deactivation_hook(__FILE__, array($this, 'clear_schedule'));
	register_deactivation_hook(__FILE__, array($this, 'delete_option'));
	if (function_exists('get_blog_list')) {
		add_action('activate_sitewide_plugin', array($this, 'started_sitewidely'));
		add_action('deactivate_sitewide_plugin', array($this, 'stopped_sitewidely'));
	}
	return;
}

/* ==================================================
 * @param	string   $domain
 * @param	string   $subdir
 * @return	none
 * @since	0.9.5
 */
// public 
function load_textdomain() {
	$lang_dir = $this->plugin_dir . $this->domain_path;
	$plugin_path = defined('PLUGINDIR') ? PLUGINDIR . '/': 'wp-content/plugins/';
	load_plugin_textdomain($this->text_domain, $plugin_path . $lang_dir, $lang_dir);
}

/* ==================================================
 * @param	string   $version
 * @param	string   $operator
 * @return	boolean  $result
 */
// private 
function check_wp_version($version, $operator = '>=') {
	if (! JSeriesNotifier::$wp_vers) {
		JSeriesNotifier::$wp_vers = get_bloginfo('version');
		if (! is_numeric(JSeriesNotifier::$wp_vers)) {
			JSeriesNotifier::$wp_vers = preg_replace('/[^.0-9]/', '', JSeriesNotifier::$wp_vers);  // strip 'ME'
		}
	}
	return version_compare(JSeriesNotifier::$wp_vers, $version, $operator);
}

/* ==================================================
 * @param	none
 * @return	object $jseries_plugins
 * @since   0.9.5
 */
public function get_jseries_plugins() {
	if ( !isset(JSeriesNotifier::$jseries_plugins)) {
		JSeriesNotifier::$jseries_plugins = get_option('jseries_plugins');
	}
	return JSeriesNotifier::$jseries_plugins;
}

/* ==================================================
 * @param	object $jseries_plugins
 * @return	none
 * @since   0.9.5
 */
public function set_jseries_plugins($jseries_plugins) {
	update_option('jseries_plugins', $jseries_plugins);
	JSeriesNotifier::$jseries_plugins = get_option('jseries_plugins'); // clone of the object for both PHP4 and PHP5
}

/* ==================================================
 * @param	none
 * @return	none
 * @since   0.9.5
 */
public function start_schedule() {
	wp_schedule_event(time(), JN_RETRIEVE_INTERVAL, $this->hook_name);
}

/* ==================================================
 * @param	none
 * @return	none
 * @since   0.9.5
 */
public function started_sitewidely() {
	$blogs = get_blog_list(0, 'all', false);
	if (is_array($blogs)) {
		reset($blogs);
		foreach((array) $blogs as $key => $details) {
			switch_to_blog($details['blog_id']);
			$this->start_schedule();
			$this->delete_option();
			restore_current_blog();
		}
	}
}

/* ==================================================
 * @param	none
 * @return	none
 * @since   0.9.5
 */
public function clear_schedule() {
	wp_clear_scheduled_hook($this->hook_name);
}

/* ==================================================
 * @param	none
 * @return	none
 * @since   0.9.5
 */
public function stopped_sitewidely() {
	$blogs = get_blog_list(0, 'all', false);
	if (is_array($blogs)) {
		reset($blogs);
		foreach((array) $blogs as $key => $details) {
			switch_to_blog($details['blog_id']);
			$this->clear_schedule();
			$this->delete_option();
			restore_current_blog();
		}
	}
}

/* ==================================================
 * @param	none
 * @return	none
 */
// public 
function delete_option() {
	delete_option('jseries_plugins');
}

/* ==================================================
 * @param	none
 * @return	array  $versions
 * based on wp_update_plugins() at wp-admin/includes/update.php of WP 2.3.1
 */
// public 
function maybe_check_updates() {
	$current = $this->get_jseries_plugins();
	$schedules = wp_get_schedules();
	if ( isset( $current->last_checked ) && ($this->interval / 2) > ( time() - $current->last_checked ) ) {
		return;
	}
	$this->check_updates();
}

/* ==================================================
 * @param	none
 * @return	array  $versions
 * based on wp_update_plugins() at wp-admin/includes/update.php of WP 2.3.1
 */
// public 
function check_updates() {
	if (! function_exists('get_plugins') && file_exists(ABSPATH . 'wp-admin/includes/plugin.php')) {
		require_once(ABSPATH . 'wp-admin/includes/plugin.php');
	}

	$plugins = get_plugins();
	$current = $this->get_jseries_plugins();
	$versions = array();

	$this_plugin = plugin_basename(__FILE__);
//	$this->plugin_name = $plugins[$this_plugin]['Name'];
	if (isset($plugins[$this_plugin]['PluginURI'])) {
		$this->plugin_dist_uri = $plugins[$this_plugin]['PluginURI'];
	} elseif (preg_match('/a href="([^"]*)"/', $plugins[$this_plugin]['Title'], $url)) {
		$this->plugin_dist_uri = $url[1];
	} else {
		$this->plugin_dist_uri = JSERIES_PROJECT_WEB;
	}

	$new_option = '';
	$new_option->last_checked = time();

	$plugin_changed = false;
	foreach ( $plugins as $file => $p ) {
//		$versions[$p['Name']] = $p['Version'];
		$versions[$p['Name']][$file] = $p['Version'];
		$new_option->checked[ $file ] = $p['Version'];

		if ( ! isset( $current->checked[ $file ] ) ) {
			$plugin_changed = true;
			continue;
		}

		if ( $current->checked[ $file ] != $p['Version'] )
			$plugin_changed = true;
	}
	$this->versions = $versions;

	if (
		isset( $current->last_checked ) &&
		$this->interval > ( time() - $current->last_checked ) &&
		! $plugin_changed
	) {
		return false;
	}
	
	$js = $this->retrive_latest_versions();
	if ($js) {
		$response = $this->pickup_versions($js);
		if ( $response ) {
			$new_option->response = $response;
		}
	}
	$this->set_jseries_plugins($new_option);

}

/* ==================================================
 * @param	none
 * @return	mix     $response
 * based on wp_update_plugins() at wp-admin/includes/update.php of WP 2.3.1
 */
// private 
function retrive_latest_versions() {
	$response = '';
	if ( defined('JN_USE_LOCAL_FILE') && JN_USE_LOCAL_FILE ) {
		$response = file_get_contents(dirname(__FILE__) . '/compact.js');
	} elseif ( function_exists('wp_remote_get') ) {
		$http_result = wp_remote_get('http://' . JSERIES_RELEASE_HOST . JSERIES_RELEASE_JS_PATH);
		if ( is_wp_error($http_result) ) {
			$response = $http_result;
		} else {
			$response = $http_result['body'];
		}
	} elseif ( function_exists('fsockopen') ) {
		global $wp_version;
		$http_request = array();
		$http_request[] = 'GET ' . JSERIES_RELEASE_JS_PATH . ' HTTP/1.0';
		$http_request[] = 'Host: ' . JSERIES_RELEASE_HOST;
		$http_request[] = 'User-Agent: WordPress/' . $wp_version . '; ' . get_bloginfo('url');
	
		if ( false !== ( $fs = @fsockopen( JSERIES_RELEASE_HOST, 80, $errno, $errstr, 3) ) && is_resource($fs) ) {
			fwrite($fs, implode("\r\n", preg_replace('/[\r\n]/', '', $http_request)) . "\r\n\r\n");
	
			while ( !feof($fs) ) {
				$response .= fgets($fs, 1160); // One TCP-IP packet
			}
			fclose($fs);
		}
	}
	return $response;
}

/* ==================================================
 * @param	string  $js
 * @return	mix     $response
 */
// private 
function pickup_versions($js) {
	if ( empty($js) ) {
		return $this->set_error(JN_SFJP_NO_RESPONSE);
	}
	if ( is_wp_error($js) ) {
		return $this->set_error(JN_SFJP_NO_RESPONSE, $js->get_error_message());
	}
	$pos = strpos($js, JSERIES_RELEASE_JSON, 0);
	if ( false === $pos ) {
		return $this->set_error(JN_JSON_PARSE_ERROR);
	}
	$response = array();
	if ( preg_match_all(
		'!\'id\': (\d+),\s*\'package\': "([^"]*)",\s*\'version\': "([^"]*)",\s*\'date\': (\d+),\s*\'link\': "([^"]*)"!s', 
		$js, $plugins, PREG_SET_ORDER, $pos) ) {
		foreach ( $plugins as $p ) {
			$id          = $p[1];
			$name        = $p[2];
			$new_version = $p[3];
			$date        = $p[4];
			$link        = $p[5];
			if (isset($this->versions[$name])) {
				foreach ( (array) $this->versions[$name] as $file => $version ) {
					if (version_compare($version, $new_version, '<')) {
						$response[$file]->id = $id;
						$response[$file]->name = $name;
						$response[$file]->slug = preg_replace('#(/.*|\.php)$#', '', $file);
						$response[$file]->new_version = $new_version;
						$response[$file]->url = $link;
					}
				}

			}
		}
	} else {
		return $this->set_error(JN_JSON_PARSE_ERROR);
	}

	return $response;
}

/* ==================================================
 * @param	int      $code
 * @param	string   $http_error
 * @return	array    $response
 */
// private  
function set_error($code, $http_error = '') {
	$this_plugin = plugin_basename(__FILE__);
	$response[$this_plugin]->url = $this->plugin_dist_uri;
	$response[$this_plugin]->new_version = JN_BAD_VERSION;
	$response[$this_plugin]->error_code = $code;
	$response[$this_plugin]->http_error = $http_error;
	return $response;
}

/* ==================================================
 * @param	mix  $value
 * @return	mix  $value
 * @since	0.9.5
 */
// public  
function add_update_plugins($value) {
	$updates = $this->get_jseries_plugins();
	if ( !isset($updates->response) ) {
		return $value;
	}
	foreach ($updates->response as $file => $plugin) {
		$value->response[$file] = $plugin;
	}
	return $value;
}

/* ==================================================
 * @param	none
 * @return	none
 * @since	0.9.5
 */
// public  
function replace_update_row() {
	$updates = get_option('jseries_plugins');
	if ( !isset($updates->response) ) {
		return;
	}
	foreach ($updates->response as $file => $plugin) {
		remove_action( "after_plugin_row_$file", 'wp_plugin_update_row', 10, 2);
		add_action( "after_plugin_row_$file", array($this, 'plugin_update_row'), 10, 2);
	}
	return $value;
}

/* ==================================================
 * @param	string  $file
 * @return	none
 */
// public  
function plugin_update_row23($file) {
	global $plugin_data;
	$this->plugin_update_row($file, $plugin_data);
}

/* ==================================================
 * @param	string  $file
 * @return	none
 * based on wp_plugin_update_row() at wp-admin/includes/update.php
 */
// public  
function plugin_update_row($file, $plugin_data) {
	$current = $this->get_jseries_plugins();
	if ( !isset( $current->response[$file]) ) {
		return false;
	}
	$r = $current->response[$file];

	if ( JN_BAD_VERSION == $r->new_version ) {
		$error = __('<strong>An error occured!</strong> ', 'jseries-notifier') . $this->get_error_message($r->error_code);
		if (isset($r->http_error)) {
			$error .= $r->http_error;
		}
	} else {
		$error = '';
	}
	if ( $this->check_wp_version('2.8') ) {
		$before = '<tr class="plugin-update-tr"><td colspan="3" class="plugin-update"><div class="update-message">';
		$format = str_replace(' class="thickbox"', '', __('There is a new version of %1$s available. <a href="%2$s" class="thickbox" title="%1$s">View version %3$s Details</a>.'));
		$after = '</div></td></tr>';
	} elseif ( $this->check_wp_version('2.7') ) {
		$before =  "<tr><td colspan='5' class='plugin-update'>";
		$format = str_replace(' class="thickbox"', '', __('There is a new version of %1$s available. <a href="%2$s" class="thickbox" title="%1$s">View version %3$s Details</a>.'));
		$after = '</td></tr>';
	} elseif ( $this->check_wp_version('2.5') ) { // for WP 2.5 to 2.6
		$before = "<tr><td colspan='5' class='plugin-update'>";
		$format = __('There is a new version of %1$s available. <a href="%2$s">Download version %3$s here</a>.');
		$after = '</td></tr>';
	} else { // for WP 2.3
		$before = "<tr><td colspan='5' class='plugin-update'>";
		$format = __('There is a new version of %s available. <a href="%s">Download version %s here</a>.');
		$after = '</td></tr>';
	}
	echo $before 
	. sprintf( ($error ? $error : $format), $plugin_data['Name'], attribute_escape($r->url), attribute_escape($r->new_version) )
	. $after;
}

/* ==================================================
 * @param	int      $code
 * @return	string   $message
 */
// private  
function get_error_message($code) {
	$message = '';
	switch ($code) {
	case JN_SFJP_NO_RESPONSE:
		$message = __('No response from SourceForge.jp.', 'jseries-notifier');
		break;
	case JN_JSON_PARSE_ERROR:
		$message = __('The format of release updates has been changed. Please <a href="%2$s">confirm</a> a new version of %1$s that can parse the new format.', 'jseries-notifier');
		break;
	}
	return $message;
}

// ===== End of class ====================
}

$JS_Notifier = new JSeriesNotifier;
endif; // WP_INSTALLING
?>