<?php

/*
Plugin Name: Sidebar Modules
Plugin URI: http://nybblelabs.org.uk/projects/sidebar-modules/
Description: Add sidebar modules to your sidebar!
Version: 1.0.0
Author: Ben Sherratt
Author URI: http://nybblelabs.org.uk/
*/

/**
 * Sidebar module plugin
 * Copyright (c) 2006 Ben Sherratt
 *
 * REDISTRIBUTION OF THIS FILE OR ANY FILE DERIVED FROM IT IS AGAINST COPYRIGHT LAW
 **/

/* Constants **************************************************************************************/

// Rather messy, but it's cross platform!
define('SBMPLUGINPATH', (DIRECTORY_SEPARATOR != '/') ? str_replace(DIRECTORY_SEPARATOR, '/', dirname(__FILE__)) : dirname(__FILE__));

/* Globals ****************************************************************************************/

// The registered sidebars
$sbm_registered_sidebars = array();

// The active modules
$sbm_active_modules = array();

// The disabled modules
$sbm_disabled_modules = array();

// The registered sidebar modules
$sbm_registered_modules = array();

// The module currently being manipulated
$sbm_current_module = false;

// Error text for XML
$sbm_error_text = '';


/* Classes ****************************************************************************************/

/**
 * The main class
 **/
class sbm {
	/* Install functions **********************************************************************/

	/**
	 * The installation function
	 **/
	function install() {
		// Add the options
		add_option('sbm_modules_active', array());
		add_option('sbm_modules_disabled', array());
		add_option('sbm_modules_next_id', 1);
	}

	/**
	 * The uninstallation function
	 **/
	function uninstall() {
		// Remove the options
		delete_option('sbm_modules_active');
		delete_option('sbm_modules_disabled');
		delete_option('sbm_modules_next_id');
	}


	/* Interface functions ********************************************************************/

	/**
	 * The function to bootstrap for the WP interface
	 **/
	function wp_bootstrap() {
		// Register activation stuff
		register_activation_hook(__FILE__, array('sbm', 'activate'));
		register_deactivation_hook(__FILE__, array('sbm', 'deactivate'));

		sbm::pre_bootstrap();

		// Add menus
		add_action('admin_menu', array('sbm', 'add_menus'));

		// Check if this page is the one being shown, if so then add stuff to the header
		if($_GET['page'] == 'sbm-modules') {
			add_action('admin_head', array('sbm', 'module_admin_head'));
		}

		// Post-bootstrap when everything is loaded
		add_action('init', array('sbm', 'post_bootstrap'));

		// Output the CSS files
		add_action('wp_head', array('sbm', 'output_module_css_files'));
	}

	/**
	 * The function to bootstrap for the direct interface
	 **/
	function direct_bootstrap() {
		global $sbm_registered_modules, $sbm_active_modules, $sbm_disabled_modules, $sbm_error_text;

		// You MUST be an admin to access this stuff
		auth_redirect();

		sbm::pre_bootstrap();
		sbm::post_bootstrap();

		// Check for specific actions that return a HTML response
		if($_GET['action'] == 'control-show') {
			if(isset($_POST['module_id'])) {
				$all_modules = sbm::get_all_modules();
				$all_modules[$_POST['module_id']]->displayControl();
			} else {
				echo(false);
			}
		} elseif($_GET['action'] == 'control-post-list-show') {
			if(isset($_POST['module_id'])) {
				$all_modules = sbm::get_all_modules();
				$all_modules[$_POST['module_id']]->displayPostList();
			} else {
				echo(false);
			}
		} elseif($_GET['action'] == 'control-page-list-show') {
			if(isset($_POST['module_id'])) {
				$all_modules = sbm::get_all_modules();
				$all_modules[$_POST['module_id']]->displayPageList();
			} else {
				echo(false);
			}
		} else {
			// Set the output type
			header('Content-type: text/xml');

			// XML prelude
			echo('<?xml version="1.0"?>');

			// Begin the response
			echo('<response>');

			// Check what the action is
			switch($_GET['action']) {
				// List of the modules in the sidebar
				case 'list':
					$tmp_modules = $sbm_active_modules;
					$tmp_modules[] = $sbm_disabled_modules;

					if($tmp_modules) {
						// Output the modules
						foreach($tmp_modules as $modules) {
							echo('<modules>');
							foreach($modules as $module) {
								echo('<module id="' . $module->id . '">' . $module->name . '</module>');
							}
							echo('</modules>');
						}
					}

					break;

				// Add a module to the sidebar
				case 'add':
					// Check the title was correct
					if(isset($_POST['add_name']) && trim((string)($_POST['add_name'])) != '') {
						sbm::add_module($_POST['add_name'], $_POST['add_type'], $_POST['add_sidebar']);
					} else {
						sbm::set_error_text('You must specify a valid module name');
					}

					break;

				// Update a module
				case 'update':
					if(isset($_POST['sidebar_id']) && isset($_POST['module_id'])) {
						sbm::update_module($_POST['sidebar_id'], $_POST['module_id']);
					} else {
						sbm::set_error_text('Missing sidebar and module ids');
					}

					break;

				// Remove a module from the sidebar
				case 'remove':
					if(isset($_POST['sidebar_id']) && isset($_POST['module_id'])) {
						sbm::remove_module($_POST['sidebar_id'], $_POST['module_id']);
					} else {
						sbm::set_error_text('Missing sidebar and module ids');
					}

					break;

				// Re-order the modules in the sidebar
				case 'reorder':
					if(isset($_POST['sidebar_ordering'])) {
						sbm::reorder_sidebar($_POST['sidebar_ordering']);
					} else {
						sbm::set_error_text('Missing ordering data');
					}

					break;

				// Error
				default:
					sbm::set_error_text('Invalid call');
					break;
			}

			if($sbm_error_text != null) {
				echo('<error>' . $sbm_error_text . '</error>');
				echo(false);
			} else {
				echo(true);
			}

			// End the response
			echo('</response>');

			// Safeguard
			wp_cache_flush();
		}
	}

	/**
	 * The pre-bootstrap function
	 **/
	function pre_bootstrap() {
		// Load the modules
		sbm::load_modules();

		// Scan for in-built modules
		sbm::module_scan();

	}

	/**
	 * The post-bootstrap function
	 **/
	function post_bootstrap() {
		// Allow the Widgets and SBM defined in plugins & themes to be loaded
		do_action('sbm_init');
		do_action('widgets_init');
	}

	/**
	 * The activation function
	 **/
	function activate() {
		// Install
		sbm::install();
	}

	/**
	 * The deactivation function
	 **/
	function deactivate() {
		sbm::uninstall();
	}

	/**
	 * The function to add menus
	 **/
	function add_menus() {
		// Add the submenus
		add_submenu_page('themes.php', __('Sidebar Modules'), __('Sidebar Modules'), 5, 'sbm-modules', array('sbm', 'module_admin'));
	}

	/**
	 * The function to show the module admin interface
	 **/
	function module_admin() {
		global $sbm_registered_sidebars, $sbm_registered_modules;

		if(count($sbm_registered_sidebars) == 0) {
		?>
			<div class="wrap">現在使用中のテーマはサイドバーモジュールと互換性がありません。</div>
		<?php
		} elseif(count($sbm_registered_modules) == 0) {
		?>
			<div class="wrap">インストールまたは有効になっているモジュールやウィジットはありません。</div>
		<?php
		} else {
			include(SBMPLUGINPATH . '/display/module-form.php');
		}
	}

	/**
	 * The function to add stuff to the WP admin header
	 **/
	function module_admin_head() {
	?>
		<link type="text/css" rel="stylesheet" href="<?php sbm::output_url(); ?>/css/sbm.css" />

		<script type="text/javascript">
			//<![CDATA[
				var sbm_baseUrl = '<?php sbm::output_url(); ?>/sbm.php';
			//]]>
		</script>

		<script type="text/javascript" src="<?php sbm::output_url(); ?>/js/prototype.js"></script>
		<script type="text/javascript" src="<?php sbm::output_url(); ?>/js/effects.js"></script>
		<script type="text/javascript" src="<?php sbm::output_url(); ?>/js/dragdrop.js"></script>
		<script type="text/javascript" src="<?php sbm::output_url(); ?>/js/sbm.js"></script>
	<?php
	}

	/**
	 * The function to output the URL of the plugin's files
	 **/
	function output_url() {
		echo(sbm::get_url());
	}

	/**
	 * The function to get the URL of the plugin's files
	 **/
	function get_url() {
		static $plugin_url;

		// Only re-calculate this URL if required
		if(!isset($plugin_url)) {
			$abspath = (DIRECTORY_SEPARATOR != '/') ? str_replace(DIRECTORY_SEPARATOR, '/', ABSPATH) : ABSPATH;
			$plugin_url = str_replace($abspath, get_settings('siteurl').'/', SBMPLUGINPATH);
		}

		return $plugin_url;
	}

	/**
	 * The function to set the XML error text for a direct request
	 *
	 * $text - The new error text
	 **/
	function set_error_text($text) {
		global $sbm_error_text;

		$sbm_error_text = $text;
	}


	/* Sidebar functions **********************************************************************/

	/**
	 * Function to register a sidebar
	 *
	 * $args - Sidebar's arguments
	 **/
	function register_sidebar($args = array()) {
		global $sbm_registered_sidebars;

		// Just in case they have not yet been loaded
		sbm::load_modules();

		// Apparently, WPW lets you pass arguments as a string
		if(is_string($args)) {
			parse_str($args, $args);
		}

		// Check the default arguments are there
		$args['name'] = isset($args['name']) ? $args['name'] : sprintf(__('Sidebar %d'), count($sbm_registered_sidebars) + 1);
		$args['before_widget'] = isset($args['before_widget']) ? $args['before_widget'] : '<li id="%1$s" class="widget %2$s">';
		$args['after_widget'] = isset($args['after_widget']) ? $args['after_widget'] : "</li>\n";
		$args['before_title'] = isset($args['before_title']) ? $args['before_title'] : '<h2 class="widgettitle">';
		$args['after_title'] = isset($args['after_title']) ? $args['after_title'] : "</h2>\n";

		$sidebar = new sbmSidebar($args['name'], $args['before_widget'], $args['after_widget'], $args['before_title'], $args['after_title']);

		// Add the sidebar to the list
		$sbm_registered_sidebars[$sidebar->id] = $sidebar;
	}

	/**
	 * Function to unregister a sidebar
	 **/
	function unregister_sidebar($name) {
		global $sbm_registered_sidebars;

		$id = sbm::name_to_id($name);

		// Unregister the sidebar
		unset($sbm_registered_sidebars[$id]);
	}

	/**
	 * Function to get the list of sidebars
	 **/
	function get_sidebars() {
		global $sbm_registered_sidebars;

		return $sbm_registered_sidebars;
	}

	/**
	 * Function to register n sidebars
	 *
	 * $count - # of sidebars to register
	 * $args - Arguments for the sidebars
	 **/
	function register_sidebars($count = 1, $args = array()) {
		// Apparently, WPW lets you pass arguments as a string
		if(is_string($args)) {
			parse_str($args, $args);
		}

		// Check for a name
		$arg_name = isset($args['name']) ? $args['name'] : __('Sidebar %d');

		// Check there is a count in the name
		if(!strstr($arg_name, '%d')) {
			$arg_name += __(' %d');
		}

		// Register the sidebars
		for($i = 0; $i < $count; $i++) {
			$args['name'] = sprintf($arg_name, $i + 1);

			register_sidebar($args);
		}
	}

	/**
	 * Function to show a sidebar
	 *
	 * $name - Name of the sidebar to show
	 **/
	function dynamic_sidebar($name = 1) {
		global $sbm_registered_sidebars;

		$return = false;

		if(count($sbm_registered_sidebars) > 0) {
			// Check if this is an integer ID of a sidebar
			if(is_int($name)) {
				$name = __('Sidebar ' . $name);
			}

			// Get the sidebar
			$id = sbm::name_to_id($name);

			if(isset($sbm_registered_sidebars[$id])) {
				$return = $sbm_registered_sidebars[$id]->display();
			}
		}

		return $return;
	}


	/* Module functions ***********************************************************************/

	/**
	 * Function to load modules
	 **/
	function load_modules() {
		global $sbm_active_modules, $sbm_disabled_modules;

		if(empty($sbm_active_modules) && empty($sbm_disabled_modules)) {
			$sbm_active_modules = get_option('sbm_modules_active');
			$sbm_disabled_modules = get_option('sbm_modules_disabled');
		}
	}

	/**
	 * Function to save modules
	 **/
	function save_modules() {
		global $sbm_active_modules, $sbm_disabled_modules;

		update_option('sbm_modules_active', $sbm_active_modules);
		update_option('sbm_modules_disabled', $sbm_disabled_modules);
	}

	/**
	 * Function to register a module function
	 *
	 * $name - Module's name
	 * $callback - Callback function
	 * $css_class - The CSS class of this module
	 * $options - The module's default options
	 **/
	function register_sidebar_module($name, $callback, $css_class = '', $options = array()) {
		global $sbm_registered_modules;

		// Another odd bit of WPW code
		// Better include it for the sake of Widget developers
		if(is_array($name)) {
			$id = sbm::name_to_id(sprintf($name[0], $name[2]));
			$name = sprintf(__($name[0], $name[1]), $name[2]);
		} else {
			$id = sbm::name_to_id($name);
			$name = __($name);
		}

		$css_class = (string)$css_class == '' ? (string)$callback : $css_class;

		// Add the module to the array
		$sbm_registered_modules[$id] = array(
			'name' => $name,
			'callback' => $callback,
			'control_callback' => '',
			'css_class' => $css_class,
			'options' => $options
		);
	}

	/**
	 * Function to unregister a module
	 *
	 * $name - Name of the module to unregister
	 **/
	function unregister_sidebar_module($name) {
		global $sbm_registered_modules;

		$id = sbm::name_to_id($name);

		// Unset the module
		unset($sbm_registered_modules[$id]);
	}

	/**
	 * Function to check if a module is active
	 *
	 * $callback - Callback to check for
	 **/
	function is_active_module($callback) {
		global $sbm_registered_modules, $sbm_active_modules, $wp_query;

		$active = false;

		$tmp_modules = array_values($sbm_active_modules);

		// Check if a module with this callback is active
		for($i = 0; $i < count($tmp_modules) && !$active; $i++) {
			for($j = 0; $j < count($tmp_modules[$i]) && !$active; $j++) {
				$current_module = $tmp_modules[$i][$j];

				// We can only check if the module can be displayed if $wp_query is set
				// Otherwise, just assume it can. Ugly, but true.
				if($sbm_registered_modules[$current_module->type]['callback'] == $callback
					&& (!$wp_query || $current_module->canDisplay())
				) {
					$active = true;
				}
			}
		}

		return $active;
	}

	function register_sidebar_module_control($name, $callback) {
		global $sbm_registered_modules;

		// Another odd bit of WPW code
		// Better include it for the sake of Widget developers
		if(is_array($name)) {
			$id = sbm::name_to_id(sprintf($name[0], $name[2]));
			$name = sprintf(__($name[0], $name[1]), $name[2]);
		} else {
			$id = sbm::name_to_id($name);
			$name = __($name);
		}

		// Add the module control to the array
		if($sbm_registered_modules[$id]) {
			$sbm_registered_modules[$id]['control_callback'] = $callback;
		}
	}

	/**
	 * Function to unregister a module control
	 *
	 * $name - Name of the module control to unregister
	 **/
	function unregister_sidebar_module_control($name) {
		global $sbm_registered_modules;

		$id = sbm::name_to_id($name);

		// Unset the module control
		// Add the module control to the array
		if($sbm_registered_modules[$id]) {
			$sbm_registered_modules[$id]['control_callback'] = '';
		}
	}

	/**
	 * Function to get all the installed modules
	 **/
	function get_installed_modules() {
		global $sbm_registered_modules;

		// Sort the list of registered modules
		asort($sbm_registered_modules);

		// Return the list
		return $sbm_registered_modules;
	}

	/**
	 * Function to scan for all the modules
	 **/
	function module_scan() {
		sbm::module_scan_dir(SBMPLUGINPATH . '/modules/');
	}

	/**
	 * Function to scan a directory for usable modules
	 *
	 * $directory_path - The path to scan
	 **/
	function module_scan_dir($directory_path) {
		// Open the module directory
		$dir = dir($directory_path);

		// Get all the files from the directory
		while(($file = $dir->read()) !== false) {
			// Check the file is a module file
			if(is_file($directory_path . $file) && preg_match('/^(.+)\.php$/i', $file)) {
				// Include the file
				require_once($directory_path . $file);
			}
		}

		// Close the widget directory
		$dir->close();
	}

	/**
	 * Function to get all modules, reguardless of sidebar
	 **/
	function get_all_modules() {
		global $sbm_active_modules, $sbm_disabled_modules;

		$all_modules = array();

		if($sbm_active_modules) {
			foreach($sbm_active_modules as $sidebar_modules) {
				foreach($sidebar_modules as $sidebar_module) {
					$all_modules[$sidebar_module->id] = $sidebar_module;
				}
			}
		}

		if($sbm_disabled_modules) {
			foreach($sbm_disabled_modules as $sidebar_module) {
				$all_modules[$sidebar_module->id] = $sidebar_module;
			}
		}

		return $all_modules;
	}

	/**
	 * Function to add a module
	 *
	 * $name - Name of the module
	 * $type - Type of the module
	 * $sidebar - Sidebar to add the module to
	 **/
	function add_module($name, $type, $sidebar) {
		global $sbm_registered_modules, $sbm_active_modules, $sbm_disabled_modules;

		$module_id = sbm::name_to_id($type);

		// Load the base module
		$base_module = $sbm_registered_modules[$module_id];

		// Check the base module is registered
		if($base_module) {
			// Create the ID for the module
			// Quick & cheap
			$next_id = get_option('sbm_modules_next_id');
			$module_id = 'module-' . $next_id;
			update_option('sbm_modules_next_id', ++$next_id);

			// Create the new module
			$new_module = new sbmModule($module_id, $name, $type, $base_module['options']);

			// Add the module to the list
			if($sidebar == 'disabled') {
				$sbm_disabled_modules[] = $new_module;
			} else {
				$sbm_active_modules[sbm::name_to_id($sidebar)][] = $new_module;
			}

			sbm::save_modules();
		}
	}

	/**
	 * Function to update module
	 *
	 * $sidebar_id - The ID of the sidebar the module resides on
	 * $module_id - The ID of the module itself
	 **/
	function update_module($sidebar_id, $module_id) {
		global $sbm_disabled_modules, $sbm_active_modules;

		// Start the capture
		ob_start();

		if($sidebar_id == 'disabled') {
			foreach($sbm_disabled_modules as $key => $module) {
				if($module->id == $module_id) {
					$sbm_disabled_modules[$key]->displayControl();
				}
			}
		} else {
			foreach($sbm_active_modules[$sidebar_id] as $key => $module) {
				if($module->id == $module_id) {
					$sbm_active_modules[$sidebar_id][$key]->displayControl();
				}
			}
		}

		sbm::save_modules();

		// Junk the capture
		ob_end_clean();
	}

	/**
	 * Function to remove module
	 *
	 * $sidebar_id - The ID of the sidebar the module resides on
	 * $module_id - The ID of the module itself
	 **/
	function remove_module($sidebar_id, $module_id) {
		global $sbm_disabled_modules, $sbm_active_modules;

		if($sidebar_id == 'disabled') {
			foreach($sbm_disabled_modules as $key => $module) {
				if($module->id == $module_id) {
					unset($sbm_disabled_modules[$key]);
				}
			}
		} else {
			foreach($sbm_active_modules[$sidebar_id] as $key => $module) {
				if($module->id == $module_id) {
					unset($sbm_active_modules[$sidebar_id][$key]);
				}
			}
		}

		sbm::save_modules();
	}

	/**
	 * Function to re-order modules
	 *
	 * $ordering - Array of modules' order in sidebars
	 **/
	function reorder_sidebar($ordering) {
		global $sbm_disabled_modules, $sbm_active_modules;

		$all_modules = sbm::get_all_modules();

		$sbm_disabled_modules = array();
		$sbm_active_modules = array();

		foreach($ordering as $sidebar_id => $modules) {
			if($sidebar_id == 'disabled') {
				foreach($modules as $module_id) {
					$sbm_disabled_modules[] = $all_modules[$module_id];
				}
			} else {
				foreach($modules as $module_id) {
					$sbm_active_modules[$sidebar_id][] = $all_modules[$module_id];
				}
			}
		}

		sbm::save_modules();
	}

	/**
	 * Function to output the CSS files of the modules to the header of the site
	 **/
	function output_module_css_files() {
		global $sbm_active_modules;

		$css_files = array();

		foreach($sbm_active_modules as $modules) {
			foreach($modules as $module) {
				// If this module has a CSS file to show, and will be shown on this page
				// then get the file
				if($module->output['css_file'] && $module->canDisplay()) {
					$css_files[] = $module->output['css_file'];
				}
			}
		}

		// Strip duplicates
		$css_files = array_unique($css_files);

		// Output the links
		if($css_files) {
			foreach($css_files as $css_file) {
				echo('<link rel="stylesheet" href="' . $css_file . '" type="text/css" media="screen" />');
			}
		}
	}


	/* Generic functions **********************************************************************/

	/**
	 * Function to change a name into an ID
	 *
	 * $name - Name to convert to an ID
	 **/
	function name_to_id($name) {
		// Use the WP function to do this
		return sanitize_title($name);
	}
}

/**
 * Class to represent a sidebar
 **/
class sbmSidebar {
	var $id;
	var $name;
	var $before_module;
	var $after_module;
	var $before_title;
	var $after_title;

	var $modules;

	/**
	 * Sidebar constructor
	 *
	 * $name - The name of the sidebar
	 * $before_module - The HTML code to display before each module
	 * $after_module - The HTML code to display after each module
	 * $before_title - The HTML code to display before each module title
	 * $after_title - The HTML code to display after each module title
	 **/
	function sbmSidebar($name, $before_module, $after_module, $before_title, $after_title) {
		global $sbm_active_modules;

		// Set the generic data from the parameters
		$this->id = sbm::name_to_id($name);
		$this->name = $name;
		$this->before_module = $before_module;
		$this->after_module = $after_module;
		$this->before_title = $before_title;
		$this->after_title = $after_title;

		$this->modules = array();

		// Load the modules
		if($sbm_active_modules[$this->id]) {
			foreach($sbm_active_modules[$this->id] as $module_id => $module) {
				$this->modules[$module_id] = $module;
			}
		}
	}

	/**
	 * Displays the sidebar by outputting it's modules
	 **/
	function display() {
		// Check there are some modules present
		if(count($this->modules) > 0) {
			$return = false;

			// Output the modules
			foreach($this->modules as $module) {
				$return |= $module->display($this);
			}

			return $return;
		} else {
			return false;
		}
	}
}

/**
 * Class to represent a module
 **/
class sbmModule {
	var $id;
	var $name;
	var $type;

	var $display;
	var $output;
	var $options;

	/**
	 * Module constructor
	 *
	 * $id - The ID of the module
	 * $name - The name of the module
	 * $type - The base type of the module, used for accessing data such as callbacks
	 * $options - The default module options
	 **/
	function sbmModule($id, $name, $type, $options) {
		// Set the generic data from the parameters
		$this->id = $id;
		$this->name = $name;
		$this->type = $type;

		$this->display = array(
			'home' => true,
			'archives' => true,
			'post' => true,
			'post_id' => array('show' => 'show', 'ids' => false),
			'search' => true,
			'pages' => true,
			'page_id' => array('show' => 'show', 'ids' => false),
			'error' => true
		);
		$this->output = array('show_title' => true, 'css_file' => false);
		$this->options = $options;
	}

	/**
	 * Displays a module
	 *
	 * $sidebar - The sidebar the module belongs to
	 **/
	function display($sidebar) {
		global $sbm_registered_modules, $sbm_current_module, $post;
		static $sbm_count_id;

		// Get the base module details
		$base_module = $sbm_registered_modules[$this->type];

		// Check that the function exists & that this module is to be displayed
		if(function_exists($base_module['callback'])) {
			if($this->canDisplay()) {
				$sbm_current_module = $this;
				$id = sbm::name_to_id($this->name);

				$sbm_count_id[$id]++;

				$id = $id . ($sbm_count_id[$id] > 1 ? '-' . $sbm_count_id[$id] : '');

				// Call the display callback
				$params[0] = array(
					'before_module' => sprintf($sidebar->before_module, $id, $base_module['css_class']),
					'after_module' => $sidebar->after_module
				);

				// Allow the user to hide the title, simplest method is to unset the title elements
				if($this->output['show_title']) {
					$params[0]['before_title'] = $sidebar->before_title;
					$params[0]['title'] = $this->name;
					$params[0]['after_title'] = $sidebar->after_title;
				} else {
					$params[0]['before_title'] = '';
					$params[0]['title'] = '';
					$params[0]['after_title'] = '';
				}

				$params[0]['before_widget'] = $params[0]['before_module'];
				$params[0]['after_widget'] = $params[0]['after_module'];
				call_user_func_array($base_module['callback'], $params);

				// Update options in any PHP < 5
				if(version_compare(PHP_VERSION, '5.0') < 0) {
					foreach($sbm_current_module->options as $key => $value) {
						$this->update_option($key, $value);
					}
				}

				$sbm_current_module = false;

				return true;
			}
		} else {
			// Remove this module - it dosn't exist properly
			sbm::remove_module($sidebar->id, $this->id);
		}

		return false;
	}

	/**
	 * Display's the module's controls for editing
	 **/
	function displayControl() {
		global $sbm_registered_modules, $sbm_current_module;

		// Handle default control stuff

		// Handle the module name form
		if(isset($_POST['module_name']) && trim((string)$_POST['module_name']) != '') {
			$this->name = stripslashes((string)$_POST['module_name']);
		} else {
			sbm::set_error_text('You must specify a valid module name');
		}

		// Handle the advanced output options form
		if(isset($_POST['output'])) {
			// Don't set anything...
			foreach($this->output as $key => $value) {
				$this->output[$key] = false;
			}

			// ...unless given
			foreach($_POST['output'] as $key => $value) {
				$this->output[$key] = $value;
			}
		}

		// Handle the module display form
		if(isset($_POST['display'])) {
			// Store the page and post IDs, AJAX mess
			$old_post_id = $this->display['post_id'];
			$old_page_id = $this->display['page_id'];

			// Don't display anything...
			foreach($this->display as $page => $display) {
				$this->display[$page] = false;
			}

			// ...unless specified
			foreach($_POST['display'] as $page => $display) {
				$this->display[$page] = $display;
			}

			// Add the exceptional circumstances, if required
			if(!isset($_POST['display']['post_id'])) {
				$this->display['post_id'] = $old_post_id;
			}

			if(!isset($_POST['display']['page_id'])) {
				$this->display['page_id'] = $old_page_id;
			}
		}

		// Display the generic edit form
		extract(array('module' => $this));
		include(SBMPLUGINPATH . '/display/edit-module-form.php');

		// Get the base module details
		$base_module = $sbm_registered_modules[$this->type];

		if(function_exists($base_module['control_callback'])) {
			$sbm_current_module = $this;

			// Call the control callback
			call_user_func($base_module['control_callback']);

			// Update options in any PHP < 5
			if(version_compare(PHP_VERSION, '5.0') < 0) {
				foreach($sbm_current_module->options as $key => $value) {
					$this->update_option($key, $value);
				}
			}

			$sbm_current_module = false;

			return true;
		} else {
			return false;
		}
	}

	/**
	 * Displays the checkbox list of specific posts this module will appear on
	 **/
	function displayPostList() {
		// Display the generic post list
		extract(array('module' => $this));
		include(SBMPLUGINPATH . '/display/edit-module-posts-form.php');
	?>
		
	<?php
	}

	/**
	 * Displays the checkbox list of specific pages this module will appear on
	 **/
	function displayPageList() {
		// Display the generic post list
		extract(array('module' => $this));
		include(SBMPLUGINPATH . '/display/edit-module-pages-form.php');
	}

	/**
	 * Returns whether this module is to be displayed on the current blog page
	 **/
	function canDisplay() {
		global $post;

		return ($this->display['home'] && is_home())
			|| ($this->display['archives'] && is_archive())
			|| ($this->display['post'] && is_single() && (
				   !$this->display['post_id']['ids']
				|| ($this->display['post_id']['show'] == 'show' && $this->display['post_id']['ids'][$post->ID])
				|| ($this->display['post_id']['show'] == 'hide' && !$this->display['post_id']['ids'][$post->ID]))
			)
			|| ($this->display['search'] && is_search())
			|| ($this->display['pages'] && is_page() && (
				   !$this->display['page_id']['ids']
				|| ($this->display['page_id']['show'] == 'show' && $this->display['page_id']['ids'][$post->ID])
				|| ($this->display['page_id']['show'] == 'hide' && !$this->display['page_id']['ids'][$post->ID]))
			)
			|| ($this->display['error'] && (is_404() || !($post || have_posts()))
		);
	}

	/**
	 * Gets the value of an option relating to this module
	 *
	 * $name - The name of the option
	 **/
	function get_option($name) {
		return $this->options[$name];
	}

	/**
	 * Adds an option relating to this module
	 *
	 * $name - The name of the option
	 * $value - The value of the option
	 **/
	function add_option($name, $value = '') {
		$this->options[$name] = $value;
	}

	/**
	 * Updates an option relating to this module
	 *
	 * $name - The name of the option
	 * $value - The value of the option
	 **/
	function update_option($name, $newvalue) {
		$this->options[$name] = $newvalue;
	}

	/**
	 * Deletes an option relating to this module
	 *
	 * $name - The name of the option
	 **/
	function delete_option($name) {
		unset($this->options[$name]);
	}
}


/* Helper functions *******************************************************************************/

/**
 * Helper function to set an option
 **/
function sbm_get_option($name) {
	global $sbm_current_module;

	return $sbm_current_module->get_option($name);
}

/**
 * Helper function to add an option
 **/
function sbm_add_option($name, $value = '', $description = '', $autoload = 'yes') {
	global $sbm_current_module;

	$sbm_current_module->add_option($name, $value, $description);
}

/**
 * Helper function to update an option
 **/
function sbm_update_option($name, $newvalue) {
	global $sbm_current_module;

	$sbm_current_module->update_option($name, $newvalue);
}

/**
 * Helper function to delete an option
 **/
function sbm_delete_option($name) {
	global $sbm_current_module;

	$sbm_current_module->delete_option($name);
}

/**
 * Helper function to register a sidebar
 **/
function register_sidebar($args = array()) {
	sbm::register_sidebar($args);
}

/**
 * Helper function to unregister a sidebar
 **/
function unregister_sidebar($name) {
	sbm::unregister_sidebar($name);
}

/**
 * Helper function to register n sidebars
 **/
function register_sidebars($count = 1, $args = array()) {
	sbm::register_sidebars($count, $args);
}

/**
 * Helper function to show a sidebar
 **/
function dynamic_sidebar($name = 1) {
	return sbm::dynamic_sidebar($name);
}

/**
 * Helper function to register a module
 **/
function register_sidebar_module($name, $callback, $css_class = '', $options = array()) {
	sbm::register_sidebar_module($name, $callback, $css_class, $options);
}

/**
 * Helper function to unregister a module
 **/
function unregister_sidebar_module($name) {
	sbm::unregister_sidebar_module($name);
}

/**
 * Helper function to check if a module is active
 **/
function is_active_module($callback) {
	return sbm::is_active_module($callback);
}

/**
 * Helper function to register a module's control
 **/
function register_sidebar_module_control($name, $callback) {
	sbm::register_sidebar_module_control($name, $callback);
}

/**
 * Helper function to unregister a module's control
 **/
function unregister_sidebar_module_control($name) {
	sbm::unregister_sidebar_module_control($name);
}



/**
 * Until SBM takes over the world of Wordpress sidebars ;), it's nice to allow Widgets to be transparently allowed.
 * Therefore, these helper functions are wrappers for WPW's hooks that allow SMB to use WPW widgets.
 *
 * Sorry about the daft acronyms. ;)
 **/

/**
 * WPW function to register a widget
 **/
function register_sidebar_widget($name, $callback, $classname = '') {
	sbm::register_sidebar_module($name, $callback, $classname);
}

/**
 * WPW function to unregister a widget
 **/
function unregister_sidebar_widget($name) {
	sbm::unregister_sidebar_module($name);
}

/**
 * WPW function to check if a widget is active
 **/
function is_active_widget($callback) {
	return sbm::is_active_module($callback);
}

/**
 * WPW function to register a widget's control
 **/
function register_widget_control($name, $callback, $width = false, $height = false) {
	// Chop off W & H, not needed
	sbm::register_sidebar_module_control($name, $callback);
}

/**
 * WPW function to unregister a widget's control
 **/
function unregister_widget_control($name, $callback) {
	sbm::unregister_sidebar_module_control($name);
}


/* Check if this is a direct request & load accordingly **************************************************************/

// Fix for PHP CGI
if(!isset($_SERVER['SCRIPT_FILENAME']) || strpos($_SERVER['SCRIPT_FILENAME'], 'php.cgi') == strlen($_SERVER['SCRIPT_FILENAME']) - 7) {
	$_SERVER['SCRIPT_FILENAME'] = $_SERVER['PATH_TRANSLATED'];
}

if(realpath($_SERVER['SCRIPT_FILENAME']) == realpath(SBMPLUGINPATH . '/sbm.php')) {
	require_once(dirname(dirname(dirname(dirname(__FILE__)))) . '/wp-blog-header.php');
	sbm::direct_bootstrap();
} else {
	sbm::wp_bootstrap();
}

?>
