/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.hello;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import woolpack.action.ActionDef;
import woolpack.action.ActionInvoker;
import woolpack.action.ForwardDef;
import woolpack.container.ComponentDef;
import woolpack.container.ComponentScope;
import woolpack.container.ScopeContainer;
import woolpack.dom.ResetId;
import woolpack.dom.DelegateDomExpression;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.FormatId;
import woolpack.dom.Serial;
import woolpack.ee.ActionBuilder;
import woolpack.ee.HttpSessionMap;
import woolpack.ee.ServletContextMap;
import woolpack.ee.ServletInputStreamFactory;
import woolpack.ee.ServletRequestAttributeMap;
import woolpack.ee.ValidatorRuntimeException;
import woolpack.html.ToNode;
import woolpack.samples.ActionDefMaker;
import woolpack.text.RegExpFormat;
import woolpack.utils.InputStreamReaderFactory;
import woolpack.utils.OGE;

/**
 * Hello World Web アプリケーション。
 * @author nakamura
 *
 */
public class HelloServlet extends HttpServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public final Map<String,ComponentDef> componentDefMap;
	public final DelegateDomExpression toNode;
	public final DomExpression domExpression;
	
	public HelloServlet(){
		// コンテナの定義。
		componentDefMap = new HashMap<String,ComponentDef>();
		componentDefMap.put("helloComponent", new ComponentDef(ComponentScope.REQUEST){
			@Override
			protected Object newInstance() {
				return new HelloComponent();
			}
		});

		// アクションの定義
		final ActionDefMaker maker = new ActionDefMaker();
		maker.put("index", new ActionDef(OGE.NULL, new ForwardDef("index", new OGE("container.helloComponent"))));
		final ActionBuilder actionBuilder = new ActionBuilder(new ActionInvoker(maker.getMap()), Arrays.asList("name", "id"));
		
		// サーブレットを使用せずにテストを可能にするため、 HTML ファイルを読む込む定義を後で設定するための定義。
		toNode = new DelegateDomExpression();

		// コンストラクタツリー
		domExpression = new Serial(
				// id から無駄な情報を削除する。
				new FormatId(new RegExpFormat("^.*/([^/]+)$", "$1")),
				// Model を呼び出す。
				actionBuilder.getActionExpression(),
				// HTML ファイルを読み込む。
				toNode,
				// Modelの出力を HTML に流し込む。
				actionBuilder.getAutoUpdateExpression()
		);
	}


	@Override public void init(final ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		
		// サーブレットのリソースから HTML ファイルを読み込む定義。
		toNode.setExpression(new ResetId(new Serial(
				new FormatId(new RegExpFormat("^(.*)$", "/html/sample/hello/$1.html")),
				new ToNode(new InputStreamReaderFactory(new ServletInputStreamFactory(servletConfig.getServletContext()), "UTF-8")))));
	}

	@Override protected void service(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException{
		// リクエストごとに Context 役を生成する。
		final DomContext domContext = new DomContext();
		
		// URI を id として Context 役に設定する。
		domContext.setId(request.getRequestURI());

		// 入力値を Context 役に設定する。
		domContext.setInput(request.getParameterMap());
		
		// 各スコープを Context 役に設定する。
		domContext.setRequest(new ServletRequestAttributeMap(request));
		domContext.setSession(new HttpSessionMap(request.getSession()));
		domContext.setApplication(new ServletContextMap(request.getSession().getServletContext()));
		
		// コンテナを Context役に設定する。好みのコンテナで置き換えることができる。
		domContext.setContainer(new ScopeContainer(domContext.getRequest(), domContext.getSession(), domContext.getApplication(), componentDefMap));
		
		// Context 役を Expression 役に作用させる。
		try{
			domExpression.interpret(domContext);
		}catch(final ValidatorRuntimeException e){
			// 値検証で機能を逸脱するエラーが発生した場合は値検証のコンテキストをダンプ情報として出力する。
			final StringBuilder sb = new StringBuilder();
			e.getContext().appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}catch(final RuntimeException e){
			// 機能を逸脱するエラーが発生した場合はコンテキストをダンプ情報として出力する。
			final StringBuilder sb = new StringBuilder();
			domContext.appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}
		
		// HTMLを出力する。
		final Writer w = new BufferedWriter(new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
		try{
			DomConstants.write(domContext.getNode(), w);
		}finally{
			w.close();
		}
	}
}
