/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.utils.OGE;

public class ValidatorContextTest extends TestCase {

	public void testAdd(){
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.add("message00");
		context.add("message01");
		context.setTmpKey("key1");
		context.add("message10");

		assertTrue(TestUtils.equals("#{\"key0\":{\"message00\",\"message01\"},\"key1\":{\"message10\"}}", 
				context.getMessageMap()));
		assertTrue(TestUtils.equals("{\"message00\",\"message01\",\"message10\"}", 
				context.getMessageList()));
	}

	public void testAppendToInitial() throws IOException{
		final ValidatorContext context = new ValidatorContext();
		final StringBuilder sb = new StringBuilder();
		context.appendTo(sb);
		assertEquals("ValidatorContext dump information:\n,id:\n,inputMap:\n,messageList:[]\n,messageMap:{}" +
				"\n,tmpKey:\n,tmpIndex:0", sb.toString());
	}

	public void testAppendToFull() throws IOException{
		final ValidatorContext context = new ValidatorContext();
		context.setId("id0");
		new OGE("inputMap = #{\"inputKey0\":{\"inputValue0\"}}").getValue(context);
		context.setTmpKey("tmpKey0");
		context.setTmpIndex(1);
		context.add("message0");
		
		final StringBuilder sb = new StringBuilder();
		context.appendTo(sb);
		assertEquals("ValidatorContext dump information:" +
				"\n,id:id0" +
				"\n,inputMap:{inputKey0=[inputValue0]}" +
				"\n,messageList:[message0]" +
				"\n,messageMap:{tmpKey0=[message0]}" +
				"\n,tmpKey:tmpKey0" +
				"\n,tmpIndex:1", sb.toString());
	}
	
	public void testTmp(){
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(new HashMap<String,List<Object>>());
		
		context.setTmpKey("key0");
		assertNull(context.getTmpValue());
		
		context.setTmpValue("value0");
		assertEquals("value0", context.getTmpValue());
		assertTrue(TestUtils.equals("#{\"key0\":{\"value0\"}}", context.getInputMap()));
		
		context.setTmpValue("value1");
		assertEquals("value1", context.getTmpValue());
		assertTrue(TestUtils.equals("#{\"key0\":{\"value1\"}}", context.getInputMap()));
		
		context.setTmpKey("key1");
		assertNull(context.getTmpValue());
		
		context.setTmpValue("value0");
		assertEquals("value0", context.getTmpValue());
		assertTrue(TestUtils.equals("#{\"key0\":{\"value1\"}, \"key1\":{\"value0\"}}", context.getInputMap()));
		
		context.setTmpValue("value1");
		assertEquals("value1", context.getTmpValue());
		assertTrue(TestUtils.equals("#{\"key0\":{\"value1\"}, \"key1\":{\"value1\"}}", context.getInputMap())); 
	}
}
