/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.locale;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Locale;

import junit.framework.TestCase;
import woolpack.utils.MapBuilder;
import woolpack.validator.CountValidator;
import woolpack.validator.ValidatorConstants;
import woolpack.validator.ValidatorContext;
import woolpack.validator.ValidatorExpression;

public class BranchByLocaleValidatorTest extends TestCase {

	public void testConstructor(){
		try{
			new BranchByLocaleValidator(null, new ThreadLocal<Locale>());
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new BranchByLocaleValidator(new HashMap<Locale,ValidatorExpression>(), null);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testNormal(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ThreadLocal<Locale> threadLocal = new ThreadLocal<Locale>();
		final ValidatorExpression expression = new BranchByLocaleValidator(new MapBuilder<Locale,ValidatorExpression>()
				.put(Locale.JAPANESE, counter0)
				.put(Locale.ENGLISH, counter1)
				.getMap(), threadLocal);
		final ValidatorContext context = new ValidatorContext();
		
		threadLocal.set(Locale.JAPANESE);
		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getIntepretCount());
		assertEquals(0, counter1.getIntepretCount());
		
		threadLocal.set(Locale.ENGLISH);
		assertTrue(expression.interpret(context));
		assertEquals(1, counter0.getIntepretCount());
		assertEquals(1, counter1.getIntepretCount());
	}
	
	public void testAppendTo() throws IOException{
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.TRUE);
		final ThreadLocal<Locale> threadLocal = new ThreadLocal<Locale>();
		final ValidatorExpression expression = new BranchByLocaleValidator(new MapBuilder<Locale,ValidatorExpression>()
				.put(Locale.JAPANESE, counter0)
				.put(Locale.ENGLISH, counter1)
				.getMap(), threadLocal);
		
		final StringBuilder sb = new StringBuilder();
		expression.appendTo(sb);
		assertEquals("new BranchByLocaleValidator({\"ja\":new FalseValidator(),\"en\":new TrueValidator()})", sb.toString());
		assertEquals(1, counter0.getAppendToCount());
		assertEquals(1, counter1.getAppendToCount());
		assertTrue(sb == counter0.getSbList().getLast());
		assertTrue(sb == counter1.getSbList().getLast());
	}
	
	public void testAddMessageTo(){
		final CountValidator counter0 = new CountValidator(ValidatorConstants.FALSE);
		final CountValidator counter1 = new CountValidator(ValidatorConstants.FALSE);
		final ThreadLocal<Locale> threadLocal = new ThreadLocal<Locale>();
		final ValidatorExpression expression = new BranchByLocaleValidator(new MapBuilder<Locale,ValidatorExpression>()
				.put(Locale.JAPANESE, counter0)
				.put(Locale.ENGLISH, counter1)
				.getMap(), threadLocal);
		
		final Collection<String> expectedCollection = new ArrayList<String>();
		expression.addMessageTo(expectedCollection);
		assertEquals(1, counter0.getAddMessageToCount());
		assertEquals(1, counter1.getAddMessageToCount());
		assertTrue(expectedCollection == counter0.getCollectionList().getLast());
		assertTrue(expectedCollection == counter1.getCollectionList().getLast());
	}
}
