/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.io.IOException;
import java.util.HashMap;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.utils.OGE;

public class DomContextTest extends TestCase {
	
	public void testCopy(){
		final DomExpression expression = TestUtils.getToNodeForm("<INPUT type=\"text\" name=\"name0\" value=\"value0\" />");
		
		final DomContext context0 = new DomContext();
		context0.setId("id0");
		context0.setConfig(new HashMap<String,Object>());
		context0.setContainer(new HashMap());
		context0.setInput(new HashMap());
		context0.setRequest(new HashMap<String,Object>());
		context0.setSession(new HashMap<String,Object>());
		context0.setApplication(new HashMap<String,Object>());
		expression.interpret(context0);
		
		final DomContext context1 = context0.copy();
		
		assertEquals(context0.getId(), context1.getId());
		assertEquals(context0.getConfig(), context1.getConfig());
		assertEquals(context0.getContainer(), context1.getContainer());
		assertEquals(context0.getInput(), context1.getInput());
		assertEquals(context0.getRequest(), context1.getRequest());
		assertEquals(context0.getSession(), context1.getSession());
		assertEquals(context0.getApplication(), context1.getApplication());
		assertEquals(context0.getNode(), context1.getNode());
	}
	
	public void testTmp(){
		final DomContext context0 = new DomContext();
		context0.getLocal().put("key0", "value0");
		
		final DomContext context1 = context0.copy();
		assertEquals("value0", context1.getLocal().get("key0"));
		
		context1.getLocal().put("key1", "value1");
		assertEquals("value1", context1.getLocal().get("key1"));
		assertNull(context0.getLocal().get("key1"));
		
		context1.getLocal().put("key0", "value2");
		assertEquals("value2", context1.getLocal().get("key0"));
		assertEquals("value0", context0.getLocal().get("key0"));
	}
	
	public void testTmp2(){
		final DomContext context0 = new DomContext();
		context0.getLocal().put("key0", "value0");
		final DomContext context1 = context0.copy();
		
		// ----
		final DomContext context2 = context0.copy();
		assertEquals("value0", context2.getLocal().get("key0"));
		
		context2.getLocal().put("key1", "value1");
		assertEquals("value1", context2.getLocal().get("key1"));
		assertNull(context0.getLocal().get("key1"));
		
		context2.getLocal().put("key0", "value2");
		assertEquals("value2", context2.getLocal().get("key0"));
		assertEquals("value0", context0.getLocal().get("key0"));
		// ----
		
		assertEquals("value0", context1.getLocal().get("key0"));
		
		context1.getLocal().put("key1", "value1");
		assertEquals("value1", context1.getLocal().get("key1"));
		assertNull(context0.getLocal().get("key1"));
		
		context1.getLocal().put("key0", "value2");
		assertEquals("value2", context1.getLocal().get("key0"));
		assertEquals("value0", context0.getLocal().get("key0"));
	}
	
	public void testAppendToInitial() throws IOException{
		final DomContext context = new DomContext();
		final StringBuilder sb = new StringBuilder();
		context.appendTo(sb);
		assertEquals("DomContext dump information:\n,id:\n,config:\n,container:\n,input:" +
				"\n,request:\n,session:\n,application:\n,local:\n,node:", sb.toString());
	}
	
	public void testAppendToFull() throws IOException{
		final DomContext context = new DomContext();
		context.setId("id0");
		new OGE("config = #{\"configKey0\":\"configValue0\"}").getValue(context);
		new OGE("container = #{\"containerKey0\":\"containerValue0\"}").getValue(context);
		new OGE("input = #{\"inputKey0\":{\"inputValue0\"}}").getValue(context);
		new OGE("request = #{\"requestKey0\":\"requestValue0\"}").getValue(context);
		new OGE("session = #{\"sessionKey0\":\"sessionValue0\"}").getValue(context);
		new OGE("application = #{\"applicationKey0\":\"applicationValue0\"}").getValue(context);
		new OGE("local.localKey0 = \"localValue0\"").getValue(context);
		TestUtils.getToNode("<HTML><BODY>hoge</BODY></HTML>").interpret(context);
		
		final StringBuilder sb = new StringBuilder();
		context.appendTo(sb);
		assertEquals("DomContext dump information:" +
				"\n,id:id0" +
				"\n,config:{configKey0=configValue0}" +
				"\n,container:{containerKey0=containerValue0}" +
				"\n,input:{inputKey0=[inputValue0]}" +
				"\n,request:{requestKey0=requestValue0}" +
				"\n,session:{sessionKey0=sessionValue0}" +
				"\n,application:{applicationKey0=applicationValue0}" +
				"\n,local:{localKey0=localValue0}" +
				"\n,node:<HTML>\n<BODY>hoge</BODY>\n</HTML>\n", sb.toString());
	}
}
