/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Arrays;
import java.util.Map;

import woolpack.dom.BranchByAttrValue;
import woolpack.dom.BranchByNodeName;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.EqualsAttrValue;
import woolpack.dom.RemoveAttr;
import woolpack.dom.UpdateAttrValue;
import woolpack.dom.XPath;
import woolpack.utils.MapBuilder;
import woolpack.utils.OGE;
import woolpack.utils.UtilsConstants;


/**
 * 値を再生成する{@link DomExpression}。
 * ノードの種類がラジオボタン・チェックボックス・セレクトで selectFlag の場合、selected 属性の有無または checked 属性の有無を変更する。
 * ノードの種類がラジオボタン・チェックボックス・セレクトで selectFlag でないか、
 * ラジオボタン・チェックボックス・テキストエリア以外の入力部品の場合、value 属性値を変更する。
 * ノードの種類が入力部品以外であるかテキストエリアの場合、子ノードを値のテキストで置き換える。
 * @author nakamura
 *
 */
public class UpdateValue implements DomExpression {
	private static final String VALUE = "value";
	private static final String SELECTED = "selected";
	private static final String CHECKED = "checked";
	private static final OGE TRUE_OGE = new OGE("\"true\"");

	private final DomExpression root;
	
	/**
	 * コンストラクタ。
	 * @param valueOGE 値の取得先。
	 * @param mapOGE 値とラベルの{@link Map}の取得先。
	 * @param selectFlag selected または checked 属性の有無を変更するなら true。value の属性値を変更するなら false。
	 * @throws NullPointerException valueOGE が null の場合。
	 */
	public UpdateValue(
			final OGE valueOGE, 
			final OGE mapOGE, 
			final boolean selectFlag){
		
		final DomExpression updateValueAttr = new UpdateAttrValue(VALUE, valueOGE);
		
		final DomExpression expressionForOption = selectFlag?
				new EqualsAttrValue(
						VALUE, 
						valueOGE, 
						new UpdateAttrValue(SELECTED, TRUE_OGE), 
						new RemoveAttr(SELECTED))
				:updateValueAttr;
		
						// TODO buttonのテスト。
						// TODO passwordの取り扱いの検討。
		root = new BranchByNodeName(UtilsConstants.defaultValueMap(
				new MapBuilder<String,DomExpression>()
				.put("OPTION", expressionForOption)
				.put("SELECT", new XPath(HtmlConstants.XPATH_OPTION, expressionForOption))
				.put("INPUT", selectFlag?
						new BranchByAttrValue(
								Arrays.asList("type"), 
								new MapBuilder<String,DomExpression>()
								.put("radio", new EqualsAttrValue(
										VALUE, 
										valueOGE, 
										new UpdateAttrValue(CHECKED, TRUE_OGE), 
										new RemoveAttr(CHECKED)))
								.put("checkbox")
								.getMap(), 
								updateValueAttr)
						:updateValueAttr)
				.put("BUTTON", updateValueAttr)
				.getMap(), 
				new UpdateToSelectedValue(valueOGE, mapOGE)));
	}
	
	/**
	 * コンストラクタ。
	 * @param valueOGE 値の取得先。
	 * @param mapOGE 値とラベルの{@link Map}の取得先。
	 * @throws NullPointerException valueOGE が null の場合。
	 */
	public UpdateValue(final OGE valueOGE, final OGE mapOGE){
		this(valueOGE, mapOGE, true);
	}
	
	/**
	 * コンストラクタ。
	 * @param valueOGE 値の取得先。
	 * @throws NullPointerException valueOGE が null の場合。
	 */
	public UpdateValue(final OGE valueOGE){
		this(valueOGE, null, true);
	}
	
	public void interpret(final DomContext context) {
		root.interpret(context);
	}
}
