/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Map;


import org.w3c.dom.Node;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.utils.UtilsConstants;
import woolpack.utils.OGE;

/**
 * 値に対応するラベルを再生成する{@link DomExpression}。
 * 値が複数の場合は複数のラベルを再生成する。
 * {@link Map}が存在しない場合または{@link Map}に対応するキーが存在しない場合は値をそのまま表示する。
 * 前の画面のラジオボタン・チェックボックス・セレクトで選択した値を確認画面で表示するために使用する。
 * @author nakamura
 *
 */
public class UpdateToSelectedValue implements DomExpression {
	private final OGE valueOGE;
	private final OGE mapOGE;
	
	/**
	 * コンストラクタ。
	 * mapOGE は null を許可する。{@link Map}が存在しない場合または{@link Map}に対応するキーが存在しない場合は値をそのまま表示する。
	 * @param valueOGE 値の取得先。
	 * @param mapOGE 値とラベルの{@link Map}の取得先。キー・値とも{@link Object#toString()}で文字列として扱う。
	 * @throws NullPointerException valueOGE が null の場合。
	 */
	public UpdateToSelectedValue(final OGE valueOGE, final OGE mapOGE){
		valueOGE.getClass();
		this.valueOGE = valueOGE;
		this.mapOGE = mapOGE;
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}または取得した{@link Map}のキーか値が null の場合。
	 * 
	 */
	public void interpret(final DomContext context) {
		final Object value1 = valueOGE.getValue(context);
		final Iterable iterable = UtilsConstants.toIterable(value1);
		for(final Object value:iterable){
			if(value == null){
				continue;
			}
			final String valueString = value.toString();
			
			final Node newNode = context.getNode().cloneNode(true);
			DomConstants.removeChildren(newNode);
			try{
				final Map map = (Map)mapOGE.getValue(context);
				if(map.containsKey(valueString)){
					// mapOGE が null の場合もここに処理が遷移する。
					DomConstants.appendText(newNode, map.get(valueString).toString());
				}
			}catch(final RuntimeException e){
				DomConstants.appendText(newNode, valueString);
			}
			context.getNode().getParentNode().insertBefore(newNode, context.getNode());
		}
		DomConstants.removeThis(context.getNode());
	}
}
