/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import woolpack.action.ActionInvoker;
import woolpack.action.ActionResult;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.html.AutoUpdate;
import woolpack.utils.OGE;
import woolpack.utils.UtilsConstants;

/**
 * アクションを実行する{@link DomExpression}・
 * アクションの実行結果を DOM ノードに自動生成する{@link DomExpression}・
 * アクションが返したエラーのメッセージを DOM ノードに設定する{@link DomExpression}の生成器。
 * @author nakamura
 *
 */
public class ActionBuilder {

	/**
	 * 遷移先のコンポーネントの取得先のデフォルト値。
	 */
	public static final OGE DEFAULT_FORWARD_COMPONENT_OGE_OGE = new OGE("local[\"woolpack.ee.FORWARD_COMPONENT\"]");
	
	/**
	 * 返却値の取得先のデフォルト値。
	 */
	public static final OGE DEFAULT_RETURN_OGE = new OGE("local.returnMessages");
	
	private final ActionInvoker actionDefs;
	private final Iterable<String> attrNames;
	private final OGE forwardComponentOGEOGE;
	private final OGE returnOGE;

	/**
	 * コンストラクタ。
	 * @param actionDefs アクション定義の一覧と遷移先定義の一覧の集合。
	 * @param attrNames 属性名の一覧。
	 * @param forwardComponentOGEOGE 遷移先のコンポーネントの取得先。
	 * @param returnOGE 返却値の取得先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ActionBuilder(
			final ActionInvoker actionDefs, 
			final Iterable<String> attrNames,
			final OGE forwardComponentOGEOGE,
			final OGE returnOGE){
		actionDefs.getClass();
		attrNames.getClass();
		forwardComponentOGEOGE.getClass();
		returnOGE.getClass();
		
		this.actionDefs = actionDefs;
		this.attrNames = attrNames;
		this.forwardComponentOGEOGE = forwardComponentOGEOGE;
		this.returnOGE = returnOGE;
	}
	

	/**
	 * コンストラクタ。
	 * 遷移先のコンポーネントの取得先として{@link ActionBuilder#DEFAULT_FORWARD_COMPONENT_OGE_OGE}を使用する。
	 * 返却値の取得先として{@link ActionBuilder#DEFAULT_RETURN_OGE}を使用する。
	 * @param actionDefs アクション定義の一覧と遷移先定義の一覧の集合。
	 * @param attrNames 属性名の一覧。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ActionBuilder(final ActionInvoker actionDefs, final Iterable<String> attrNames){
		this(
				actionDefs, 
				attrNames, 
				DEFAULT_FORWARD_COMPONENT_OGE_OGE, 
				DEFAULT_RETURN_OGE);
	}
	
	/**
	 * アクションを実行し実行結果を{@link #getReturnOGE()}の位置に設定する{@link DomExpression}を返す。
	 * @return アクションを実行する{@link DomExpression}。
	 */
	public DomExpression getActionExpression(){
		return new DomExpression(){
			public void interpret(final DomContext context) {
				final ActionResult actionResult = actionDefs.invoke(context, context.getId(), context.getInput());
				context.setId(actionResult.getForwardDef().getId());
				forwardComponentOGEOGE.setValue(context, actionResult.getForwardDef().getComponentOGE());
				if(actionResult.getReturnedObject() instanceof Throwable){
					returnOGE.setValue(context, UtilsConstants.toMessageList((Throwable)actionResult.getReturnedObject()));
				}else{
					returnOGE.setValue(context, actionResult.getReturnedObject());
				}
			}
		};
	}
	
	/**
	 * コンポーネントの取得先をアクションの実行結果から取得し、
	 * 属性値をプロパティ名としてコンポーネントから値を取得しDOMノードに自動設定する{@link DomExpression}を返す。
	 * @return アクションの実行結果を DOM ノードに自動生成する{@link DomExpression}。
	 */
	public DomExpression getAutoUpdateExpression(){
		return new DomExpression(){
			public void interpret(final DomContext context) {
				final OGE o = (OGE)forwardComponentOGEOGE.getValue(context);
				if(o != null){
					new AutoUpdate(attrNames, o).interpret(context);
				}
			}
		};
	}
	
	/**
	 * 遷移先のコンポーネントの取得先を返す。
	 * @return 遷移先のコンポーネントの取得先。
	 */
	public OGE getForwardComponentOGEOGE() {
		return forwardComponentOGEOGE;
	}


	/**
	 * 返却値の取得先を返す。
	 * @return 返却値の取得先。
	 */
	public OGE getReturnOGE() {
		return returnOGE;
	}
}
