/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.Map;

import org.w3c.dom.Element;

/**
 * DOMエレメントの属性値で委譲先を分岐する{@link DomExpression}。
 * @author nakamura
 *
 */
public class BranchByAttrValue implements DomExpression {
	private final Iterable<String> attrNames;
	private final Map<String,DomExpression> map;
	private final DomExpression other;
	
	/**
	 * コンストラクタ。
	 * 一致する属性値が{@link Map}に存在しない場合はなにもしない。
	 * @param attrNames 属性名の一覧。
	 * @param map 属性値と委譲先の{@link Map}。ステートレスであるべき。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public BranchByAttrValue(final Iterable<String> attrNames, final Map<String,DomExpression> map){
		this(attrNames, map, DomConstants.NULL);
	}
	
	/**
	 * コンストラクタ。
	 * @param attrNames 属性名の一覧。
	 * @param map 属性値と委譲先の{@link Map}。ステートレスであるべき。
	 * @param other 一致する属性値が{@link Map}に存在しない場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public BranchByAttrValue(final Iterable<String> attrNames, final Map<String,DomExpression> map, final DomExpression other){
		attrNames.getClass();
		map.getClass();
		other.getClass();
		
		this.attrNames = attrNames;
		this.map = map;
		this.other = other;
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}が null の場合。
	 * @throws ClassCastException {@link DomContext#getNode()}が{@link Element}でない場合。
	 * 
	 */
	public void interpret(final DomContext context) {
		final Element element = (Element)context.getNode();
		for(final String attrName:attrNames){
			final String value = element.getAttribute(attrName);
			if(value != null){
				final DomExpression expression = map.get(value);
				if(expression != null){
					expression.interpret(context);
					return;
				}
			}
		}
		other.interpret(context);
	}
}
