/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import woolpack.utils.OGE;

/**
 * 遷移先の定義。
 * 本クラスのインスタンスは複数のスレッドで同時に使用できる。
 * @author nakamura
 *
 */
public class ForwardDef {
	private final String id;
	private final OGE componentOGE;
	private final ForwardMatchable matchable;

	/**
	 * コンストラクタ。
	 * @param id 遷移先 id。
	 * @param componentOGE 遷移先のコンポーネントの取得先。
	 * @param matchable マッチ操作。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id, final OGE componentOGE, final ForwardMatchable matchable){
		id.charAt(0);
		componentOGE.getClass();
		matchable.getClass();
		
		this.id = id;
		this.componentOGE = componentOGE;
		this.matchable = matchable;
	}

	/**
	 * コンストラクタ。
	 * 遷移先のコンポーネントの取得先を{@link OGE#NULL}とする。
	 * @param id 遷移先 id。
	 * @param matchable マッチ操作。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id, final ForwardMatchable matchable){
		this(id, OGE.NULL, matchable);
	}

	/**
	 * コンストラクタ。
	 * マッチ操作を{@link ActionConstants#NOT_THROWABLE}とする。
	 * @param id 遷移先 id。
	 * @param componentOGE 遷移先のコンポーネントの取得先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id, final OGE componentOGE){
		this(id, componentOGE, ActionConstants.NOT_THROWABLE);
	}

	/**
	 * コンストラクタ。
	 * 遷移先のコンポーネントの取得先を{@link OGE#NULL}とする。
	 * マッチ操作を{@link ActionConstants#NOT_THROWABLE}とする。
	 * @param id 遷移先 id。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id){
		this(id, OGE.NULL, ActionConstants.NOT_THROWABLE);
	}
	
	/**
	 * 遷移先 id を返す。
	 * @return 遷移先 id。
	 */
	public String getId(){
		return id;
	}

	/**
	 * 遷移先のコンポーネントの取得先を返す。
	 * @return 遷移先のコンポーネントの取得先。
	 */
	public OGE getComponentOGE() {
		return componentOGE;
	}
	
	/**
	 * マッチ操作を返す。
	 * @return マッチ操作。
	 */
	public ForwardMatchable getMatchable(){
		return matchable;
	}
}
