/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;

import woolpack.utils.OGE;


/**
 * アクションの定義。
 * @author nakamura
 *
 */
public class ActionDef {
	
	private final OGE componentOGE;
	private final OGE methodOGE;
	private final Iterable<ForwardDef> forwardDefIterable;
	
	/**
	 * コンストラクタ。
	 * @param componentOGE コンポーネントの取得先。
	 * @param methodOGE 実行するメソッドの取得先。
	 * @param forwardDefIterable 遷移先定義の一覧。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final OGE componentOGE, final OGE methodOGE, final Iterable<ForwardDef> forwardDefIterable){
		componentOGE.getClass();
		methodOGE.getClass();
		forwardDefIterable.getClass();
		
		this.componentOGE = componentOGE;
		this.methodOGE = methodOGE;
		this.forwardDefIterable = forwardDefIterable;
	}

	/**
	 * コンストラクタ。
	 * コンポーネントの取得先は null を表す{@link OGE}とする。
	 * @param methodOGE 実行するメソッドの取得先。
	 * @param forwardDefIterable 遷移先定義の一覧。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final OGE methodOGE, final Iterable<ForwardDef> forwardDefIterable){
		this(OGE.NULL, methodOGE, forwardDefIterable);
	}
	
	/**
	 * コンストラクタ。
	 * @param componentOGE コンポーネントの取得先。
	 * @param methodOGE 実行するメソッドの取得先。
	 * @param forwardDefArray 遷移先定義の一覧。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final OGE componentOGE, final OGE methodOGE, final ForwardDef... forwardDefArray){
		this(componentOGE, methodOGE, Arrays.asList(forwardDefArray));
	}

	/**
	 * コンストラクタ。
	 * コンポーネントの取得先は null を表す{@link OGE}とする。
	 * @param methodOGE 実行するメソッドの取得先。
	 * @param forwardDefArray 遷移先定義の一覧。ステートレスであるべき。
	 * @throws 引数のいずれかが null の場合。
	 */
	public ActionDef(final OGE methodOGE, final ForwardDef... forwardDefArray){
		this(OGE.NULL, methodOGE, forwardDefArray);
	}
	
	/**
	 * コンポーネントの位置を返す。
	 * @return コンポーネントの位置。
	 */
	public OGE getComponentOGE() {
		return componentOGE;
	}

	/**
	 * メソッドの位置を返す。
	 * @return メソッドの位置。
	 */
	public OGE getMethodOGE() {
		return methodOGE;
	}

	/**
	 * 遷移先定義の一覧を返す。
	 * @return 遷移先定義の一覧。
	 */
	public Iterable<ForwardDef> getForwardDefIterable() {
		return forwardDefIterable;
	}
}
