! sieve_module.F -*-f90-*-
! Time-stamp: <2007-11-14 16:09:59 takeshi>
! Author: Takeshi NISHIMATSU
!!
#if defined HAVE_CONFIG_H
#  include "config.h"
#endif

module sieve_module
  type sieve
#  if defined(SR11000)
#    define allocated associated
     integer, pointer     :: vec(:) => null()
#  else
     integer, allocatable :: vec(:)
#  endif
     integer*8 :: vec_integer_bit
     integer*8 :: vec_integer_byte
     integer*8 first_odd
     integer*8 last_odd
  end type sieve

  private primep, check_composite_numbers, count_primes

contains
  subroutine eratosthenes(memory_size, n, largest_prime_less_than_n, n_prime)
    ! The return values are correct only if mpi_sp_n_processor==mpi_sp_n_processor-1
    use i64_module
    use mpi_sp_module
    implicit none
    integer*8, intent(inout) :: memory_size
    integer*8, intent(out)   :: n, largest_prime_less_than_n, n_prime
    integer*8 vec_size, vec_bit_length2, sqrt_n, i, n_in_a_sieve
    integer i_error, rank
    logical l_prime
    type(sieve) :: s

    s%vec_integer_bit = bit_size(s%vec)
    s%vec_integer_byte = s%vec_integer_bit/8

    memory_size = (memory_size / s%vec_integer_byte) * s%vec_integer_byte
    if (memory_size<s%vec_integer_byte) memory_size=s%vec_integer_byte

    vec_size = memory_size / s%vec_integer_byte
    vec_bit_length2 = vec_size * s%vec_integer_bit * 2
    n = vec_bit_length2 * mpi_sp_n_processor
    sqrt_n = i64sqrt(n)

    allocate( s%vec(0:vec_size-1) )
    s%vec(:) = 0
    s%first_odd = vec_bit_length2 *  mpi_sp_my_rank    + 1
    s%last_odd  = vec_bit_length2 * (mpi_sp_my_rank+1) - 1

    !write(6,'(a,i3,a,i10,a,i10)') 'rank=', mpi_sp_my_rank, '     first_odd=', s%first_odd, '     last_odd=', s%last_odd
    call flush(6)
    call MPI_Barrier(MPI_COMM_WORLD, i_error)

    do i = 3, sqrt_n, 2
       rank = i/vec_bit_length2
       if (rank.eq.mpi_sp_my_rank) l_prime=primep(s,i)
       call MPI_Barrier(MPI_COMM_WORLD, i_error)
#    ifdef MPI_PARALLEL
       call MPI_Bcast(l_prime, 1, MPI_LOGICAL, rank, MPI_COMM_WORLD, i_error)
#    endif
       if (l_prime) call check_composite_numbers(s,i)
       call MPI_Barrier(MPI_COMM_WORLD, i_error)
    end do

    call MPI_Barrier(MPI_COMM_WORLD, i_error)
    n_in_a_sieve = count_primes(s)
    call MPI_Barrier(MPI_COMM_WORLD, i_error)
# ifdef MPI_PARALLEL
    call MPI_Reduce(n_in_a_sieve, n_prime, 1, MPI_INTEGER8, MPI_SUM, mpi_sp_n_processor-1, MPI_COMM_WORLD, i_error)
# else
    n_prime = n_in_a_sieve
# endif
    if (mpi_sp_my_rank.eq.mpi_sp_n_processor-1) largest_prime_less_than_n = largest(s)
  end subroutine eratosthenes

  integer*8 function largest(s)
    use i64_module
    implicit none
    type(sieve), intent(in) :: s
    integer*8               :: i
    largest = 0_i64
    do i = s%last_odd, s%first_odd, -2
       if (primep(s,i)) then
          largest = i
          return
       end if
    end do
  end function largest

  integer*8 function count_primes(s)
    use i64_module
    implicit none
    type(sieve), intent(in) :: s
    integer*8               :: i
    count_primes = 0_i64
    do i = s%first_odd, s%last_odd, 2
       if (primep(s,i)) count_primes = count_primes + 1_i64
    end do
  end function count_primes

  logical function primep(s,i)
    use i64_module
    implicit none
    type(sieve), intent(in) :: s
    integer*8,   intent(in) :: i
    integer*8 bit_position
    bit_position = (i-s%first_odd)/2
    if ( iand(          s%vec(bit_position/s%vec_integer_bit), &
              ishft(1,int(mod(bit_position,s%vec_integer_bit))) ) .eq. 0 ) then
       primep = .true.
    else
       primep = .false.
    end if
  end function primep

  subroutine check_composite_numbers(s,i)
    use i64_module
    implicit none
    type(sieve), intent(inout) :: s
    integer*8 i,j,k
    integer*8 bit_position
    do j = i*i, s%last_odd, i*2
       if (j < s%first_odd) cycle
       bit_position = (j-s%first_odd)/2
       k = bit_position/s%vec_integer_bit
       s%vec(k) = ior(s%vec(k),ishft(1,int(mod(bit_position,s%vec_integer_bit))))
    end do
  end subroutine check_composite_numbers

end module sieve_module
