/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    playmidi.c -- random stuff in need of rearrangement

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef __W32__
#include "interface.h"
#endif
#include <stdio.h>
#include <stdlib.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <math.h>
#ifdef __W32__
#include <windows.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include "timidity.h"
#include "common.h"
#include "tables.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "output.h"
#include "mix.h"
#include "controls.h"
#include "miditrace.h"
#include "recache.h"
#include "arc.h"
#include "reverb.h"
#include "wrd.h"
#include "aq.h"
#include "freq.h"
#include "quantity.h"
#include "timidity_main.h"
#include "m2m.h"

#include "playmidi_prv.h" //guha

/* m2m.c */
extern void convert_mod_to_midi_file(tmdy_struct_ex_t *tmdy_struct, MidiEvent * ev);

#define ABORT_AT_FATAL 1 /*#################*/
#define MYCHECK(s) do { if(s == 0) { printf("## L %d\n", __LINE__); abort(); } } while(0)

//extern VOLATILE int intr;

/* #define SUPPRESS_CHANNEL_LAYER */

#ifdef SOLARIS
/* shut gcc warning up */
int usleep(unsigned int useconds);
#endif

#ifdef SUPPORT_SOUNDSPEC
#include "soundspec.h"
#endif /* SUPPORT_SOUNDSPEC */

#include "tables.h"

#define PLAY_INTERLEAVE_SEC		1.0
#define PORTAMENTO_TIME_TUNING		(1.0 / 5000.0)
#define PORTAMENTO_CONTROL_RATIO	256	/* controls per sec */
#define DEFAULT_CHORUS_DELAY1		0.02
#define DEFAULT_CHORUS_DELAY2		0.003
#define CHORUS_OPPOSITE_THRESHOLD	32
#define CHORUS_VELOCITY_TUNING1		0.7
#define CHORUS_VELOCITY_TUNING2		0.6
#define EOT_PRESEARCH_LEN		32
#define SPEED_CHANGE_RATE		1.0594630943592953  /* 2^(1/12) */


static void set_reverb_level(tmdy_struct_ex_t *tmdy_struct, int ch, int level);

/* for auto amplitude compensation */


static void update_portamento_controls(tmdy_struct_ex_t *tmdy_struct, int ch);
static void update_rpn_map(tmdy_struct_ex_t *tmdy_struct, int ch, int addr, int update_now);
static void ctl_prog_event(tmdy_struct_ex_t *tmdy_struct, int ch);
static void ctl_timestamp(tmdy_struct_ex_t *tmdy_struct);
static void ctl_updatetime(tmdy_struct_ex_t *tmdy_struct, int32 samples);
static void ctl_pause_event(tmdy_struct_ex_t *tmdy_struct, int pause, int32 samples);
static void update_legato_controls(tmdy_struct_ex_t *tmdy_struct, int ch);
static void update_channel_freq(tmdy_struct_ex_t *tmdy_struct, int ch);
static void set_single_note_tuning(tmdy_struct_ex_t *tmdy_struct, int, int, int, int);

#define IS_SYSEX_EVENT_TYPE(type) ((type) == ME_NONE || (type) >= ME_RANDOM_PAN)

static char *event_name(tmdy_struct_ex_t *tmdy_struct, int type)
{
#define EVENT_NAME(X) case X: return #X
	switch (type) {
	EVENT_NAME(ME_NONE);
	EVENT_NAME(ME_NOTEOFF);
	EVENT_NAME(ME_NOTEON);
	EVENT_NAME(ME_KEYPRESSURE);
	EVENT_NAME(ME_PROGRAM);
	EVENT_NAME(ME_CHANNEL_PRESSURE);
	EVENT_NAME(ME_PITCHWHEEL);
	EVENT_NAME(ME_TONE_BANK_MSB);
	EVENT_NAME(ME_TONE_BANK_LSB);
	EVENT_NAME(ME_MODULATION_WHEEL);
	EVENT_NAME(ME_BREATH);
	EVENT_NAME(ME_FOOT);
	EVENT_NAME(ME_MAINVOLUME);
	EVENT_NAME(ME_BALANCE);
	EVENT_NAME(ME_PAN);
	EVENT_NAME(ME_EXPRESSION);
	EVENT_NAME(ME_SUSTAIN);
	EVENT_NAME(ME_PORTAMENTO_TIME_MSB);
	EVENT_NAME(ME_PORTAMENTO_TIME_LSB);
	EVENT_NAME(ME_PORTAMENTO);
	EVENT_NAME(ME_PORTAMENTO_CONTROL);
	EVENT_NAME(ME_DATA_ENTRY_MSB);
	EVENT_NAME(ME_DATA_ENTRY_LSB);
	EVENT_NAME(ME_SOSTENUTO);
	EVENT_NAME(ME_SOFT_PEDAL);
	EVENT_NAME(ME_LEGATO_FOOTSWITCH);
	EVENT_NAME(ME_HOLD2);
	EVENT_NAME(ME_HARMONIC_CONTENT);
	EVENT_NAME(ME_RELEASE_TIME);
	EVENT_NAME(ME_ATTACK_TIME);
	EVENT_NAME(ME_BRIGHTNESS);
	EVENT_NAME(ME_REVERB_EFFECT);
	EVENT_NAME(ME_TREMOLO_EFFECT);
	EVENT_NAME(ME_CHORUS_EFFECT);
	EVENT_NAME(ME_CELESTE_EFFECT);
	EVENT_NAME(ME_PHASER_EFFECT);
	EVENT_NAME(ME_RPN_INC);
	EVENT_NAME(ME_RPN_DEC);
	EVENT_NAME(ME_NRPN_LSB);
	EVENT_NAME(ME_NRPN_MSB);
	EVENT_NAME(ME_RPN_LSB);
	EVENT_NAME(ME_RPN_MSB);
	EVENT_NAME(ME_ALL_SOUNDS_OFF);
	EVENT_NAME(ME_RESET_CONTROLLERS);
	EVENT_NAME(ME_ALL_NOTES_OFF);
	EVENT_NAME(ME_MONO);
	EVENT_NAME(ME_POLY);
#if 0
	EVENT_NAME(ME_VOLUME_ONOFF);		/* Not supported */
#endif
	EVENT_NAME(ME_SCALE_TUNING);
	EVENT_NAME(ME_BULK_TUNING_DUMP);
	EVENT_NAME(ME_SINGLE_NOTE_TUNING);
	EVENT_NAME(ME_RANDOM_PAN);
	EVENT_NAME(ME_SET_PATCH);
	EVENT_NAME(ME_DRUMPART);
	EVENT_NAME(ME_KEYSHIFT);
	EVENT_NAME(ME_PATCH_OFFS);
	EVENT_NAME(ME_TEMPO);
	EVENT_NAME(ME_CHORUS_TEXT);
	EVENT_NAME(ME_LYRIC);
	EVENT_NAME(ME_GSLCD);
	EVENT_NAME(ME_MARKER);
	EVENT_NAME(ME_INSERT_TEXT);
	EVENT_NAME(ME_TEXT);
	EVENT_NAME(ME_KARAOKE_LYRIC);
	EVENT_NAME(ME_MASTER_VOLUME);
	EVENT_NAME(ME_RESET);
	EVENT_NAME(ME_NOTE_STEP);
	EVENT_NAME(ME_TIMESIG);
	EVENT_NAME(ME_KEYSIG);
	EVENT_NAME(ME_TEMPER_KEYSIG);
	EVENT_NAME(ME_TEMPER_TYPE);
	EVENT_NAME(ME_MASTER_TEMPER_TYPE);
	EVENT_NAME(ME_SYSEX_LSB);
	EVENT_NAME(ME_SYSEX_MSB);
	EVENT_NAME(ME_SYSEX_GS_LSB);
	EVENT_NAME(ME_SYSEX_GS_MSB);
	EVENT_NAME(ME_SYSEX_XG_LSB);
	EVENT_NAME(ME_SYSEX_XG_MSB);
	EVENT_NAME(ME_WRD);
	EVENT_NAME(ME_SHERRY);
	EVENT_NAME(ME_BARMARKER);
	EVENT_NAME(ME_STEP);
	EVENT_NAME(ME_LAST);
	EVENT_NAME(ME_EOT);
	}
	return "Unknown";
#undef EVENT_NAME
}

static void adjust_amplification(tmdy_struct_ex_t *tmdy_struct)
{
    /* compensate master volume */
    (PM->master_volume) = (double)((PM->amplification)) / 100.0 *
	((double)(PM->master_volume_ratio) * ((PM->compensation_ratio)/0xFFFF));
}

static int new_vidq(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    int i;

    if((PM->opt_overlap_voice_allow))
    {
	i = ch * 128 + note;
	return (PM->vidq_head)[i]++;
    }
    return 0;
}

static int last_vidq(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    int i;

    if((PM->opt_overlap_voice_allow))
    {
	i = ch * 128 + note;
	if((PM->vidq_head)[i] == (PM->vidq_tail)[i])
	{
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_DEBUG_SILLY,
		      "channel=%d, note=%d: Voice is already OFF", ch, note);
	    return -1;
	}
	return (PM->vidq_tail)[i]++;
    }
    return 0;
}

static void reset_voices(tmdy_struct_ex_t *tmdy_struct)
{
    int i;
    for(i = 0; i < PM->max_voices; i++)
    {
	(PM->voice)[i].status = VOICE_FREE;
	(PM->voice)[i].temper_instant = 0;
	(PM->voice)[i].chorus_link = i;
    }
    (PM->upper_voices) = 0;
    memset(PM->vidq_head, 0, sizeof(PM->vidq_head));
    memset(PM->vidq_tail, 0, sizeof(PM->vidq_tail));
}

static void kill_note(tmdy_struct_ex_t *tmdy_struct, int i)
{
    (PM->voice)[i].status = VOICE_DIE;
    if(!PM->prescanning_flag){
	ctl_note_event(tmdy_struct, i);
	}
}

static void kill_all_voices(tmdy_struct_ex_t *tmdy_struct)
{
    int i, uv = (PM->upper_voices);

    for(i = 0; i < uv; i++)
	if((PM->voice)[i].status & ~(VOICE_FREE | VOICE_DIE))
	    kill_note(tmdy_struct, i);
    memset(PM->vidq_head, 0, sizeof(PM->vidq_head));
    memset(PM->vidq_tail, 0, sizeof(PM->vidq_tail));

}

static void reset_drum_controllers(tmdy_struct_ex_t *tmdy_struct, struct DrumParts *d[], int note)
{
    int i,j;

    if(note == -1)
    {
	for(i = 0; i < 128; i++)
	    if(d[i] != NULL)
	    {
		d[i]->drum_panning = NO_PANNING;
		for(j=0;j<6;j++) {d[i]->drum_envelope_rate[j] = -1;}
		d[i]->pan_random = 0;
		d[i]->drum_level = 1.0f;
		d[i]->coarse = 0;
		d[i]->fine = 0;
		d[i]->delay_level = 0;
		d[i]->chorus_level = 0;
		d[i]->reverb_level = 0;
		d[i]->play_note = -1;
		d[i]->rx_note_off = 0;
		d[i]->drum_cutoff_freq = 0;
		d[i]->drum_resonance = 0;
	    }
    }
    else
    {
	d[note]->drum_panning = NO_PANNING;
	for(j=0;j<6;j++) {d[note]->drum_envelope_rate[j] = -1;}
	d[note]->pan_random = 0;
	d[note]->drum_level = 1.0f;
	d[note]->coarse = 0;
	d[note]->fine = 0;
	d[note]->delay_level = 0;
	d[note]->chorus_level = 0;
	d[note]->reverb_level = 0;
	d[note]->play_note = -1;
	d[note]->rx_note_off = 0;
	d[note]->drum_cutoff_freq = 0;
	d[note]->drum_resonance = 0;
    }
}

static void reset_nrpn_controllers(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i,j;

  /* NRPN */
  reset_drum_controllers(tmdy_struct, (PM->channel)[c].drums, -1);
  (PM->channel)[c].vibrato_ratio = 0;
  (PM->channel)[c].vibrato_depth = 0;
  (PM->channel)[c].vibrato_delay = 0;
  (PM->channel)[c].param_cutoff_freq = 0;
  (PM->channel)[c].param_resonance = 0;
  (PM->channel)[c].cutoff_freq_coef = 1.0;
  (PM->channel)[c].resonance_dB = 0;

  /* GS & XG System Exclusive */
  (PM->channel)[c].eq_on = 1;
  (PM->channel)[c].insertion_effect = 0;
  (PM->channel)[c].velocity_sense_depth = 0x40;
  (PM->channel)[c].velocity_sense_offset = 0x40;
  (PM->channel)[c].pitch_offset_fine = 0;
  (PM->channel)[c].legato = 0;
  (PM->channel)[c].assign_mode = 1;
  if(TMDY_READMIDI->play_system_mode == GS_SYSTEM_MODE) {
	  (PM->channel)[c].bank_lsb = (PM->channel)[c].tone_map0_number;
  }
	for (i = 0; i < 12; i++)
		(PM->channel)[c].scale_tuning[i] = 0;
	(PM->channel)[c].prev_scale_tuning = 0;
	(PM->channel)[c].temper_type = 0;

  TMDY_READMIDI->init_channel_layer(tmdy_struct, c);

  /* channel pressure & polyphonic key pressure control */
  (PM->channel)[c].caf_lfo1_rate_ctl = 0;

  (PM->channel)[c].sysex_gs_msb_addr = (PM->channel)[c].sysex_gs_msb_val =
	(PM->channel)[c].sysex_xg_msb_addr = (PM->channel)[c].sysex_xg_msb_val =
	(PM->channel)[c].sysex_msb_addr = (PM->channel)[c].sysex_msb_val = 0;
}

/* Process the Reset All Controllers event */
static void reset_controllers(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i,j;
  
    /* Some standard says, although the SCC docs say 0. */
    
  if(TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE)
      (PM->channel)[c].volume = 100;
  else
      (PM->channel)[c].volume = 90;
  if (PM->prescanning_flag) {
    if ((PM->channel)[c].volume > (PM->mainvolume_max)) {	/* pick maximum value of mainvolume */
      (PM->mainvolume_max) = (PM->channel)[c].volume;
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_DEBUG,"ME_MAINVOLUME/max (CH:%d VAL:%#x)",c,(PM->mainvolume_max));
    }
  }

  (PM->channel)[c].expression=127; /* SCC-1 does this. */
  (PM->channel)[c].sustain=0;
  (PM->channel)[c].pitchbend=0x2000;
  (PM->channel)[c].pitchfactor=0; /* to be computed */
  (PM->channel)[c].modulation_wheel = 0;
  (PM->channel)[c].portamento_time_lsb = 0;
  (PM->channel)[c].portamento_time_msb = 0;
  (PM->channel)[c].porta_control_ratio = 0;
  (PM->channel)[c].portamento = 0;
  (PM->channel)[c].last_note_fine = -1;
  for(j=0;j<6;j++) {(PM->channel)[c].envelope_rate[j] = -1;}
  update_portamento_controls(tmdy_struct, c);
  set_reverb_level(tmdy_struct, c, -1);
  if((PM->opt_chorus_control) == 1)
      (PM->channel)[c].chorus_level = 0;
  else
      (PM->channel)[c].chorus_level = -(PM->opt_chorus_control);
  (PM->channel)[c].mono = 0;
  (PM->channel)[c].delay_level = 0;
}

static void redraw_controllers(tmdy_struct_ex_t *tmdy_struct, int c)
{
    ctl_mode_event(tmdy_struct, CTLE_VOLUME, 1, c, (PM->channel)[c].volume);
    ctl_mode_event(tmdy_struct, CTLE_EXPRESSION, 1, c, (PM->channel)[c].expression);
    ctl_mode_event(tmdy_struct, CTLE_SUSTAIN, 1, c, (PM->channel)[c].sustain);
    ctl_mode_event(tmdy_struct, CTLE_MOD_WHEEL, 1, c, (PM->channel)[c].modulation_wheel);
    ctl_mode_event(tmdy_struct, CTLE_PITCH_BEND, 1, c, (PM->channel)[c].pitchbend);
    ctl_prog_event(tmdy_struct, c);
    ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 1, c, (PM->channel)[c].temper_type);
    ctl_mode_event(tmdy_struct, CTLE_MUTE, 1,
    		c, (IS_SET_CHANNELMASK(PM->channel_mute, c)) ? 1 : 0);
    ctl_mode_event(tmdy_struct, CTLE_CHORUS_EFFECT, 1, c, get_chorus_level(tmdy_struct, c));
    ctl_mode_event(tmdy_struct, CTLE_REVERB_EFFECT, 1, c, get_reverb_level(tmdy_struct, c));
}

static void reset_midi(tmdy_struct_ex_t *tmdy_struct, int playing)
{
	int i, cnt;
	
	for (i = 0; i < MAX_CHANNELS; i++) {
		reset_controllers(tmdy_struct, i);
		reset_nrpn_controllers(tmdy_struct, i);
		/* The rest of these are unaffected
		 * by the Reset All Controllers event
		 */
		(PM->channel)[i].program = TMDY_INSTRUM->default_program[i];
		(PM->channel)[i].panning = NO_PANNING;
		(PM->channel)[i].pan_random = 0;
		/* tone bank or drum set */
		if (ISDRUMCHANNEL(i)) {
			(PM->channel)[i].bank = 0;
			(PM->channel)[i].altassign = TMDY_INSTRUM->drumset[0]->alt;
		} else {
			if ((PM->special_tonebank) >= 0)
				(PM->channel)[i].bank = (PM->special_tonebank);
			else
				(PM->channel)[i].bank = (PM->default_tonebank);
		}
		(PM->channel)[i].bank_lsb = (PM->channel)[i].bank_msb =
				(PM->channel)[i].tone_map0_number = 0;
		if (TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE && i % 16 == 9)
			(PM->channel)[i].bank_msb = 127;	/* Use MSB=127 for XG */
		update_rpn_map(tmdy_struct,i, RPN_ADDR_FFFF, 0);
		(PM->channel)[i].special_sample = 0;
		(PM->channel)[i].key_shift = 0;
		(PM->channel)[i].mapID = TMDY_READMIDI->get_default_mapID(tmdy_struct, i);
		(PM->channel)[i].lasttime = 0;
	}
	if (playing) {
		kill_all_voices(tmdy_struct);
		if ((PM->temper_type_mute)) {
			if ((PM->temper_type_mute) & 1)
				FILL_CHANNELMASK(PM->channel_mute);
			else
				CLEAR_CHANNELMASK(PM->channel_mute);
		}
		for (i = 0; i < MAX_CHANNELS; i++)
			redraw_controllers(tmdy_struct, i);
		if (PM->midi_streaming && (TMDY_MAIN->free_instruments_afterwards)) {
			TMDY_INSTRUM->free_instruments(tmdy_struct, 0);
			/* free unused memory */
			cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);
			if (cnt > 0)
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
						"%d memory blocks are free", cnt);
		}
	} else
		reset_voices(tmdy_struct);
	(PM->master_volume_ratio) = 0xffff;
	adjust_amplification(tmdy_struct);
	init_freq_table_tuning();
	if (TMDY_READMIDI->current_file_info) {
		COPY_CHANNELMASK((PM->drumchannels), TMDY_READMIDI->current_file_info->drumchannels);
		COPY_CHANNELMASK((PM->drumchannel_mask),
				TMDY_READMIDI->current_file_info->drumchannel_mask);
	} else {
		COPY_CHANNELMASK((PM->drumchannels), (PM->default_drumchannels));
		COPY_CHANNELMASK((PM->drumchannel_mask), (PM->default_drumchannel_mask));
	}
	ctl_mode_event(tmdy_struct, CTLE_MASTER_VOLUME, 0, (PM->amplification), 0);
	ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 0, (PM->note_key_offset), 0);
	ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (PM->midi_time_ratio) + 0.5, 0);
}

void recompute_freq(tmdy_struct_ex_t *tmdy_struct, int v)
{
	int i;
	int ch = (PM->voice)[v].channel;
	int note = (PM->voice)[v].note;
	int32 tuning = 0;
	int8 st = (PM->channel)[ch].scale_tuning[note % 12];
	int8 tt = (PM->channel)[ch].temper_type;
	uint8 tp = (PM->channel)[ch].rpnmap[RPN_ADDR_0003];
	int32 f;
	int pb = (PM->channel)[ch].pitchbend;
	int32 tmp;
	FLOAT_T pf, root_freq;
	int32 a;
	
	if (! (PM->voice)[v].sample->sample_rate){
		return;
	}
	if (! (PM->opt_modulation_wheel))
		(PM->voice)[v].modulation_wheel = 0;
	if (! PM->opt_portamento)
		(PM->voice)[v].porta_control_ratio = 0;
	(PM->voice)[v].vibrato_control_ratio = (PM->voice)[v].orig_vibrato_control_ratio;
	if ((PM->voice)[v].vibrato_control_ratio || (PM->voice)[v].modulation_wheel > 0) {
		/* This instrument has vibrato. Invalidate any precomputed
		 * sample_increments.
		 */
		if ((PM->voice)[v].modulation_wheel > 0) {
			(PM->voice)[v].vibrato_control_ratio = (TMDY_OUTPUT->play_mode)->rate / 2.0
					* MODULATION_WHEEL_RATE / VIBRATO_SAMPLE_INCREMENTS;
			(PM->voice)[v].vibrato_delay = 0;
		}
		for (i = 0; i < VIBRATO_SAMPLE_INCREMENTS; i++)
			(PM->voice)[v].vibrato_sample_increment[i] = 0;
		(PM->voice)[v].cache = NULL;
	}
	/* fine: [0..128] => [-256..256]
	 * 1 coarse = 256 fine (= 1 note)
	 * 1 fine = 2^5 tuning
	 */
	tuning = ((PM->channel)[ch].rpnmap[RPN_ADDR_0001] - 0x40
			+ ((PM->channel)[ch].rpnmap[RPN_ADDR_0002] - 0x40) * 64) << 7;
	/* for NRPN Coarse Pitch of Drum (GS) & Fine Pitch of Drum (XG) */
	if (ISDRUMCHANNEL(ch) && (PM->channel)[ch].drums[note] != NULL
			&& ((PM->channel)[ch].drums[note]->fine
			|| (PM->channel)[ch].drums[note]->coarse)) {
		tuning += ((PM->channel)[ch].drums[note]->fine
				+ (PM->channel)[ch].drums[note]->coarse * 64) << 7;
	}
	if ((PM->opt_modulation_envelope)) {
		if ((PM->voice)[v].sample->tremolo_to_pitch)
			tuning += lookup_triangular((PM->voice)[v].tremolo_phase >> RATE_SHIFT)
					* ((PM->voice)[v].sample->tremolo_to_pitch << 13) / 100.0 + 0.5;
		if ((PM->voice)[v].sample->modenv_to_pitch)
			tuning += (PM->voice)[v].last_modenv_volume
					* ((PM->voice)[v].sample->modenv_to_pitch << 13) / 100.0 + 0.5;
	}
	/* GS/XG - Scale Tuning */
	if (! ISDRUMCHANNEL(ch)) {
		tuning += (st << 13) / 100.0 + 0.5;
		if (st != (PM->channel)[ch].prev_scale_tuning) {
			(PM->channel)[ch].pitchfactor = 0;
			(PM->channel)[ch].prev_scale_tuning = st;
		}
	}
	if (! (PM->opt_pure_intonation)
			&& (PM->opt_temper_control) && (PM->voice)[v].temper_instant) {
		switch (tt) {
		case 0:
			f = freq_table_tuning[tp][note];
			break;
		case 1:
			if (PM->current_temper_keysig < 8)
				f = freq_table_pytha[PM->current_freq_table][note];
			else
				f = freq_table_pytha[PM->current_freq_table + 12][note];
			break;
		case 2:
			if ((PM->current_temper_keysig) < 8)
				f = freq_table_meantone[PM->current_freq_table
						+ ((PM->temper_adj) ? 36 : 0)][note];
			else
				f = freq_table_meantone[PM->current_freq_table
						+ ((PM->temper_adj) ? 24 : 12)][note];
			break;
		case 3:
			if ((PM->current_temper_keysig) < 8)
				f = freq_table_pureint[PM->current_freq_table
						+ ((PM->temper_adj) ? 36 : 0)][note];
			else
				f = freq_table_pureint[PM->current_freq_table
						+ ((PM->temper_adj) ? 24 : 12)][note];
			break;
		default:	/* user-defined temperaments */
			if ((tt -= 0x40) >= 0 && tt < 4) {
				if ((PM->current_temper_keysig) < 8)
					f = freq_table_user[tt][PM->current_freq_table
							+ ((PM->temper_adj) ? 36 : 0)][note];
				else
					f = freq_table_user[tt][PM->current_freq_table
+							+ ((PM->temper_adj) ? 24 : 12)][note];
			} else
				f = freq_table[note];
			break;
		}
		(PM->voice)[v].orig_frequency = f;
	}
	if (! (PM->voice)[v].porta_control_ratio) {
		if (tuning == 0 && pb == 0x2000)
			(PM->voice)[v].frequency = (PM->voice)[v].orig_frequency;
		else {
			pb -= 0x2000;
			if (! (PM->channel)[ch].pitchfactor) {
				/* Damn.  Somebody bent the pitch. */
				tmp = pb * (PM->channel)[ch].rpnmap[RPN_ADDR_0000] + tuning;
				if (tmp >= 0)
					(PM->channel)[ch].pitchfactor = bend_fine[tmp >> 5 & 0xff]
							* bend_coarse[tmp >> 13 & 0x7f];
				else
					(PM->channel)[ch].pitchfactor = 1.0 /
							(bend_fine[-tmp >> 5 & 0xff]
							* bend_coarse[-tmp >> 13 & 0x7f]);
			}
			(PM->voice)[v].frequency =
					(PM->voice)[v].orig_frequency * (PM->channel)[ch].pitchfactor;
			if ((PM->voice)[v].frequency != (PM->voice)[v].orig_frequency)
				(PM->voice)[v].cache = NULL;
		}
	} else {	/* Portamento */
		pb -= 0x2000;
		tmp = pb * (PM->channel)[ch].rpnmap[RPN_ADDR_0000]
				+ ((PM->voice)[v].porta_pb << 5) + tuning;
		if (tmp >= 0)
			pf = bend_fine[tmp >> 5 & 0xff]
					* bend_coarse[tmp >> 13 & 0x7f];
		else
			pf = 1.0 / (bend_fine[-tmp >> 5 & 0xff]
					* bend_coarse[-tmp >> 13 & 0x7f]);
		(PM->voice)[v].frequency = (PM->voice)[v].orig_frequency * pf;
		(PM->voice)[v].cache = NULL;
	}
	if (ISDRUMCHANNEL(ch) && (PM->channel)[ch].drums[note] != NULL
			&& (PM->channel)[ch].drums[note]->play_note != -1)
		root_freq = (PM->voice)[v].sample->root_freq
				* (double) freq_table[(PM->channel)[ch].drums[note]->play_note]
				/ (PM->voice)[v].orig_frequency;
	else
		root_freq = (PM->voice)[v].sample->root_freq;
	a = TIM_FSCALE(((double) (PM->voice)[v].sample->sample_rate
			* (PM->voice)[v].frequency + (PM->channel)[ch].pitch_offset_fine)
			/ (root_freq * (TMDY_OUTPUT->play_mode)->rate), FRACTION_BITS) + 0.5;
	/* need to preserve the loop direction */
	(PM->voice)[v].sample_increment = ((PM->voice)[v].sample_increment >= 0) ? a : -a;
#ifdef ABORT_AT_FATAL
	if ((PM->voice)[v].sample_increment == 0) {
		fprintf(stderr, "Invalid sample increment a=%e %ld %ld %ld %ld%s\n",
				(double)a, (long) (PM->voice)[v].sample->sample_rate,
				(long) (PM->voice)[v].frequency, (long) (PM->voice)[v].sample->root_freq,
				(long) (TMDY_OUTPUT->play_mode)->rate, ((PM->voice)[v].cache) ? " (Cached)" : "");
		abort();
	}
#endif	/* ABORT_AT_FATAL */
}

static int32 calc_velocity(tmdy_struct_ex_t *tmdy_struct, int32 ch,int32 vel)
{
	int32 velocity;
	velocity = (PM->channel)[ch].velocity_sense_depth * vel / 64 + ((PM->channel)[ch].velocity_sense_offset - 64) * 2;
	if(velocity > 127) {velocity = 127;}
	return velocity;
}

static void recompute_amp(tmdy_struct_ex_t *tmdy_struct, int v)
{
	FLOAT_T tempamp;

	/* master_volume and sample->volume are percentages, used to scale
	 *  amplitude directly, NOT perceived volume
	 *
	 * all other MIDI volumes are linear in perceived volume, 0-127
	 * use a lookup table for the non-linear scalings
	 */
	if (PM->opt_user_volume_curve) {
	tempamp = PM->master_volume *
		   PM->voice[v].sample->volume *
		   user_vol_table[calc_velocity(tmdy_struct, PM->voice[v].channel,PM->voice[v].velocity)] *
		   user_vol_table[PM->channel[PM->voice[v].channel].volume] *
		   user_vol_table[PM->channel[PM->voice[v].channel].expression]; /* 21 bits */
	} else if(TMDY_READMIDI->play_system_mode == GS_SYSTEM_MODE) {	/* use measured curve */ 
	tempamp = (PM->master_volume) *
		   (PM->voice)[v].sample->volume *
		   sc_vel_table[calc_velocity(tmdy_struct, (PM->voice)[v].channel,(PM->voice)[v].velocity)] *
		   sc_vol_table[(PM->channel)[(PM->voice)[v].channel].volume] *
		   sc_vol_table[(PM->channel)[(PM->voice)[v].channel].expression]; /* 21 bits */
	} else if (IS_CURRENT_MOD_FILE) {	/* use linear curve */
	tempamp = (PM->master_volume) *
		  (PM->voice)[v].sample->volume *
		  calc_velocity(tmdy_struct, (PM->voice)[v].channel,(PM->voice)[v].velocity) *
		  (PM->channel)[(PM->voice)[v].channel].volume *
		  (PM->channel)[(PM->voice)[v].channel].expression; /* 21 bits */
	} else {	/* use generic exponential curve */
	tempamp = (PM->master_volume) *
		  (PM->voice)[v].sample->volume *
		  perceived_vol_table[calc_velocity(tmdy_struct, (PM->voice)[v].channel,(PM->voice)[v].velocity)] *
		  perceived_vol_table[(PM->channel)[(PM->voice)[v].channel].volume] *
		  perceived_vol_table[(PM->channel)[(PM->voice)[v].channel].expression]; /* 21 bits */
	}

	/* every digital effect increases amplitude,
	 * so that it must be reduced in advance.
	 */
	if (! (TMDY_OUTPUT->play_mode->encoding & PE_MONO)
			&& (PM->opt_reverb_control || PM->opt_chorus_control || PM->opt_delay_control
			|| PM->opt_eq_control && (TMDY_REVERB->eq_status.low_gain != 0x40
			|| TMDY_REVERB->eq_status.high_gain != 0x40) || PM->opt_delay_control))
		tempamp *= 1.35f * 0.55f;
	else
		tempamp *= 1.35f;
	

	/* NRPN - drum instrument tva level */
	if(ISDRUMCHANNEL((PM->voice)[v].channel)) {
		if((PM->channel)[(PM->voice)[v].channel].drums[(PM->voice)[v].note] != NULL) {
			tempamp *= (PM->channel)[(PM->voice)[v].channel].drums[(PM->voice)[v].note]->drum_level;
		}
		tempamp *= (double)(PM->opt_drum_power) * 0.01f;
	}

	if(!((TMDY_OUTPUT->play_mode)->encoding & PE_MONO))
    	{
		if((PM->voice)[v].panning == 64)
		{
			(PM->voice)[v].panned = PANNED_CENTER;
			(PM->voice)[v].left_amp = (PM->voice)[v].right_amp = TIM_FSCALENEG(tempamp * sc_pan_table[64], 27);
		}
		else if ((PM->voice)[v].panning < 2)
		{
			(PM->voice)[v].panned = PANNED_LEFT;
			(PM->voice)[v].left_amp = TIM_FSCALENEG(tempamp, 20);
			(PM->voice)[v].right_amp = 0;
		}
		else if((PM->voice)[v].panning == 127)
		{
#ifdef SMOOTH_MIXING
			if((PM->voice)[v].panned == PANNED_MYSTERY) {
				(PM->voice)[v].old_left_mix = (PM->voice)[v].old_right_mix;
				(PM->voice)[v].old_right_mix = 0;
			}
#endif
			(PM->voice)[v].panned = PANNED_RIGHT;
			(PM->voice)[v].left_amp =  TIM_FSCALENEG(tempamp, 20);
			(PM->voice)[v].right_amp = 0;
		}
		else
		{
#ifdef SMOOTH_MIXING
			if((PM->voice)[v].panned == PANNED_RIGHT) {
				(PM->voice)[v].old_right_mix = (PM->voice)[v].old_left_mix;
				(PM->voice)[v].old_left_mix = 0;
			}
#endif
			(PM->voice)[v].panned = PANNED_MYSTERY;
			(PM->voice)[v].left_amp = TIM_FSCALENEG(tempamp * sc_pan_table[127 - (PM->voice)[v].panning], 27);
			(PM->voice)[v].right_amp = TIM_FSCALENEG(tempamp * sc_pan_table[(PM->voice)[v].panning], 27);
		}
    	}
    	else
    	{
		(PM->voice)[v].panned = PANNED_CENTER;
		(PM->voice)[v].left_amp = TIM_FSCALENEG(tempamp, 21);
    	}
}

void recompute_channel_filter(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
	int ch = e->channel, note, prog, bk;
	double coef = 1.0f, reso = 0;
	ToneBankElement *elm;

	if((PM->channel)[ch].special_sample > 0) {
		return;
	}

	note = MIDI_EVENT_NOTE(e);
	prog = (PM->channel)[ch].program;
	bk = (PM->channel)[ch].bank;
	elm = &(TMDY_INSTRUM->tonebank)[bk]->tone[prog];

	/* Soft Pedal */
	if((PM->channel)[ch].soft_pedal > 63) {
		if(note > 49) {	/* tre corde */
			coef *= 1.0 - 0.20 * ((double)(PM->channel)[ch].soft_pedal - 64) / 63.0f;
		} else {	/* una corda (due corde) */
			coef *= 1.0 - 0.25 * ((double)(PM->channel)[ch].soft_pedal - 64) / 63.0f;
		}
	}

	if(!ISDRUMCHANNEL(ch)) {
		/* NRPN Filter Cutoff */
		coef *= pow(1.26, (double)((PM->channel)[ch].param_cutoff_freq) / 8.0f);
		/* NRPN Resonance */
		reso = (double)(PM->channel)[ch].param_resonance * 0.5f;
	}

	(PM->channel)[ch].cutoff_freq_coef = coef;
	(PM->channel)[ch].resonance_dB = reso;

/*	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Cutoff Frequency (CH:%d VAL:%f)",ch,coef);
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Resonance (CH:%d VAL:%f)",ch,reso);*/
}

void recompute_voice_filter(tmdy_struct_ex_t *tmdy_struct, int v)
{
	int ch = (PM->voice)[v].channel, note = (PM->voice)[v].note;
	double coef = 1.0, reso = 0, cent = 0;
	int32 freq;
	FilterCoefficients *fc = &((PM->voice)[v].fc);
	Sample *sp = (Sample *) &(PM->voice)[v].sample;

	if(fc->freq == -1) {return;}

	coef = (PM->channel)[ch].cutoff_freq_coef;

	if(ISDRUMCHANNEL(ch) && (PM->channel)[ch].drums[note] != NULL) {
		/* NRPN Drum Instrument Filter Cutoff */
		coef *= pow(1.26, (double)((PM->channel)[ch].drums[note]->drum_cutoff_freq) / 8.0f);
		/* NRPN Drum Instrument Filter Resonance */
		reso += (double)(PM->channel)[ch].drums[note]->drum_resonance * 0.5f;
	}

	if(sp->vel_to_fc) {	/* velocity to filter cutoff frequency */
		if((PM->voice)[v].velocity > sp->vel_to_fc_threshold)
			cent += sp->vel_to_fc * (double)(127 - (PM->voice)[v].velocity) / 127.0f;
		else
			coef += -1200;
	}
	if(sp->vel_to_resonance) {	/* velocity to filter resonance */
		reso += (double)(PM->voice)[v].velocity * sp->vel_to_resonance / 127.0f / 10.0f;
	}
	if(sp->key_to_fc) {	/* filter cutoff key-follow */
		cent += sp->vel_to_fc * (double)((PM->voice)[v].note - sp->key_to_fc_bpo);
	}

	if((PM->opt_modulation_envelope)) {
		if((PM->voice)[v].sample->tremolo_to_fc) {
			cent += (double)(PM->voice)[v].sample->tremolo_to_fc * lookup_triangular((PM->voice)[v].tremolo_phase >> RATE_SHIFT);
		}
		if((PM->voice)[v].sample->modenv_to_fc) {
			cent += (double)(PM->voice)[v].sample->modenv_to_fc * (PM->voice)[v].last_modenv_volume;
		}
	}

	if(cent != 0) {coef *= pow(2.0, cent / 1200.0f);}

	freq = (double)fc->orig_freq * coef;

	if (fc->filter_calculated == 0 && freq > (TMDY_OUTPUT->play_mode)->rate / 2) {
		fc->freq = -1;
		return;
	}
	else if(freq < 5) {freq = 5;}
	else if(freq > 20000) {freq = 20000;}
	fc->freq = freq;

	fc->reso_dB = fc->orig_reso_dB + (PM->channel)[ch].resonance_dB + reso;
	if(fc->reso_dB < 0.0f) {fc->reso_dB = 0.0f;}
	else if(fc->reso_dB > 96.0f) {fc->reso_dB = 96.0f;}
	fc->reso_dB -= 3.01f;
}

FLOAT_T calc_drum_tva_level(tmdy_struct_ex_t *tmdy_struct, int ch,int note,int level)
{
	int def_level,nbank,nprog;
	ToneBank *bank;

	if((PM->channel)[ch].special_sample > 0) {
		return 1.0;
	}
	
	nbank = (PM->channel)[ch].bank;
	nprog = note;
	TMDY_INSTRUM->instrument_map(tmdy_struct, (PM->channel)[ch].mapID, &nbank, &nprog);

	if(ISDRUMCHANNEL(ch)) {
		bank = TMDY_INSTRUM->drumset[nbank];
		if(bank == NULL) {bank = TMDY_INSTRUM->drumset[0];}
	} else {
		return 1.0;
	}

	def_level = bank->tone[nprog].tva_level;

	if(def_level == -1 || def_level == 0) {def_level = 127;}
	else if(def_level > 127) {def_level = 127;}

	return (sc_drum_level_table[level] / sc_drum_level_table[def_level]);
}

void recompute_bank_parameter(tmdy_struct_ex_t *tmdy_struct, int ch,int note)
{
	int prog;
	ToneBank *bank;

	if((PM->channel)[ch].special_sample > 0) {
		return;
	}

	prog = (PM->channel)[ch].program;

	if(ISDRUMCHANNEL(ch)) {
		return;
	} else {
		if(prog == SPECIAL_PROGRAM) {
			return;
		}
		bank = TMDY_INSTRUM->tonebank[(int)(PM->channel)[ch].bank];
		if(bank == NULL) {bank = TMDY_INSTRUM->tonebank[0];}
	}

	(PM->channel)[ch].legato = bank->tone[prog].legato;
}

static void *memdup(tmdy_struct_ex_t *tmdy_struct, void *s,size_t size)
{
	void *p;

	p = TMDY_COMMON->safe_malloc(tmdy_struct, size);
	if(p != NULL) {memcpy(p,s,size);}

	return p;
}

void dup_tone_bank_element(tmdy_struct_ex_t *tmdy_struct, int dr, int bk, int prog)
{
	ToneBank **bank = ((dr) ? TMDY_INSTRUM->drumset : TMDY_INSTRUM->tonebank);
	ToneBankElement *elm = &bank[bk]->tone[prog];
	int i;
	
	if (elm->name)
		elm->name = TMDY_COMMON->safe_strdup(tmdy_struct, elm->name);
	if (elm->comment)
		elm->comment = TMDY_COMMON->safe_strdup(tmdy_struct, elm->comment);
	if (elm->tunenum)
		elm->tune = (float *) memdup(tmdy_struct, 
				elm->tune, elm->tunenum * sizeof(float));
	if (elm->envratenum) {
		elm->envrate = (int **) memdup(tmdy_struct, 
				elm->envrate, elm->envratenum * sizeof(int *));
		for (i = 0; i < elm->envratenum; i++)
			elm->envrate[i] = (int *) memdup(tmdy_struct, elm->envrate[i], 6 * sizeof(int));
	}
	if (elm->envofsnum) {
		elm->envofs = (int **) memdup(tmdy_struct, 
				elm->envofs, elm->envofsnum * sizeof(int *));
		for (i = 0; i < elm->envofsnum; i++)
			elm->envofs[i] = (int *) memdup(tmdy_struct, elm->envofs[i], 6 * sizeof(int));
	}
	if (elm->tremnum) {
		elm->trem = (Quantity **) memdup(tmdy_struct, elm->trem, elm->tremnum * sizeof(Quantity *));
		for (i = 0; i < elm->tremnum; i++)
			elm->trem[i] = (Quantity *) memdup(tmdy_struct, elm->trem[i], 3 * sizeof(Quantity));
	}
	if (elm->vibnum) {
		elm->vib = (Quantity **) memdup(tmdy_struct, elm->vib, elm->vibnum * sizeof(Quantity *));
		for (i = 0; i < elm->vibnum; i++)
			elm->vib[i] = (Quantity *) memdup(tmdy_struct, elm->vib[i], 3 * sizeof(Quantity));
	}
}

void free_tone_bank_element(tmdy_struct_ex_t *tmdy_struct, int dr, int bk, int prog)
{
	ToneBank **bank = (dr) ? TMDY_INSTRUM->drumset : TMDY_INSTRUM->tonebank;
	ToneBankElement *elm = &bank[bk]->tone[prog];
	int i;
	
	if (elm->name) {
		free(elm->name);
		elm->name = NULL;
	}
	if (elm->comment) {
		free(elm->comment);
		elm->comment = NULL;
	}
	if (elm->tune) {
		free(elm->tune);
		elm->tune = NULL;
	}
	if (elm->envratenum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->envrate, elm->envratenum);
		elm->envrate = NULL;
		elm->envratenum = 0;
	}
	if (elm->envofsnum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->envofs, elm->envofsnum);
		elm->envofs = NULL;
		elm->envofsnum = 0;
	}
	if (elm->tremnum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->trem, elm->tremnum);
		elm->trem = NULL;
		elm->tremnum = 0;
	}
	if (elm->vibnum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->vib, elm->vibnum);
		elm->vib = NULL;
		elm->vibnum = 0;
	}
	elm->instype = 0;
}

Instrument *play_midi_load_instrument(tmdy_struct_ex_t *tmdy_struct, int dr, int bk, int prog)
{
	ToneBank **bank = (dr) ? TMDY_INSTRUM->drumset : TMDY_INSTRUM->tonebank;
	Instrument *ip;
	ToneBankElement *elm;
	int load_success = 0;
	
	if (bank[bk] == NULL)
		bk = 0;
	if ((ip = bank[bk]->tone[prog].instrument) == MAGIC_LOAD_INSTRUMENT) {
		if (ip = bank[bk]->tone[prog].instrument =
				TMDY_INSTRUM->load_instrument(tmdy_struct, dr, bk, prog))
			load_success = 1;
#ifndef SUPPRESS_CHANNEL_LAYER
	} else if (ip == NULL) {
		if (ip = bank[bk]->tone[prog].instrument =
				TMDY_INSTRUM->load_instrument(tmdy_struct, dr, bk, prog))
			load_success = 1;
#endif
	}
	if (ip == NULL && bk) {
		/* Instrument is not found.
		 * Retry to load the instrument from bank 0
		 */
		if ((ip = bank[0]->tone[prog].instrument) == NULL
				|| ip == MAGIC_LOAD_INSTRUMENT)
			ip = bank[0]->tone[prog].instrument =
					TMDY_INSTRUM->load_instrument(tmdy_struct, dr, 0, prog);
		if (ip) {
			/* duplicate tone bank parameter */
			elm = &bank[bk]->tone[prog];
			memcpy(elm, &bank[0]->tone[prog], sizeof(ToneBankElement));
			elm->instrument = ip;
			dup_tone_bank_element(tmdy_struct, dr, bk, prog);
			load_success = 1;
		}
	}
	if (load_success)
		(TMDY_AQ->aq_add)(tmdy_struct,NULL, 0);	/* Update software buffer */
	if (ip == MAGIC_ERROR_INSTRUMENT)
		return NULL;
	if (ip == NULL)
		bank[bk]->tone[prog].instrument = MAGIC_ERROR_INSTRUMENT;
	return ip;
}

#if 0
/* reduce_voice_CPU() may not have any speed advantage over reduce_voice().
 * So this function is not used, now.
 */

/* The goal of this routine is to free as much CPU as possible without
   loosing too much sound quality.  We would like to know how long a note
   has been playing, but since we usually can't calculate this, we guess at
   the value instead.  A bad guess is better than nothing.  Notes which
   have been playing a short amount of time are killed first.  This causes
   decays and notes to be cut earlier, saving more CPU time.  It also causes
   notes which are closer to ending not to be cut as often, so it cuts
   a different note instead and saves more CPU in the long run.  ON voices
   are treated a little differently, since sound quality is more important
   than saving CPU at this point.  Duration guesses for loop regions are very
   crude, but are still better than nothing, they DO help.  Non-looping ON
   notes are cut before looping ON notes.  Since a looping ON note is more
   likely to have been playing for a long time, we want to keep it because it
   sounds better to keep long notes.
*/
static int reduce_voice_CPU(tmdy_struct_ex_t *tmdy_struct)
{
    int32 lv, v, vr;
    int i, j, lowest=-0x7FFFFFFF;
    int32 duration;

    i = (PM->upper_voices);
    lv = 0x7FFFFFFF;
    
    /* Look for the decaying note with the longest remaining decay time */
    /* Protect drum decays.  They do not take as much CPU (?) and truncating
       them early sounds bad, especially on snares and cymbals */
    for(j = 0; j < i; j++)
    {
	if((PM->voice)[j].status & VOICE_FREE || (PM->voice)[j].cache != NULL)
	    continue;
	/* skip notes that don't need resampling (most drums) */
	if ((PM->voice)[j].sample->note_to_use)
	    continue;
	if((PM->voice)[j].status & ~(VOICE_ON | VOICE_DIE | VOICE_SUSTAINED))
	{
	    /* Choose note with longest decay time remaining */
	    /* This frees more CPU than choosing lowest volume */
	    if (!(PM->voice)[j].envelope_increment) duration = 0;
	    else duration =
	    	((PM->voice)[j].envelope_target - (PM->voice)[j].envelope_volume) /
	    	(PM->voice)[j].envelope_increment;
	    v = -duration;
	    if(v < lv)
	    {
		lv = v;
		lowest = j;
	    }
	}
    }
    if(lowest != -0x7FFFFFFF)
    {
	/* This can still cause a click, but if we had a free voice to
	   spare for ramping down this note, we wouldn't need to kill it
	   in the first place... Still, this needs to be fixed. Perhaps
	   we could use a reserve of voices to play dying notes only. */

	(PM->cut_notes)++;
	return lowest;
    }

    /* try to remove VOICE_DIE before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -1;
    for(j = 0; j < i; j++)
    {
      if((PM->voice)[j].status & VOICE_FREE || (PM->voice)[j].cache != NULL)
	    continue;
      if((PM->voice)[j].status & ~(VOICE_ON | VOICE_SUSTAINED))
      {
	/* continue protecting non-resample decays */
	if ((PM->voice)[j].status & ~(VOICE_DIE) && (PM->voice)[j].sample->note_to_use)
		continue;

	/* choose note which has been on the shortest amount of time */
	/* this is a VERY crude estimate... */
	if ((PM->voice)[j].sample->modes & MODES_LOOPING)
	    duration = (PM->voice)[j].sample_offset - (PM->voice)[j].sample->loop_start;
	else
	    duration = (PM->voice)[j].sample_offset;
	if ((PM->voice)[j].sample_increment > 0)
	    duration /= (PM->voice)[j].sample_increment;
	v = duration;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -1)
    {
	(PM->cut_notes)++;
	return lowest;
    }

    /* try to remove VOICE_SUSTAINED before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((PM->voice)[j].status & VOICE_FREE || (PM->voice)[j].cache != NULL)
	    continue;
      if((PM->voice)[j].status & VOICE_SUSTAINED)
      {
	/* choose note which has been on the shortest amount of time */
	/* this is a VERY crude estimate... */
	if ((PM->voice)[j].sample->modes & MODES_LOOPING)
	    duration = (PM->voice)[j].sample_offset - (PM->voice)[j].sample->loop_start;
	else
	    duration = (PM->voice)[j].sample_offset;
	if ((PM->voice)[j].sample_increment > 0)
	    duration /= (PM->voice)[j].sample_increment;
	v = duration;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(PM->cut_notes)++;
	return lowest;
    }

    /* try to remove chorus before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((PM->voice)[j].status & VOICE_FREE || (PM->voice)[j].cache != NULL)
	    continue;
      if((PM->voice)[j].chorus_link < j)
      {
	/* score notes based on both volume AND duration */
	/* this scoring function needs some more tweaking... */
	if ((PM->voice)[j].sample->modes & MODES_LOOPING)
	    duration = (PM->voice)[j].sample_offset - (PM->voice)[j].sample->loop_start;
	else
	    duration = (PM->voice)[j].sample_offset;
	if ((PM->voice)[j].sample_increment > 0)
	    duration /= (PM->voice)[j].sample_increment;
	v = (PM->voice)[j].left_mix * duration;
	vr = (PM->voice)[j].right_mix * duration;
	if((PM->voice)[j].panned == PANNED_MYSTERY && vr > v)
	    v = vr;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(PM->cut_notes)++;

	/* hack - double volume of chorus partner, fix pan */
	j = (PM->voice)[lowest].chorus_link;
	(PM->voice)[j].velocity <<= 1;
    	(PM->voice)[j].panning = (PM->channel)[(PM->voice)[lowest].channel].panning;
    	recompute_amp(tmdy_struct, j);
    	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, j);

	return lowest;
    }

    (PM->lost_notes)++;

    /* try to remove non-looping voices first */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((PM->voice)[j].status & VOICE_FREE || (PM->voice)[j].cache != NULL)
	    continue;
      if(!((PM->voice)[j].sample->modes & MODES_LOOPING))
      {
	/* score notes based on both volume AND duration */
	/* this scoring function needs some more tweaking... */
	duration = (PM->voice)[j].sample_offset;
	if ((PM->voice)[j].sample_increment > 0)
	    duration /= (PM->voice)[j].sample_increment;
	v = (PM->voice)[j].left_mix * duration;
	vr = (PM->voice)[j].right_mix * duration;
	if((PM->voice)[j].panned == PANNED_MYSTERY && vr > v)
	    v = vr;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	return lowest;
    }

    lv = 0x7FFFFFFF;
    lowest = 0;
    for(j = 0; j < i; j++)
    {
	if((PM->voice)[j].status & VOICE_FREE || (PM->voice)[j].cache != NULL)
	    continue;
	if (!((PM->voice)[j].sample->modes & MODES_LOOPING)) continue;

	/* score notes based on both volume AND duration */
	/* this scoring function needs some more tweaking... */
	duration = (PM->voice)[j].sample_offset - (PM->voice)[j].sample->loop_start;
	if ((PM->voice)[j].sample_increment > 0)
	    duration /= (PM->voice)[j].sample_increment;
	v = (PM->voice)[j].left_mix * duration;
	vr = (PM->voice)[j].right_mix * duration;
	if((PM->voice)[j].panned == PANNED_MYSTERY && vr > v)
	    v = vr;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
    }
    return lowest;
}
#endif

/* this reduces voices while maintaining sound quality */
static int reduce_voice(tmdy_struct_ex_t *tmdy_struct)
{
    int32 lv, v;
    int i, j, lowest=-0x7FFFFFFF;

    i = (PM->upper_voices);
    lv = 0x7FFFFFFF;
    
    /* Look for the decaying note with the smallest volume */
    /* Protect drum decays.  Truncating them early sounds bad, especially on
       snares and cymbals */
    for(j = 0; j < i; j++)
    {
	if((PM->voice)[j].status & VOICE_FREE ||
	   ((PM->voice)[j].sample->note_to_use && ISDRUMCHANNEL((PM->voice)[j].channel)))
	    continue;
	
	if((PM->voice)[j].status & ~(VOICE_ON | VOICE_DIE | VOICE_SUSTAINED))
	{
	    /* find lowest volume */
	    v = (PM->voice)[j].left_mix;
	    if((PM->voice)[j].panned == PANNED_MYSTERY && (PM->voice)[j].right_mix > v)
	    	v = (PM->voice)[j].right_mix;
	    if(v < lv)
	    {
		lv = v;
		lowest = j;
	    }
	}
    }
    if(lowest != -0x7FFFFFFF)
    {
	/* This can still cause a click, but if we had a free voice to
	   spare for ramping down this note, we wouldn't need to kill it
	   in the first place... Still, this needs to be fixed. Perhaps
	   we could use a reserve of voices to play dying notes only. */

	(PM->cut_notes)++;
	free_voice(tmdy_struct, lowest);
	if(!PM->prescanning_flag){
	    ctl_note_event(tmdy_struct, lowest);
	}
	return lowest;
    }

    /* try to remove VOICE_DIE before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -1;
    for(j = 0; j < i; j++)
    {
      if((PM->voice)[j].status & VOICE_FREE)
	    continue;
      if((PM->voice)[j].status & ~(VOICE_ON | VOICE_SUSTAINED))
      {
	/* continue protecting drum decays */
	if ((PM->voice)[j].status & ~(VOICE_DIE) &&
	    ((PM->voice)[j].sample->note_to_use && ISDRUMCHANNEL((PM->voice)[j].channel)))
		continue;
	/* find lowest volume */
	v = (PM->voice)[j].left_mix;
	if((PM->voice)[j].panned == PANNED_MYSTERY && (PM->voice)[j].right_mix > v)
	    v = (PM->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -1)
    {
	(PM->cut_notes)++;
	free_voice(tmdy_struct, lowest);
	if(!PM->prescanning_flag){
	    ctl_note_event(tmdy_struct, lowest);
	}
	return lowest;
    }

    /* try to remove VOICE_SUSTAINED before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((PM->voice)[j].status & VOICE_FREE)
	    continue;
      if((PM->voice)[j].status & VOICE_SUSTAINED)
      {
	/* find lowest volume */
	v = (PM->voice)[j].left_mix;
	if((PM->voice)[j].panned == PANNED_MYSTERY && (PM->voice)[j].right_mix > v)
	    v = (PM->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(PM->cut_notes)++;
	free_voice(tmdy_struct, lowest);
	if(!PM->prescanning_flag){
	    ctl_note_event(tmdy_struct, lowest);
	}
	return lowest;
    }

    /* try to remove chorus before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((PM->voice)[j].status & VOICE_FREE)
	    continue;
      if((PM->voice)[j].chorus_link < j)
      {
	/* find lowest volume */
	v = (PM->voice)[j].left_mix;
	if((PM->voice)[j].panned == PANNED_MYSTERY && (PM->voice)[j].right_mix > v)
	    v = (PM->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(PM->cut_notes)++;

	/* hack - double volume of chorus partner, fix pan */
	j = (PM->voice)[lowest].chorus_link;
	(PM->voice)[j].velocity <<= 1;
    	(PM->voice)[j].panning = (PM->channel)[(PM->voice)[lowest].channel].panning;
    	recompute_amp(tmdy_struct, j);
    	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, j);

	free_voice(tmdy_struct, lowest);
	if(!PM->prescanning_flag){
	    ctl_note_event(tmdy_struct, lowest);
	}
	return lowest;
    }

    (PM->lost_notes)++;

    /* remove non-drum VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
        if((PM->voice)[j].status & VOICE_FREE ||
	   ((PM->voice)[j].sample->note_to_use && ISDRUMCHANNEL((PM->voice)[j].channel)))
	   	continue;

	/* find lowest volume */
	v = (PM->voice)[j].left_mix;
	if((PM->voice)[j].panned == PANNED_MYSTERY && (PM->voice)[j].right_mix > v)
	    v = (PM->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
    }
    if(lowest != -0x7FFFFFFF)
    {
	free_voice(tmdy_struct, lowest);
	if(!PM->prescanning_flag){
	    ctl_note_event(tmdy_struct, lowest);
	}
	return lowest;
    }

    /* remove all other types of notes */
    lv = 0x7FFFFFFF;
    lowest = 0;
    for(j = 0; j < i; j++)
    {
	if((PM->voice)[j].status & VOICE_FREE)
	    continue;
	/* find lowest volume */
	v = (PM->voice)[j].left_mix;
	if((PM->voice)[j].panned == PANNED_MYSTERY && (PM->voice)[j].right_mix > v)
	    v = (PM->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
    }

    free_voice(tmdy_struct, lowest);
    if(!PM->prescanning_flag){
		ctl_note_event(tmdy_struct, lowest);
	}

    return lowest;
}



/* Only one instance of a note can be playing on a single channel. */
static int find_voice(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
  int i, j, lowest=-1, note, ch, status_check, mono_check;
  AlternateAssign *altassign;

  note = MIDI_EVENT_NOTE(e);
  ch = e->channel;

  if((PM->opt_overlap_voice_allow))
      status_check = (VOICE_OFF | VOICE_SUSTAINED);
  else
      status_check = 0xFF;
  mono_check = (PM->channel)[ch].mono;
  altassign = TMDY_INSTRUM->find_altassign(tmdy_struct, (PM->channel)[ch].altassign, note);

  i = (PM->upper_voices);
  for(j = 0; j < i; j++)
      if((PM->voice)[j].status == VOICE_FREE)
      {
	  lowest = j; /* lower volume */
	  break;
      }

	for (j = 0; j < i; j++)
		if ((PM->voice)[j].status != VOICE_FREE && (PM->voice)[j].channel == ch
				&& (((PM->voice)[j].status & status_check) && (PM->voice)[j].note == note
				|| mono_check
				|| (altassign && TMDY_INSTRUM->find_altassign(tmdy_struct, altassign, (PM->voice)[j].note))))
			kill_note(tmdy_struct, j);
		else if ((PM->voice)[j].status != VOICE_FREE && (PM->voice)[j].channel == ch
				&& (PM->voice)[j].note == note && (((PM->channel)[ch].assign_mode == 1
				&& (PM->voice)[j].proximate_flag == 0)
				|| (PM->channel)[ch].assign_mode == 0))
			kill_note(tmdy_struct, j);
	for (j = 0; j < i; j++)
		if ((PM->voice)[j].channel == ch && (PM->voice)[j].note == note)
			(PM->voice)[j].proximate_flag = 0;

  if(lowest != -1)
  {
      /* Found a free voice. */
      if((PM->upper_voices) <= lowest)
	  (PM->upper_voices) = lowest + 1;
  
      return lowest;
  }

  if(i < (PM->voices)){
 
     return (PM->upper_voices)++;
  }
  return reduce_voice(tmdy_struct);
}

void free_voice(tmdy_struct_ex_t *tmdy_struct, int v1)
{
    int v2;

    v2 = (PM->voice)[v1].chorus_link;
    if(v1 != v2)
    {
	/* Unlink chorus link */
	(PM->voice)[v1].chorus_link = v1;
	(PM->voice)[v2].chorus_link = v2;
    }
    (PM->voice)[v1].status = VOICE_FREE;
    (PM->voice)[v1].temper_instant = 0;
}

static int find_free_voice(tmdy_struct_ex_t *tmdy_struct)
{
    int i, nv = (PM->voices), lowest;
    int32 lv, v;

    for(i = 0; i < nv; i++)
	if((PM->voice)[i].status == VOICE_FREE)
	{
	    if((PM->upper_voices) <= i)
		  (PM->upper_voices) = i + 1;
	
	    return i;
	}

    (PM->upper_voices) = (PM->voices);

    /* Look for the decaying note with the lowest volume */
    lv = 0x7FFFFFFF;
    lowest = -1;
    for(i = 0; i < nv; i++)
    {
	if((PM->voice)[i].status & ~(VOICE_ON | VOICE_DIE) &&
	   !((PM->voice)[i].sample->note_to_use && ISDRUMCHANNEL((PM->voice)[i].channel)))
	{
	    v = (PM->voice)[i].left_mix;
	    if(((PM->voice)[i].panned==PANNED_MYSTERY) && ((PM->voice)[i].right_mix>v))
		v = (PM->voice)[i].right_mix;
	    if(v<lv)
	    {
		lv = v;
		lowest = i;
	    }
	}
    }
    if(lowest != -1 && !PM->prescanning_flag)
    {
	free_voice(tmdy_struct, lowest);
	ctl_note_event(tmdy_struct, lowest);
    }

    return lowest;
}

static int select_play_sample(tmdy_struct_ex_t *tmdy_struct, Sample *splist,
		int nsp, int note, int *vlist, MidiEvent *e)
{
	int32 f, fs, ft, fst, fc, cdiff, diff;
	int8 tt = (PM->channel)[e->channel].temper_type;
	uint8 tp = (PM->channel)[e->channel].rpnmap[RPN_ADDR_0003];
	Sample *sp, *spc;
	int16 st;
	double ratio;
	int i, j, nv, vel;
	
	if ((PM->opt_pure_intonation)) {
		if ((PM->current_keysig) < 8)
			f = freq_table_pureint[(PM->current_freq_table)][note];
		else
			f = freq_table_pureint[(PM->current_freq_table) + 12][note];
	} else if ((PM->opt_temper_control))
		switch (tt) {
		case 0:
			f = freq_table_tuning[tp][note];
			break;
		case 1:
			if (PM->current_temper_keysig < 8)
				f = freq_table_pytha[PM->current_freq_table][note];
			else
				f = freq_table_pytha[PM->current_freq_table + 12][note];
			break;
		case 2:
			if ((PM->current_temper_keysig) < 8)
				f = freq_table_meantone[PM->current_freq_table
						+ ((PM->temper_adj) ? 36 : 0)][note];
			else
				f = freq_table_meantone[PM->current_freq_table
						+ ((PM->temper_adj) ? 24 : 12)][note];
			break;
		case 3:
			if ((PM->current_temper_keysig) < 8)
				f = freq_table_pureint[PM->current_freq_table
						+ ((PM->temper_adj) ? 36 : 0)][note];
			else
				f = freq_table_pureint[PM->current_freq_table
						+ ((PM->temper_adj) ? 24 : 12)][note];
			break;
		default:	/* user-defined temperaments */
			if ((tt -= 0x40) >= 0 && tt < 4) {
				if ((PM->current_temper_keysig) < 8)
					f = freq_table_user[tt][PM->current_freq_table
							+ ((PM->temper_adj) ? 36 : 0)][note];
				else
					f = freq_table_user[tt][PM->current_freq_table
							+ ((PM->temper_adj) ? 24 : 12)][note];
			} else
				f = freq_table[note];
			break;
		}
	else
		f = freq_table[note];
	if ((PM->opt_temper_control))
		fs = (tt) ? freq_table[note] : freq_table_tuning[tp][note];
	else
		fs = freq_table[note];
	vel = e->b;
	nv = 0;
	for (i = 0, sp = splist; i < nsp; i++, sp++) {
		/* SF2 - Scale Tuning */
		if ((st = sp->scale_tuning) != 100) {
			ratio = pow(2.0, (note - 60) * (st - 100) / 1200.0);
			ft = f * ratio + 0.5, fst = fs * ratio + 0.5;
		} else
			ft = f, fst = fs;
		if (sp->low_freq <= fst && sp->high_freq >= fst
				&& sp->low_vel <= vel && sp->high_vel >= vel) {
			j = vlist[nv] = find_voice(tmdy_struct, e);
			(PM->voice)[j].orig_frequency = ft;
			MYCHECK((PM->voice)[j].orig_frequency);
			(PM->voice)[j].sample = sp;
			(PM->voice)[j].status = VOICE_ON;
			nv++;
		}
	}
	if (nv == 0) {
		cdiff = 0x7fffffff;
		for (i = 0, sp = splist; i < nsp; i++, sp++) {
			/* SF2 - Scale Tuning */
			if ((st = sp->scale_tuning) != 100) {
				ratio = pow(2.0, (note - 60) * (st - 100) / 1200.0);
				ft = f * ratio + 0.5, fst = fs * ratio + 0.5;
			} else
				ft = f, fst = fs;
			diff = abs(sp->root_freq - fst);
			if (diff < cdiff) {
				fc = ft;
				spc = sp;
				cdiff = diff;
			}
		}
		j = vlist[nv] = find_voice(tmdy_struct, e);
		(PM->voice)[j].orig_frequency = fc;
		MYCHECK((PM->voice)[j].orig_frequency);
		(PM->voice)[j].sample = spc;
		(PM->voice)[j].status = VOICE_ON;
		nv++;
	}
	return nv;
}

static int find_samples(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e, int *vlist)
{
	Instrument *ip;
	int i, nv, note, ch, prog, bk;

	ch = e->channel;
	if((PM->channel)[ch].special_sample > 0)
	{
	    SpecialPatch *s;

	    s = TMDY_INSTRUM->special_patch[(PM->channel)[ch].special_sample];
	    if(s == NULL)
	    {
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
			  "Strange: Special patch %d is not installed",
			  (PM->channel)[ch].special_sample);
		return 0;
	    }
	    note = e->a + (PM->channel)[ch].key_shift + (PM->note_key_offset) + (PM->key_adjust);

	    if(note < 0)
		note = 0;
	    else if(note > 127)
		note = 127;	
	    return select_play_sample(tmdy_struct, s->sample, s->samples, note, vlist, e);
	}

	bk = (PM->channel)[ch].bank;
	if(ISDRUMCHANNEL(ch))
	{
	    note = e->a & 0x7F;
	    TMDY_INSTRUM->instrument_map(tmdy_struct, (PM->channel)[ch].mapID, &bk, &note);
	    if(!(ip = play_midi_load_instrument(tmdy_struct, 1, bk, note))){
			return 0;	/* No instrument? Then we can't play. */
		}
		
		if(ip->type == INST_GUS && ip->samples != 1)
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
			  "Strange: percussion instrument with %d samples!",
			  ip->samples);
		if(ip->sample->note_to_use)
		note = ip->sample->note_to_use;
		if(ip->type == INST_SF2) {
			nv = select_play_sample(tmdy_struct, ip->sample, ip->samples, note, vlist, e);
			/* Replace the sample if the sample is cached. */
			if(!PM->prescanning_flag)
			{
				if(ip->sample->note_to_use)
				note = MIDI_EVENT_NOTE(e);

				for(i = 0; i < nv; i++)
				{
				int j;

				j = vlist[i];
				if(!(PM->opt_realtime_playing) && TMDY_RECACHE->allocate_cache_size > 0 &&
				   !(PM->channel)[ch].portamento)
				{
					(PM->voice)[j].cache = TMDY_RECACHE->resamp_cache_fetch(tmdy_struct, (PM->voice)[j].sample, note);
					if((PM->voice)[j].cache) /* cache hit */
					(PM->voice)[j].sample = (PM->voice)[j].cache->resampled;
				}
				else
					(PM->voice)[j].cache = NULL;
				}
			}
			return nv;
		} else {
			i = vlist[0] = find_voice(tmdy_struct, e);
			(PM->voice)[i].orig_frequency = freq_table[note];
			(PM->voice)[i].sample = ip->sample;
			(PM->voice)[i].status = VOICE_ON;
			return 1;
		}
	}

	prog = (PM->channel)[ch].program;
	if(prog == SPECIAL_PROGRAM)
	    ip = TMDY_INSTRUM->default_instrument;
	else
	{
	    TMDY_INSTRUM->instrument_map(tmdy_struct, (PM->channel)[ch].mapID, &bk, &prog);
	    if(!(ip = play_midi_load_instrument(tmdy_struct, 0, bk, prog))){
			return 0;	/* No instrument? Then we can't play. */
		}
	}

	if(ip->sample->note_to_use)
	    note = ip->sample->note_to_use + (PM->channel)[ch].key_shift + (PM->note_key_offset) + (PM->key_adjust);
	else
	    note = e->a + (PM->channel)[ch].key_shift + (PM->note_key_offset) + (PM->key_adjust);
	if(note < 0)
	    note = 0;
	else if(note > 127)
	    note = 127;

	nv = select_play_sample(tmdy_struct, ip->sample, ip->samples, note, vlist, e);

	/* Replace the sample if the sample is cached. */
	if(!PM->prescanning_flag)
	{
	    if(ip->sample->note_to_use)
		note = MIDI_EVENT_NOTE(e);

	    for(i = 0; i < nv; i++)
	    {
		int j;

		j = vlist[i];
		if(!(PM->opt_realtime_playing) && TMDY_RECACHE->allocate_cache_size > 0 &&
		   !(PM->channel)[ch].portamento)
		{
		    (PM->voice)[j].cache = TMDY_RECACHE->resamp_cache_fetch(tmdy_struct, (PM->voice)[j].sample, note);
		    if((PM->voice)[j].cache) /* cache hit */
			(PM->voice)[j].sample = (PM->voice)[j].cache->resampled;
		}
		else
		    (PM->voice)[j].cache = NULL;
	    }
	}
	return nv;
}

static int get_panning(tmdy_struct_ex_t *tmdy_struct, int ch, int note,int v)
{
    int i, pan;

	if((PM->voice)[v].sample_panning_average == -1) {	/* mono sample */
		if((PM->channel)[ch].panning != NO_PANNING) {pan = (int)(PM->channel)[ch].panning - 64;}
		else {pan = 0;}
		if(ISDRUMCHANNEL(ch) &&
		 (PM->channel)[ch].drums[note] != NULL &&
		 (PM->channel)[ch].drums[note]->drum_panning != NO_PANNING) {
			pan += (PM->channel)[ch].drums[note]->drum_panning;
		} else {
			pan += (PM->voice)[v].sample->panning;
		}
	} else {	/* stereo sample */
		if((PM->channel)[ch].panning != NO_PANNING) {pan = (int)(PM->channel)[ch].panning - 64;}
		else {pan = 0;}
		if(ISDRUMCHANNEL(ch) &&
		 (PM->channel)[ch].drums[note] != NULL &&
		 (PM->channel)[ch].drums[note]->drum_panning != NO_PANNING) {
			pan += (PM->channel)[ch].drums[note]->drum_panning - 64;
		}
		pan += (PM->voice)[v].sample->panning - (PM->voice)[v].sample_panning_average + 64;
	}

	if (pan > 127) pan = 127;
	else if (pan < 0) pan = 0;

	return pan;
}

static void calc_sample_panning_average(tmdy_struct_ex_t *tmdy_struct, int nv,int *vlist)
{
	int i, v, average = 0;

	if(!nv) {
		return;
	}	/* error! */
	else if(nv == 1) {	/* mono sample */
		v = vlist[0];
		(PM->voice)[v].sample_panning_average = -1;
		return;
	}

	for(i=0;i<nv;i++) {
		v = vlist[i];
		average += (PM->voice)[v].sample->panning;
	}
	average /= nv;

	for(i=0;i<nv;i++) {
		v = vlist[i];
		(PM->voice)[v].sample_panning_average = average;
	}
}

static void start_note(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e, int i, int vid, int cnt)
{
  int j, ch, note, pan;
#ifdef ENABLE_PAN_DELAY
  double pan_delay_diff;  
#endif

  ch = e->channel;

  note = MIDI_EVENT_NOTE(e);
  (PM->voice)[i].status = VOICE_ON;
  (PM->voice)[i].channel = ch;
  (PM->voice)[i].note = note;
  (PM->voice)[i].velocity = e->b;
  (PM->voice)[i].chorus_link = i;	/* No link */
  (PM->voice)[i].proximate_flag = 1;

  j = (PM->channel)[ch].special_sample;
  if(j == 0 || TMDY_INSTRUM->special_patch[j] == NULL)
      (PM->voice)[i].sample_offset = 0;
  else
  {
      (PM->voice)[i].sample_offset =
	  TMDY_INSTRUM->special_patch[j]->sample_offset << FRACTION_BITS;
      if((PM->voice)[i].sample->modes & MODES_LOOPING)
      {
	  if((PM->voice)[i].sample_offset > (PM->voice)[i].sample->loop_end)
	      (PM->voice)[i].sample_offset = (PM->voice)[i].sample->loop_start;
      }
      else if((PM->voice)[i].sample_offset > (PM->voice)[i].sample->data_length)
      {
	  free_voice(tmdy_struct, i);
	  return;
      }
  }
  (PM->voice)[i].sample_increment=0; /* make sure it isn't negative */
  (PM->voice)[i].modulation_wheel=(PM->channel)[ch].modulation_wheel;
  (PM->voice)[i].delay = (PM->voice)[i].sample->envelope_delay;
  (PM->voice)[i].modenv_delay = (PM->voice)[i].sample->modenv_delay;
  (PM->voice)[i].tremolo_delay = (PM->voice)[i].sample->tremolo_delay;
  (PM->voice)[i].vid=vid;

  (PM->voice)[i].tremolo_phase=0;
  (PM->voice)[i].tremolo_phase_increment=(PM->voice)[i].sample->tremolo_phase_increment;
  (PM->voice)[i].tremolo_sweep=(PM->voice)[i].sample->tremolo_sweep_increment;
  (PM->voice)[i].tremolo_sweep_position=0;
  (PM->voice)[i].tremolo_depth = (PM->voice)[i].sample->tremolo_depth;

  (PM->voice)[i].vibrato_sweep=(PM->voice)[i].sample->vibrato_sweep_increment;
  (PM->voice)[i].vibrato_sweep_position=0;

  (PM->voice)[i].delay_counter = 0;

  memset(&((PM->voice)[i].fc), 0, sizeof(FilterCoefficients));
  if((PM->opt_lpf_def) && (PM->voice)[i].sample->cutoff_freq) {
	  (PM->voice)[i].fc.orig_freq = (PM->voice)[i].sample->cutoff_freq;
	  (PM->voice)[i].fc.orig_reso_dB = (double)(PM->voice)[i].sample->resonance / 10.0f;
  } else {
	  (PM->voice)[i].fc.freq = -1;
  }

  if((PM->opt_nrpn_vibrato))
  {
	  if((PM->channel)[ch].vibrato_ratio) {
	      (PM->voice)[i].vibrato_control_ratio = (PM->voice)[i].sample->vibrato_control_ratio + (PM->channel)[ch].vibrato_ratio;
		  if((PM->voice)[i].vibrato_control_ratio < 0) {(PM->voice)[i].vibrato_control_ratio = 0;}
	  }
	  if((PM->channel)[ch].vibrato_depth) {
	      (PM->voice)[i].vibrato_depth = (PM->voice)[i].sample->vibrato_depth + (PM->channel)[ch].vibrato_depth;
		  if((PM->voice)[i].vibrato_depth > 255) {(PM->voice)[i].vibrato_depth = 255;}
		  else if((PM->voice)[i].vibrato_depth < -255) {(PM->voice)[i].vibrato_depth = -255;}
	  }
      (PM->voice)[i].vibrato_delay = (PM->voice)[i].sample->vibrato_delay + (PM->channel)[ch].vibrato_delay;
  }
  else
  {
      (PM->voice)[i].vibrato_control_ratio = (PM->voice)[i].sample->vibrato_control_ratio;
      (PM->voice)[i].vibrato_depth = (PM->voice)[i].sample->vibrato_depth;
      (PM->voice)[i].vibrato_delay = (PM->voice)[i].sample->vibrato_delay;
  }
  (PM->voice)[i].orig_vibrato_control_ratio = (PM->voice)[i].sample->vibrato_control_ratio;

  (PM->voice)[i].vibrato_control_counter=(PM->voice)[i].vibrato_phase=0;
  for (j=0; j<VIBRATO_SAMPLE_INCREMENTS; j++)
    (PM->voice)[i].vibrato_sample_increment[j]=0;

  /* Pan */
  (PM->voice)[i].panning = get_panning(tmdy_struct, ch, note, i);
  
#ifdef ENABLE_PAN_DELAY
  PM->voice[i].pan_delay_rpt = 0;
  if(PM->opt_pan_delay) {
	  if(PM->voice[i].panning == 64) {PM->voice[i].delay += pan_delay_table[64] * TMDY_OUTPUT->play_mode->rate / 1000;}
	  else {
		  if(pan_delay_table[PM->voice[i].panning] > pan_delay_table[127 - PM->voice[i].panning]) {
			  pan_delay_diff = pan_delay_table[PM->voice[i].panning] - pan_delay_table[127 - PM->voice[i].panning];
			  PM->voice[i].delay += (pan_delay_table[PM->voice[i].panning] - pan_delay_diff) * TMDY_OUTPUT->play_mode->rate / 1000;
		  } else {
			  pan_delay_diff = pan_delay_table[127 - PM->voice[i].panning] - pan_delay_table[PM->voice[i].panning];
			  PM->voice[i].delay += (pan_delay_table[127 - PM->voice[i].panning] - pan_delay_diff) * TMDY_OUTPUT->play_mode->rate / 1000;
		  }
		  PM->voice[i].pan_delay_rpt = pan_delay_diff * TMDY_OUTPUT->play_mode->rate / 1000;
	  }
	  memset(PM->voice[i].pan_delay_buf, 0, sizeof(PM->voice[i].pan_delay_buf));
	  if(PM->voice[i].pan_delay_rpt < 1) {PM->voice[i].pan_delay_rpt = 0;}
	  PM->voice[i].pan_delay_wpt = PM->voice[i].pan_delay_rpt - 1;
  }
#endif	/* ENABLE_PAN_DELAY */


  (PM->voice)[i].porta_control_counter = 0;
  if((PM->channel)[ch].legato && (PM->channel)[ch].legato_flag) {
	  update_legato_controls(tmdy_struct, ch);
  }
  else if((PM->channel)[ch].portamento && !(PM->channel)[ch].porta_control_ratio)
      update_portamento_controls(tmdy_struct, ch);
  if((PM->channel)[ch].porta_control_ratio)
  {
      if((PM->channel)[ch].last_note_fine == -1)
      {
	  /* first on */
	  (PM->channel)[ch].last_note_fine = (PM->voice)[i].note * 256;
	  (PM->channel)[ch].porta_control_ratio = 0;
      }
      else
      {
	  (PM->voice)[i].porta_control_ratio = (PM->channel)[ch].porta_control_ratio;
	  (PM->voice)[i].porta_dpb = (PM->channel)[ch].porta_dpb;
	  (PM->voice)[i].porta_pb = (PM->channel)[ch].last_note_fine -
	      (PM->voice)[i].note * 256;
	  if((PM->voice)[i].porta_pb == 0)
	      (PM->voice)[i].porta_control_ratio = 0;
      }
  }

  if(cnt == 0)
      (PM->channel)[ch].last_note_fine = (PM->voice)[i].note * 256;

  recompute_amp(tmdy_struct, i);
  if ((PM->voice)[i].sample->modes & MODES_ENVELOPE)
    {
      /* Ramp up from 0 */
	  (PM->voice)[i].envelope_stage=0;
      (PM->voice)[i].envelope_volume=0;
      (PM->voice)[i].control_counter=0;
      TMDY_MIX->recompute_envelope(tmdy_struct, i);
	  TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	  (PM->voice)[i].modenv_stage = 0;
      (PM->voice)[i].modenv_volume = 0;
      TMDY_MIX->recompute_modulation_envelope(tmdy_struct, i);
	  TMDY_MIX->apply_modulation_envelope(tmdy_struct, i);

    }
  else
    {
      (PM->voice)[i].envelope_increment=0;
	  (PM->voice)[i].modenv_increment=0;
      TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	  TMDY_MIX->apply_modulation_envelope(tmdy_struct, i);
    }
  recompute_freq(tmdy_struct, i);
  recompute_voice_filter(tmdy_struct, i);

  (PM->voice)[i].timeout = -1;
  if(!PM->prescanning_flag){
  
      ctl_note_event(tmdy_struct, i);
  }
}

static void finish_note(tmdy_struct_ex_t *tmdy_struct, int i)
{
    if ((PM->voice)[i].sample->modes & MODES_ENVELOPE)
    {
	/* We need to get the envelope out of Sustain stage. */
	/* Note that (PM->voice)[i].envelope_stage < 3 */
	(PM->voice)[i].status=VOICE_OFF;
	(PM->voice)[i].envelope_stage=3;
	TMDY_MIX->recompute_envelope(tmdy_struct, i);
	(PM->voice)[i].modenv_stage=3;
	TMDY_MIX->recompute_modulation_envelope(tmdy_struct, i);
	TMDY_MIX->apply_modulation_envelope(tmdy_struct, i);
	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	ctl_note_event(tmdy_struct, i);
	}
    else
    {
	if(TMDY_READMIDI->current_file_info->pcm_mode != PCM_MODE_NON)
	{
	    free_voice(tmdy_struct, i);	
	    ctl_note_event(tmdy_struct, i);
	}
	else
	{
	    /* Set status to OFF so resample_voice() will let this voice out
		of its loop, if any. In any case, this voice dies when it
		    hits the end of its data (ofs>=data_length). */
	    if((PM->voice)[i].status != VOICE_OFF)
	    {
		(PM->voice)[i].status = VOICE_OFF;
		ctl_note_event(tmdy_struct, i);
	    }
	}
    }
}

static void set_envelope_time(tmdy_struct_ex_t *tmdy_struct, int ch,int val,int stage)
{
	val = val & 0x7F;
	switch(stage) {
	case 0:	/* Attack */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Attack Time (CH:%d VALUE:%d)",ch,val);
		break;
	case 2: /* Decay */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Decay Time (CH:%d VALUE:%d)",ch,val);
		break;
	case 3:	/* Release */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Release Time (CH:%d VALUE:%d)",ch,val);
		break;
	default:
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"? Time (CH:%d VALUE:%d)",ch,val);
	}
	(PM->channel)[ch].envelope_rate[stage] = val;
}

static void new_delay_voice(tmdy_struct_ex_t *tmdy_struct, int v1, int level)
{
    int v2,ch=(PM->voice)[v1].channel;
	FLOAT_T delay,vol;
	FLOAT_T threshold = 1.0;

	/* NRPN Delay Send Level of Drum */
	if(ISDRUMCHANNEL(ch) &&	(PM->channel)[ch].drums[(PM->voice)[v1].note] != NULL) {
		level *= (FLOAT_T)(PM->channel)[ch].drums[(PM->voice)[v1].note]->delay_level / 127.0;
	}

	vol = (PM->voice)[v1].velocity * level / 127.0 * TMDY_REVERB->delay_status.level_ratio_c;

	if (vol > threshold) {
		delay = 0;
		if((v2 = find_free_voice(tmdy_struct)) == -1) {
			return;
		}
		(PM->voice)[v2].cache = NULL;
		delay += TMDY_REVERB->delay_status.time_center;
		(PM->voice)[v2] = (PM->voice)[v1];	/* copy all parameters */
		(PM->voice)[v2].velocity = (uint8)vol;
		(PM->voice)[v2].delay += (int32)((TMDY_OUTPUT->play_mode)->rate * delay / 1000);
		recompute_amp(tmdy_struct, v2);
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);
		recompute_freq(tmdy_struct, v2);
	}
}


static void new_chorus_voice(tmdy_struct_ex_t *tmdy_struct, int v1, int level)
{
    int v2, ch;
    uint8 vol;

    if((v2 = find_free_voice(tmdy_struct)) == -1){
		return;
	}
    ch = (PM->voice)[v1].channel;
    vol = (PM->voice)[v1].velocity;
    (PM->voice)[v2] = (PM->voice)[v1];	/* copy all parameters */

	/* NRPN Chorus Send Level of Drum */
	if(ISDRUMCHANNEL(ch) && (PM->channel)[ch].drums[(PM->voice)[v1].note] != NULL) {
		level *= (FLOAT_T)(PM->channel)[ch].drums[(PM->voice)[v1].note]->chorus_level / 127.0;
	}

    /* Choose lower voice index for base voice (v1) */
    if(v1 > v2)
    {
	int tmp;
	tmp = v1;
	v1 = v2;
	v2 = v1;
    }

    /* v1: Base churos voice
     * v2: Sub chorus voice (detuned)
     */

    (PM->voice)[v1].velocity = (uint8)(vol * CHORUS_VELOCITY_TUNING1);
    (PM->voice)[v2].velocity = (uint8)(vol * CHORUS_VELOCITY_TUNING2);

    /* Make doubled link v1 and v2 */
    (PM->voice)[v1].chorus_link = v2;
    (PM->voice)[v2].chorus_link = v1;

    level >>= 2;		     /* scale level to a "better" value */
    if((PM->channel)[ch].pitchbend + level < 0x2000)
        (PM->voice)[v2].orig_frequency *= bend_fine[level];
    else
	(PM->voice)[v2].orig_frequency /= bend_fine[level];

    MYCHECK((PM->voice)[v2].orig_frequency);

    (PM->voice)[v2].cache = NULL;

    /* set panning & delay */
    if(!((TMDY_OUTPUT->play_mode)->encoding & PE_MONO))
    {
	double delay;

	if((PM->voice)[v2].panned == PANNED_CENTER)
	{
	    (PM->voice)[v2].panning = 64 + TMDY_COMMON->int_rand(tmdy_struct, 40) - 20; /* 64 +- rand(20) */
	    delay = 0;
	}
	else
	{
	    int panning = (PM->voice)[v2].panning;

	    if(panning < CHORUS_OPPOSITE_THRESHOLD)
	    {
		(PM->voice)[v2].panning = 127;
		delay = DEFAULT_CHORUS_DELAY1;
	    }
	    else if(panning > 127 - CHORUS_OPPOSITE_THRESHOLD)
	    {
		(PM->voice)[v2].panning = 0;
		delay = DEFAULT_CHORUS_DELAY1;
	    }
	    else
	    {
		(PM->voice)[v2].panning = (panning < 64 ? 0 : 127);
		delay = DEFAULT_CHORUS_DELAY2;
	    }
	}
	(PM->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
    }

    recompute_amp(tmdy_struct, v1);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v1);
    recompute_amp(tmdy_struct, v2);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);

    /* (PM->voice)[v2].orig_frequency is changed.
     * Update the depened parameters.
     */
    recompute_freq(tmdy_struct, v2);
}

/* Yet another chorus implementation
 *	by Eric A. Welsh <ewelsh@gpc.wustl.edu>.
 */
static void new_chorus_voice_alternate(tmdy_struct_ex_t *tmdy_struct, int v1, int level)
{
    int v2, ch, panlevel;
    uint8 vol, pan;
    double delay;

    if((v2 = find_free_voice(tmdy_struct)) == -1){
		return;
	}
    ch = (PM->voice)[v1].channel;
    (PM->voice)[v2] = (PM->voice)[v1];

	/* NRPN Chorus Send Level of Drum */
	if(ISDRUMCHANNEL(ch) && (PM->channel)[ch].drums[(PM->voice)[v1].note] != NULL) {
		level *= (FLOAT_T)(PM->channel)[ch].drums[(PM->voice)[v1].note]->chorus_level / 127.0;
	}

    /* for our purposes, hard left will be equal to 1 instead of 0 */
    pan = (PM->voice)[v1].panning;
    if (!pan) pan = 1;

    /* Choose lower voice index for base voice (v1) */
    if(v1 > v2)
    {
	int tmp;
	tmp = v1;
	v1 = v2;
	v2 = v1;
    }

    /* lower the volumes so that the two notes add to roughly the orig. vol */
    vol = (PM->voice)[v1].velocity;
    (PM->voice)[v1].velocity  = (uint8)(vol * CHORUS_VELOCITY_TUNING1);
    (PM->voice)[v2].velocity  = (uint8)(vol * CHORUS_VELOCITY_TUNING1);

    /* Make doubled link v1 and v2 */
    (PM->voice)[v1].chorus_link = v2;
    (PM->voice)[v2].chorus_link = v1;

    /* detune notes for chorus effect */
    level >>= 2;		/* scale to a "better" value */
    if (level)
    {
        if((PM->channel)[ch].pitchbend + level < 0x2000)
            (PM->voice)[v2].orig_frequency *= bend_fine[level];
        else
	    (PM->voice)[v2].orig_frequency /= bend_fine[level];
        (PM->voice)[v2].cache = NULL;
    }

    MYCHECK((PM->voice)[v2].orig_frequency);

    delay = 0.003;

    /* Try to keep the delayed voice from cancelling out the other voice */
    /* Don't bother with trying to figure out drum pitches... */
    /* Don't bother with mod files for the same reason... */
    /* Drums and mods could be fixed, but pitch detection is too expensive */
    if (!ISDRUMCHANNEL((PM->voice)[v1].channel) && !(IS_CURRENT_MOD_FILE))
    {
    	double freq, frac;
    
    	freq = TMDY_FREQ->pitch_freq_table[(PM->voice)[v1].note];
    	delay *= freq;
    	frac = delay - floor(delay);

	/* force the delay away from 0.5 period */
    	if (frac < 0.5 && frac > 0.40)
    	{
    	    delay = (floor(delay) + 0.40) / freq;
    	    if ((TMDY_OUTPUT->play_mode)->encoding & ~PE_MONO)
    	    	delay += (0.5 - frac) * (1.0 - labs(64 - pan) / 63.0) / freq;
    	}
    	else if (frac >= 0.5 && frac < 0.60)
    	{
    	    delay = (floor(delay) + 0.60) / freq;
    	    if ((TMDY_OUTPUT->play_mode)->encoding & ~PE_MONO)
    	    	delay += (0.5 - frac) * (1.0 - labs(64 - pan) / 63.0) / freq;
    	}
    	else
	    delay = 0.003;
    }

    /* set panning & delay for pseudo-surround effect */
    if((TMDY_OUTPUT->play_mode)->encoding & PE_MONO)    /* delay sounds good */
        (PM->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
    else
    {
        panlevel = 63;
        if (pan - panlevel < 1) panlevel = pan - 1;
        if (pan + panlevel > 127) panlevel = 127 - pan;
        (PM->voice)[v1].panning -= panlevel;
        (PM->voice)[v2].panning += panlevel;

        /* choose which voice is delayed based on panning */
        if ((PM->voice)[v1].panned == PANNED_CENTER) {
            /* randomly choose which voice is delayed */
            if (TMDY_COMMON->int_rand(tmdy_struct, 2))
                (PM->voice)[v1].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
            else
                (PM->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
        }
        else if (pan - 64 < 0) {
            (PM->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
        }
        else {
            (PM->voice)[v1].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
        }
    }

    recompute_amp(tmdy_struct, v1);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v1);
    recompute_amp(tmdy_struct, v2);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);
    if (level) recompute_freq(tmdy_struct, v2);
}

static void note_on(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
    int i, nv, v, ch, note;
    int vlist[32];
    int vid;

    if((nv = find_samples(tmdy_struct, e, vlist)) == 0){
		return;
	}

    note = MIDI_EVENT_NOTE(e);
    vid = new_vidq(tmdy_struct, e->channel, note);
    ch = e->channel;

	recompute_bank_parameter(tmdy_struct, ch,note);
	recompute_channel_filter(tmdy_struct, e);
	calc_sample_panning_average(tmdy_struct, nv, vlist);

    for(i = 0; i < nv; i++)
    {
	v = vlist[i];
	if(ISDRUMCHANNEL(ch) &&
	   (PM->channel)[ch].drums[note] != NULL &&
	   (PM->channel)[ch].drums[note]->pan_random)
	    (PM->channel)[ch].drums[note]->drum_panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	else if((PM->channel)[ch].pan_random)
	{
	    (PM->channel)[ch].panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	    ctl_mode_event(tmdy_struct, CTLE_PANNING, 1, ch, (PM->channel)[ch].panning);
	}
	start_note(tmdy_struct, e, v, vid, nv - i - 1);
#ifdef SMOOTH_MIXING
	(PM->voice)[v].old_left_mix = (PM->voice)[v].old_right_mix =
	(PM->voice)[v].left_mix_inc = (PM->voice)[v].left_mix_offset =
	(PM->voice)[v].right_mix_inc = (PM->voice)[v].right_mix_offset = 0;
#endif
#ifdef USE_DSP_EFFECT
	if((PM->opt_surround_chorus))
	    new_chorus_voice_alternate(tmdy_struct, v, 0);
#else
	if(((PM->channel)[ch].chorus_level || (PM->opt_surround_chorus)))
	{
	    if((PM->opt_surround_chorus))
		new_chorus_voice_alternate(tmdy_struct, v, (PM->channel)[ch].chorus_level);
	    else
		new_chorus_voice(tmdy_struct, v, (PM->channel)[ch].chorus_level);
	}
	if((PM->channel)[ch].delay_level)
	{
	    new_delay_voice(tmdy_struct, v, (PM->channel)[ch].delay_level);
	}
#endif
    }

    (PM->channel)[ch].legato_flag = 1;

}

static void set_voice_timeout(tmdy_struct_ex_t *tmdy_struct, Voice *vp, int ch, int note)
{
    int prog;
    ToneBank *bank;

    if((PM->channel)[ch].special_sample > 0){
		return;
	}
	
    if(ISDRUMCHANNEL(ch))
    {
	prog = note;
	bank = TMDY_INSTRUM->drumset[(int)(PM->channel)[ch].bank];
	if(bank == NULL)
	    bank = TMDY_INSTRUM->drumset[0];
    }
    else
    {
	if((prog = (PM->channel)[ch].program) == SPECIAL_PROGRAM){
	
	    return;
	}
	bank = TMDY_INSTRUM->tonebank[(int)(PM->channel)[ch].bank];
	if(bank == NULL)
	    bank = TMDY_INSTRUM->tonebank[0];
    }

    if(bank->tone[prog].loop_timeout > 0)
	vp->timeout = (int32)(bank->tone[prog].loop_timeout
			      * (TMDY_OUTPUT->play_mode)->rate * (PM->midi_time_ratio)
			      + (PM->current_sample));
}

static void note_off(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
  int uv = (PM->upper_voices), i;
  int ch, note, vid, sustain;

  ch = e->channel;
  note = MIDI_EVENT_NOTE(e);

  if(ISDRUMCHANNEL(ch) && (PM->channel)[ch].drums[note] != NULL &&
     !(PM->channel)[ch].drums[note]->rx_note_off)
  {
      ToneBank *bank;

      bank = TMDY_INSTRUM->drumset[(PM->channel)[ch].bank];
      if(bank == NULL) bank = TMDY_INSTRUM->drumset[0];
      
      /* uh oh, this drum doesn't have an instrument loaded yet */
      if (bank->tone[note].instrument == NULL){
      
          return;
      }

      /* this drum is not loaded for some reason (error occured?) */
      if (IS_MAGIC_INSTRUMENT(bank->tone[note].instrument)){
      
          return;
      }

      /* only disallow Note Off if the drum sample is not looped */
      if (!(bank->tone[note].instrument->sample->modes & MODES_LOOPING)){
	  	return;	/* Note Off is not allowed. */
	  }
  }

  if((vid = last_vidq(tmdy_struct, ch, note)) == -1){
  
      return;
  }
  sustain = (PM->channel)[ch].sustain;
  for(i = 0; i < uv; i++)
  {
      if((PM->voice)[i].status==VOICE_ON &&
	 (PM->voice)[i].channel==ch &&
	 (PM->voice)[i].note==note &&
	 (PM->voice)[i].vid==vid)
      {
	  if(sustain)
	  {
	      (PM->voice)[i].status=VOICE_SUSTAINED;
	      set_voice_timeout(tmdy_struct, PM->voice + i, ch, note);
	  
		  ctl_note_event(tmdy_struct, i);

	  }
	  else
	      finish_note(tmdy_struct, i);
      }
  }

  (PM->channel)[ch].legato_flag = 0;
}

/* Process the All Notes Off event */
static void all_notes_off(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (PM->upper_voices);

  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "All notes off on channel %d", c);
  for(i = 0; i < uv; i++)
    if ((PM->voice)[i].status==VOICE_ON &&
	(PM->voice)[i].channel==c)
      {
	if ((PM->channel)[c].sustain)
	  {
	    (PM->voice)[i].status=VOICE_SUSTAINED;
	    set_voice_timeout(tmdy_struct, PM->voice + i, c, (PM->voice)[i].note);
	    ctl_note_event(tmdy_struct, i);
	  }
	else
	  finish_note(tmdy_struct, i);
      }
  for(i = 0; i < 128; i++)
      (PM->vidq_head)[c * 128 + i] = (PM->vidq_tail)[c * 128 + i] = 0;
}

/* Process the All Sounds Off event */
static void all_sounds_off(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (PM->upper_voices);
  
  for(i = 0; i < uv; i++)
    if ((PM->voice)[i].channel==c &&
	((PM->voice)[i].status & ~(VOICE_FREE | VOICE_DIE)))
      {
	kill_note(tmdy_struct, i);
      }
  for(i = 0; i < 128; i++)
      (PM->vidq_head)[c * 128 + i] = (PM->vidq_tail)[c * 128 + i] = 0;
  }

static void adjust_pressure(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
 /*   int i, uv = (PM->upper_voices);
    int note, ch;
	FLOAT_T pressure, amp_ctl, rate_ctl1, pitch_depth1, cutoff_ctl, coef;

	ch = e->channel;
	pressure = (FLOAT_T)e->b / 127.0f;
	amp_ctl = (PM->channel)[ch].caf_amp_ctl * pressure;
	rate_ctl1 = (PM->channel)[ch].caf_rate_ctl1 * pressure + 1.0f;
	pitch_depth1 = (PM->channel)[ch].caf_pitch_depth1 * pressure + 1.0f;
	cutoff_ctl = (PM->channel)[ch].caf_cutoff_ctl * pressure + 1.0f;
    note = MIDI_EVENT_NOTE(e);

    for(i = 0; i < uv; i++)
    if((PM->voice)[i].status == VOICE_ON &&
       (PM->voice)[i].channel == ch &&
       (PM->voice)[i].note == note)
    {
		recompute_amp(tmdy_struct, i);
		(PM->voice)[i].tremolo_depth = amp_ctl * 255;
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		(PM->voice)[i].vibrato_control_ratio *= rate_ctl1;
		(PM->voice)[i].vibrato_depth *= pitch_depth1;
		recompute_freq(tmdy_struct, i);
		if((PM->opt_lpf_def) && (PM->voice)[i].sample->cutoff_freq) {
			coef = (PM->channel)[ch].cutoff_freq_coef;
			(PM->channel)[ch].cutoff_freq_coef *= cutoff_ctl;
			recompute_voice_filter(tmdy_struct, i);
			(PM->channel)[ch].cutoff_freq_coef = coef;
		}
    }
    */
}

static void adjust_channel_pressure(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
 /*
 	if((PM->opt_channel_pressure))
    {
	int i, uv = (PM->upper_voices);
	int ch;
	FLOAT_T pressure, amp_ctl, rate_ctl1, pitch_depth1, cutoff_ctl, coef;

	ch = e->channel;
	pressure = (FLOAT_T)e->a / 127.0f;
	amp_ctl = (PM->channel)[ch].caf_amp_ctl * pressure;
	rate_ctl1 = (PM->channel)[ch].caf_rate_ctl1 * pressure + 1.0f;
	pitch_depth1 = (PM->channel)[ch].caf_pitch_depth1 * pressure + 1.0f;
	cutoff_ctl = (PM->channel)[ch].caf_cutoff_ctl * pressure + 1.0f;
	  
	for(i = 0; i < uv; i++)
	{
	    if((PM->voice)[i].status == VOICE_ON && (PM->voice)[i].channel == ch)
	    {
		recompute_amp(tmdy_struct, i);
		(PM->voice)[i].tremolo_depth = amp_ctl * 255;
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		(PM->voice)[i].vibrato_control_ratio *= rate_ctl1;
		(PM->voice)[i].vibrato_depth *= pitch_depth1;
		recompute_freq(tmdy_struct, i);
		if((PM->opt_lpf_def) && (PM->voice)[i].sample->cutoff_freq) {
			coef = (PM->channel)[ch].cutoff_freq_coef;
			(PM->channel)[ch].cutoff_freq_coef *= cutoff_ctl;
			recompute_voice_filter(tmdy_struct, i);
			(PM->channel)[ch].cutoff_freq_coef = coef;
		}
	    }
	}
    }
    */
}

static void adjust_panning(tmdy_struct_ex_t *tmdy_struct, int c)
{
    int i, uv = (PM->upper_voices), pan = (PM->channel)[c].panning;
    
    for(i = 0; i < uv; i++)
    {
	if (((PM->voice)[i].channel==c) &&
	    ((PM->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED)))
	{
            /* adjust pan to include drum/sample pan offsets */
			pan = get_panning(tmdy_struct, c,i,i);

	    /* Hack to handle -EFchorus=2 in a "reasonable" way */
#ifdef USE_DSP_EFFECT
	    if((PM->opt_surround_chorus) && (PM->voice)[i].chorus_link != i)
#else
	    if(((PM->channel)[c].chorus_level || (PM->opt_surround_chorus)) &&
	       (PM->voice)[i].chorus_link != i)
#endif
	    {
		int v1, v2;

		if(i >= (PM->voice)[i].chorus_link)
		    /* `i' is not base chorus voice.
		     *  This sub voice is already updated.
		     */
		    continue;

		v1 = i;				/* base voice */
		v2 = (PM->voice)[i].chorus_link;	/* sub voice (detuned) */

		if((PM->opt_surround_chorus)) /* Surround chorus mode by Eric. */
		{
		    int panlevel;

		    if (!pan) pan = 1;	/* make hard left be 1 instead of 0 */
		    panlevel = 63;
		    if (pan - panlevel < 1) panlevel = pan - 1;
		    if (pan + panlevel > 127) panlevel = 127 - pan;
		    (PM->voice)[v1].panning = pan - panlevel;
		    (PM->voice)[v2].panning = pan + panlevel;
		}
		else
		{
		    (PM->voice)[v1].panning = pan;
		    if(pan > 60 && pan < 68) /* PANNED_CENTER */
			(PM->voice)[v2].panning =
			    64 + TMDY_COMMON->int_rand(tmdy_struct, 40) - 20; /* 64 +- rand(20) */
		    else if(pan < CHORUS_OPPOSITE_THRESHOLD)
			(PM->voice)[v2].panning = 127;
		    else if(pan > 127 - CHORUS_OPPOSITE_THRESHOLD)
			(PM->voice)[v2].panning = 0;
		    else
			(PM->voice)[v2].panning = (pan < 64 ? 0 : 127);
		}
		recompute_amp(tmdy_struct, v2);
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);
		/* v1 == i, so v1 will be updated next */
	    }
	    else
		(PM->voice)[i].panning = pan;

	    recompute_amp(tmdy_struct, i);
	    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	}
    }
}

void play_midi_setup_drums(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    (PM->channel)[ch].drums[note] = (struct DrumParts *)
	TMDY_UTILS->mblock->new_segment(tmdy_struct, &PM->playmidi_pool, sizeof(struct DrumParts));
    reset_drum_controllers(tmdy_struct, (PM->channel)[ch].drums, note);
}

static void adjust_drum_panning(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    int i, uv = (PM->upper_voices), pan;

    for(i = 0; i < uv; i++) {
		if((PM->voice)[i].channel == ch &&
		   (PM->voice)[i].note == note &&
		   ((PM->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED)))
		{
			(PM->voice)[i].panning = get_panning(tmdy_struct, ch,note,i);
			recompute_amp(tmdy_struct, i);
			TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		}
	}
}

static void drop_sustain(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (PM->upper_voices);
  for(i = 0; i < uv; i++)
    if ((PM->voice)[i].status==VOICE_SUSTAINED && (PM->voice)[i].channel==c)
      finish_note(tmdy_struct, i);
}

static void adjust_pitch(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (PM->upper_voices);
  for(i = 0; i < uv; i++)
    if ((PM->voice)[i].status!=VOICE_FREE && (PM->voice)[i].channel==c)
	recompute_freq(tmdy_struct, i);
}

static void adjust_volume(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (PM->upper_voices);
  for(i = 0; i < uv; i++)
    if ((PM->voice)[i].channel==c &&
	((PM->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED)))
      {
	recompute_amp(tmdy_struct, i);
	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
      }
}

static void set_reverb_level(tmdy_struct_ex_t *tmdy_struct, int ch, int level)
{
	if (level == -1) {
		PM->channel[ch].reverb_level = PM->channel[ch].reverb_id =
				(PM->opt_reverb_control < 0)
				? -PM->opt_reverb_control & 0x7f : DEFAULT_REVERB_SEND_LEVEL;
		PM->make_rvid_flag = 1;
		return;
	}
	PM->channel[ch].reverb_level = level;
	PM->make_rvid_flag = 0;	/* to update reverb_id */
}

int get_reverb_level(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	if (PM->channel[ch].reverb_level == -1)
		return (PM->opt_reverb_control < 0)
			? -PM->opt_reverb_control & 0x7f : DEFAULT_REVERB_SEND_LEVEL;
	return PM->channel[ch].reverb_level;
}

int get_chorus_level(tmdy_struct_ex_t *tmdy_struct, int ch)
{
#ifdef DISALLOW_DRUM_BENDS
    if(ISDRUMCHANNEL(ch))
	return 0; /* Not supported drum channel chorus */
#endif
    if((PM->opt_chorus_control) == 1){
		return (PM->channel)[ch].chorus_level;
	}
    return -(PM->opt_chorus_control);
}

static void make_rvid(tmdy_struct_ex_t *tmdy_struct)
{
    int i, j, lv, maxrv;

    for(maxrv = MAX_CHANNELS - 1; maxrv >= 0; maxrv--)
    {
	if((PM->channel)[maxrv].reverb_level == -1)
	    (PM->channel)[maxrv].reverb_id = -1;
	else if((PM->channel)[maxrv].reverb_level >= 0)
	    break;
    }

    /* collect same reverb level. */
    for(i = 0; i <= maxrv; i++)
    {
	if((lv = (PM->channel)[i].reverb_level) == -1)
	{
	    (PM->channel)[i].reverb_id = -1;
	    continue;
	}
	(PM->channel)[i].reverb_id = i;
	for(j = 0; j < i; j++)
	{
	    if((PM->channel)[j].reverb_level == lv)
	    {
		(PM->channel)[i].reverb_id = j;
		break;
	    }
	}
    }
}

static void adjust_master_volume(tmdy_struct_ex_t *tmdy_struct)
{
  int i, uv = (PM->upper_voices);

  adjust_amplification(tmdy_struct);
  for(i = 0; i < uv; i++)
      if((PM->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED))
      {
	  recompute_amp(tmdy_struct, i);
	  TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
      }
}

int midi_drumpart_change(tmdy_struct_ex_t *tmdy_struct, int ch, int isdrum)
{
    if(IS_SET_CHANNELMASK((PM->drumchannel_mask), ch)){
		return 0;
	}
    if(isdrum)
    {
	SET_CHANNELMASK((PM->drumchannels), ch);
	SET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, ch);
    }
    else
    {
	UNSET_CHANNELMASK((PM->drumchannels), ch);
	UNSET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, ch);
    }


    return 1;
}

void midi_program_change(tmdy_struct_ex_t *tmdy_struct, int ch, int prog)
{
	int dr = ISDRUMCHANNEL(ch);
	int newbank, b, p;
	
	switch (TMDY_READMIDI->play_system_mode) {
	case GS_SYSTEM_MODE:	/* GS */
		switch ((PM->channel)[ch].bank_lsb) {
		case 0:		/* No change */
			break;
		case 1:
			(PM->channel)[ch].mapID = (dr) ? SC_55_DRUM_MAP : SC_55_TONE_MAP;
			break;
		case 2:
			(PM->channel)[ch].mapID = (dr) ? SC_88_DRUM_MAP : SC_88_TONE_MAP;
			break;
		case 3:
			(PM->channel)[ch].mapID = (dr) ? SC_88PRO_DRUM_MAP : SC_88PRO_TONE_MAP;
			break;
		default:
			break;
		}
		newbank = (PM->channel)[ch].bank_msb;
		break;
	case XG_SYSTEM_MODE:	/* XG */
		switch ((PM->channel)[ch].bank_msb) {
		case 0:		/* Normal */
			if (ch == 9 && (PM->channel)[ch].bank_lsb == 127
					&& (PM->channel)[ch].mapID == XG_DRUM_MAP)
				/* FIXME: Why this part is drum?  Is this correct? */
				break;
			midi_drumpart_change(tmdy_struct, ch, 0);
			(PM->channel)[ch].mapID = XG_NORMAL_MAP;
			break;
		case 64:	/* SFX voice */
			midi_drumpart_change(tmdy_struct, ch, 0);
			(PM->channel)[ch].mapID = XG_SFX64_MAP;
			break;
		case 126:	/* SFX kit */
			midi_drumpart_change(tmdy_struct, ch, 1);
			(PM->channel)[ch].mapID = XG_SFX126_MAP;
			break;
		case 127:	/* Drumset */
			midi_drumpart_change(tmdy_struct, ch, 1);
			(PM->channel)[ch].mapID = XG_DRUM_MAP;
			break;
		default:
			break;
		}
		newbank = (PM->channel)[ch].bank_lsb;
		break;
	default:
		newbank = (PM->channel)[ch].bank_msb;
		break;
	}
	if (dr) {
		(PM->channel)[ch].bank = prog;	/* newbank is ignored */
		(PM->channel)[ch].program = prog;
		if (TMDY_INSTRUM->drumset[prog] == NULL || TMDY_INSTRUM->drumset[prog]->alt == NULL)
			(PM->channel)[ch].altassign = TMDY_INSTRUM->drumset[0]->alt;
		else
			(PM->channel)[ch].altassign = TMDY_INSTRUM->drumset[prog]->alt;
		ctl_mode_event(tmdy_struct, CTLE_DRUMPART, 1, ch, 1);
	} else {
//		(PM->channel)[ch].bank = ((PM->special_tonebank) >= 0)
//				? (PM->special_tonebank) : newbank;
		(PM->channel)[ch].program = (TMDY_INSTRUM->default_program[ch] == SPECIAL_PROGRAM)
				? SPECIAL_PROGRAM : prog;
		(PM->channel)[ch].altassign = NULL;
		ctl_mode_event(tmdy_struct, CTLE_DRUMPART, 1, ch, 0);
		if ((PM->opt_realtime_playing) && ((TMDY_OUTPUT->play_mode)->flag & PF_PCM_STREAM)) {
			b = (PM->channel)[ch].bank, p = prog;
			TMDY_INSTRUM->instrument_map(tmdy_struct, (PM->channel)[ch].mapID, &b, &p);
			play_midi_load_instrument(tmdy_struct, 0, b, p);
		}
	}
}

static void process_sysex_event(tmdy_struct_ex_t *tmdy_struct, int ev,int ch,int val,int b)
{
	if(ev == ME_SYSEX_GS_LSB)
	{
		switch(b)
		{
		case 0x00:	/* EQ ON/OFF */
			if(!(PM->opt_eq_control)) {break;}
			if((PM->channel)[ch].eq_on != val) {
				if(val) {
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ ON (CH:%d)",ch);
				} else {
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ OFF (CH:%d)",ch);
				}
			}
			(PM->channel)[ch].eq_on = val;
			break;
		case 0x01:	/* EQ LOW FREQ */
			if(!(PM->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.low_freq = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ LOW FREQ (%d)",val);
			break;
		case 0x02:	/* EQ LOW GAIN */
			if(!(PM->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.low_gain = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ LOW GAIN (%d dB)",val - 0x40);
			break;
		case 0x03:	/* EQ HIGH FREQ */
			if(!(PM->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.high_freq = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ HIGH FREQ (%d)",val);
			break;
		case 0x04:	/* EQ HIGH GAIN */
			if(!(PM->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.high_gain = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ HIGH GAIN (%d dB)",val - 0x40);
			break;
		case 0x05:	/* Reverb Macro */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Macro (%d)",val);
			TMDY_READMIDI->set_reverb_macro(tmdy_struct, val);
			TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			break;
		case 0x06:	/* Reverb Character */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Character (%d)",val);
			if(TMDY_REVERB->reverb_status.character != val) {
				TMDY_REVERB->reverb_status.character = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
				TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			}
			break;
		case 0x07:	/* Reverb Pre-LPF */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Pre-LPF (%d)",val);
			if(TMDY_REVERB->reverb_status.pre_lpf != val) {
				TMDY_REVERB->reverb_status.pre_lpf = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			}
			break;
		case 0x08:	/* Reverb Level */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Level (%d)",val);
			if(TMDY_REVERB->reverb_status.level != val) {
				TMDY_REVERB->reverb_status.level = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
				TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			}
			break;
		case 0x09:	/* Reverb Time */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Time (%d)",val);
			if(TMDY_REVERB->reverb_status.time != val) {
				TMDY_REVERB->reverb_status.time = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
				TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			}
			break;
		case 0x0A:	/* Reverb Delay Feedback */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Delay Feedback (%d)",val);
			if(TMDY_REVERB->reverb_status.delay_feedback != val) {
				TMDY_REVERB->reverb_status.delay_feedback = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			}
			break;
		case 0x0C:	/* Reverb Predelay Time */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Predelay Time (%d)",val);
			if(TMDY_REVERB->reverb_status.pre_delay_time != val) {
				TMDY_REVERB->reverb_status.pre_delay_time = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			}
			break;
		case 0x0D:	/* Chorus Macro */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Macro (%d)",val);
			TMDY_REVERB->init_ch_chorus(tmdy_struct);
			TMDY_READMIDI->set_chorus_macro(tmdy_struct, val);
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x0E:	/* Chorus Pre-LPF */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Pre-LPF (%d)",val);
			TMDY_REVERB->chorus_param.chorus_pre_lpf = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x0F:	/* Chorus Level */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Level (%d)",val);
			TMDY_REVERB->chorus_param.chorus_level = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x10:	/* Chorus Feedback */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Feedback (%d)",val);
			TMDY_REVERB->chorus_param.chorus_feedback = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x11:	/* Chorus Delay */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Delay (%d)",val);
			TMDY_REVERB->chorus_param.chorus_delay = val;
			TMDY_REVERB->init_ch_chorus(tmdy_struct);
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x12:	/* Chorus Rate */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Rate (%d)",val);
			TMDY_REVERB->chorus_param.chorus_rate = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x13:	/* Chorus Depth */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Depth (%d)",val);
			TMDY_REVERB->chorus_param.chorus_depth = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x14:	/* Chorus Send Level to Reverb */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send Level to Reverb (%d)",val);
			TMDY_REVERB->chorus_param.chorus_send_level_to_reverb = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x15:	/* Chorus Send Level to Delay */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send Level to Delay (%d)",val);
			TMDY_REVERB->chorus_param.chorus_send_level_to_delay = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x16:	/* Delay Macro */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Macro (%d)",val);
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_READMIDI->set_delay_macro(tmdy_struct, val);
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x17:	/* Delay Pre-LPF */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Pre-LPF (%d)",val);
			TMDY_REVERB->delay_status.pre_lpf = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x18:	/* Delay Time Center */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Time Center (%d)",val);
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_REVERB->delay_status.time_center = delay_time_center_table[val > 0x73 ? 0x73 : val];
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x19:	/* Delay Time Ratio Left */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Time Ratio Left (%d)",val);
			if(val == 0) {val = 1;}
			TMDY_REVERB->delay_status.time_ratio_left = (double)val / 24;
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1A:	/* Delay Time Ratio Right */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Time Ratio Right (%d)",val);
			if(val == 0) {val = 1;}
			TMDY_REVERB->delay_status.time_ratio_right = (double)val / 24;
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1B:	/* Delay Level Center */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level Center (%d)",val);
			TMDY_REVERB->delay_status.level_center = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1C:	/* Delay Level Left */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level Left (%d)",val);
			TMDY_REVERB->delay_status.level_left = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1D:	/* Delay Level Right */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level Right (%d)",val);
			TMDY_REVERB->delay_status.level_right = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1E:	/* Delay Level */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level (%d)",val);
			TMDY_REVERB->delay_status.level = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1F:	/* Delay Feedback */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Feedback (%d)",val);
			TMDY_REVERB->delay_status.feedback = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x20:	/* Delay Send Level to Reverb */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Send Level to Reverb (%d)",val);
			TMDY_REVERB->delay_status.send_reverb = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x21:	/* Velocity Sense Depth */
			(PM->channel)[ch].velocity_sense_depth = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Velocity Sense Depth (CH:%d VAL:%d)",ch,val);
			break;
		case 0x22:	/* Velocity Sense Offset */
			(PM->channel)[ch].velocity_sense_offset = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Velocity Sense Offset (CH:%d VAL:%d)",ch,val);
			break;
		case 0x23:	/* Insertion Effect ON/OFF */
			if(!(PM->opt_insertion_effect)) {break;}
			if((PM->channel)[ch].insertion_effect != val) {
				if(val) {(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EFX ON (CH:%d)",ch);}
				else {(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EFX OFF (CH:%d)",ch);}
			}
			(PM->channel)[ch].insertion_effect = val;
			break;
		case 0x24:	/* Assign Mode */
			(PM->channel)[ch].assign_mode = val;
			if(val == 0) {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Assign Mode: Single (CH:%d)",ch);
			} else if(val == 1) {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Assign Mode: Limited-Multi (CH:%d)",ch);
			} else if(val == 2) {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Assign Mode: Full-Multi (CH:%d)",ch);
			}
			break;
		case 0x25:	/* TONE MAP-0 NUMBER */
			(PM->channel)[ch].tone_map0_number = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Tone Map-0 Number (CH:%d VAL:%d)",ch,val);
			break;
		case 0x26:	/* Pitch Offset Fine */
			(PM->channel)[ch].pitch_offset_fine = (FLOAT_T)((((int32)val << 4) | (int32)val) - 0x80) / 10.0;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Pitch Offset Fine (CH:%d %3fHz)",ch,(PM->channel)[ch].pitch_offset_fine);
			break;
		case 0x27:	/* Insertion Effect Parameter */
			if(!(PM->opt_insertion_effect)) {break;}
			gs_ieffect.type_msb = val;
			break;
		case 0x28:	/* Insertion Effect Parameter */
			if(!(PM->opt_insertion_effect)) {break;}
			gs_ieffect.type_lsb = val;
			gs_ieffect.type = ((int32)gs_ieffect.type_msb << 8) | (int32)gs_ieffect.type_lsb;
			TMDY_READMIDI->set_insertion_effect_default_parameter(tmdy_struct);
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EFX TYPE (%02X %02X)",gs_ieffect.type_msb,gs_ieffect.type_lsb);
			break;
		case 0x29:
			gs_ieffect.parameter[0] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2A:
			gs_ieffect.parameter[1] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2B:
			gs_ieffect.parameter[2] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2C:
			gs_ieffect.parameter[3] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2D:
			gs_ieffect.parameter[4] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2E:
			gs_ieffect.parameter[5] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2F:
			gs_ieffect.parameter[6] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x30:
			gs_ieffect.parameter[7] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x31:
			gs_ieffect.parameter[8] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x32:
			gs_ieffect.parameter[9] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x33:
			gs_ieffect.parameter[10] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x34:
			gs_ieffect.parameter[11] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x35:
			gs_ieffect.parameter[12] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x36:
			gs_ieffect.parameter[13] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x37:
			gs_ieffect.parameter[14] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x38:
			gs_ieffect.parameter[15] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x39:
			gs_ieffect.parameter[16] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3A:
			gs_ieffect.parameter[17] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3B:
			gs_ieffect.parameter[18] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3C:
			gs_ieffect.parameter[19] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3D:
			gs_ieffect.send_reverb = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3E:
			gs_ieffect.send_chorus = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3F:
			gs_ieffect.send_delay = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x40:
			gs_ieffect.control_source1 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x41:
			gs_ieffect.control_depth1 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x42:
			gs_ieffect.control_source2 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x43:
			gs_ieffect.control_depth2 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x44:
			gs_ieffect.send_eq_switch = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x45:	/* Rx. Channel */
			reset_controllers(tmdy_struct, ch);
			redraw_controllers(tmdy_struct, ch);
			all_notes_off(tmdy_struct, ch);
			if (val == 0x80)
				TMDY_READMIDI->remove_channel_layer(tmdy_struct, ch);
			else
				TMDY_READMIDI->add_channel_layer(tmdy_struct, ch, val);
			break;
		case 0x46:	/* Channel Msg Rx Port */
			reset_controllers(tmdy_struct, ch);
			redraw_controllers(tmdy_struct, ch);
			all_notes_off(tmdy_struct, ch);
			(PM->channel)[ch].port_select = val;
			break;
			
			/* MOD PITCH CONTROL */
			/* 0x45~ MOD, CAF, PAF */
		default:
			break;
		}
	} else if(ev == ME_SYSEX_XG_LSB) {
		switch(b)
		{
		case 0x00:	/* Reverb Return */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Return (%d)",val);
			TMDY_REVERB->reverb_status.level = val;
			TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			break;
		case 0x01:	/* Chorus Return */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Return (%d)",val);
			TMDY_REVERB->chorus_param.chorus_level = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x65:	/* Rcv CHANNEL */
			if (val == 0x7f)
				TMDY_READMIDI->remove_channel_layer(tmdy_struct, ch);
			else
				TMDY_READMIDI->add_channel_layer(tmdy_struct, ch, val);
			break;
		default:
			break;
		}
	}
}

static void play_midi_prescan(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
    int i, j, k, ch, orig_ch, port_ch, offset, layered;
    
    if((PM->opt_amp_compensation)) {(PM->mainvolume_max) = 0;}
    else {(PM->mainvolume_max) = 0x7f;}
    (PM->compensation_ratio) = 1.0;

    PM->prescanning_flag = 1;
    TMDY_READMIDI->change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);
    reset_midi(tmdy_struct, 0);
    TMDY_RECACHE->resamp_cache_reset(tmdy_struct);

	while (ev->type != ME_EOT) {
#ifndef SUPPRESS_CHANNEL_LAYER
		orig_ch = ev->channel;
		layered = ! IS_SYSEX_EVENT_TYPE(ev->type);
		for (j = 0; j < MAX_CHANNELS; j += 16) {
			port_ch = (orig_ch + j) % MAX_CHANNELS;
			offset = (port_ch < 16) ? 0 : 16;
			for (k = offset; k < offset + 16; k++) {
				if (! layered && (j || k != offset))
					continue;
				if (layered) {
					if (! IS_SET_CHANNELMASK(
							(PM->channel)[k].channel_layer, port_ch)
							|| (PM->channel)[k].port_select ^ (orig_ch >= 16))
						continue;
					ev->channel = k;
				}
#endif
	ch = ev->channel;

	switch(ev->type)
	{
	  case ME_NOTEON:
	    if(((PM->channel)[ch].portamento_time_msb |
		(PM->channel)[ch].portamento_time_lsb) == 0 ||
	       (PM->channel)[ch].portamento == 0)
	    {
		int nv;
		int vlist[32];
		Voice *vp;

		nv = find_samples(tmdy_struct, ev, vlist);
		for(i = 0; i < nv; i++)
		{
		    vp = PM->voice + vlist[i];
		    start_note(tmdy_struct, ev, vlist[i], 0, nv - i - 1);
		    TMDY_RECACHE->resamp_cache_refer_on(tmdy_struct, vp, ev->time);
		    vp->status = VOICE_FREE;
		    vp->temper_instant = 0;
		}
	    }
	    break;

	  case ME_NOTEOFF:
	    TMDY_RECACHE->resamp_cache_refer_off(tmdy_struct, ch, MIDI_EVENT_NOTE(ev), ev->time);
	    break;

	  case ME_PORTAMENTO_TIME_MSB:
	    (PM->channel)[ch].portamento_time_msb = ev->a;
	    break;

	  case ME_PORTAMENTO_TIME_LSB:
	    (PM->channel)[ch].portamento_time_lsb = ev->a;
	    break;

	  case ME_PORTAMENTO:
	    (PM->channel)[ch].portamento = (ev->a >= 64);

	  case ME_RESET_CONTROLLERS:
	    reset_controllers(tmdy_struct, ch);
	    TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
	    break;

	  case ME_PROGRAM:
	    midi_program_change(tmdy_struct, ch, ev->a);
	    break;

	  case ME_TONE_BANK_MSB:
	    (PM->channel)[ch].bank_msb = ev->a;
	    break;

	  case ME_TONE_BANK_LSB:
	    (PM->channel)[ch].bank_lsb = ev->a;
	    break;

	  case ME_RESET:
	    TMDY_READMIDI->change_system_mode(tmdy_struct, ev->a);
	    reset_midi(tmdy_struct, 0);
	    break;

	  case ME_PITCHWHEEL:
	  case ME_ALL_NOTES_OFF:
	  case ME_ALL_SOUNDS_OFF:
	  case ME_MONO:
	  case ME_POLY:
	    TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
	    break;

	  case ME_DRUMPART:
	    if(midi_drumpart_change(tmdy_struct, ch, ev->a))
		midi_program_change(tmdy_struct, ch, (PM->channel)[ch].program);
	    break;

	  case ME_KEYSHIFT:
	    TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
	    (PM->channel)[ch].key_shift = (int)ev->a - 0x40;
	    break;

	  case ME_SCALE_TUNING:
		TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
		(PM->channel)[ch].scale_tuning[ev->a] = ev->b;
		break;

	  case ME_MAINVOLUME:
	    if (ev->a > (PM->mainvolume_max)) {
	      (PM->mainvolume_max) = ev->a;
	      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_DEBUG,"ME_MAINVOLUME/max (CH:%d VAL:%#x)",ev->channel,ev->a);
	    }
	    break;
	}
#ifndef SUPPRESS_CHANNEL_LAYER
			}
		}
		ev->channel = orig_ch;
#endif
	ev++;
    }

    /* calculate compensation ratio */
    if (0 < (PM->mainvolume_max) && (PM->mainvolume_max) < 0x7f) {
      (PM->compensation_ratio) = pow((double)0x7f/(double)(PM->mainvolume_max), 4);
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_DEBUG,"Compensation ratio:%lf",(PM->compensation_ratio));
    }

    for(i = 0; i < MAX_CHANNELS; i++)
	TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, i, ev->time);
    TMDY_RECACHE->resamp_cache_create(tmdy_struct);
    PM->prescanning_flag = 0;
}

static int32 midi_cnv_vib_rate(tmdy_struct_ex_t *tmdy_struct, int rate)
{
    return (int32)((double)(TMDY_OUTPUT->play_mode)->rate * MODULATION_WHEEL_RATE
		   / (midi_time_table[rate] *
		      2.0 * VIBRATO_SAMPLE_INCREMENTS));
}

static int midi_cnv_vib_depth(tmdy_struct_ex_t *tmdy_struct, int depth)
{
    return (int)(depth * VIBRATO_DEPTH_TUNING);
}

static int32 midi_cnv_vib_delay(tmdy_struct_ex_t *tmdy_struct, int delay)
{
    return (int32)(midi_time_table[delay]);
}

static int last_rpn_addr(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	int lsb, msb, addr, i;
	struct rpn_tag_map_t *addrmap;
	struct rpn_tag_map_t {
		int addr, mask, tag;
	};
	static struct rpn_tag_map_t nrpn_addr_map[] = {
		{0x0108, 0xffff, NRPN_ADDR_0108},
		{0x0109, 0xffff, NRPN_ADDR_0109},
		{0x010a, 0xffff, NRPN_ADDR_010A},
		{0x0120, 0xffff, NRPN_ADDR_0120},
		{0x0121, 0xffff, NRPN_ADDR_0121},
		{0x0163, 0xffff, NRPN_ADDR_0163},
		{0x0164, 0xffff, NRPN_ADDR_0164},
		{0x0166, 0xffff, NRPN_ADDR_0166},
		{0x1400, 0xff00, NRPN_ADDR_1400},
		{0x1500, 0xff00, NRPN_ADDR_1500},
		{0x1600, 0xff00, NRPN_ADDR_1600},
		{0x1700, 0xff00, NRPN_ADDR_1700},
		{0x1800, 0xff00, NRPN_ADDR_1800},
		{0x1900, 0xff00, NRPN_ADDR_1900},
		{0x1a00, 0xff00, NRPN_ADDR_1A00},
		{0x1c00, 0xff00, NRPN_ADDR_1C00},
		{0x1d00, 0xff00, NRPN_ADDR_1D00},
		{0x1e00, 0xff00, NRPN_ADDR_1E00},
		{0x1f00, 0xff00, NRPN_ADDR_1F00},
		{-1, -1, 0}
	};
	static struct rpn_tag_map_t rpn_addr_map[] = {
		{0x0000, 0xffff, RPN_ADDR_0000},
		{0x0001, 0xffff, RPN_ADDR_0001},
		{0x0002, 0xffff, RPN_ADDR_0002},
		{0x0003, 0xffff, RPN_ADDR_0003},
		{0x0004, 0xffff, RPN_ADDR_0004},
		{0x7f7f, 0xffff, RPN_ADDR_7F7F},
		{0xffff, 0xffff, RPN_ADDR_FFFF},
		{-1, -1}
	};
	
	if ((PM->channel)[ch].nrpn == -1){
		return -1;
	}
	lsb = (PM->channel)[ch].lastlrpn;
	msb = (PM->channel)[ch].lastmrpn;
	if (lsb == 0xff || msb == 0xff){
		return -1;
	}
	addr = (msb << 8 | lsb);
	if ((PM->channel)[ch].nrpn)
		addrmap = nrpn_addr_map;
	else
		addrmap = rpn_addr_map;
	for (i = 0; addrmap[i].addr != -1; i++)
		if (addrmap[i].addr == (addr & addrmap[i].mask)){
			return addrmap[i].tag;
		}
	return -1;
}

static void update_channel_freq(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	int i, uv = (PM->upper_voices);
	for (i = 0; i < uv; i++)
		if ((PM->voice)[i].status != VOICE_FREE && (PM->voice)[i].channel == ch)
	recompute_freq(tmdy_struct, i);
}

static void update_rpn_map(tmdy_struct_ex_t *tmdy_struct, int ch, int addr, int update_now)
{
	int val, drumflag, i, note;
	
	val = (PM->channel)[ch].rpnmap[addr];
	drumflag = 0;
	switch (addr) {
	case NRPN_ADDR_0108:	/* Vibrato Rate */
		if ((PM->opt_nrpn_vibrato)) {
			/* from -10.72 Hz to +10.72 Hz. */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Vibrato Rate (CH:%d VAL:%d)", ch, val - 64);
			(PM->channel)[ch].vibrato_ratio = 168 * (val - 64)
					* (VIBRATO_RATE_TUNING * (TMDY_OUTPUT->play_mode)->rate)
					/ (2 * VIBRATO_SAMPLE_INCREMENTS);
		}
		if (update_now)
			update_channel_freq(tmdy_struct, ch);
		break;
	case NRPN_ADDR_0109:	/* Vibrato Depth */
		if ((PM->opt_nrpn_vibrato)) {
			/* from -10cents to +10cents. */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Vibrato Depth (CH:%d VAL:%d)", ch, val - 64);
			(PM->channel)[ch].vibrato_depth =
					(double) (val - 64) * 0.15625 * 256 / 400;
		}
		if (update_now)
			update_channel_freq(tmdy_struct, ch);
		break;
	case NRPN_ADDR_010A:	/* Vibrato Delay */
		if ((PM->opt_nrpn_vibrato)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Vibrato Delay (CH:%d VAL:%d)", ch, val);
			(PM->channel)[ch].vibrato_delay =
					(TMDY_OUTPUT->play_mode)->rate * delay_time_center_table[val] * 0.001;
		}
		if (update_now)
			update_channel_freq(tmdy_struct, ch);
		break;
	case NRPN_ADDR_0120:	/* Filter Cutoff Frequency */
		if ((PM->opt_lpf_def)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Filter Cutoff (CH:%d VAL:%d)", ch, val - 64);
			(PM->channel)[ch].param_cutoff_freq = val - 64;
		}
		break;
	case NRPN_ADDR_0121:	/* Filter Resonance */
		if ((PM->opt_lpf_def)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Filter Resonance (CH:%d VAL:%d)", ch, val - 64);
			(PM->channel)[ch].param_resonance = val - 64;
		}
		break;
	case NRPN_ADDR_0163:	/* Attack Time */
		if ((PM->opt_tva_attack)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Attack Time (CH:%d VAL:%d)", ch, val);
			set_envelope_time(tmdy_struct, ch, val, 0);
		}
		break;
	case NRPN_ADDR_0164:	/* EG Decay Time */
		if ((PM->opt_tva_decay)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"EG Decay Time (CH:%d VAL:%d)", ch, val);
			set_envelope_time(tmdy_struct, ch, val, 2);
		}
		break;
	case NRPN_ADDR_0166:	/* EG Release Time */
		if ((PM->opt_tva_release)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"EG Release Time (CH:%d VAL:%d)", ch, val);
			set_envelope_time(tmdy_struct, ch, val, 3);
		}
		break;
	case NRPN_ADDR_1400:	/* Drum Filter Cutoff (XG) */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument Filter Cutoff (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(PM->channel)[ch].drums[note]->drum_cutoff_freq = val - 64;
		break;
	case NRPN_ADDR_1500:	/* Drum Filter Resonance (XG) */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument Filter Resonance (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(PM->channel)[ch].drums[note]->drum_resonance = val - 64;
		break;
	case NRPN_ADDR_1600:	/* Drum EG Attack Time (XG) */
		drumflag = 1;
		if ((PM->opt_tva_attack)) {
			val = val & 0x7f;
			note = (PM->channel)[ch].lastlrpn;
			if ((PM->channel)[ch].drums[note] == NULL)
				play_midi_setup_drums(tmdy_struct, ch, note);
			val	-= 64;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"XG Drum Attack Time (CH:%d NOTE:%d VALUE:%d)",
					ch, note, val);
			(PM->channel)[ch].drums[note]->drum_envelope_rate[0] = val;
		}
		break;
	case NRPN_ADDR_1700:	/* Drum EG Decay Time (XG) */
		drumflag = 1;
		if ((PM->opt_tva_decay)) {
			val = val & 0x7f;
			note = (PM->channel)[ch].lastlrpn;
			if ((PM->channel)[ch].drums[note] == NULL)
				play_midi_setup_drums(tmdy_struct, ch, note);
			val	-= 64;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"XG Drum Decay Time (CH:%d NOTE:%d VALUE:%d)",
					ch, note, val);
			(PM->channel)[ch].drums[note]->drum_envelope_rate[2] = val;
		}
		break;
	case NRPN_ADDR_1800:	/* Coarse Pitch of Drum (GS) */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		if ((PM->current_event)->b == 0x01) {
			(PM->channel)[ch].drums[note]->play_note = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
			"Drum Instrument Play Note (CH:%d NOTE:%d VALUE:%d)",
			ch, note, (PM->channel)[ch].drums[note]->play_note);
		} else {
			(PM->channel)[ch].drums[note]->coarse = val - 64;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
			"Drum Instrument Pitch Coarse (CH:%d NOTE:%d VALUE:%d)",
			ch, note, (PM->channel)[ch].drums[note]->coarse);
		}
		(PM->channel)[ch].pitchfactor = 0;
		break;
	case NRPN_ADDR_1900:	/* Fine Pitch of Drum (XG) */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(PM->channel)[ch].drums[note]->fine = val - 64;
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument Pitch Fine (CH:%d NOTE:%d VALUE:%d)",
				ch, note, (PM->channel)[ch].drums[note]->fine);
		(PM->channel)[ch].pitchfactor = 0;
		break;
	case NRPN_ADDR_1A00:	/* Level of Drum */	 
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument TVA Level (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(PM->channel)[ch].drums[note]->drum_level =
				calc_drum_tva_level(tmdy_struct, ch, note, val);
		break;
	case NRPN_ADDR_1C00:	/* Panpot of Drum */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		if(val == 0) {
			val = TMDY_COMMON->int_rand(tmdy_struct, 128);
			(PM->channel)[ch].drums[note]->pan_random = 1;
		} else
			(PM->channel)[ch].drums[note]->pan_random = 0;
		(PM->channel)[ch].drums[note]->drum_panning = val;
		if (update_now && (PM->adjust_panning_immediately)
				&& ! (PM->channel)[ch].pan_random)
			adjust_drum_panning(tmdy_struct, ch, note);
		break;
	case NRPN_ADDR_1D00:	/* Reverb Send Level of Drum */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Reverb Send Level of Drum (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(PM->channel)[ch].drums[note]->reverb_level = val;
		break;
	case NRPN_ADDR_1E00:	/* Chorus Send Level of Drum */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Chorus Send Level of Drum (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(PM->channel)[ch].drums[note]->chorus_level = val;
		break;
	case NRPN_ADDR_1F00:	/* Variation Send Level of Drum */
		drumflag = 1;
		note = (PM->channel)[ch].lastlrpn;
		if ((PM->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Delay Send Level of Drum (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(PM->channel)[ch].drums[note]->delay_level = val;
		break;
	case RPN_ADDR_0000:		/* Pitch bend sensitivity */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Pitch Bend Sensitivity (CH:%d VALUE:%d)", ch, val);
		/* for mod2mid.c, arpeggio */
		if (! IS_CURRENT_MOD_FILE && (PM->channel)[ch].rpnmap[RPN_ADDR_0000] > 24)
			(PM->channel)[ch].rpnmap[RPN_ADDR_0000] = 24;
		(PM->channel)[ch].pitchfactor = 0;
		break;
	case RPN_ADDR_0001:		/* Master Fine Tuning */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Master Fine Tuning (CH:%d VALUE:%d)", ch, val);
		(PM->channel)[ch].pitchfactor = 0;
		break;
	case RPN_ADDR_0002:		/* Master Coarse Tuning */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Master Coarse Tuning (CH:%d VALUE:%d)", ch, val);
		(PM->channel)[ch].pitchfactor = 0;
		break;
	case RPN_ADDR_0003:		/* Tuning Program Select */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Tuning Program Select (CH:%d VALUE:%d)", ch, val);
		for (i = 0; i < (PM->upper_voices); i++)
			if ((PM->voice)[i].status != VOICE_FREE) {
				(PM->voice)[i].temper_instant = 1;
				recompute_freq(tmdy_struct, i);
			}
		break;
	case RPN_ADDR_0004:		/* Tuning Bank Select */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Tuning Bank Select (CH:%d VALUE:%d)", ch, val);
		for (i = 0; i < (PM->upper_voices); i++)
			if ((PM->voice)[i].status != VOICE_FREE) {
				(PM->voice)[i].temper_instant = 1;
				recompute_freq(tmdy_struct, i);
			}
		break;
	case RPN_ADDR_7F7F:		/* RPN reset */
		(PM->channel)[ch].rpn_7f7f_flag = 1;
		break;
	case RPN_ADDR_FFFF:		/* RPN initialize */
		/* All reset to defaults */
		(PM->channel)[ch].rpn_7f7f_flag = 0;
		memset((PM->channel)[ch].rpnmap, 0, sizeof((PM->channel)[ch].rpnmap));
		(PM->channel)[ch].lastlrpn = (PM->channel)[ch].lastmrpn = 0;
		(PM->channel)[ch].nrpn = 0;
		(PM->channel)[ch].rpnmap[RPN_ADDR_0000] = 2;
		(PM->channel)[ch].rpnmap[RPN_ADDR_0001] = 0x40;
		(PM->channel)[ch].rpnmap[RPN_ADDR_0002] = 0x40;
		(PM->channel)[ch].pitchfactor = 0;
		break;
	}
	drumflag = 0;
	if (drumflag && midi_drumpart_change(tmdy_struct, ch, 1)) {
		midi_program_change(tmdy_struct, ch, (PM->channel)[ch].program);
		if (update_now)
			ctl_prog_event(tmdy_struct, ch);
	}
}

static void seek_forward(tmdy_struct_ex_t *tmdy_struct, int32 until_time)
{
    int32 i;
    int j, k, ch, orig_ch, port_ch, offset, layered;

    (PM->playmidi_seek_flag) = 1;
    TMDY_WRD->wrd_midi_event(tmdy_struct, WRD_START_SKIP, WRD_NOARG);
	while (MIDI_EVENT_TIME((PM->current_event)) < until_time) {
#ifndef SUPPRESS_CHANNEL_LAYER
		orig_ch = (PM->current_event)->channel;
		layered = ! IS_SYSEX_EVENT_TYPE((PM->current_event)->type);
		for (j = 0; j < MAX_CHANNELS; j += 16) {
			port_ch = (orig_ch + j) % MAX_CHANNELS;
			offset = (port_ch < 16) ? 0 : 16;
			for (k = offset; k < offset + 16; k++) {
				if (! layered && (j || k != offset))
					continue;
				if (layered) {
					if (! IS_SET_CHANNELMASK(
							(PM->channel)[k].channel_layer, port_ch)
							|| (PM->channel)[k].port_select ^ (orig_ch >=16))
						continue;
					(PM->current_event)->channel = k;
				}
#endif
	ch = (PM->current_event)->channel;
	
	switch((PM->current_event)->type)
	{
	  case ME_PITCHWHEEL:
	    (PM->channel)[ch].pitchbend = (PM->current_event)->a + (PM->current_event)->b * 128;
	    (PM->channel)[ch].pitchfactor=0;
	    break;

	  case ME_MAINVOLUME:
	    (PM->channel)[ch].volume = (PM->current_event)->a;
	    break;

	  case ME_MASTER_VOLUME:
	    (PM->master_volume_ratio) =
		(int32)(PM->current_event)->a + 256 * (int32)(PM->current_event)->b;
	    break;

	  case ME_PAN:
	    (PM->channel)[ch].panning = (PM->current_event)->a;
	    (PM->channel)[ch].pan_random = 0;
	    break;

	  case ME_EXPRESSION:
	    (PM->channel)[ch].expression=(PM->current_event)->a;
	    break;

	  case ME_PROGRAM:
	    midi_program_change(tmdy_struct, ch, (PM->current_event)->a);
	    break;

	  case ME_SUSTAIN:
	    (PM->channel)[ch].sustain = ((PM->current_event)->a >= 64);
	    break;

	  case ME_SOSTENUTO:
	    break;

	  case ME_LEGATO_FOOTSWITCH:
        (PM->channel)[ch].legato = ((PM->current_event)->a >= 64);
	    break;

      case ME_HOLD2:
        break;

	  case ME_FOOT:
	    break;

	  case ME_BREATH:
	    break;

	  case ME_BALANCE:
	    break;

	  case ME_RESET_CONTROLLERS:
	    reset_controllers(tmdy_struct, ch);
	    break;

	  case ME_TONE_BANK_MSB:
	    (PM->channel)[ch].bank_msb = (PM->current_event)->a;
	    break;

	  case ME_TONE_BANK_LSB:
	    (PM->channel)[ch].bank_lsb = (PM->current_event)->a;
	    break;

	  case ME_MODULATION_WHEEL:
	    (PM->channel)[ch].modulation_wheel =
		midi_cnv_vib_depth(tmdy_struct, (PM->current_event)->a);
	    break;

	  case ME_PORTAMENTO_TIME_MSB:
	    (PM->channel)[ch].portamento_time_msb = (PM->current_event)->a;
	    break;

	  case ME_PORTAMENTO_TIME_LSB:
	    (PM->channel)[ch].portamento_time_lsb = (PM->current_event)->a;
	    break;

	  case ME_PORTAMENTO:
	    (PM->channel)[ch].portamento = ((PM->current_event)->a >= 64);
	    break;

	  case ME_MONO:
	    (PM->channel)[ch].mono = 1;
	    break;

	  case ME_POLY:
	    (PM->channel)[ch].mono = 0;
	    break;

	  case ME_SOFT_PEDAL:
		  if((PM->opt_lpf_def)) {
			  (PM->channel)[ch].soft_pedal = (PM->current_event)->a;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Soft Pedal (CH:%d VAL:%d)",ch,(PM->channel)[ch].soft_pedal);
		  }
		  break;

	  case ME_HARMONIC_CONTENT:
		  if((PM->opt_lpf_def)) {
			  (PM->channel)[ch].param_resonance = (PM->current_event)->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Harmonic Content (CH:%d VAL:%d)",ch,(PM->channel)[ch].param_resonance);
		  }
		  break;

	  case ME_BRIGHTNESS:
		  if((PM->opt_lpf_def)) {
			  (PM->channel)[ch].param_cutoff_freq = (PM->current_event)->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Brightness (CH:%d VAL:%d)",ch,(PM->channel)[ch].param_cutoff_freq);
		  }
		  break;

	    /* RPNs */
	  case ME_NRPN_LSB:
	    (PM->channel)[ch].lastlrpn = (PM->current_event)->a;
	    (PM->channel)[ch].nrpn = 1;
	    break;
	  case ME_NRPN_MSB:
	    (PM->channel)[ch].lastmrpn = (PM->current_event)->a;
	    (PM->channel)[ch].nrpn = 1;
	    break;
	  case ME_RPN_LSB:
	    (PM->channel)[ch].lastlrpn = (PM->current_event)->a;
	    (PM->channel)[ch].nrpn = 0;
	    break;
	  case ME_RPN_MSB:
	    (PM->channel)[ch].lastmrpn = (PM->current_event)->a;
	    (PM->channel)[ch].nrpn = 0;
	    break;
	  case ME_RPN_INC:
	    if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	    {
		if((PM->channel)[ch].rpnmap[i] < 127)
		    (PM->channel)[ch].rpnmap[i]++;
		update_rpn_map(tmdy_struct, ch, i, 0);
	    }
	    break;
	case ME_RPN_DEC:
	    if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	    {
		if((PM->channel)[ch].rpnmap[i] > 0)
		    (PM->channel)[ch].rpnmap[i]--;
		update_rpn_map(tmdy_struct, ch, i, 0);
	    }
	    break;
	  case ME_DATA_ENTRY_MSB:
	    if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	    {
		(PM->channel)[ch].rpnmap[i] = (PM->current_event)->a;
		update_rpn_map(tmdy_struct, ch, i, 0);
	    }
	    break;
	  case ME_DATA_ENTRY_LSB:
	    if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    /* Ignore */
	    (PM->channel)[ch].nrpn = -1;
	    break;

	  case ME_REVERB_EFFECT:
	    set_reverb_level(tmdy_struct, ch, (PM->current_event)->a);
	    break;

	  case ME_CHORUS_EFFECT:
	    if((PM->opt_chorus_control) == 1)
		(PM->channel)[ch].chorus_level = (PM->current_event)->a;
	    else
		(PM->channel)[ch].chorus_level = -(PM->opt_chorus_control);

		if((PM->current_event)->a) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send (CH:%d LEVEL:%d)",ch,(PM->current_event)->a);
		}
		break;

	  case ME_TREMOLO_EFFECT:
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Tremolo Send (CH:%d LEVEL:%d)",ch,(PM->current_event)->a);
		break;

	  case ME_CELESTE_EFFECT:
		if((PM->opt_delay_control)) {
			(PM->channel)[ch].delay_level = (PM->current_event)->a;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Send (CH:%d LEVEL:%d)",ch,(PM->current_event)->a);
		}
	    break;

	  case ME_ATTACK_TIME:
	  	if(!(PM->opt_tva_attack)) { break; }
		set_envelope_time(tmdy_struct, ch,(PM->current_event)->a,0);
		break;

	  case ME_RELEASE_TIME:
	  	if(!(PM->opt_tva_release)) { break; }
		set_envelope_time(tmdy_struct, ch,(PM->current_event)->a,3);
		break;

	  case ME_PHASER_EFFECT:
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Phaser Send (CH:%d LEVEL:%d)",ch,(PM->current_event)->a);
		break;

	  case ME_RANDOM_PAN:
	    (PM->channel)[ch].panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	    (PM->channel)[ch].pan_random = 1;
	    break;

	  case ME_SET_PATCH:
	    i = (PM->channel)[ch].special_sample = (PM->current_event)->a;
	    if(TMDY_INSTRUM->special_patch[i] != NULL)
		TMDY_INSTRUM->special_patch[i]->sample_offset = 0;
	    break;

	  case ME_TEMPO:
	    (PM->current_play_tempo) = ch +
		(PM->current_event)->b * 256 + (PM->current_event)->a * 65536;
	    break;

	  case ME_RESET:
	    TMDY_READMIDI->change_system_mode(tmdy_struct, (PM->current_event)->a);
	    reset_midi(tmdy_struct, 0);
	    break;

	  case ME_PATCH_OFFS:
	    i = (PM->channel)[ch].special_sample;
	    if(TMDY_INSTRUM->special_patch[i] != NULL)
		TMDY_INSTRUM->special_patch[i]->sample_offset =
		    ((PM->current_event)->a | 256 * (PM->current_event)->b);
	    break;

	  case ME_WRD:
	    TMDY_WRD->wrd_midi_event(tmdy_struct, ch, (PM->current_event)->a | 256 * (PM->current_event)->b);
	    break;

	  case ME_SHERRY:
	    TMDY_WRD->wrd_sherry_event(tmdy_struct, ch |
			     ((PM->current_event)->a<<8) |
			     ((PM->current_event)->b<<16));
	    break;

	  case ME_DRUMPART:
	    if(midi_drumpart_change(tmdy_struct, ch, (PM->current_event)->a))
		midi_program_change(tmdy_struct, ch, (PM->channel)[ch].program);
	    break;

	  case ME_KEYSHIFT:
	    (PM->channel)[ch].key_shift = (int)(PM->current_event)->a - 0x40;
	    break;

	case ME_KEYSIG:
		(PM->current_keysig) = (PM->current_event)->a + (PM->current_event)->b * 16;
		break;

	case ME_SCALE_TUNING:
		(PM->channel)[ch].scale_tuning[(PM->current_event)->a] = (PM->current_event)->b;
		break;

	case ME_BULK_TUNING_DUMP:
		set_single_note_tuning(tmdy_struct, ch, (PM->current_event)->a, (PM->current_event)->b, 0);
		break;

	case ME_SINGLE_NOTE_TUNING:
		set_single_note_tuning(tmdy_struct, ch, (PM->current_event)->a, (PM->current_event)->b, 0);
		break;

	case ME_TEMPER_KEYSIG:
		PM->current_temper_keysig = (PM->current_event->a + 8) % 32 - 8;
		PM->temper_adj = (PM->current_event->a + 8 & 0x20) ? 1 : 0;
		break;

	case ME_TEMPER_TYPE:
		(PM->channel)[ch].temper_type = (PM->current_event)->a;
		break;

	case ME_MASTER_TEMPER_TYPE:
		for (i = 0; i < MAX_CHANNELS; i++)
			(PM->channel)[i].temper_type = (PM->current_event)->a;
		break;

	  case ME_SYSEX_LSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_LSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	  case ME_SYSEX_MSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_MSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	  case ME_SYSEX_GS_LSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_GS_LSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	  case ME_SYSEX_GS_MSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_GS_MSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	  case ME_SYSEX_XG_LSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_XG_LSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	  case ME_SYSEX_XG_MSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_XG_MSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	  case ME_EOT:
	    (PM->current_sample) = (PM->current_event)->time;
	    (PM->playmidi_seek_flag) = 0;
	
	    return;
	}
#ifndef SUPPRESS_CHANNEL_LAYER
			}
		}
		(PM->current_event)->channel = orig_ch;
#endif
	(PM->current_event)++;
    }
    TMDY_WRD->wrd_midi_event(tmdy_struct, WRD_END_SKIP, WRD_NOARG);

    (PM->playmidi_seek_flag) = 0;
    if((PM->current_event) != (PM->event_list))
	(PM->current_event)--;
    (PM->current_sample) = until_time;
}

static void skip_to(tmdy_struct_ex_t *tmdy_struct, int32 until_time)
{
  int ch;

  TMDY_MIDITRACE->trace_flush(tmdy_struct);
  (PM->current_event) = NULL;

  if ((PM->current_sample) > until_time)
    (PM->current_sample)=0;

  TMDY_READMIDI->change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);
  reset_midi(tmdy_struct, 0);

  (PM->buffered_count)=0;
  (PM->buffer_pointer)=(PM->common_buffer);
  (PM->current_event)=(PM->event_list);
  (PM->current_play_tempo) = 500000; /* 120 BPM */

  if (until_time)
    seek_forward(tmdy_struct, until_time);
  for(ch = 0; ch < MAX_CHANNELS; ch++)
      (PM->channel)[ch].lasttime = (PM->current_sample);

  ctl_mode_event(tmdy_struct, CTLE_RESET, 0, 0, 0);
  TMDY_MIDITRACE->trace_offset(tmdy_struct, until_time);

#ifdef SUPPORT_SOUNDSPEC
  TMDY_SOUNDSPEC->soundspec_update_wave(tmdy_struct, NULL, 0);
#endif /* SUPPORT_SOUNDSPEC */
}

static int32 sync_restart(tmdy_struct_ex_t *tmdy_struct, int only_trace_ok)
{
    int32 cur;

    cur = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
    if(cur == -1)
    {
	if(only_trace_ok){
	
	    return -1;
	}
	cur = (PM->current_sample);
    }
    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
    skip_to(tmdy_struct, cur);

    return cur;
}

static int playmidi_change_rate(tmdy_struct_ex_t *tmdy_struct, int32 rate, int restart)
{
    int arg;

    if(rate == (TMDY_OUTPUT->play_mode)->rate){
		return 1; /* Not need to change */
	}

    if(rate < MIN_OUTPUT_RATE || rate > MAX_OUTPUT_RATE)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "Out of sample rate: %d", rate);
	return -1;
    }

    if(restart)
    {
	if(((PM->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct)) == -1)
	    (PM->midi_restart_time) = (PM->current_sample);
    }
    else
	(PM->midi_restart_time) = 0;

    arg = (int)rate;
    if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_RATE, &arg) == -1)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "Can't change sample rate to %d", rate);
	return -1;
    }

    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
    (TMDY_AQ->aq_setup)(tmdy_struct);
    (TMDY_AQ->aq_set_soft_queue)(tmdy_struct, -1.0, -1.0);
    TMDY_INSTRUM->free_instruments(tmdy_struct,1);
#ifdef SUPPORT_SOUNDSPEC
    TMDY_SOUNDSPEC->soundspec_reinit(tmdy_struct);
#endif /* SUPPORT_SOUNDSPEC */

    return 0;
}

void playmidi_output_changed(tmdy_struct_ex_t *tmdy_struct, int play_state)
{
    if((TMDY_OUTPUT->target_play_mode) == NULL)
	return;
    (TMDY_OUTPUT->play_mode) = (TMDY_OUTPUT->target_play_mode);

    if(play_state == 0)
    {
	/* Playing */
	if(((PM->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct)) == -1)
	    (PM->midi_restart_time) = (PM->current_sample);
    }
    else /* Not playing */
	(PM->midi_restart_time) = 0;

    if(play_state != 2)
    {
	(TMDY_AQ->aq_flush)(tmdy_struct, 1);
	(TMDY_AQ->aq_setup)(tmdy_struct);
	(TMDY_AQ->aq_set_soft_queue)(tmdy_struct, -1.0, -1.0);
	TMDY_INSTRUM->clear_magic_instruments(tmdy_struct);
    }
    TMDY_INSTRUM->free_instruments(tmdy_struct,1);
#ifdef SUPPORT_SOUNDSPEC
    TMDY_SOUNDSPEC->soundspec_reinit(tmdy_struct);
#endif /* SUPPORT_SOUNDSPEC */
    (TMDY_OUTPUT->target_play_mode) = NULL;
}

int check_apply_control(tmdy_struct_ex_t *tmdy_struct)
{
    int rc;
    int32 val;

    if((PM->file_from_stdin)){
		return RC_NONE;
	}
    rc = (TMDY_CONTROLS->ctl)->read(tmdy_struct, &val);
    switch(rc)
    {
      case RC_CHANGE_VOLUME:
	if (val>0 || (PM->amplification) > -val)
	    (PM->amplification) += val;
	else
	    (PM->amplification)=0;
	if ((PM->amplification) > MAX_AMPLIFICATION)
	    (PM->amplification)=MAX_AMPLIFICATION;
	adjust_amplification(tmdy_struct);
	ctl_mode_event(tmdy_struct, CTLE_MASTER_VOLUME, 0, (PM->amplification), 0);
	break;
      case RC_SYNC_RESTART:
	(TMDY_AQ->aq_flush)(tmdy_struct, 1);
	break;
      case RC_TOGGLE_PAUSE:
	(PM->play_pause_flag) = !(PM->play_pause_flag);
	ctl_pause_event(tmdy_struct, (PM->play_pause_flag), 0);
	return RC_NONE;
      case RC_TOGGLE_SNDSPEC:
#ifdef SUPPORT_SOUNDSPEC
	if(TMDY_SOUNDSPEC->view_soundspec_flag)
	    TMDY_SOUNDSPEC->close_soundspec(tmdy_struct);
	else
	    TMDY_SOUNDSPEC->open_soundspec(tmdy_struct);
	if(TMDY_SOUNDSPEC->view_soundspec_flag || TMDY_SOUNDSPEC->ctl_speana_flag)
	    TMDY_SOUNDSPEC->soundspec_update_wave(tmdy_struct, NULL, -1);
	return RC_NONE;
      case RC_TOGGLE_CTL_SPEANA:
	TMDY_SOUNDSPEC->ctl_speana_flag = !TMDY_SOUNDSPEC->ctl_speana_flag;
	if(TMDY_SOUNDSPEC->view_soundspec_flag || TMDY_SOUNDSPEC->ctl_speana_flag)
	    TMDY_SOUNDSPEC->soundspec_update_wave(tmdy_struct, NULL, -1);
	return RC_NONE;
#endif /* SUPPORT_SOUNDSPEC */
      case RC_CHANGE_RATE:
	if(playmidi_change_rate(tmdy_struct, val, 0)){
	    return RC_NONE;
	}
	return RC_RELOAD;
      case RC_OUTPUT_CHANGED:
	playmidi_output_changed(tmdy_struct, 1);
	return RC_RELOAD;
    }
    return rc;
}

static void voice_increment(tmdy_struct_ex_t *tmdy_struct, int n)
{
    int i;
    
    for(i = 0; i < n; i++)
    {
	if((PM->voices) == PM->max_voices)
	    break;
	(PM->voice)[(PM->voices)].status = VOICE_FREE;
	(PM->voice)[(PM->voices)].temper_instant = 0;
	(PM->voice)[(PM->voices)].chorus_link = (PM->voices);
	(PM->voices)++;
    }
    if(n > 0)
	ctl_mode_event(tmdy_struct, CTLE_MAXVOICES, 1, (PM->voices), 0);
}

static void voice_decrement(tmdy_struct_ex_t *tmdy_struct, int n)
{
    int i, j, lowest;
    int32 lv, v;

    /* decrease voice */
    for(i = 0; i < n && (PM->voices) > 0; i++)
    {
	(PM->voices)--;
	if((PM->voice)[(PM->voices)].status == VOICE_FREE)
	    continue;	/* found */

	for(j = 0; j < (PM->voices); j++)
	    if((PM->voice)[j].status == VOICE_FREE)
		break;
	if(j != (PM->voices))
	{
	    (PM->voice)[j] = (PM->voice)[(PM->voices)];
	    continue;	/* found */
	}

	/* Look for the decaying note with the lowest volume */
	lv = 0x7FFFFFFF;
	lowest = -1;
	for(j = 0; j <= (PM->voices); j++)
	{
	    if((PM->voice)[j].status & ~(VOICE_ON | VOICE_DIE))
	    {
		v = (PM->voice)[j].left_mix;
		if(((PM->voice)[j].panned==PANNED_MYSTERY) &&
		   ((PM->voice)[j].right_mix > v))
		    v = (PM->voice)[j].right_mix;
		if(v < lv)
		{
		    lv = v;
		    lowest = j;
		}
	    }
	}

	if(lowest != -1)
	{
	    (PM->cut_notes)++;
	    free_voice(tmdy_struct, lowest);
	    ctl_note_event(tmdy_struct, lowest);
	    (PM->voice)[lowest] = (PM->voice)[(PM->voices)];
	}
	else
	    (PM->lost_notes)++;
    }
    if((PM->upper_voices) > (PM->voices))
	(PM->upper_voices) = (PM->voices);
    if(n > 0)
	ctl_mode_event(tmdy_struct, CTLE_MAXVOICES, 1, (PM->voices), 0);
}

/* EAW -- do not throw away good notes, stop decrementing */
static void voice_decrement_conservative(tmdy_struct_ex_t *tmdy_struct, int n)
{
    int i, j, lowest, finalnv;
    int32 lv, v;

    /* decrease voice */
    finalnv = (PM->voices) - n;
    for(i = 1; i <= n && (PM->voices) > 0; i++)
    {
	if((PM->voice)[(PM->voices)-1].status == VOICE_FREE) {
	    (PM->voices)--;
	    continue;	/* found */
	}

	for(j = 0; j < finalnv; j++)
	    if((PM->voice)[j].status == VOICE_FREE)
		break;
	if(j != finalnv)
	{
	    (PM->voice)[j] = (PM->voice)[(PM->voices)-1];
	    (PM->voices)--;
	    continue;	/* found */
	}

	/* Look for the decaying note with the lowest volume */
	lv = 0x7FFFFFFF;
	lowest = -1;
	for(j = 0; j < (PM->voices); j++)
	{
	    if((PM->voice)[j].status & ~(VOICE_ON | VOICE_DIE) &&
	       !((PM->voice)[j].sample->note_to_use &&
	         ISDRUMCHANNEL((PM->voice)[j].channel)))
	    {
		v = (PM->voice)[j].left_mix;
		if(((PM->voice)[j].panned==PANNED_MYSTERY) &&
		   ((PM->voice)[j].right_mix > v))
		    v = (PM->voice)[j].right_mix;
		if(v < lv)
		{
		    lv = v;
		    lowest = j;
		}
	    }
	}

	if(lowest != -1)
	{
	    (PM->voices)--;
	    (PM->cut_notes)++;
	    free_voice(tmdy_struct, lowest);
	
	    ctl_note_event(tmdy_struct, lowest);
	    (PM->voice)[lowest] = (PM->voice)[(PM->voices)];
	}
	else break;
    }
    if((PM->upper_voices) > (PM->voices))
	(PM->upper_voices) = (PM->voices);
}

void restore_voices(tmdy_struct_ex_t *tmdy_struct, int save_voices)
{
#ifdef REDUCE_VOICE_TIME_TUNING
//    static int old_voices = -1;
    if(PM->restore_voices_old_voices == -1 || save_voices)
	PM->restore_voices_old_voices = (PM->voices);
    else if ((PM->voices) < PM->restore_voices_old_voices)
	voice_increment(tmdy_struct, PM->restore_voices_old_voices - (PM->voices));
    else
	voice_decrement(tmdy_struct, (PM->voices) - PM->restore_voices_old_voices);
#endif /* REDUCE_VOICE_TIME_TUNING */
}
	

static int apply_controls(tmdy_struct_ex_t *tmdy_struct)
{
    int rc, i, jump_flag = 0;
    int32 val, cur;
    FLOAT_T r;
    ChannelBitMask tmp_chbitmask;

    /* ASCII renditions of CD player pictograms indicate approximate effect */
    do
    {
	switch(rc=(TMDY_CONTROLS->ctl)->read(tmdy_struct, &val))
	{
	  case RC_STOP:
	  case RC_QUIT:		/* [] */
	  case RC_LOAD_FILE:
	  case RC_NEXT:		/* >>| */
	  case RC_REALLY_PREVIOUS: /* |<< */
	  case RC_TUNE_END:	/* skip */
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	
	    return rc;

	  case RC_CHANGE_VOLUME:
	    if (val>0 || (PM->amplification) > -val)
		(PM->amplification) += val;
	    else
		(PM->amplification)=0;
	    if ((PM->amplification) > MAX_AMPLIFICATION)
		(PM->amplification)=MAX_AMPLIFICATION;
	    adjust_amplification(tmdy_struct);
	    for (i=0; i<(PM->upper_voices); i++)
		if ((PM->voice)[i].status != VOICE_FREE)
		{
		    recompute_amp(tmdy_struct, i);
		    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		}
	    ctl_mode_event(tmdy_struct, CTLE_MASTER_VOLUME, 0, (PM->amplification), 0);
	    continue;

	  case RC_CHANGE_REV_EFFB:
	  case RC_CHANGE_REV_TIME:
	    TMDY_REVERB->reverb_rc_event(tmdy_struct, rc, val);
	    sync_restart(tmdy_struct, 0);
	    continue;

	  case RC_PREVIOUS:	/* |<< */
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if ((PM->current_sample) < 2*(TMDY_OUTPUT->play_mode->rate)){
			return RC_REALLY_PREVIOUS;
		}
	
	    return RC_RESTART;

	  case RC_RESTART:	/* |<< */
	    if((PM->play_pause_flag))
	    {
		(PM->midi_restart_time) = 0;
		ctl_pause_event(tmdy_struct, 1, 0);
		continue;
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    skip_to(tmdy_struct, 0);
	    ctl_updatetime(tmdy_struct, 0);
	    jump_flag = 1;
		(PM->midi_restart_time) = 0;
	    continue;

	  case RC_JUMP:
	    if((PM->play_pause_flag))
	    {
		(PM->midi_restart_time) = val;
		ctl_pause_event(tmdy_struct, 1, val);
		continue;
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if (val >= (PM->sample_count)){
			return RC_TUNE_END;
		}
	    skip_to(tmdy_struct, val);
	    ctl_updatetime(tmdy_struct, val);
	
	    return rc;

	  case RC_FORWARD:	/* >> */
	    if((PM->play_pause_flag))
	    {
		(PM->midi_restart_time) += val;
		if((PM->midi_restart_time) > (PM->sample_count))
		    (PM->midi_restart_time) = (PM->sample_count);
		ctl_pause_event(tmdy_struct, 1, (PM->midi_restart_time));
		continue;
	    }
	    cur = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if(cur == -1)
		cur = (PM->current_sample);
	    if(val + cur >= (PM->sample_count)){
			return RC_TUNE_END;
		}
	    skip_to(tmdy_struct, val + cur);
	    ctl_updatetime(tmdy_struct, val + cur);
	
	    return RC_JUMP;

	  case RC_BACK:		/* << */
	    if((PM->play_pause_flag))
	    {
		(PM->midi_restart_time) -= val;
		if((PM->midi_restart_time) < 0)
		    (PM->midi_restart_time) = 0;
		ctl_pause_event(tmdy_struct, 1, (PM->midi_restart_time));
		continue;
	    }
	    cur = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if(cur == -1)
		cur = (PM->current_sample);
	    if(cur > val)
	    {
		skip_to(tmdy_struct, cur - val);
		ctl_updatetime(tmdy_struct, cur - val);
	    }
	    else
	    {
		skip_to(tmdy_struct, 0);
		ctl_updatetime(tmdy_struct, 0);
		(PM->midi_restart_time) = 0;
	    }
	
	    return RC_JUMP;

	  case RC_TOGGLE_PAUSE:
	    if((PM->play_pause_flag))
	    {
		(PM->play_pause_flag) = 0;
		skip_to(tmdy_struct, (PM->midi_restart_time));
	    }
	    else
	    {
		(PM->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
		if((PM->midi_restart_time) == -1)
		    (PM->midi_restart_time) = (PM->current_sample);
		(TMDY_AQ->aq_flush)(tmdy_struct, 1);
		(PM->play_pause_flag) = 1;
	    }
	    ctl_pause_event(tmdy_struct, (PM->play_pause_flag), (PM->midi_restart_time));
	    jump_flag = 1;
	    continue;

	  case RC_KEYUP:
	  case RC_KEYDOWN:
	    (PM->note_key_offset) += val;
	    (PM->current_freq_table) += val;
	    (PM->current_freq_table) -= floor((PM->current_freq_table) / 12.0) * 12;
	    if(sync_restart(tmdy_struct, 1) != -1)
		jump_flag = 1;
	    ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 0, (PM->note_key_offset), 0);
	    continue;

	  case RC_SPEEDUP:
	    r = 1.0;
	    for(i = 0; i < val; i++)
		r *= SPEED_CHANGE_RATE;
	    sync_restart(tmdy_struct, 0);
	    (PM->midi_time_ratio) /= r;
	    (PM->current_sample) = (int32)((PM->current_sample) / r + 0.5);
	    TMDY_MIDITRACE->trace_offset(tmdy_struct, (PM->current_sample));
	    jump_flag = 1;
	    ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (PM->midi_time_ratio) + 0.5, 0);
	    continue;

	  case RC_SPEEDDOWN:
	    r = 1.0;
	    for(i = 0; i < val; i++)
		r *= SPEED_CHANGE_RATE;
	    sync_restart(tmdy_struct, 0);
	    (PM->midi_time_ratio) *= r;
	    (PM->current_sample) = (int32)((PM->current_sample) * r + 0.5);
	    TMDY_MIDITRACE->trace_offset(tmdy_struct, (PM->current_sample));
	    jump_flag = 1;
	    ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (PM->midi_time_ratio) + 0.5, 0);
	    continue;

	  case RC_VOICEINCR:
	    restore_voices(tmdy_struct, 0);
	    voice_increment(tmdy_struct, val);
	    if(sync_restart(tmdy_struct, 1) != -1)
		jump_flag = 1;
	    restore_voices(tmdy_struct, 1);
	    continue;

	  case RC_VOICEDECR:
	    restore_voices(tmdy_struct, 0);
	    if(sync_restart(tmdy_struct, 1) != -1)
	    {
		(PM->voices) -= val;
		if((PM->voices) < 0)
		    (PM->voices) = 0;
		jump_flag = 1;
	    }
	    else
		voice_decrement(tmdy_struct, val);
	    restore_voices(tmdy_struct, 1);
	    continue;

	  case RC_TOGGLE_DRUMCHAN:
	    (PM->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    if((PM->midi_restart_time) == -1)
		(PM->midi_restart_time) = (PM->current_sample);
	    SET_CHANNELMASK((PM->drumchannel_mask), val);
	    SET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannel_mask, val);
	    if(IS_SET_CHANNELMASK((PM->drumchannels), val))
	    {
		UNSET_CHANNELMASK((PM->drumchannels), val);
		UNSET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, val);
	    }
	    else
	    {
		SET_CHANNELMASK((PM->drumchannels), val);
		SET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, val);
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	
	    return RC_RELOAD;

	  case RC_TOGGLE_SNDSPEC:
#ifdef SUPPORT_SOUNDSPEC
	    if(TMDY_SOUNDSPEC->view_soundspec_flag)
		TMDY_SOUNDSPEC->close_soundspec(tmdy_struct);
	    else
		TMDY_SOUNDSPEC->open_soundspec(tmdy_struct);
	    if(TMDY_SOUNDSPEC->view_soundspec_flag || TMDY_SOUNDSPEC->ctl_speana_flag)
	    {
		sync_restart(tmdy_struct, 0);
		TMDY_SOUNDSPEC->soundspec_update_wave(tmdy_struct, NULL, -1);
	    }
#endif /* SUPPORT_SOUNDSPEC */
	    continue;

	  case RC_TOGGLE_CTL_SPEANA:
#ifdef SUPPORT_SOUNDSPEC
	    TMDY_SOUNDSPEC->ctl_speana_flag = !TMDY_SOUNDSPEC->ctl_speana_flag;
	    if(TMDY_SOUNDSPEC->view_soundspec_flag || TMDY_SOUNDSPEC->ctl_speana_flag)
	    {
		sync_restart(tmdy_struct, 0);
		TMDY_SOUNDSPEC->soundspec_update_wave(tmdy_struct, NULL, -1);
	    }
#endif /* SUPPORT_SOUNDSPEC */
	    continue;

	  case RC_SYNC_RESTART:
	    sync_restart(tmdy_struct, val);
	    jump_flag = 1;
	    continue;

	  case RC_RELOAD:
	    (PM->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    if((PM->midi_restart_time) == -1)
		(PM->midi_restart_time) = (PM->current_sample);
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    return RC_RELOAD;

	  case RC_CHANGE_RATE:
	    if(playmidi_change_rate(tmdy_struct, val, 1)){
			return RC_NONE;
		}
	    return RC_RELOAD;

	  case RC_OUTPUT_CHANGED:
	    playmidi_output_changed(tmdy_struct, 0);
	    return RC_RELOAD;

	case RC_TOGGLE_MUTE:
		TOGGLE_CHANNELMASK(PM->channel_mute, val);
		sync_restart(tmdy_struct, 0);
		jump_flag = 1;
		ctl_mode_event(tmdy_struct, CTLE_MUTE, 0,
				val, (IS_SET_CHANNELMASK(PM->channel_mute, val)) ? 1 : 0);
		continue;

	case RC_SOLO_PLAY:
		COPY_CHANNELMASK(tmp_chbitmask, PM->channel_mute);
		FILL_CHANNELMASK(PM->channel_mute);
		UNSET_CHANNELMASK(PM->channel_mute, val);
		if (! COMPARE_CHANNELMASK(tmp_chbitmask, PM->channel_mute)) {
			sync_restart(tmdy_struct, 0);
			jump_flag = 1;
			for (i = 0; i < MAX_CHANNELS; i++)
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, i, 1);
			ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, val, 0);
		}
		continue;

	case RC_MUTE_CLEAR:
		COPY_CHANNELMASK(tmp_chbitmask, PM->channel_mute);
		CLEAR_CHANNELMASK(PM->channel_mute);
		if (! COMPARE_CHANNELMASK(tmp_chbitmask, PM->channel_mute)) {
			sync_restart(tmdy_struct, 0);
			jump_flag = 1;
			for (i = 0; i < MAX_CHANNELS; i++)
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, i, 0);
		}
		continue;
	}
	if(TMDY_MAIN->intr){
	    return RC_QUIT;
	}
	if((PM->play_pause_flag))
	    usleep(300000);
    } while (rc != RC_NONE || (PM->play_pause_flag));
    return jump_flag ? RC_JUMP : RC_NONE;
}

#ifdef USE_DSP_EFFECT
/* do_compute_data_midi() for new chorus */
static void do_compute_data_midi(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
	int i, j, uv, stereo, n, ch, note;
	int32 *vpblist[MAX_CHANNELS];
	int vc[MAX_CHANNELS];
	int channel_effect,channel_reverb,channel_chorus,channel_delay,channel_eq;
	int32 cnt = count * 2;
	
	stereo = ! ((TMDY_OUTPUT->play_mode->encoding) & PE_MONO);
	n = count * ((stereo) ? 8 : 4); /* in bytes */

	memset((PM->buffer_pointer), 0, n);  
	memset((PM->insertion_effect_buffer), 0, n);

	/* are effects valid? / don't supported in mono */
	channel_reverb = (stereo && (PM->opt_reverb_control == 1
			|| PM->opt_reverb_control == 3
			|| PM->opt_reverb_control < 0 && PM->opt_reverb_control & 0x80));
	channel_chorus = (stereo && PM->opt_chorus_control);
	channel_delay = (stereo && PM->opt_delay_control > 0);

	/* is EQ valid? */
	channel_effect = (stereo && (channel_reverb || channel_chorus
			|| channel_delay || channel_eq || PM->opt_delay_control));

	uv = (PM->upper_voices);
	for(i=0;i<uv;i++) {
		if((PM->voice)[i].status != VOICE_FREE) {
			(PM->channel)[(PM->voice)[i].channel].lasttime = (PM->current_sample) + count;
		}
	}

	if(channel_effect) {
		int buf_index = 0;
		
		if((PM->reverb_buffer) == NULL) {	/* allocating buffer for channel effect */
			(PM->reverb_buffer) = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, MAX_CHANNELS * AUDIO_BUFFER_SIZE * 8);
		}

		for(i=0;i<MAX_CHANNELS;i++) {
			if(ISDRUMCHANNEL(i) && (PM->opt_drum_effect) == 0) {
				vpblist[i] = (PM->buffer_pointer);
			} else if((PM->opt_insertion_effect) && (PM->channel)[i].insertion_effect) {
				vpblist[i] = (PM->insertion_effect_buffer);
			} else if((PM->channel)[i].eq_on || ((PM->channel)[i].reverb_level >= 0
					&& (PM->current_sample) - (PM->channel)[i].lasttime < REVERB_MAX_DELAY_OUT)
					|| (PM->channel)[i].chorus_level > 0 || (PM->channel)[i].delay_level > 0) {
				vpblist[i] = (int32*)((PM->reverb_buffer) + buf_index);
				buf_index += n;
			} else {
				vpblist[i] = (PM->buffer_pointer);
			}
		}

		if(buf_index) {memset((PM->reverb_buffer), 0, buf_index);}
	}

	for(i=0;i<uv;i++) {
		if((PM->voice)[i].status != VOICE_FREE) {
			int32 *vpb;
			
			if(channel_effect) {
				ch = (PM->voice)[i].channel;
				vpb = vpblist[ch];
			} else {
				vpb = (PM->buffer_pointer);
			}

			if(!IS_SET_CHANNELMASK(PM->channel_mute, (PM->voice)[i].channel)) {
				TMDY_MIX->mix_voice(tmdy_struct, vpb, i, count);
			} else {
				free_voice(tmdy_struct, i);
				ctl_note_event(tmdy_struct, i);
			}

			if((PM->voice)[i].timeout > 0 && (PM->voice)[i].timeout < (PM->current_sample)) {
				/* timeout (See also "#extension timeout" line in *.cfg file */
				if((PM->voice)[i].timeout > 1) {
					finish_note(tmdy_struct, i);
				} else {
					free_voice(tmdy_struct, i);
					ctl_note_event(tmdy_struct, i);
				}
			}
		}
	}

	while(uv > 0 && (PM->voice)[uv - 1].status == VOICE_FREE)	{uv--;}
	(PM->upper_voices) = uv;

	if(channel_effect) {
		if((PM->opt_insertion_effect)) { 	/* insertion effect */
			/* applying insertion effect */
			TMDY_REVERB->do_insertion_effect(tmdy_struct, (PM->insertion_effect_buffer), cnt);
			/* sending insertion effect voice to channel effect */
			TMDY_REVERB->set_ch_chorus(tmdy_struct, (PM->insertion_effect_buffer), cnt, gs_ieffect.send_chorus);
			TMDY_REVERB->set_ch_delay(tmdy_struct, (PM->insertion_effect_buffer), cnt, gs_ieffect.send_delay);
			TMDY_REVERB->set_ch_reverb(tmdy_struct, (PM->insertion_effect_buffer), cnt,	gs_ieffect.send_reverb);
			if(gs_ieffect.send_eq_switch && channel_eq) {
				TMDY_REVERB->set_ch_eq(tmdy_struct, (PM->insertion_effect_buffer), cnt);
			} else {
				TMDY_REVERB->set_dry_signal(tmdy_struct, (PM->insertion_effect_buffer), cnt);
			}
		}

		for(i=0;i<MAX_CHANNELS;i++) {	/* system effects */
			int32 *p;	
			p = vpblist[i];
			if(p != (PM->buffer_pointer) && p != (PM->insertion_effect_buffer)) {
				if(channel_chorus && (PM->channel)[i].chorus_level > 0) {
					TMDY_REVERB->set_ch_chorus(tmdy_struct, p, cnt, (PM->channel)[i].chorus_level);
				}
				if(channel_delay && (PM->channel)[i].delay_level > 0) {
					TMDY_REVERB->set_ch_delay(tmdy_struct, p, cnt, (PM->channel)[i].delay_level);
				}
				if(channel_reverb && (PM->channel)[i].reverb_level > 0
					&& (PM->current_sample) - (PM->channel)[i].lasttime < REVERB_MAX_DELAY_OUT) {
					TMDY_REVERB->set_ch_reverb(tmdy_struct, p, cnt, (PM->channel)[i].reverb_level);
				}
				if(channel_eq && (PM->channel)[i].eq_on) {
					TMDY_REVERB->set_ch_eq(tmdy_struct, p, cnt);
				} else {
					TMDY_REVERB->set_dry_signal(tmdy_struct, p, cnt);
				}
			}
		}
		
		if(channel_reverb) {
			TMDY_REVERB->set_ch_reverb(tmdy_struct, (PM->buffer_pointer), cnt, DEFAULT_REVERB_SEND_LEVEL);
		}
		TMDY_REVERB->set_dry_signal(tmdy_struct, (PM->buffer_pointer), cnt);

		/* mixing signal and applying system effects */ 
		TMDY_REVERB->mix_dry_signal(tmdy_struct, (PM->buffer_pointer), cnt);
		if(channel_eq) {TMDY_REVERB->do_ch_eq(tmdy_struct, (PM->buffer_pointer), cnt);}
		if(channel_chorus) {TMDY_REVERB->do_ch_chorus(tmdy_struct, (PM->buffer_pointer), cnt);}
		if(channel_delay) {TMDY_REVERB->do_ch_delay(tmdy_struct, (PM->buffer_pointer), cnt);}
		if(channel_reverb) {TMDY_REVERB->do_ch_reverb(tmdy_struct, (PM->buffer_pointer), cnt);}
	}

	(PM->current_sample) += count;
}

#else
/* do_compute_data_midi() for traditionally chorus */
static void do_compute_data_midi(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
	int i, j, uv, stereo, n, ch, note;
	int32 *vpblist[MAX_CHANNELS];
	int vc[MAX_CHANNELS];
	int channel_reverb;
	int channel_effect;
	int32 cnt = count * 2;
	
	stereo = ! ((TMDY_OUTPUT->play_mode)->encoding & PE_MONO);
	n = count * ((stereo) ? 8 : 4); /* in bytes */
	/* don't supported in mono */
	channel_reverb = (stereo && (PM->opt_reverb_control == 1
			|| PM->opt_reverb_control == 3
			|| PM->opt_reverb_control < 0 && PM->opt_reverb_control & 0x80));
	memset((PM->buffer_pointer), 0, n);

	channel_effect = (stereo && (PM->opt_reverb_control || PM->opt_chorus_control
			|| PM->opt_delay_control || opt_eq_control || PM->opt_delay_control));

	uv = (PM->upper_voices);
	for (i = 0; i < uv; i++)
		if ((PM->voice)[i].status != VOICE_FREE)
			(PM->channel)[(PM->voice)[i].channel].lasttime = (PM->current_sample) + count;

	if (channel_reverb) {
		int chbufidx;
		
		if (! (PM->make_rvid_flag)) {
			make_rvid(tmdy_struct);
			(PM->make_rvid_flag) = 1;
		}
		chbufidx = 0;
		for (i = 0; i < MAX_CHANNELS; i++) {
			vc[i] = 0;
			if ((PM->channel)[i].reverb_id != -1
					&& (PM->current_sample) - (PM->channel)[i].lasttime
					< REVERB_MAX_DELAY_OUT) {
				if ((PM->reverb_buffer) == NULL)
					(PM->reverb_buffer) = (char *) TMDY_COMMON->safe_malloc(tmdy_struct, MAX_CHANNELS
							* AUDIO_BUFFER_SIZE * 8);
				if ((PM->channel)[i].reverb_id != i)
					vpblist[i] = vpblist[(PM->channel)[i].reverb_id];
				else {
					vpblist[i] = (int32 *) ((PM->reverb_buffer) + chbufidx);
					chbufidx += n;
				}
			} else
				vpblist[i] = (PM->buffer_pointer);
		}
		if (chbufidx)
			memset((PM->reverb_buffer), 0, chbufidx);
	}
	for (i = 0; i < uv; i++)
		if ((PM->voice)[i].status != VOICE_FREE) {
			int32 *vpb;
			
			if (channel_reverb) {
				int ch = (PM->voice)[i].channel;
				
				vpb = vpblist[ch];
				vc[ch] = 1;
			} else
				vpb = (PM->buffer_pointer);
			if (! IS_SET_CHANNELMASK((PM->channel_mute), (PM->voice)[i].channel))
				TMDY_MIX->mix_voice(tmdy_struct, vpb, i, count);
			else {
				free_voice(tmdy_struct, i);
				ctl_note_event(tmdy_struct, i);
			}
			if ((PM->voice)[i].timeout > 0 && (PM->voice)[i].timeout < (PM->current_sample)) {
				if ((PM->voice)[i].timeout > 1)
					finish_note(tmdy_struct, i);
						/* timeout (See also "#extension timeout"
						line in *.cfg file */
				else {
					free_voice(tmdy_struct, i);
					ctl_note_event(tmdy_struct, i);
				}
			}
		}

	while (uv > 0 && (PM->voice)[uv - 1].status == VOICE_FREE)
		uv--;
	(PM->upper_voices) = uv;

	if (channel_reverb) {
		int k;
		
		k = count * 2; /* calclated buffer length in int32 */
		for (i = 0; i < MAX_CHANNELS; i++) {
			int32 *p;
			
			p = vpblist[i];
			if (p != (PM->buffer_pointer) && (PM->channel)[i].reverb_id == i)
				TMDY_REVERB->set_ch_reverb(tmdy_struct, p, k, (PM->channel)[i].reverb_level);
		}
		TMDY_REVERB->set_ch_reverb(tmdy_struct, (PM->buffer_pointer), k, DEFAULT_REVERB_SEND_LEVEL);
		TMDY_REVERB->do_ch_reverb(tmdy_struct, (PM->buffer_pointer), k);
	}
	(PM->current_sample) += count;
}
#endif

static void do_compute_data_wav(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
    int i, stereo, n, file_byte, samples;

    stereo = !((TMDY_OUTPUT->play_mode)->encoding & PE_MONO);
    samples = (stereo ? (count * 2) : count );
    n = samples*4; /* in bytes */
    file_byte = samples*2; /*regard as 16bit*/

    memset((PM->buffer_pointer), 0, n);

    TMDY_COMMON->tf_read(tmdy_struct, (PM->wav_buffer), 1, file_byte, TMDY_READMIDI->current_file_info->pcm_tf);
    for( i=0; i<samples; i++ ){
    	(PM->buffer_pointer)[i] = (LE_SHORT((PM->wav_buffer)[i])) << 16;
    	(PM->buffer_pointer)[i] /=4; /*level down*/
    }

    (PM->current_sample) += count;

}

static void do_compute_data_aiff(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
    int i, stereo, n, file_byte, samples;

    stereo = !((TMDY_OUTPUT->play_mode)->encoding & PE_MONO);
    samples = (stereo ? (count * 2) : count );
    n = samples*4; /* in bytes */
    file_byte = samples*2; /*regard as 16bit*/

    memset((PM->buffer_pointer), 0, n);

    TMDY_COMMON->tf_read(tmdy_struct, (PM->wav_buffer), 1, file_byte, TMDY_READMIDI->current_file_info->pcm_tf);
    for( i=0; i<samples; i++ ){
    	(PM->buffer_pointer)[i] = (BE_SHORT((PM->wav_buffer)[i])) << 16;
    	(PM->buffer_pointer)[i] /=4; /*level down*/
    }


    (PM->current_sample) += count;
}

static void do_compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
   	switch(TMDY_READMIDI->current_file_info->pcm_mode)
    {
     	 case PCM_MODE_NON:
    		do_compute_data_midi(tmdy_struct, count);
      		break;
      	case PCM_MODE_WAV:
    		do_compute_data_wav(tmdy_struct, count);
        	break;
      	case PCM_MODE_AIFF:
    		do_compute_data_aiff(tmdy_struct, count);
        	break;
      	case PCM_MODE_AU:
        	break;
      	case PCM_MODE_MP3:
        	break;	
    } 
}

static int check_midi_play_end(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e, int len)
{
    int i, type;

    for(i = 0; i < len; i++)
    {
	type = e[i].type;
	if(type == ME_NOTEON || type == ME_LAST || type == ME_WRD || type == ME_SHERRY)
	    return 0;
	if(type == ME_EOT)
	    return i + 1;
    }
    return 0;
}

int compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count);
static int midi_play_end(tmdy_struct_ex_t *tmdy_struct)
{
    int i, rc = RC_TUNE_END;

    (PM->check_eot_flag) = 0;

    if((PM->opt_realtime_playing) && (PM->current_sample) == 0)
    {
	reset_voices(tmdy_struct);
	return RC_TUNE_END;
    }

    if((PM->upper_voices) > 0)
    {
	int fadeout_cnt;

	rc = compute_data(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;

	for(i = 0; i < (PM->upper_voices); i++)
	    if((PM->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED))
		finish_note(tmdy_struct, i);
	if((PM->opt_realtime_playing))
	    fadeout_cnt = 3;
	else
	    fadeout_cnt = 6;
	for(i = 0; i < fadeout_cnt && (PM->upper_voices) > 0; i++)
	{
	    rc = compute_data(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate / 2);
	    if(RC_IS_SKIP_FILE(rc))
		goto midi_end;
	}

	/* kill voices */
	kill_all_voices(tmdy_struct);
	rc = compute_data(tmdy_struct, MAX_DIE_TIME);
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;
	(PM->upper_voices) = 0;
    }

    /* clear reverb echo sound */
    TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
    for(i = 0; i < MAX_CHANNELS; i++)
    {
	(PM->channel)[i].reverb_level = -1;
	(PM->channel)[i].reverb_id = -1;
	(PM->make_rvid_flag) = 1;
    }

    /* output null sound */
    if((PM->opt_realtime_playing))
	rc = compute_data(tmdy_struct, (int32)((TMDY_OUTPUT->play_mode)->rate * PLAY_INTERLEAVE_SEC/2));
    else
	rc = compute_data(tmdy_struct, (int32)((TMDY_OUTPUT->play_mode)->rate * PLAY_INTERLEAVE_SEC));
    if(RC_IS_SKIP_FILE(rc))
	goto midi_end;

    compute_data(tmdy_struct, 0); /* flush buffer to device */

    if((TMDY_CONTROLS->ctl)->trace_playing)
    {
	rc = (TMDY_AQ->aq_flush)(tmdy_struct, 0); /* Wait until play out */
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;
    }
    else
    {
	TMDY_MIDITRACE->trace_flush(tmdy_struct);
	rc = (TMDY_AQ->aq_soft_flush)(tmdy_struct);
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;
    }

  midi_end:
    if(RC_IS_SKIP_FILE(rc))
	(TMDY_AQ->aq_flush)(tmdy_struct, 1);

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "Playing time: ~%d seconds",
	      (PM->current_sample)/(TMDY_OUTPUT->play_mode)->rate+2);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "Notes cut: %d",
	      (PM->cut_notes));
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "Notes lost totally: %d",
	      (PM->lost_notes));
    if(RC_IS_SKIP_FILE(rc)){
		return rc;
	}
    return RC_TUNE_END;
}

/* count=0 means flush remaining buffered data to output device, then
   flush the device itself */
int compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
  int rc;

  if (!count)
    {
      if ((PM->buffered_count))
      {
	  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		    "output data (%d)", (PM->buffered_count));

#ifdef SUPPORT_SOUNDSPEC
	  TMDY_SOUNDSPEC->soundspec_update_wave(tmdy_struct, (PM->common_buffer), (PM->buffered_count));
#endif /* SUPPORT_SOUNDSPEC */

	  if((TMDY_AQ->aq_add)(tmdy_struct,(PM->common_buffer), (PM->buffered_count)) == -1){
	  
	      return RC_ERROR;
	  }
      }
      (PM->buffer_pointer)=(PM->common_buffer);
      (PM->buffered_count)=0;
  
      return RC_NONE;
    }

  while ((count+(PM->buffered_count)) >= audio_buffer_size)
    {
      int i;

      if((rc = apply_controls(tmdy_struct)) != RC_NONE){
	  	return rc;
      }
      do_compute_data(tmdy_struct, audio_buffer_size-(PM->buffered_count));
      count -= audio_buffer_size-(PM->buffered_count);
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		"output data (%d)", audio_buffer_size);

#ifdef SUPPORT_SOUNDSPEC
      TMDY_SOUNDSPEC->soundspec_update_wave(tmdy_struct, (PM->common_buffer), audio_buffer_size);
#endif /* SUPPORT_SOUNDSPEC */

      /* fall back to linear interpolation when queue < 100% */
      if (! (PM->opt_realtime_playing) && ((TMDY_OUTPUT->play_mode)->flag & PF_CAN_TRACE)) {
	  if (!(TMDY_AQ->aq_fill_buffer_flag) &&
	      100 * ((double)((TMDY_AQ->aq_filled)(tmdy_struct) + TMDY_AQ->aq_soft_filled(tmdy_struct)) /
		     (TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct)) < 99)
	      (PM->reduce_quality_flag) = 1;
	  else
	      (PM->reduce_quality_flag) = (PM->no_4point_interpolation);
      }

#ifdef REDUCE_VOICE_TIME_TUNING
      /* Auto voice reduce implementation by Masanao Izumo */
      if((PM->reduce_voice_threshold) &&
	 ((TMDY_OUTPUT->play_mode)->flag & PF_CAN_TRACE) &&
	 !(TMDY_AQ->aq_fill_buffer_flag) &&
	 (TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct) > 0)
      {
	  /* Reduce voices if there is not enough audio device buffer */

          int nv, filled, filled_limit, rate, rate_limit;
//          static int last_filled;

	  filled = (TMDY_AQ->aq_filled)(tmdy_struct);

	  rate_limit = 75;
	  if((PM->reduce_voice_threshold) >= 0)
	  {
	      filled_limit = (TMDY_OUTPUT->play_mode)->rate * (PM->reduce_voice_threshold) / 1000
		  + 1; /* +1 disable zero */
	  }
	  else /* Use default threshold */
	  {
	      int32 maxfill;
	      maxfill = (TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct);
	      filled_limit = REDUCE_VOICE_TIME_TUNING;
	      if(filled_limit > maxfill / 5) /* too small audio buffer */
	      {
		  rate_limit -= 100 * audio_buffer_size / maxfill / 5;
		  filled_limit = 1;
	      }
	  }

	  /* Calculate rate as it is displayed in ncurs_c.c */
	  /* The old method of calculating rate resulted in very low values
	     when using the new high order interplation methods on "slow"
	     CPUs when the queue was being drained WAY too quickly.  This
	     caused premature voice reduction under Linux, even if the queue
	     was over 2000%, leading to major voice lossage. */
	  rate = (int)(((double)((TMDY_AQ->aq_filled)(tmdy_struct) + TMDY_AQ->aq_soft_filled(tmdy_struct)) /
                  	(TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct)) * 100 + 0.5);

          for(i = nv = 0; i < (PM->upper_voices); i++)
	      if((PM->voice)[i].status != VOICE_FREE)
	          nv++;

	  if(! (PM->opt_realtime_playing))
	  {
	      /* calculate ok_nv, the "optimum" max polyphony */
	      if ((PM->auto_reduce_polyphony) && rate < 85) {
		/* average in current nv */
	        if ((rate == (PM->old_rate) && nv > (PM->min_bad_nv)) ||
	            (rate >= (PM->old_rate) && rate < 20)) {
	        	(PM->ok_nv_total) += nv;
	        	(PM->ok_nv_counts)++;
	        }
	        /* increase polyphony when it is too low */
	        else if (nv == (PM->voices) &&
	                 (rate > (PM->old_rate) && filled > PM->compute_data_last_filled)) {
	          		(PM->ok_nv_total) += nv + 1;
	          		(PM->ok_nv_counts)++;
	        }
	        /* reduce polyphony when loosing buffer */
	        else if (rate < 75 &&
	        	 (rate < (PM->old_rate) && filled < PM->compute_data_last_filled)) {
	        	(PM->ok_nv_total) += (PM->min_bad_nv);
	    		(PM->ok_nv_counts)++;
	        }
	        else goto NO_RESCALE_NV;

		/* rescale ok_nv stuff every 1 seconds */
		if ((PM->current_sample) >= (PM->ok_nv_sample) && (PM->ok_nv_counts) > 1) {
			(PM->ok_nv_total) >>= 1;
			(PM->ok_nv_counts) >>= 1;
			(PM->ok_nv_sample) = (PM->current_sample) + ((TMDY_OUTPUT->play_mode)->rate);
		}

		NO_RESCALE_NV:;
	      }
	  }

	  /* EAW -- if buffer is < 75%, start reducing some voices to
	     try to let it recover.  This really helps a lot, preserves
	     decent sound, and decreases the frequency of lost ON notes */
	  if ((! (PM->opt_realtime_playing) && rate < rate_limit)
	      || filled < filled_limit)
	  {
	      if(filled <= PM->compute_data_last_filled)
	      {
	          int v, kill_nv, temp_nv;

		  /* set bounds on "good" and "bad" nv */
		  if (! (PM->opt_realtime_playing) && rate > 20 &&
		      nv < (PM->min_bad_nv)) {
		  	(PM->min_bad_nv) = nv;
	                if ((PM->max_good_nv) < (PM->min_bad_nv))
	                	(PM->max_good_nv) = (PM->min_bad_nv);
	          }

		  /* EAW -- count number of !ON voices */
		  /* treat chorus notes as !ON */
		  for(i = kill_nv = 0; i < (PM->upper_voices); i++) {
		      if((PM->voice)[i].status & VOICE_FREE ||
		         (PM->voice)[i].cache != NULL)
		      		continue;
		      
		      if(((PM->voice)[i].status & ~(VOICE_ON|VOICE_SUSTAINED) &&
			  !((PM->voice)[i].status & ~(VOICE_DIE) &&
			    (PM->voice)[i].sample->note_to_use)))
				kill_nv++;
		  }

		  /* EAW -- buffer is dangerously low, drasticly reduce
		     voices to a hopefully "safe" amount */
		  if (filled < filled_limit &&
		      ((PM->opt_realtime_playing) || rate < 10)) {
		      FLOAT_T n;

		      /* calculate the drastic voice reduction */
		      if(nv > kill_nv) /* Avoid division by zero */
		      {
			  n = (FLOAT_T) nv / (nv - kill_nv);
			  temp_nv = (int)(nv - nv / (n + 1));

			  /* reduce by the larger of the estimates */
			  if (kill_nv < temp_nv && temp_nv < nv)
			      kill_nv = temp_nv;
		      }
		      else kill_nv = nv - 1; /* do not kill all the voices */
		  }
		  else {
		      /* the buffer is still high enough that we can throw
		         fewer voices away; keep the ON voices, use the
		         minimum "bad" nv as a floor on voice reductions */
		      temp_nv = nv - (PM->min_bad_nv);
		      if (kill_nv > temp_nv)
		          kill_nv = temp_nv;
		  }

		  for(i = 0; i < kill_nv; i++)
		      v = reduce_voice(tmdy_struct);

		  /* lower max # of allowed voices to let the buffer recover */
		  if ((PM->auto_reduce_polyphony)) {
		  	temp_nv = nv - kill_nv;
		  	(PM->ok_nv) = (PM->ok_nv_total) / (PM->ok_nv_counts);

		  	/* decrease it to current nv left */
		  	if ((PM->voices) > temp_nv && temp_nv > (PM->ok_nv))
			    voice_decrement_conservative(tmdy_struct, (PM->voices) - temp_nv);
			/* decrease it to ok_nv */
		  	else if ((PM->voices) > (PM->ok_nv) && temp_nv <= (PM->ok_nv))
			    voice_decrement_conservative(tmdy_struct, (PM->voices) - (PM->ok_nv));
		  	/* increase the polyphony */
		  	else if ((PM->voices) < (PM->ok_nv))
			    voice_increment(tmdy_struct, (PM->ok_nv) - (PM->voices));
		  }

		  while((PM->upper_voices) > 0 &&
			(PM->voice)[(PM->upper_voices) - 1].status == VOICE_FREE)
		      (PM->upper_voices)--;
	      }
	      PM->compute_data_last_filled = filled;
	  }
	  else {
	      if (! (PM->opt_realtime_playing) && rate >= rate_limit &&
	          filled > PM->compute_data_last_filled) {

		    /* set bounds on "good" and "bad" nv */
		    if (rate > 85 && nv > (PM->max_good_nv)) {
		  	(PM->max_good_nv) = nv;
		  	if ((PM->min_bad_nv) > (PM->max_good_nv))
		  	    (PM->min_bad_nv) = (PM->max_good_nv);
		    }

		    if ((PM->auto_reduce_polyphony)) {
		    	/* reset ok_nv stuff when out of danger */
		    	(PM->ok_nv_total) = (PM->max_good_nv) * (PM->ok_nv_counts);
			if ((PM->ok_nv_counts) > 1) {
			    (PM->ok_nv_total) >>= 1;
			    (PM->ok_nv_counts) >>= 1;
			}

		    	/* restore max # of allowed voices to normal */
			restore_voices(tmdy_struct, 0);
		    }
	      }

	      PM->compute_data_last_filled = filled_limit;
          }
          (PM->old_rate) = rate;
      }
#endif

      if((TMDY_AQ->aq_add)(tmdy_struct, (PM->common_buffer), audio_buffer_size) == -1){
	  	return RC_ERROR;
	  }

      (PM->buffer_pointer)=(PM->common_buffer);
      (PM->buffered_count)=0;
      if((PM->current_event)->type != ME_EOT)
	  ctl_timestamp(tmdy_struct);

      /* check break signals */
      VOLATILE_TOUCH(TMDY_MAIN->intr);
      if(TMDY_MAIN->intr){
	  	return RC_QUIT;
      }
      if((PM->upper_voices) == 0 && (PM->check_eot_flag) &&
	 (i = check_midi_play_end(tmdy_struct, (PM->current_event), EOT_PRESEARCH_LEN)) > 0)
      {
	  if(i > 1)
	      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
			"Last %d MIDI events are ignored", i - 1);
  
	  return midi_play_end(tmdy_struct);
      }
    }
  if (count>0)
    {
      do_compute_data(tmdy_struct, count);
      (PM->buffered_count) += count;
      (PM->buffer_pointer) += ((TMDY_OUTPUT->play_mode)->encoding & PE_MONO) ? count : count*2;
    }
  return RC_NONE;
}

static void update_modulation_wheel(tmdy_struct_ex_t *tmdy_struct, int ch, int val)
{
    int i, uv = (PM->upper_voices);
    
    for(i = 0; i < uv; i++)
	if((PM->voice)[i].status != VOICE_FREE && (PM->voice)[i].channel == ch)
	{
	    /* Set/Reset mod-wheel */
	    (PM->voice)[i].modulation_wheel = val;
	    (PM->voice)[i].vibrato_delay = 0;
	    recompute_freq(tmdy_struct, i);
	}
}

static void drop_portamento(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    int i, uv = (PM->upper_voices);

    (PM->channel)[ch].porta_control_ratio = 0;
    for(i = 0; i < uv; i++)
	if((PM->voice)[i].status != VOICE_FREE &&
	   (PM->voice)[i].channel == ch &&
	   (PM->voice)[i].porta_control_ratio)
	{
	    (PM->voice)[i].porta_control_ratio = 0;
	    recompute_freq(tmdy_struct, i);
	}
    (PM->channel)[ch].last_note_fine = -1;
}

static void update_portamento_controls(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    if(!(PM->channel)[ch].portamento ||
       ((PM->channel)[ch].portamento_time_msb | (PM->channel)[ch].portamento_time_lsb)
       == 0)
	drop_portamento(tmdy_struct, ch);
    else
    {
	double mt, dc;
	int d;

	mt = midi_time_table[(PM->channel)[ch].portamento_time_msb & 0x7F] *
	    midi_time_table2[(PM->channel)[ch].portamento_time_lsb & 0x7F] *
		PORTAMENTO_TIME_TUNING;
	dc = (TMDY_OUTPUT->play_mode)->rate * mt;
	d = (int)(1.0 / (mt * PORTAMENTO_CONTROL_RATIO));
	d++;
	(PM->channel)[ch].porta_control_ratio = (int)(d * dc + 0.5);
	(PM->channel)[ch].porta_dpb = d;
    }

}

static void update_portamento_time(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    int i, uv = (PM->upper_voices);
    int dpb;
    int32 ratio;

    update_portamento_controls(tmdy_struct, ch);
    dpb = (PM->channel)[ch].porta_dpb;
    ratio = (PM->channel)[ch].porta_control_ratio;

    for(i = 0; i < uv; i++)
    {
	if((PM->voice)[i].status != VOICE_FREE &&
	   (PM->voice)[i].channel == ch &&
	   (PM->voice)[i].porta_control_ratio)
	{
	    (PM->voice)[i].porta_control_ratio = ratio;
	    (PM->voice)[i].porta_dpb = dpb;
	    recompute_freq(tmdy_struct, i);
	}
    }

}

static void update_legato_controls(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	double mt, dc;
	int d;

	mt = 0.06250 * PORTAMENTO_TIME_TUNING * 0.3;
	dc = (TMDY_OUTPUT->play_mode)->rate * mt;
	d = (int)(1.0 / (mt * PORTAMENTO_CONTROL_RATIO));
	d++;
	(PM->channel)[ch].porta_control_ratio = (int)(d * dc + 0.5);
	(PM->channel)[ch].porta_dpb = d;
}

int play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
    int32 i, j, cet;
    int k, l, ch, orig_ch, port_ch, offset, layered;

    if((TMDY_OUTPUT->play_mode)->flag & PF_MIDI_EVENT){
		return (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_MIDI, ev);
	}
    if(!((TMDY_OUTPUT->play_mode)->flag & PF_PCM_STREAM)){
		return RC_NONE;
	}
	
    (PM->current_event) = ev;
    cet = MIDI_EVENT_TIME(ev);

    if((TMDY_CONTROLS->ctl)->verbosity >= VERB_DEBUG_SILLY)
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		  "Midi Event %d: %s %d %d %d", cet,
		  event_name(tmdy_struct, ev->type), ev->channel, ev->a, ev->b);
    if(cet > (PM->current_sample))
    {
	int rc;


    if(PM->midi_streaming!=0){

    	if ( (cet - (PM->current_sample)) * 1000 / (TMDY_OUTPUT->play_mode)->rate > (PM->stream_max_compute) ) {
			kill_all_voices(tmdy_struct);
			/* reset_voices(tmdy_struct); *
/* 			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY, "play_event: discard %d samples", cet - (PM->current_sample)); */
			(PM->current_sample) = cet;
		}
    	
    }

	rc = compute_data(tmdy_struct, cet - (PM->current_sample));
	ctl_mode_event(tmdy_struct, CTLE_REFRESH, 0, 0, 0);
    if(rc == RC_JUMP)
	{
		ctl_timestamp(tmdy_struct);
		return RC_NONE;
	}
	if(rc != RC_NONE){
	    return rc;
	}
	}

#ifndef SUPPRESS_CHANNEL_LAYER
	orig_ch = ev->channel;
	layered = ! IS_SYSEX_EVENT_TYPE(ev->type);
	for (k = 0; k < MAX_CHANNELS; k += 16) {
		port_ch = (orig_ch + k) % MAX_CHANNELS;
		offset = (port_ch < 16) ? 0 : 16;
		for (l = offset; l < offset + 16; l++) {
			if (! layered && (k || l != offset))
				continue;
			if (layered) {
				if (! IS_SET_CHANNELMASK((PM->channel)[l].channel_layer, port_ch)
						|| (PM->channel)[l].port_select ^ (orig_ch >= 16))
					continue;
				ev->channel = l;
			}
#endif
	ch = ev->channel;

    switch(ev->type)
    {
	/* MIDI Events */
      case ME_NOTEOFF:
	note_off(tmdy_struct, ev);
	break;

      case ME_NOTEON:
	note_on(tmdy_struct, ev);
	break;

      case ME_KEYPRESSURE:
	adjust_pressure(tmdy_struct, ev);
	break;

      case ME_PROGRAM:
	midi_program_change(tmdy_struct, ch, ev->a);
	ctl_prog_event(tmdy_struct, ch);
	break;

      case ME_CHANNEL_PRESSURE:
	adjust_channel_pressure(tmdy_struct, ev);
	break;

      case ME_PITCHWHEEL:
	(PM->channel)[ch].pitchbend = ev->a + ev->b * 128;
	(PM->channel)[ch].pitchfactor = 0;
	/* Adjust pitch for notes already playing */
	adjust_pitch(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_PITCH_BEND, 1, ch, (PM->channel)[ch].pitchbend);
	break;

	/* Controls */
      case ME_TONE_BANK_MSB:
	(PM->channel)[ch].bank_msb = ev->a;
	break;

      case ME_TONE_BANK_LSB:
	(PM->channel)[ch].bank_lsb = ev->a;
	break;

      case ME_MODULATION_WHEEL:
	(PM->channel)[ch].modulation_wheel =
	    midi_cnv_vib_depth(tmdy_struct, ev->a);
	update_modulation_wheel(tmdy_struct, ch, (PM->channel)[ch].modulation_wheel);
	ctl_mode_event(tmdy_struct, CTLE_MOD_WHEEL, 1, ch, (PM->channel)[ch].modulation_wheel);
	break;

      case ME_MAINVOLUME:
	(PM->channel)[ch].volume = ev->a;
	adjust_volume(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_VOLUME, 1, ch, ev->a);
	break;

      case ME_PAN:
	(PM->channel)[ch].panning = ev->a;
	(PM->channel)[ch].pan_random = 0;
	if((PM->adjust_panning_immediately) && !(PM->channel)[ch].pan_random)
	    adjust_panning(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_PANNING, 1, ch, ev->a);
	break;

      case ME_EXPRESSION:
	(PM->channel)[ch].expression = ev->a;
	adjust_volume(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_EXPRESSION, 1, ch, ev->a);
	break;

      case ME_SUSTAIN:
	(PM->channel)[ch].sustain = (ev->a >= 64);
	if(!ev->a)
	    drop_sustain(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_SUSTAIN, 1, ch, ev->a >= 64);
	break;

      case ME_SOSTENUTO:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Sostenuto - this function is not supported.");
	break;

      case ME_LEGATO_FOOTSWITCH:
    (PM->channel)[ch].legato = (ev->a >= 64);
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Legato Footswitch (CH:%d VAL:%d)",ch,(PM->channel)[ch].legato);
	break;

      case ME_HOLD2:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Hold2 - this function is not supported.");
	break;

      case ME_BREATH:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Breath - this function is not supported.");
	break;

      case ME_FOOT:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Foot - this function is not supported.");
	break;

      case ME_BALANCE:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Balance - this function is not supported.");
	break;

      case ME_PORTAMENTO_TIME_MSB:
	(PM->channel)[ch].portamento_time_msb = ev->a;
	update_portamento_time(tmdy_struct, ch);
	break;

      case ME_PORTAMENTO_TIME_LSB:
	(PM->channel)[ch].portamento_time_lsb = ev->a;
	update_portamento_time(tmdy_struct, ch);
	break;

      case ME_PORTAMENTO:
	(PM->channel)[ch].portamento = (ev->a >= 64);
	if(!(PM->channel)[ch].portamento)
	    drop_portamento(tmdy_struct, ch);
	break;

	  case ME_SOFT_PEDAL:
		  if((PM->opt_lpf_def)) {
			  (PM->channel)[ch].soft_pedal = ev->a;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Soft Pedal (CH:%d VAL:%d)",ch,(PM->channel)[ch].soft_pedal);
		  }
		  break;

	  case ME_HARMONIC_CONTENT:
		  if((PM->opt_lpf_def)) {
			  (PM->channel)[ch].param_resonance = ev->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Harmonic Content (CH:%d VAL:%d)",ch,(PM->channel)[ch].param_resonance);
		  }
		  break;

	  case ME_BRIGHTNESS:
		  if((PM->opt_lpf_def)) {
			  (PM->channel)[ch].param_cutoff_freq = ev->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Brightness (CH:%d VAL:%d)",ch,(PM->channel)[ch].param_cutoff_freq);
		  }
		  break;

      case ME_DATA_ENTRY_MSB:
	if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	{
	    (PM->channel)[ch].rpnmap[i] = ev->a;
	    update_rpn_map(tmdy_struct, ch, i, 1);
	}
	break;

      case ME_DATA_ENTRY_LSB:
	if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	/* Ignore */
	(PM->channel)[ch].nrpn = -1;
	break;

      case ME_REVERB_EFFECT:
	if((PM->opt_reverb_control))
	{
	    set_reverb_level(tmdy_struct, ch, ev->a);
	    ctl_mode_event(tmdy_struct, CTLE_REVERB_EFFECT, 1, ch, get_reverb_level(tmdy_struct, ch));
	}
	break;

      case ME_CHORUS_EFFECT:
	if((PM->opt_chorus_control))
	{
	    if((PM->opt_chorus_control) == 1)
		(PM->channel)[ch].chorus_level = ev->a;
	    else
		(PM->channel)[ch].chorus_level = -(PM->opt_chorus_control);
	    ctl_mode_event(tmdy_struct, CTLE_CHORUS_EFFECT, 1, ch, get_chorus_level(tmdy_struct, ch));
		if(ev->a) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send (CH:%d LEVEL:%d)",ch,ev->a);
		}
	}
	break;

      case ME_TREMOLO_EFFECT:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Tremolo Send (CH:%d LEVEL:%d)",ch,ev->a);
	break;

      case ME_CELESTE_EFFECT:
	if((PM->opt_delay_control)) {
		(PM->channel)[ch].delay_level = ev->a;
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Send (CH:%d LEVEL:%d)",ch,ev->a);
	}
	break;

	  case ME_ATTACK_TIME:
  	if(!(PM->opt_tva_attack)) { break; }
	set_envelope_time(tmdy_struct, ch,ev->a,0);
	break;

	  case ME_RELEASE_TIME:
  	if(!(PM->opt_tva_release)) { break; }
	set_envelope_time(tmdy_struct, ch,ev->a,3);
	break;

      case ME_PHASER_EFFECT:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Phaser Send (CH:%d LEVEL:%d)",ch,ev->a);
	break;

      case ME_RPN_INC:
	if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	{
	    if((PM->channel)[ch].rpnmap[i] < 127)
		(PM->channel)[ch].rpnmap[i]++;
	    update_rpn_map(tmdy_struct, ch, i, 1);
	}
	break;

      case ME_RPN_DEC:
	if((PM->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	{
	    if((PM->channel)[ch].rpnmap[i] > 0)
		(PM->channel)[ch].rpnmap[i]--;
	    update_rpn_map(tmdy_struct, ch, i, 1);
	}
	break;

      case ME_NRPN_LSB:
	(PM->channel)[ch].lastlrpn = ev->a;
	(PM->channel)[ch].nrpn = 1;
	break;

      case ME_NRPN_MSB:
	(PM->channel)[ch].lastmrpn = ev->a;
	(PM->channel)[ch].nrpn = 1;
	break;

      case ME_RPN_LSB:
	(PM->channel)[ch].lastlrpn = ev->a;
	(PM->channel)[ch].nrpn = 0;
	break;

      case ME_RPN_MSB:
	(PM->channel)[ch].lastmrpn = ev->a;
	(PM->channel)[ch].nrpn = 0;
	break;

      case ME_ALL_SOUNDS_OFF:
	all_sounds_off(tmdy_struct, ch);
	break;

      case ME_RESET_CONTROLLERS:
	reset_controllers(tmdy_struct, ch);
	redraw_controllers(tmdy_struct, ch);
	break;

      case ME_ALL_NOTES_OFF:
	all_notes_off(tmdy_struct, ch);
	break;

      case ME_MONO:
	(PM->channel)[ch].mono = 1;
	all_notes_off(tmdy_struct, ch);
	break;

      case ME_POLY:
	(PM->channel)[ch].mono = 0;
	all_notes_off(tmdy_struct, ch);
	break;

	/* TiMidity Extensionals */
      case ME_RANDOM_PAN:
	(PM->channel)[ch].panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	(PM->channel)[ch].pan_random = 1;
	if((PM->adjust_panning_immediately) && !(PM->channel)[ch].pan_random)
	    adjust_panning(tmdy_struct, ch);
	break;

      case ME_SET_PATCH:
	i = (PM->channel)[ch].special_sample = (PM->current_event)->a;
	if(TMDY_INSTRUM->special_patch[i] != NULL)
	    TMDY_INSTRUM->special_patch[i]->sample_offset = 0;
	ctl_prog_event(tmdy_struct, ch);
	break;

      case ME_TEMPO:
	(PM->current_play_tempo) = ch + ev->b * 256 + ev->a * 65536;
	ctl_mode_event(tmdy_struct, CTLE_TEMPO, 1, (PM->current_play_tempo), 0);
	break;

      case ME_CHORUS_TEXT:
      case ME_LYRIC:
      case ME_MARKER:
      case ME_INSERT_TEXT:
      case ME_TEXT:
      case ME_KARAOKE_LYRIC:
	i = ev->a | ((int)ev->b << 8);
	ctl_mode_event(tmdy_struct, CTLE_LYRIC, 1, i, 0);
	break;

      case ME_GSLCD:
	i = ev->a | ((int)ev->b << 8);
	ctl_mode_event(tmdy_struct, CTLE_GSLCD, 1, i, 0);
	break;

      case ME_MASTER_VOLUME:
	(PM->master_volume_ratio) = (int32)ev->a + 256 * (int32)ev->b;
	adjust_master_volume(tmdy_struct);
	break;

      case ME_RESET:
	TMDY_READMIDI->change_system_mode(tmdy_struct, ev->a);
	reset_midi(tmdy_struct, 1);
	break;

      case ME_PATCH_OFFS:
	i = (PM->channel)[ch].special_sample;
	if(TMDY_INSTRUM->special_patch[i] != NULL)
	    TMDY_INSTRUM->special_patch[i]->sample_offset =
		((PM->current_event)->a | 256 * (PM->current_event)->b);
	break;

      case ME_WRD:
	TMDY_MIDITRACE->push_midi_trace2(tmdy_struct, TMDY_WRD->wrd_midi_event,
			 ch, (PM->current_event)->a | ((PM->current_event)->b << 8));
	break;

      case ME_SHERRY:
	TMDY_MIDITRACE->push_midi_trace1(tmdy_struct, TMDY_WRD->wrd_sherry_event,
			 ch | ((PM->current_event)->a<<8) | ((PM->current_event)->b<<16));
	break;

      case ME_DRUMPART:
	if(midi_drumpart_change(tmdy_struct, ch, (PM->current_event)->a))
	{
	    /* Update bank information */
	    midi_program_change(tmdy_struct, ch, (PM->channel)[ch].program);
	    ctl_mode_event(tmdy_struct, CTLE_DRUMPART, 1, ch, ISDRUMCHANNEL(ch));
	    ctl_prog_event(tmdy_struct, ch);
	}
	break;

      case ME_KEYSHIFT:
	i = (int)(PM->current_event)->a - 0x40;
	if(i != (PM->channel)[ch].key_shift)
	{
	    all_sounds_off(tmdy_struct, ch);
	    (PM->channel)[ch].key_shift = (int8)i;
	}
	break;

	case ME_KEYSIG:
		(PM->current_keysig) = (PM->current_event)->a + (PM->current_event)->b * 16;
		ctl_mode_event(tmdy_struct, CTLE_KEYSIG, 1, (PM->current_keysig), 0);
		if ((PM->opt_force_keysig) != 8) {
			i = (PM->current_keysig) + (((PM->current_keysig) < 8) ? 7 : -6);
			(PM->note_key_offset) -= floor((PM->note_key_offset) / 12.0) * 12;
			for (j = 0; j < (PM->note_key_offset); j++)
				i += (i > 10) ? -5 : 7;
			j = (PM->opt_force_keysig) + (((PM->current_keysig) < 8) ? 7 : 10);
			while (i != j && i != j + 12) {
				if (++(PM->note_key_offset) > 6)
					(PM->note_key_offset) -= 12;
				i += (i > 10) ? -5 : 7;
			}
			kill_all_voices(tmdy_struct);
			ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 1, (PM->note_key_offset), 0);
		}
		i = (PM->current_keysig) + (((PM->current_keysig) < 8) ? 7 : -9), j = 0;
		while (i != 7 && i != 19)
			i += (i < 7) ? 5 : -7, j++;
		j += (PM->note_key_offset), j -= floor(j / 12.0) * 12;
		(PM->current_freq_table) = j;
		break;

	case ME_SCALE_TUNING:
		TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, (PM->current_event)->time);
		(PM->channel)[ch].scale_tuning[(PM->current_event)->a] = (PM->current_event)->b;
		adjust_pitch(tmdy_struct, ch);
		break;

	case ME_BULK_TUNING_DUMP:
		set_single_note_tuning(tmdy_struct, ch, (PM->current_event)->a, (PM->current_event)->b, 0);
		break;

	case ME_SINGLE_NOTE_TUNING:
		set_single_note_tuning(tmdy_struct, ch, (PM->current_event)->a, (PM->current_event)->b, 1);
		break;

	case ME_TEMPER_KEYSIG:
		PM->current_temper_keysig = (PM->current_event->a + 8) % 32 - 8;
		PM->temper_adj = (PM->current_event->a + 8 & 0x20) ? 1 : 0;
		ctl_mode_event(tmdy_struct, CTLE_TEMPER_KEYSIG, 1, PM->current_event->a, 0);
		i = (PM->current_temper_keysig) + (((PM->current_temper_keysig) < 8) ? 7 : -9);
		j = 0;
		while (i != 7 && i != 19)
			i += (i < 7) ? 5 : -7, j++;
		j += (PM->note_key_offset), j -= floor(j / 12.0) * 12;
		(PM->current_freq_table) = j;
		if ((PM->current_event)->b)
			for (i = 0; i < (PM->upper_voices); i++)
				if ((PM->voice)[i].status != VOICE_FREE) {
					(PM->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;

	case ME_TEMPER_TYPE:
		(PM->channel)[ch].temper_type = (PM->current_event)->a;
		ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 1, ch, (PM->channel)[ch].temper_type);
		if ((PM->temper_type_mute)) {
			if ((PM->temper_type_mute) & 1 << (PM->current_event)->a
					- (((PM->current_event)->a >= 0x40) ? 0x3c : 0)) {
				SET_CHANNELMASK((PM->channel_mute), ch);
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, ch, 1);
			} else {
				UNSET_CHANNELMASK((PM->channel_mute), ch);
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, ch, 0);
			}
		}
		if ((PM->current_event)->b)
			for (i = 0; i < (PM->upper_voices); i++)
				if ((PM->voice)[i].status != VOICE_FREE) {
					(PM->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;

	case ME_MASTER_TEMPER_TYPE:
		for (i = 0; i < MAX_CHANNELS; i++) {
			(PM->channel)[i].temper_type = (PM->current_event)->a;
			ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 1, i, (PM->channel)[i].temper_type);
		}
		if ((PM->temper_type_mute)) {
			if ((PM->temper_type_mute) & 1 << (PM->current_event)->a
					- (((PM->current_event)->a >= 0x40) ? 0x3c : 0)) {
				FILL_CHANNELMASK((PM->channel_mute));
				for (i = 0; i < MAX_CHANNELS; i++)
					ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, i, 1);
			} else {
				CLEAR_CHANNELMASK((PM->channel_mute));
				for (i = 0; i < MAX_CHANNELS; i++)
					ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, i, 0);
			}
		}
		if ((PM->current_event)->b)
			for (i = 0; i < (PM->upper_voices); i++)
				if ((PM->voice)[i].status != VOICE_FREE) {
					(PM->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;

	case ME_SYSEX_LSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_LSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	case ME_SYSEX_MSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_MSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	case ME_SYSEX_GS_LSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_GS_LSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	case ME_SYSEX_GS_MSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_GS_MSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	case ME_SYSEX_XG_LSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_XG_LSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	case ME_SYSEX_XG_MSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_XG_MSB,ch,(PM->current_event)->a,(PM->current_event)->b);
	    break;

	case ME_NOTE_STEP:
		i = ev->a + ((ev->b & 0x0f) << 8);
		j = ev->b >> 4;
		ctl_mode_event(tmdy_struct, CTLE_METRONOME, 1, i, j);
		if (TMDY_READMIDI->readmidi_wrd_mode)
			TMDY_WRD->wrdt->update_events(tmdy_struct);
		break;

      case ME_EOT:
	return midi_play_end(tmdy_struct);
    }
#ifndef SUPPRESS_CHANNEL_LAYER
		}
	}
	ev->channel = orig_ch;
#endif


    return RC_NONE;
}

static void set_single_note_tuning(tmdy_struct_ex_t *tmdy_struct, int part, int a, int b, int rt)
{
/*
	static int PM->set_s_n_t_tp;	* tuning program number *
	static int PM->set_s_n_t_kn;	* MIDI key number *
	static int PM->set_s_n_t_st;	* the nearest equal-tempered semitone *
*/
	double f, fst;	/* fraction of semitone */
	int i;
	
	switch (part) {
	case 0:
		PM->set_s_n_t_tp = a;
		break;
	case 1:
		PM->set_s_n_t_kn = a;
		PM->set_s_n_t_st = b;
		break;
	case 2:
		if (PM->set_s_n_t_st == 0x7f && a == 0x7f && b == 0x7f)	/* no change */
			break;
		f = 440 * pow(2.0, (PM->set_s_n_t_st - 69) / 12.0);
		fst = pow(2.0, (a << 7 | b) / 196608.0);
		freq_table_tuning[PM->set_s_n_t_tp][PM->set_s_n_t_kn] = f * fst * 1000 + 0.5;
		if (rt)
			for (i = 0; i < (PM->upper_voices); i++)
				if ((PM->voice)[i].status != VOICE_FREE) {
					(PM->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;
	}
}

static int play_midi(tmdy_struct_ex_t *tmdy_struct, MidiEvent *eventlist, int32 samples)
{
    int rc;
//    static int play_count = 0;

    if ((TMDY_OUTPUT->play_mode)->id_character == 'M') {
	int cnt;

	TMDY_M2M->convert_mod_to_midi_file(tmdy_struct, eventlist);

	PM->play_midi_play_count = 0;
	cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);	/* free unused memory */
	if(cnt > 0)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
		      "%d memory blocks are free", cnt);
	return RC_NONE;
    }

    (PM->sample_count) = samples;
    (PM->event_list) = eventlist;
    (PM->lost_notes) = (PM->cut_notes) = 0;
    (PM->check_eot_flag) = 1;

    TMDY_WRD->wrd_midi_event(tmdy_struct, -1, -1); /* For initialize */

    reset_midi(tmdy_struct, 0);
    if(!(PM->opt_realtime_playing) &&
       TMDY_RECACHE->allocate_cache_size > 0 &&
       !IS_CURRENT_MOD_FILE &&
       ((TMDY_OUTPUT->play_mode)->flag&PF_PCM_STREAM))
    {
	play_midi_prescan(tmdy_struct, eventlist);
	reset_midi(tmdy_struct, 0);
    }

    rc = (TMDY_AQ->aq_flush)(tmdy_struct, 0);
    if(RC_IS_SKIP_FILE(rc)){
		return rc;
	}

    skip_to(tmdy_struct, (PM->midi_restart_time));

    if((PM->midi_restart_time) > 0) { /* Need to update interface display */
      int i;
      for(i = 0; i < MAX_CHANNELS; i++)
	redraw_controllers(tmdy_struct, i);
    }
    rc = RC_NONE;
    for(;;)
    {
	(PM->midi_restart_time) = 1;
	rc = play_event(tmdy_struct, (PM->current_event));
	if(rc != RC_NONE)
	    break;
	if ((PM->midi_restart_time))    /* don't skip the first event if == 0 */
	    (PM->current_event)++;
    }

    if(PM->play_midi_play_count++ > 3)
    {
	int cnt;
	PM->play_midi_play_count = 0;
	cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);	/* free unused memory */
	if(cnt > 0)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
		      "%d memory blocks are free", cnt);
    }

    return rc;
}

static void read_header_wav(tmdy_struct_ex_t *tmdy_struct, struct timidity_file* tf)
{
    char buff[44];
    TMDY_COMMON->tf_read(tmdy_struct,  buff, 1, 44, tf);
}

static int read_header_aiff(tmdy_struct_ex_t *tmdy_struct, struct timidity_file* tf)
{
    char buff[5]="    ";
    int i;
    
    for( i=0; i<100; i++ ){
    	buff[0]=buff[1]; buff[1]=buff[2]; buff[2]=buff[3];
    	TMDY_COMMON->tf_read(tmdy_struct,  &buff[3], 1, 1, tf);
    	if( strcmp(buff,"SSND")==0 ){
            /*SSND chunk found */
    	    TMDY_COMMON->tf_read(tmdy_struct,  &buff[0], 1, 4, tf);
    	    TMDY_COMMON->tf_read(tmdy_struct,  &buff[0], 1, 4, tf);
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "aiff header read OK.");
	    return 0;
    	}
    }
    /*SSND chunk not found */
    return -1;
}

static int load_pcm_file_wav(tmdy_struct_ex_t *tmdy_struct)
{
    char *filename;

    if(strcmp((PM->pcm_alternate_file), "auto") == 0)
    {
	filename = TMDY_COMMON->safe_malloc(tmdy_struct, strlen(TMDY_READMIDI->current_file_info->filename)+5);
	strcpy(filename, TMDY_READMIDI->current_file_info->filename);
	strcat(filename, ".wav");
    }
    else if(strlen((PM->pcm_alternate_file)) >= 5 &&
	    strncasecmp((PM->pcm_alternate_file) + strlen((PM->pcm_alternate_file)) - 4,
			".wav", 4) == 0)
	filename = TMDY_COMMON->safe_strdup(tmdy_struct, (PM->pcm_alternate_file));
    else{
		return -1;
	}
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "wav filename: %s", filename);
    TMDY_READMIDI->current_file_info->pcm_tf = TMDY_COMMON->open_file(tmdy_struct, filename, 0, OF_SILENT);
    if( TMDY_READMIDI->current_file_info->pcm_tf ){
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open successed.");
	read_header_wav(tmdy_struct, TMDY_READMIDI->current_file_info->pcm_tf);
	TMDY_READMIDI->current_file_info->pcm_filename = filename;
	TMDY_READMIDI->current_file_info->pcm_mode = PCM_MODE_WAV;
	return 0;
    }else{
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open failed.");
	free(filename);
	TMDY_READMIDI->current_file_info->pcm_filename = NULL;
	return -1;
    }

}

static int load_pcm_file_aiff(tmdy_struct_ex_t *tmdy_struct)
{
    char *filename;

    if(strcmp((PM->pcm_alternate_file), "auto") == 0)
    {
	filename = TMDY_COMMON->safe_malloc(tmdy_struct, strlen(TMDY_READMIDI->current_file_info->filename)+6);
	strcpy(filename, TMDY_READMIDI->current_file_info->filename);
	strcat( filename, ".aiff");
    }
    else if(strlen((PM->pcm_alternate_file)) >= 6 &&
	    strncasecmp((PM->pcm_alternate_file) + strlen((PM->pcm_alternate_file)) - 5,
			".aiff", 5) == 0)
	filename = TMDY_COMMON->safe_strdup(tmdy_struct, (PM->pcm_alternate_file));
    else{
		return -1;
	}

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "aiff filename: %s", filename);
    TMDY_READMIDI->current_file_info->pcm_tf = TMDY_COMMON->open_file(tmdy_struct, filename, 0, OF_SILENT);
    if( TMDY_READMIDI->current_file_info->pcm_tf ){
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open successed.");
	read_header_aiff(tmdy_struct, TMDY_READMIDI->current_file_info->pcm_tf);
	TMDY_READMIDI->current_file_info->pcm_filename = filename;
	TMDY_READMIDI->current_file_info->pcm_mode = PCM_MODE_AIFF;
	return 0;
    }else{
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open failed.");
	free(filename);
	TMDY_READMIDI->current_file_info->pcm_filename = NULL;
	return -1;
    }

}

static void load_pcm_file(tmdy_struct_ex_t *tmdy_struct)
{
    if( load_pcm_file_wav(tmdy_struct)==0 ) return; /*load OK*/
    if( load_pcm_file_aiff(tmdy_struct)==0 ) return; /*load OK*/
}

static int play_midi_load_file(tmdy_struct_ex_t *tmdy_struct, char *fn,
			       MidiEvent **event,
			       int32 *nsamples)
{
    int rc;
    struct timidity_file *tf;
    int32 nevents;

    *event = NULL;

    if(!strcmp(fn, "-"))
	(PM->file_from_stdin) = 1;
    else
	(PM->file_from_stdin) = 0;

    ctl_mode_event(tmdy_struct, CTLE_NOW_LOADING, 0, (long)fn, 0);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "MIDI file: %s", fn);
    if((tf = TMDY_READMIDI->open_midi_file(tmdy_struct, fn, 1, OF_VERBOSE)) == NULL)
    {
	ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, -1, 0);
	return RC_ERROR;
    }

    *event = NULL;
    rc = check_apply_control(tmdy_struct);
    if(RC_IS_SKIP_FILE(rc))
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, 1, 0);
	return rc;
    }

    *event = TMDY_READMIDI->read_midi_file(tmdy_struct, tf, &nevents, nsamples, fn);
    TMDY_COMMON->close_file(tmdy_struct, tf);

    if(*event == NULL)
    {
	ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, -1, 0);
	return RC_ERROR;
    }

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
	      "%d supported events, %d samples, time %d:%02d",
	      nevents, *nsamples,
	      *nsamples / (TMDY_OUTPUT->play_mode)->rate / 60,
	      (*nsamples / (TMDY_OUTPUT->play_mode)->rate) % 60);

    TMDY_READMIDI->current_file_info->pcm_mode = PCM_MODE_NON; /*initialize*/
    if((PM->pcm_alternate_file) != NULL &&
       strcmp((PM->pcm_alternate_file), "none") != 0 &&
       ((TMDY_OUTPUT->play_mode)->flag&PF_PCM_STREAM))
	load_pcm_file(tmdy_struct);

    if(!IS_CURRENT_MOD_FILE &&
       ((TMDY_OUTPUT->play_mode)->flag&PF_PCM_STREAM))
    {
	/* FIXME: Instruments is not need for pcm_alternate_file. */

	/* Load instruments
	 * If (PM->opt_realtime_playing), the instruments will be loaded later.
	 */
	if(!(PM->opt_realtime_playing))
	{
	    rc = RC_NONE;
	    TMDY_INSTRUM->load_missing_instruments(tmdy_struct, &rc);
	    if(RC_IS_SKIP_FILE(rc))
	    {
		/* Instrument loading is terminated */
		ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, 1, 0);
		TMDY_INSTRUM->clear_magic_instruments(tmdy_struct);
		return rc;
	    }
	}
    }
    else
	TMDY_INSTRUM->clear_magic_instruments(tmdy_struct);	/* Clear load markers */

    ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, 0, 0);


    return RC_NONE;
}

int play_midi_file(tmdy_struct_ex_t *tmdy_struct, char *fn)
{
    int i, j, rc;
//     static int last_rc = RC_NONE;
   MidiEvent *event;
    int32 nsamples;

    /* Set current file information */
    TMDY_READMIDI->current_file_info = TMDY_READMIDI->get_midi_file_info(tmdy_struct, fn, 1);

    rc = check_apply_control(tmdy_struct);
    if(RC_IS_SKIP_FILE(rc) && rc != RC_RELOAD){
		return rc;
	}

    /* Reset key & speed each files */
    (PM->current_keysig)  = (PM->opt_init_keysig);
    (PM->note_key_offset) = 0;
    (PM->midi_time_ratio) = 1.0;
	for (i = 0; i < MAX_CHANNELS; i++) {
		for (j = 0; j < 12; j++)
			(PM->channel)[i].scale_tuning[j] = 0;
		(PM->channel)[i].prev_scale_tuning = 0;
		(PM->channel)[i].temper_type = 0;
	}
    CLEAR_CHANNELMASK((PM->channel_mute));
	if ((PM->temper_type_mute) & 1)
		FILL_CHANNELMASK((PM->channel_mute));

    /* Reset restart offset */
    (PM->midi_restart_time) = 0;

#ifdef REDUCE_VOICE_TIME_TUNING
    /* Reset voice reduction stuff */
    (PM->min_bad_nv) = 256;
    (PM->max_good_nv) = 1;
    (PM->ok_nv_total) = 32;
    (PM->ok_nv_counts) = 1;
    (PM->ok_nv) = 32;
    (PM->ok_nv_sample) = 0;
    (PM->old_rate) = -1;
    (PM->reduce_quality_flag) = (PM->no_4point_interpolation);
    timidity_mutex_lock(PM->busy);
    restore_voices(tmdy_struct, 0);
    timidity_mutex_unlock(PM->busy);
#endif

	PM->ctl_mode_event(tmdy_struct, CTLE_METRONOME, 0, 0, 0);
	PM->ctl_mode_event(tmdy_struct, CTLE_KEYSIG, 0, (PM->current_keysig), 0);
	PM->ctl_mode_event(tmdy_struct, CTLE_TEMPER_KEYSIG, 0, 0, 0);
	if ((PM->opt_force_keysig) != 8) {
		i = (PM->current_keysig) + (((PM->current_keysig) < 8) ? 7 : -6);
		j = (PM->opt_force_keysig) + (((PM->current_keysig) < 8) ? 7 : 10);
		while (i != j && i != j + 12) {
			if (++(PM->note_key_offset) > 6)
				(PM->note_key_offset) -= 12;
			i += (i > 10) ? -5 : 7;
		}
	}
	PM->ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 0, (PM->note_key_offset), 0);
	i = (PM->current_keysig) + (((PM->current_keysig) < 8) ? 7 : -9), j = 0;
	while (i != 7 && i != 19)
		i += (i < 7) ? 5 : -7, j++;
	j += (PM->note_key_offset), j -= floor(j / 12.0) * 12;
	(PM->current_freq_table) = j;
	PM->ctl_mode_event(tmdy_struct, CTLE_TEMPO, 0, (PM->current_play_tempo), 0);
	PM->ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (PM->midi_time_ratio) + 0.5, 0);
	for (i = 0; i < MAX_CHANNELS; i++) {
		PM->ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 0, i, (PM->channel)[i].temper_type);
		PM->ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, i, (PM->temper_type_mute) & 1);
	}
  play_reload: /* Come here to reload MIDI file */
    timidity_mutex_lock(PM->busy);
    rc = play_midi_load_file(tmdy_struct, fn, &event, &nsamples);
    timidity_mutex_unlock(PM->busy);
    if(RC_IS_SKIP_FILE(rc))
	goto play_end; /* skip playing */

//    TMDY_UTILS->mblock->init_mblock(tmdy_struct, &PM->playmidi_pool);
    PM->ctl_mode_event(tmdy_struct, CTLE_PLAY_START, 0, nsamples, 0);
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_PLAY_START, NULL);
    timidity_mutex_lock(PM->busy);
    rc = play_midi(tmdy_struct, event, nsamples);
    timidity_mutex_unlock(PM->busy);
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_PLAY_END, NULL);
    PM->ctl_mode_event(tmdy_struct, CTLE_PLAY_END, 0, 0, 0);
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &PM->playmidi_pool);

    for(i = 0; i < MAX_CHANNELS; i++)
	memset((PM->channel)[i].drums, 0, sizeof((PM->channel)[i].drums));

  play_end:
    if(TMDY_READMIDI->current_file_info->pcm_tf){
    	TMDY_COMMON->close_file(tmdy_struct, TMDY_READMIDI->current_file_info->pcm_tf);
    	TMDY_READMIDI->current_file_info->pcm_tf = NULL;
    	free( TMDY_READMIDI->current_file_info->pcm_filename );
    	TMDY_READMIDI->current_file_info->pcm_filename = NULL;
    }
    
    if(TMDY_WRD->wrdt->opened)
	TMDY_WRD->wrdt->end(tmdy_struct);

    if((TMDY_MAIN->free_instruments_afterwards))
    {
	int cnt;
	TMDY_INSTRUM->free_instruments(tmdy_struct, 0);
	cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct); /* free unused memory */
	if(cnt > 0)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "%d memory blocks are free",
		      cnt);
    }

    TMDY_INSTRUM->free_special_patch(tmdy_struct, -1);

    if(event != NULL)
	free(event);
    if(rc == RC_RELOAD)
	goto play_reload;

    if(rc == RC_ERROR)
    {
	if(TMDY_READMIDI->current_file_info->file_type == IS_OTHER_FILE)
	    TMDY_READMIDI->current_file_info->file_type = IS_ERROR_FILE;
	if(PM->last_rc == RC_REALLY_PREVIOUS){
	
	    return RC_REALLY_PREVIOUS;
	}
    }
    PM->last_rc = rc;

    return rc;
}

void dumb_pass_playing_list(tmdy_struct_ex_t *tmdy_struct, int number_of_files, char *list_of_files[])
{
    int i = 0;

    for(;;)
    {
	switch(play_midi_file(tmdy_struct, list_of_files[i]))
	{
	  case RC_REALLY_PREVIOUS:
	    if(i > 0)
		i--;
	    break;

	  default: /* An error or something */
	  case RC_NEXT:
	    if(i < number_of_files-1)
	    {
		i++;
		break;
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 0);

	    if(!((TMDY_CONTROLS->ctl)->flags & CTLF_LIST_LOOP))
		return;
	    i = 0;
	    break;

	    case RC_QUIT:
		return;
	}
    }
}

void default_ctl_lyric(tmdy_struct_ex_t *tmdy_struct, int lyricid)
{
    char *lyric;

    lyric = TMDY_READMIDI->event2string(tmdy_struct, lyricid);
    if(lyric != NULL)
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_TEXT, VERB_VERBOSE, "%s", lyric + 1);
}

void ctl_mode_event(tmdy_struct_ex_t *tmdy_struct, int type, int trace, long arg1, long arg2)
{
    CtlEvent ce;
    ce.type = type;
    ce.v1 = arg1;
    ce.v2 = arg2;
    if(trace && (TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

void ctl_note_event(tmdy_struct_ex_t *tmdy_struct, int noteID)
{
    CtlEvent ce;
    ce.type = CTLE_NOTE;
    ce.v1 = (PM->voice)[noteID].status;
    ce.v2 = (PM->voice)[noteID].channel;
    ce.v3 = (PM->voice)[noteID].note;
    ce.v4 = (PM->voice)[noteID].velocity;
    if((TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

static void ctl_timestamp(tmdy_struct_ex_t *tmdy_struct)
{
    long i, secs, voices;
    CtlEvent ce;
//    static int PM->last_secs = -1, PM->last_voices = -1;

    secs = (long)((PM->current_sample) / ((PM->midi_time_ratio) * (TMDY_OUTPUT->play_mode)->rate));
    for(i = voices = 0; i < (PM->upper_voices); i++)
	if((PM->voice)[i].status != VOICE_FREE)
	    voices++;
    if(secs == PM->last_secs && voices == PM->last_voices){
		return;
	}
    ce.type = CTLE_CURRENT_TIME;
    ce.v1 = PM->last_secs = secs;
    ce.v2 = PM->last_voices = voices;
    if((TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

static void ctl_updatetime(tmdy_struct_ex_t *tmdy_struct, int32 samples)
{
    long secs;
    secs = (long)(samples / ((PM->midi_time_ratio) * (TMDY_OUTPUT->play_mode)->rate));
    ctl_mode_event(tmdy_struct, CTLE_CURRENT_TIME, 0, secs, 0);
    ctl_mode_event(tmdy_struct, CTLE_REFRESH, 0, 0, 0);
}

static void ctl_prog_event(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    CtlEvent ce;
    int bank, prog;

    if(IS_CURRENT_MOD_FILE)
    {
	bank = 0;
	prog = (PM->channel)[ch].special_sample;
    }
    else
    {
	bank = (PM->channel)[ch].bank;
	prog = (PM->channel)[ch].program;
    }

    ce.type = CTLE_PROGRAM;
    ce.v1 = ch;
    ce.v2 = prog;
    ce.v3 = (long)channel_instrum_name(tmdy_struct, ch);
    ce.v4 = (bank |
	     ((PM->channel)[ch].bank_lsb << 8) |
	     ((PM->channel)[ch].bank_msb << 16));
    if((TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

static void ctl_pause_event(tmdy_struct_ex_t *tmdy_struct, int pause, int32 s)
{
    long secs;
    secs = (long)(s / ((PM->midi_time_ratio) * (TMDY_OUTPUT->play_mode)->rate));
    ctl_mode_event(tmdy_struct, CTLE_PAUSE, 0, pause, secs);
}

char *channel_instrum_name(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    char *comm;
    int bank, prog;

    if(ISDRUMCHANNEL(ch)) {
	bank = (PM->channel)[ch].bank;
	if (TMDY_INSTRUM->drumset[bank] == NULL){
		return "";
	}
	prog = 0;
	comm = TMDY_INSTRUM->drumset[bank]->tone[prog].comment;
	if (comm == NULL){
		return "";
	}
	return comm;
    }

    if((PM->channel)[ch].program == SPECIAL_PROGRAM){
		return "Special Program";
	}
	
    if(IS_CURRENT_MOD_FILE)
    {
	int pr;
	pr = (PM->channel)[ch].special_sample;
	if(pr > 0 &&
	   TMDY_INSTRUM->special_patch[pr] != NULL &&
	   TMDY_INSTRUM->special_patch[pr]->name != NULL){
	
	    return TMDY_INSTRUM->special_patch[pr]->name;
	   }
	return "MOD";
    }

    bank = (PM->channel)[ch].bank;
    prog = (PM->channel)[ch].program;
    TMDY_INSTRUM->instrument_map(tmdy_struct, (PM->channel)[ch].mapID, &bank, &prog);
    if(TMDY_INSTRUM->tonebank[bank] == NULL)
	bank = 0;
	comm = TMDY_INSTRUM->tonebank[bank]->tone[prog].comment;
    if(comm == NULL)
	comm = TMDY_INSTRUM->tonebank[0]->tone[prog].comment;
    return comm;
}


/*
 * For MIDI stream player.
 */
void playmidi_stream_init(tmdy_struct_ex_t *tmdy_struct)
{
    int i;
//    static int PM->p_m_s_i_first = 1;

    (PM->note_key_offset) = 0;
    (PM->midi_time_ratio) = 1.0;
    CLEAR_CHANNELMASK((PM->channel_mute));
	if ((PM->temper_type_mute) & 1)
		FILL_CHANNELMASK((PM->channel_mute));
    (PM->midi_restart_time) = 0;
    if(PM->p_m_s_i_first)
    {
	PM->p_m_s_i_first = 0;
//        TMDY_UTILS->mblock->init_mblock(tmdy_struct, &PM->playmidi_pool);
	TMDY_READMIDI->current_file_info = TMDY_READMIDI->get_midi_file_info(tmdy_struct, "TiMidity", 1);
    PM->midi_streaming=1;
    }
    else
        TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &PM->playmidi_pool);

    /* Fill in TMDY_READMIDI->current_file_info */
    TMDY_READMIDI->current_file_info->readflag = 1;
    TMDY_READMIDI->current_file_info->seq_name = TMDY_COMMON->safe_strdup(tmdy_struct, "TiMidity server");
    TMDY_READMIDI->current_file_info->karaoke_title = TMDY_READMIDI->current_file_info->first_text = NULL;
    TMDY_READMIDI->current_file_info->mid = 0x7f;
    TMDY_READMIDI->current_file_info->hdrsiz = 0;
    TMDY_READMIDI->current_file_info->format = 0;
    TMDY_READMIDI->current_file_info->tracks = 0;
    TMDY_READMIDI->current_file_info->divisions = 192; /* ?? */
    TMDY_READMIDI->current_file_info->time_sig_n = 4; /* 4/ */
    TMDY_READMIDI->current_file_info->time_sig_d = 4; /* /4 */
    TMDY_READMIDI->current_file_info->time_sig_c = 24; /* clock */
    TMDY_READMIDI->current_file_info->time_sig_b = 8;  /* q.n. */
    TMDY_READMIDI->current_file_info->samples = 0;
    TMDY_READMIDI->current_file_info->max_channel = MAX_CHANNELS;
    TMDY_READMIDI->current_file_info->compressed = 0;
    TMDY_READMIDI->current_file_info->midi_data = NULL;
    TMDY_READMIDI->current_file_info->midi_data_size = 0;
    TMDY_READMIDI->current_file_info->file_type = IS_OTHER_FILE;

    (PM->current_play_tempo) = 500000;
    (PM->check_eot_flag) = 0;

    /* Setup default drums */
   	COPY_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, PM->default_drumchannels);
	COPY_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannel_mask, PM->default_drumchannel_mask);
    for(i = 0; i < MAX_CHANNELS; i++)
	memset((PM->channel)[i].drums, 0, sizeof((PM->channel)[i].drums));
	TMDY_READMIDI->change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);
    reset_midi(tmdy_struct, 0);

    playmidi_tmr_reset(tmdy_struct);

}

void playmidi_tmr_reset(tmdy_struct_ex_t *tmdy_struct)
{
    int i;

    (TMDY_AQ->aq_flush)(tmdy_struct, 0);
    (PM->current_sample) = 0;
    (PM->buffered_count) = 0;
    (PM->buffer_pointer) = (PM->common_buffer);
    for(i = 0; i < MAX_CHANNELS; i++)
	(PM->channel)[i].lasttime = 0;
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_PLAY_START, NULL);
}
	
	
	
int ts_play_midi_file(tmdy_struct_ex_t *tmdy_struct, char *fn){
	int ts_buf;
//	timidity_mutex_lock(PM->busy);
	ts_buf=play_midi_file(tmdy_struct, fn);
//	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
void ts_dumb_pass_playing_list(tmdy_struct_ex_t *tmdy_struct, int number_of_files, char *list_of_files[]){
//	timidity_mutex_lock(PM->busy);
	dumb_pass_playing_list(tmdy_struct, number_of_files, list_of_files);
//	timidity_mutex_unlock(PM->busy);
}
void ts_default_ctl_lyric(tmdy_struct_ex_t *tmdy_struct, int lyricid){
	timidity_mutex_lock(PM->busy);
	default_ctl_lyric(tmdy_struct, lyricid);
	timidity_mutex_unlock(PM->busy);
}
int ts_check_apply_control(tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=check_apply_control(tmdy_struct);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
void ts_recompute_freq(tmdy_struct_ex_t *tmdy_struct, int v){
	timidity_mutex_lock(PM->busy);
	recompute_freq(tmdy_struct, v);
	timidity_mutex_unlock(PM->busy);
}
int ts_midi_drumpart_change(tmdy_struct_ex_t *tmdy_struct, int ch, int isdrum){
	int ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=midi_drumpart_change(tmdy_struct, ch, isdrum);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
void ts_ctl_note_event(tmdy_struct_ex_t *tmdy_struct, int noteID){
	timidity_mutex_lock(PM->busy);
	ctl_note_event(tmdy_struct, noteID);
	timidity_mutex_unlock(PM->busy);
}
void ts_ctl_mode_event(tmdy_struct_ex_t *tmdy_struct, int type, int trace, long arg1, long arg2){
	timidity_mutex_lock(PM->busy);
	ctl_mode_event(tmdy_struct, type, trace, arg1, arg2);
	timidity_mutex_unlock(PM->busy);
}
char *ts_channel_instrum_name(tmdy_struct_ex_t *tmdy_struct, int ch){
	char *ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=channel_instrum_name(tmdy_struct, ch);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
int ts_get_reverb_level(tmdy_struct_ex_t *tmdy_struct, int ch){
	int ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=get_reverb_level(tmdy_struct, ch);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
int ts_get_chorus_level(tmdy_struct_ex_t *tmdy_struct, int ch){
	int ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=get_chorus_level(tmdy_struct, ch);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
void ts_playmidi_output_changed(tmdy_struct_ex_t *tmdy_struct, int play_state){
	timidity_mutex_lock(PM->busy);
	playmidi_output_changed(tmdy_struct, play_state);
	timidity_mutex_unlock(PM->busy);
}
Instrument *ts_play_midi_load_instrument(tmdy_struct_ex_t *tmdy_struct, int dr, int bk, int prog){
	Instrument *ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=play_midi_load_instrument(tmdy_struct, dr, bk, prog);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
void ts_midi_program_change(tmdy_struct_ex_t *tmdy_struct, int ch, int prog){
	timidity_mutex_lock(PM->busy);
	midi_program_change(tmdy_struct, ch, prog);
	timidity_mutex_unlock(PM->busy);
}
void ts_free_voice(tmdy_struct_ex_t *tmdy_struct, int v){
	timidity_mutex_lock(PM->busy);
	free_voice(tmdy_struct, v);
	timidity_mutex_unlock(PM->busy);
}
void ts_play_midi_setup_drums(tmdy_struct_ex_t *tmdy_struct, int ch,int note){
	timidity_mutex_lock(PM->busy);
	play_midi_setup_drums(tmdy_struct, ch, note);
	timidity_mutex_unlock(PM->busy);
}
/* For stream player */
void ts_playmidi_stream_init(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(PM->busy);
	playmidi_stream_init(tmdy_struct);
	timidity_mutex_unlock(PM->busy);
}
void ts_playmidi_tmr_reset(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(PM->busy);
	playmidi_tmr_reset(tmdy_struct);
	timidity_mutex_unlock(PM->busy);
}
int ts_play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev){
	int ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=play_event(tmdy_struct, ev);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}
void ts_dup_tone_bank_element(tmdy_struct_ex_t *tmdy_struct, int a,int b,int c){
	timidity_mutex_lock(PM->busy);
	dup_tone_bank_element(tmdy_struct, a,b,c);
	timidity_mutex_unlock(PM->busy);
}
void ts_free_tone_bank_element(tmdy_struct_ex_t *tmdy_struct, int a,int b,int c){
	timidity_mutex_lock(PM->busy);
	free_tone_bank_element(tmdy_struct, a,b,c);
	timidity_mutex_unlock(PM->busy);
}
void ts_recompute_voice_filter(tmdy_struct_ex_t *tmdy_struct, int a){
	timidity_mutex_lock(PM->busy);
	recompute_voice_filter(tmdy_struct, a);
	timidity_mutex_unlock(PM->busy);
}
int ts_compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count){
	int ts_buf;
	timidity_mutex_lock(PM->busy);
	ts_buf=compute_data(tmdy_struct, count);
	timidity_mutex_unlock(PM->busy);
	return ts_buf;
}

playmidi_ex_t* new_playmidi(tmdy_struct_ex_t *tmdy_struct){
	playmidi_ex_t* playmidi_ex;

	playmidi_ex=(playmidi_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(playmidi_ex_t));
	timidity_mutex_init(playmidi_ex->busy);

	playmidi_ex->max_voices = DEFAULT_VOICES;
	playmidi_ex->voice = NULL;
	
	playmidi_ex->control_ratio=0;
	playmidi_ex->amplification=DEFAULT_AMPLIFICATION;
	playmidi_ex->adjust_panning_immediately=1;
	playmidi_ex->voices=DEFAULT_VOICES;
	playmidi_ex->note_key_offset = 0;	/* For key up/down */
	playmidi_ex->midi_time_ratio = 1.0;	/* For speed up/down */
#ifdef MODULATION_WHEEL_ALLOW
	playmidi_ex->opt_modulation_wheel = 1;
#else
	playmidi_ex->opt_modulation_wheel = 0;
#endif /* MODULATION_WHEEL_ALLOW */

#ifdef PORTAMENTO_ALLOW
	playmidi_ex->opt_portamento = 1;
#else
	playmidi_ex->opt_portamento = 0;
#endif /* PORTAMENTO_ALLOW */

#ifdef NRPN_VIBRATO_ALLOW
	playmidi_ex->opt_nrpn_vibrato = 1;
#else
	playmidi_ex->opt_nrpn_vibrato = 0;
#endif /* NRPN_VIBRATO_ALLOW */
#ifdef REVERB_CONTROL_ALLOW
	playmidi_ex->opt_reverb_control = 1;
#else
#ifdef FREEVERB_CONTROL_ALLOW
	playmidi_ex->opt_reverb_control = 3;
#else
	playmidi_ex->opt_reverb_control = 0;
#endif /* FREEVERB_CONTROL_ALLOW */
#endif /* REVERB_CONTROL_ALLOW */
#ifdef CHORUS_CONTROL_ALLOW
	playmidi_ex->opt_chorus_control = 1;
#else
	playmidi_ex->opt_chorus_control = 0;
#endif /* CHORUS_CONTROL_ALLOW */
#ifdef SURROUND_CHORUS_ALLOW
	playmidi_ex->opt_surround_chorus = 1;
#else
	playmidi_ex->opt_surround_chorus = 0;
#endif /* SURROUND_CHORUS_ALLOW */
#ifdef GM_CHANNEL_PRESSURE_ALLOW
	playmidi_ex->opt_channel_pressure = 1;
#else
	playmidi_ex->opt_channel_pressure = 0;
#endif /* GM_CHANNEL_PRESSURE_ALLOW */
#ifdef VOICE_BY_VOICE_LPF_ALLOW
	playmidi_ex->opt_lpf_def = 1;
#else
	playmidi_ex->opt_lpf_def = 0;
#endif /* VOICE_BY_VOICE_LPF_ALLOW */
#ifdef OVERLAP_VOICE_ALLOW
	playmidi_ex->opt_overlap_voice_allow = 1;
#else
	playmidi_ex->opt_overlap_voice_allow = 0;
#endif /* OVERLAP_VOICE_ALLOW */
#ifdef TEMPER_CONTROL_ALLOW
	playmidi_ex->opt_temper_control = 1;
#else
	playmidi_ex->opt_temper_control = 0;
#endif /* TEMPER_CONTROL_ALLOW */
	playmidi_ex->opt_tva_attack = 0;	/* attack envelope control */
	playmidi_ex->opt_tva_decay = 0;	/* decay envelope control */
	playmidi_ex->opt_tva_release = 0;	/* release envelope control */
	playmidi_ex->opt_delay_control = 0;	/* CC#94 delay(celeste) effect control */
	playmidi_ex->opt_eq_control = 0;		/* channel equalizer control */
	playmidi_ex->opt_insertion_effect = 0;	/* insertion effect control */
	playmidi_ex->opt_drum_effect = 0;	/* drumpart effect control */
	playmidi_ex->opt_drum_power = 100;		/* coef. of drum amplitude */
	playmidi_ex->opt_amp_compensation = 0;
	playmidi_ex->opt_modulation_envelope = 0;
	playmidi_ex->opt_pan_delay = 1;	/* phase difference between left ear and right ear. */
	playmidi_ex->opt_user_volume_curve = 0;
	playmidi_ex->current_play_tempo = 500000;
	playmidi_ex->opt_realtime_playing = 0;	
	playmidi_ex->reduce_voice_threshold = -1; /* msec */
	playmidi_ex->special_tonebank = -1;
	playmidi_ex->default_tonebank = 0;
	playmidi_ex->playmidi_seek_flag = 0;
	playmidi_ex->play_pause_flag = 0;
	playmidi_ex->auto_reduce_polyphony=1;
	playmidi_ex->reduce_quality_flag=0;
	playmidi_ex->no_4point_interpolation=0;
	playmidi_ex->current_keysig = 0;
	playmidi_ex->current_temper_keysig = 0;
	playmidi_ex->temper_adj = 0;
	playmidi_ex->opt_init_keysig = 0;
	playmidi_ex->opt_force_keysig = 8;
	playmidi_ex->key_adjust = 0;
	playmidi_ex->opt_pure_intonation = 0;
	playmidi_ex->current_freq_table = 0;

	playmidi_ex->play_midi_file=ts_play_midi_file;
	playmidi_ex->dumb_pass_playing_list=ts_dumb_pass_playing_list;
	playmidi_ex->default_ctl_lyric=ts_default_ctl_lyric;
	playmidi_ex->check_apply_control=ts_check_apply_control;
	playmidi_ex->recompute_freq=ts_recompute_freq;
	playmidi_ex->midi_drumpart_change=ts_midi_drumpart_change;
	playmidi_ex->ctl_note_event=ts_ctl_note_event;
	playmidi_ex->ctl_mode_event=ts_ctl_mode_event;
	playmidi_ex->channel_instrum_name=ts_channel_instrum_name;
	playmidi_ex->get_reverb_level=ts_get_reverb_level;
	playmidi_ex->get_chorus_level=ts_get_chorus_level;
	playmidi_ex->playmidi_output_changed=ts_playmidi_output_changed;
	playmidi_ex->play_midi_load_instrument=ts_play_midi_load_instrument;
	playmidi_ex->midi_program_change=ts_midi_program_change;
	playmidi_ex->free_voice=ts_free_voice;
	playmidi_ex->play_midi_setup_drums=ts_play_midi_setup_drums;

/* For stream player */
	playmidi_ex->playmidi_stream_init=ts_playmidi_stream_init;
	playmidi_ex->playmidi_tmr_reset=ts_playmidi_tmr_reset;
	playmidi_ex->play_event=ts_play_event;

	playmidi_ex->dup_tone_bank_element=ts_dup_tone_bank_element;
	playmidi_ex->free_tone_bank_element=ts_free_tone_bank_element;

	playmidi_ex->recompute_voice_filter=ts_recompute_voice_filter;
	playmidi_ex->compute_data=ts_compute_data;

/**** private valiables  ****/
#ifdef REDUCE_VOICE_TIME_TUNING
	
	playmidi_ex->max_good_nv = 1;
	playmidi_ex->min_bad_nv = 256;
	playmidi_ex->ok_nv_total = 32;
	playmidi_ex->ok_nv_counts = 1;
	playmidi_ex->ok_nv_sample = 0;
	playmidi_ex->ok_nv = 32;
	playmidi_ex->old_rate = -1;
#endif
	playmidi_ex->midi_streaming=0;
	playmidi_ex->stream_max_compute=500; /* compute time limit (in msec) when streaming */
	playmidi_ex->midi_restart_time = 0;
	TMDY_UTILS->mblock->init_mblock(tmdy_struct, &(playmidi_ex->playmidi_pool));
	playmidi_ex->make_rvid_flag = 0; /* For reverb optimization */
	/* Ring voice id for each notes.  This ID enables duplicated note. */
	playmidi_ex->voices=DEFAULT_VOICES;
	playmidi_ex->master_volume_ratio = 0xFFFF;
//	playmidi_ex->envelope_modify_rate = 1.0;

	playmidi_ex->pcm_alternate_file = NULL; /* NULL or "none": Nothing (default)
				  * "auto": Auto select
				  * filename: Use it
				  */
	playmidi_ex->reverb_buffer = NULL; /* MAX_CHANNELS*AUDIO_BUFFER_SIZE*8 */
	playmidi_ex->compensation_ratio = 1.0; /* compensation ratio */
	
//	playmidi_ex->common_buffer=malloc(sizeof(int32)*(AUDIO_BUFFER_SIZE*2));

	playmidi_ex->restore_voices_old_voices=-1;
	playmidi_ex->play_midi_play_count=0;
	playmidi_ex->last_rc = RC_NONE;
	playmidi_ex->last_secs = -1;
	playmidi_ex->last_voices = -1;
	playmidi_ex->p_m_s_i_first = 1;	
	
	return playmidi_ex;
}


void destroy_playmidi(playmidi_ex_t* playmidi){
	timidity_mutex_destroy(playmidi->busy);
	free(playmidi);
}

