﻿// WinTab.NET v1.5
//      Copyfight (C) 2006-2009 by l.p.m.11
//
// -v0.1 : first alpha release
//      programmed by l.p.m.11  : '06/01/06
// -v0.02 (file version:1.2.0)
//		                        : '06/01/07
//  v1.30 (file version:1.3.0) : first stable release
//                              : '06/01/10
//  v1.31 (file version:1.3.1)
//                              : '06/01/15
//  v1.5
//                              : '08/12/31
//
//  *Note
//    You must compile with /unsafe option.
//
//  -wintab.h : copyright 1991-1998 by LCS/Telegraphics.
//  -thanks:
//      http://www9.plala.or.jp/herm/Pages/Localized/WinTab/
//

// ↓WinTab32.dllを動的リンクするように指示します。
// このフラグを指定した場合、WinTab.LoadWinTab()でWinTabを初期化することができます。
// また参照に「WinTabDotnet」だけでなく、「WinTabInvoker」が必要です。
#define WINTAB_DYNAMIC

using System;
using System.Collections;
using System.Runtime.InteropServices;

namespace WinTabDotnet {
    using HCTX      =   IntPtr;
    using BOOL      =   Int32;
    using System.Windows.Forms;
    
    // --------------------------------------------------------------------------
    // Messages, Constants

    /// <summary>
    /// WinTabDotnetで使用される例外クラス。
    /// </summary>
    [Serializable()]
    public class WinTabException : Exception {
        /// <summary>
        /// 新しいWinTab例外クラスを生成します。
        /// </summary>
        public WinTabException() : base("WinTab internal exception occured.") {}

        /// <summary>
        /// 新しいWinTab例外クラスを生成します。
        /// </summary>
        /// <param name="es">例外の説明</param>
        public WinTabException(string es) : base(es) {}
    }

    /// <summary>
    /// WinTabをラップしたメインのクラス。
    /// <remarks>
    /// WinTabを使うために必要な関数、WinTab32.DLLの関数、定数、プロパティ値等が全て定義されています。
    /// ユーザは、このクラスを使用してペンタブレットに関する情報を得ることができます。
    /// </remarks>
    /// </summary>
    public static partial class WinTab {
        public const int WT_DEFBASE         = 0x7FF0;
        public const int WT_MAXOFFSET       = 0xF;

        /// <summary>パケットの到着メッセージ</summary>
        public const int WT_PACKET          = WT_DEFBASE + 0;
        public const int WT_CTXOPEN         = WT_DEFBASE + 1;
        public const int WT_CTXCLOSE        = WT_DEFBASE + 2;
        public const int WT_CTXUPDATE       = WT_DEFBASE + 3;
        public const int WT_CTXOVERLAP      = WT_DEFBASE + 4;
        public const int WT_PROXIMITY       = WT_DEFBASE + 5;
        public const int WT_INFOCHANGE      = WT_DEFBASE + 6;
        /// <summary>カーソルの追加メッセージ</summary>
        public const int WT_CSRCHANGE       = WT_DEFBASE + 7;
        public const int WT_MAX             = WT_DEFBASE + WT_MAXOFFSET;

        public const int LCNAMELEN          = 40;
        public const WTPKT WTPKT_ALL =
            WTPKT.CONTEXT | WTPKT.STATUS | WTPKT.TIME | WTPKT.CHANGED | WTPKT.SERIAL_NUMBER | 
            WTPKT.CURSOR | WTPKT.BUTTONS | WTPKT.X | WTPKT.Y | WTPKT.Z |
            WTPKT.NORMAL_PRESSURE | WTPKT.TANGENT_PRESSURE;
    };

    // --------------------------------------------------------------------------
    // COMMON DATA DEFS

    /// <summary>
    /// イベントパケットにおける、個々のオプションデータアイテムの属性を表すビットフィールド。
    /// </summary>
    [Flags]
    public enum WTPKT : uint /* DWORD */ {
        /// <summary>コンテキストのハンドル</summary>
        CONTEXT				= 0x0001,
        /// <summary>状態</summary>
        STATUS				= 0x0002,
        /// <summary>パケット生成時刻</summary>
        TIME				= 0x0004,
        /// <summary>変更されたパケットアイテム</summary>
        CHANGED				= 0x0008,
        /// <summary>パケットのシリアル番号</summary>
        SERIAL_NUMBER   	= 0x0010,
        /// <summary>パケットを生成したカーソル</summary>
        CURSOR				= 0x0020,
        /// <summary>ボタン情報</summary>
        BUTTONS				= 0x0040,
        /// <summary>X座標</summary>
        X					= 0x0080,
        /// <summary>Y座標</summary>
        Y					= 0x0100,
        /// <summary>Z座標</summary>
        Z					= 0x0200,
        /// <summary>筆圧</summary>
        NORMAL_PRESSURE		= 0x0400,
        /// <summary>接面筆圧</summary>
        TANGENT_PRESSURE	= 0x0800,
        /// <summary>カーソルの方向</summary>
        ORIENTATION			= 0x1000,
        /// <summary>カーソルの角度</summary>
        ROTATION            = 0x2000
    };

    // --------------------------------------------------------------------------
    // INFO DATA DEFS

    /// <summary>軸データ。</summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct AXIS {
        /// <summary>座標値データアイテムの最小値</summary>
        public int      axMin;
        /// <summary>座標値データアイテムの最大値</summary>
        public int      axMax;
        /// <summary>データアイテムの分解能の計算に使用された単位</summary>
        public uint     axUnits;
        /// <summary>物理単位ごとに増加するデータアイテム値をあらわす固定小数点数</summary>
        public Int32    axResolution;
    };

    /// <summary>分解能。</summary>
    public enum UnitSpecifies {
        /// <summary>物理的な単位では分解能情報を取得できない事を表す</summary>
        NONE        = 0,
        /// <summary>インチ単位の分解能を取得可能</summary>
        INCHES      = 1,
        /// <summary>センチメートル単位の分解能を取得可能</summary>
        CENTIMETERS = 2,
        /// <summary>1回転の弧分の分解能を取得可能</summary>
        CIRCLE      = 3
    };

    /// <summary>マウスボタンアクション。</summary>
    public enum SystemButton {
        NONE		= 0x00,
        LCLICK		= 0x01,
        LDBLCLICK	= 0x02,
        LDRAG		= 0x03,
        RCLICK		= 0x04,
        RDBLCLICK	= 0x05,
        RDRAG		= 0x06,
        MCLICK		= 0x07,
        MDBLCLICK	= 0x08,
        MDRAG		= 0x09,

        // For pen windows
        PTCLICK		= 0x10,
        PTDBLCLICK	= 0x20,
        PTDRAG		= 0x30,
        PNCLICK		= 0x40,
        PNDBLCLICK	= 0x50,
        PNDRAG		= 0x60,
        P1CLICK		= 0x70,
        P1DBLCLICK	= 0x80,
        P1DRAG		= 0x90,
        P2CLICK		= 0xA0,
        P2DBLCLICK	= 0xB0,
        P2DRAG		= 0xC0,
        P3CLICK		= 0xD0,
        P3DBLCLICK	= 0xE0,
        P3DRAG		= 0xF0
    };

    /// <summary>ハードウェア能力を表現するフラグ。</summary>
    [Flags]
    public enum HardwareCapabilities {
        /// <summary>ディスプレイとデジタイザが同じサーフェイスを共有する事を示す</summary>
        INTEGRATED		= 0x0001,
        /// <summary>カーソルが、位置を報告するデバイスと物理的に接触することを示す</summary>
        TOUCH			= 0x0002,
        /// <summary>カーソルの物理的な検知範囲において、近付いたり離れたりするイベントを生成する事が可能であること示す</summary>
        HARDPROX		= 0x0004,
        /// <summary>ハードウェアにおいてアクティブカーソルとして独自に識別が可能であること示す</summary>
        PHYSID_CURSORS	= 0x0008 /* 1.1 */
    };

    /// <summary>カーソル能力を表現するフラグ。</summary>
    [Flags]
    public enum CursorCapabilities {
        /// <summary>単体の物理的なカーソルの各モードのうちのひとつを表すカーソル形状を識別</summary>
        MULTIMODE	= 0x0001, /* 1.1 */
        /// <summary>ソフトウェアによって識別する事ができない、いくつかの物理カーソルのカーソル形状を識別</summary>
        AGGREGATE	= 0x0002, /* 1.1 */
        /// <summary>反転した方向の物理カーソルを表すカーソル形状を識別</summary>
        INVERT		= 0x0004  /* 1.1 */
    };

    /// <summary>
    /// 情報カテゴリ、WT_Info呼び出し時に使用。
    /// </summary>
    public enum InfoCategory : uint {
        /// <summary>WinTabが使用可能かチェックする場合に利用</summary>
        NONE        = 0,
        /// <summary>グローバルインタフェース識別子および能力情報</summary>
        INTERFACE   = 1,
        /// <summary>インタフェースリソースの使用統計</summary>
        STATUS      = 2,
        /// <summary>デフォルトデジタイザのための論理コンテクスト</summary>
        DEFCONTEXT  = 3,
        /// <summary>標準的なシステム論理コンテクスト</summary>
        DEFSYSCTX   = 4,
        /// <summary>デバイスの能力や状態情報</summary>
        DEVICES     = 100,
        /// <summary>カーソル形状の機能や状態情報</summary>
        CURSORS     = 200,
        /// <summary>デバイスの拡張機能についての情報</summary>
        EXTENSIONS  = 300,
        /// <summary>デフォルトデジタイザのための論理コンテクスト</summary>
        DDCTXS      = 400,
        /// <summary>デフォルトシステム論理コンテクスト</summary>
        DSCTXS      = 500
    };

    /// <summary>
    /// 情報インデックス、WT_Info呼び出し時に使用。
    /// </summary>
    public struct InfoIndex {
        /// <summary>グローバルインタフェース識別子および能力情報</summary>
        public enum INTERFACE {
            WINTABID		= 1,
            SPECVERSION		= 2,
            IMPLVERSION		= 3,
            NDEVICES		= 4,
            NCURSORS		= 5,
            NCONTEXTS		= 6,
            CTXOPTIONS		= 7,
            CTXSAVESIZE		= 8,
            NEXTENSIONS		= 9,
            NMANAGERS		= 10,
            MAX				= 10
        }

        /// <summary>インタフェースリソースの使用統計</summary>
        public enum STATUS {
            CONTEXTS		= 1,
            SYSCTXS			= 2,
            PKTRATE			= 3,
            PKTDATA			= 4,
            MANAGERS		= 5,
            SYSTEM			= 6,
            BUTTONUSE		= 7,
            SYSBTNUSE		= 8,
            MAX				= 8
        }

        /// <summary>標準的なシステム論理コンテクスト</summary>
        public enum CONTEXT {
            NAME		= 1,
            OPTIONS		= 2,
            STATUS		= 3,
            LOCKS		= 4,
            MSGBASE		= 5,
            DEVICE		= 6,
            PKTRATE		= 7,
            PKTDATA		= 8,
            PKTMODE		= 9,
            MOVEMASK	= 10,
            BTNDNMASK	= 11,
            BTNUPMASK	= 12,
            INORGX		= 13,
            INORGY		= 14,
            INORGZ		= 15,
            INEXTX		= 16,
            INEXTY		= 17,
            INEXTZ		= 18,
            OUTORGX		= 19,
            OUTORGY		= 20,
            OUTORGZ		= 21,
            OUTEXTX		= 22,
            OUTEXTY		= 23,
            OUTEXTZ		= 24,
            SENSX		= 25,
            SENSY		= 26,
            SENSZ		= 27,
            SYSMODE		= 28,
            SYSORGX		= 29,
            SYSORGY		= 30,
            SYSEXTX		= 31,
            SYSEXTY		= 32,
            SYSSENSX	= 33,
            SYSSENSY	= 34,
            MAX			= 34
        }

        /// <summary>デバイスの能力や状態情報</summary>
        public enum DEVICE {
            NAME			= 1,
            HARDWARE		= 2,
            NCSRTYPES		= 3,
            FIRSTCSR		= 4,
            PKTRATE			= 5,
            PKTDATA			= 6,
            PKTMODE			= 7,
            CSRDATA			= 8,
            XMARGIN			= 9,
            YMARGIN			= 10,
            ZMARGIN			= 11,
            X				= 12,
            Y				= 13,
            Z				= 14,
            NPRESSURE		= 15,
            TPRESSURE		= 16,
            ORIENTATION		= 17,
            ROTATION		= 18,
            PNPID			= 19,
            MAX				= 19
        }

        /// <summary>カーソル形状の機能や状態情報</summary>
        public enum CURSOR {
            NAME			= 1,
            ACTIVE			= 2,
            PKTDATA			= 3,
            BUTTONS			= 4,
            BUTTONBITS		= 5,
            BTNNAMES		= 6,
            BUTTONMAP		= 7,
            SYSBTNMAP		= 8,
            NPBUTTON		= 9,
            NPBTNMARKS		= 10,
            NPRESPONSE		= 11,
            TPBUTTON		= 12,
            TPBTNMARKS		= 13,
            TPRESPONSE		= 14,
            PHYSID			= 15,
            MODE			= 16,
            MINPKTDATA		= 17,
            MINBUTTONS		= 18,
            CAPABILITIES	= 19,
            MAX				= 19
        }

        /// <summary>デバイスの拡張機能についての情報</summary>
        public enum EXTENSION {
            NAME		= 1,
            TAG			= 2,
            MASK		= 3,
            SIZE		= 4,
            AXES		= 5,
            DEFAULT		= 6,
            DEFCONTEXT	= 7,
            DEFSYSCTX	= 8,
            CURSORS		= 9,
            MAX			= 109
        }
    };

    // --------------------------------------------------------------------------
    // CONTEXT DATA DEFS

    /// <summary>
    /// タブレットコンテクストの定義に必要な属性を保持する構造体。
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    unsafe public struct LOGCONTEXT {
        public fixed char    lcName[WinTab.LCNAMELEN];
        public ContextOption lcOptions;
        public uint	    lcStatus;
        public uint	    lcLocks;
        public uint	    lcMsgBase;
        public uint	    lcDevice;
        public uint	    lcPktRate;
        public WTPKT	lcPktData;
        public WTPKT	lcPktMode;
        public WTPKT	lcMoveMask;
        public uint	    lcBtnDnMask;
        public uint	    lcBtnUpMask;
        public int	    lcInOrgX;
        public int	    lcInOrgY;
        public int 	    lcInOrgZ;
        public int 	    lcInExtX;
        public int 	    lcInExtY;
        public int 	    lcInExtZ;
        public int 	    lcOutOrgX;
        public int 	    lcOutOrgY;
        public int	    lcOutOrgZ;
        public int 	    lcOutExtX;
        public int	    lcOutExtY;
        public int 	    lcOutExtZ;
        public Int32	lcSensX;
        public Int32	lcSensY;
        public Int32	lcSensZ;
        public int	    lcSysMode;
        public int		lcSysOrgX;
        public int		lcSysOrgY;
        public int		lcSysExtX;
        public int		lcSysExtY;
        public Int32	lcSysSensX;
        public Int32	lcSysSensY;
    }

    /// <summary>
    /// コンテキストオプションを表すフラグ。
    /// </summary>
    [Flags]
    public enum ContextOption : uint {
        /// <summary>デフォルトのオプション</summary>
        DEFAULT     = 0x0000,   // WinTab.NET extension
        /// <summary>システムカーソルコンテクスト (システムカーソルをペンに追従させるか)</summary>
        SYSTEM		= 0x0001,
        /// <summary>ペンウィンドウズがインストールされている場合、ペンウィンドウズのコンテクスト</summary>
        PEN			= 0x0002,
        /// <summary>オーナーに WT_PACKETメッセージを返すコンテクスト (自動で指定される)</summary>
        MESSAGES	= 0x0004,
        /// <summary>オーナーに WT_CSRCHANGEメッセージを返すコンテクスト (自動で指定される)</summary>
        CSRMESSAGES	= 0x0008,
        /// <summary>MARGINが指定されている場合、マージンがコンテクストの内側にあることを指定</summary>
        MGNINSIDE	= 0x4000,
        /// <summary>タブレット上の入力コンテクストがマージンを持つことを表す</summary>
        MARGIN		= 0x8000,

        /// <summary>指定したオプションをデフォルトから切る機能 (DEFAULTは無視します)</summary>
        OFFMODE     = 0x10000,  // WinTab.NET extension
    };

    /// <summary>
    /// コンテキスト状態を表すフラグ。
    /// </summary>
    [Flags]
    public enum ContextStatus {
        /// <summary>WinTabContext.Enableプロパティ(WTEnable関数)によって無効にされたコンテクストを表す</summary>
        DISABLED	= 0x0001,
        /// <summary>より高位のオーバーラップ要求によって部分的に不明瞭となっているコンテクストを表す</summary>
        OBSCURED	= 0x0002,
        /// <summary>オーバーラップ要求の最上位にあるコンテクストを表す</summary>
        ONTOP		= 0x0004
    };

    /// <summary>
    /// コンテキストのロック(変更不可)項目を示すフラグ。
    /// </summary>
    [Flags]
    public enum ContextLock {
        /// <summary>変更不可なコンテクストの入力サイズを定義</summary>
        INSIZE		= 0x0001,
        /// <summary>変更不可なコンテクストの入力アスペクト比を定義</summary>
        INASPECT	= 0x0002,
        /// <summary>変更不可なコンテクストの感度を定義</summary>
        SENSITIVITY	= 0x0004,
        /// <summary>変更不可なコンテクストのマージンオプションを定義</summary>
        MARGIN		= 0x0008,
        /// <summary>コンテクストがシステムカーソルコンテクストであった場合、システムが指し示す変更不可なコンテクストのコントロール変数の定義。</summary>
        SYSOUT		= 0x0010
    };

    // --------------------------------------------------------------------------
    // EVENT DATA DEFS

    /// <summary>
    /// Packet status values
    /// </summary>
    [Flags]
    public enum PacketStatus {
        PROXIMITY		= 0x0001,
        QUEUE_ERR		= 0x0002,
        MARGIN			= 0x0004,
        GRAB			= 0x0008,
        INVERT			= 0x0010
    }

    /// <summary>
    /// タブレットに準じたカーソルの方向。
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct ORIENTATION {
        /// <summary>z軸を通る一回転を時計回りの角度</summary>
        public int  orAzimuth;
        /// <summary>x-y平面を通る半円の角度</summary>
        public int  orAltitude;
        /// <summary>カーソルの主軸を中心とした時計回りの角度</summary>
        public int  orTwist;
    }

    /// <summary>
    /// タブレットでのカーソルの回転。
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct ROTATION { /* 1.1 */
        /// <summary>カーソルの傾斜度 (pitch)</summary>
        public int  roPitch;
        /// <summary>カーソルの回転度 (roll)</summary>
        public int  roRoll;
        /// <summary>カーソルの偏揺角度 (yaw)</summary>
        public int  roYaw;
    }

    /// <summary>
    /// 相対モードの時、ALLPACKET(pkts).pkButtonsの下位ワード。
    /// </summary>
    public enum RelativeButton {
        /// <summary>ボタンの状態変化が無いことを示す</summary>
        NONE        = 0,
        /// <summary>ボタンが上げられたことを示す</summary>
        UP          = 1,
        /// <summary>ボタンが押下されたことを示す</summary>
        DOWN        = 2
    };

    // --------------------------------------------------------------------------
    // DEVICE CONFIG CONSTANTS

    /// <summary>
    /// Device config status
    /// </summary>
    public enum WTDeviceConfig {
        NONE		= 0,
        CANCEL		= 1,
        OK			= 2,
        RESTART		= 3
    };

    // --------------------------------------------------------------------------
    // HOOK CONSTANTS

    /// <summary>
    /// Hook constants
    /// </summary>
    public enum WTHook {
        PLAYBACK		= 1,
        RECORD			= 2
    };

    /// <summary>
    /// Hook config values
    /// </summary>
    public enum WTHookConfig {
        GETLPLPFN	= -3,
        LPLPFNNEXT	= -2,
        LPFNNEXT	= -1,
        ACTION		= 0,
        GETNEXT   	= 1,
        SKIP 		= 2
    };

    // --------------------------------------------------------------------------
    // EXTENSION TAGS AND CONSTANTS
    
    /// <summary>
    /// Extension tags
    /// </summary>
    public enum WTX {
        OBT			= 0,	/// <summary>Out of bounds tracking</summary>
        FKEYS		= 1,	/// <summary>Function keys</summary>
        TILT		= 2,	/// <summary>Raw Cartesian tilt; 1.1</summary>
        CSRMASK		= 3,	/// <summary>select input by cursor type; 1.1</summary>
        XBTNMASK	= 4	    /// <summary>Extended button mask; 1.1</summary>
    };

    [StructLayout(LayoutKind.Sequential)]
    public unsafe struct XBTNMASK {
        public fixed byte xBtnDnMask[32];
        public fixed byte xBtnUpMask[32];
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct TILT {
        public int  tiltX;
        public int  tiltY;
    };

    // --------------------------------------------------------------------------
    // Functions

#if WINTAB_DYNAMIC
    static unsafe public class WinTabInvoker {
        [DllImport("kernel32",CharSet=CharSet.Unicode,SetLastError=true)]
        internal static extern IntPtr LoadLibrary(string lpFileName);
        [DllImport("kernel32",SetLastError=true)]
        internal static extern bool FreeLibrary(IntPtr hModule);
        [DllImport("kernel32",CharSet=CharSet.Ansi,SetLastError=true,ExactSpelling=false)]
        internal static extern IntPtr GetProcAddress(IntPtr hModule,string lpProcName);

        static bool m_Valid = false;
        public static bool Valid { get { return WinTabInvoker.m_Valid; } }

        public delegate IntPtr WTOpenDelegate(IntPtr hWnd,/*LOGCONTEXT*/void* lpLogCtx,BOOL fEnable);
        public delegate int WTCloseDelegate(HCTX hCTX);
        public delegate uint WTInfoDelegate(uint nCategory,uint nIndex,void* pBuf);
        public delegate int WTPacketDelegate(HCTX hCTX,uint wSerial,void* pBuf);
        public delegate int WTEnableDelegate(HCTX hCTX,BOOL fEnable);
        public delegate int WTOverlapDelegate(HCTX hCTX,BOOL fEnable);
        public delegate int WTConfigDelegate(HCTX hCTX,IntPtr hWnd);

        static public WTOpenDelegate WTOpen;
        static public WTCloseDelegate WTClose;
        static public WTInfoDelegate WTInfo;
        static public WTPacketDelegate WTPacket;
        static public WTEnableDelegate WTEnable;
        static public WTOverlapDelegate WTOverlap;
        static public WTConfigDelegate WTConfig;

        static public bool LoadWinTab() {
            IntPtr  hWinTab = LoadLibrary("WinTab32.dll");
            if (hWinTab==IntPtr.Zero) { return false; }

            try {
                IntPtr  fpWTOpen	= GetProcAddress(hWinTab,"WTOpenW");
			    IntPtr  fpWTClose	= GetProcAddress(hWinTab,"WTClose");
			    IntPtr  fpWTInfo    = GetProcAddress(hWinTab,"WTInfoW");
			    IntPtr  fpWTPacket	= GetProcAddress(hWinTab,"WTPacket");
			    IntPtr  fpWTEnable	= GetProcAddress(hWinTab,"WTEnable");
			    IntPtr  fpWTOverlap	= GetProcAddress(hWinTab,"WTOverlap");
			    IntPtr  fpWTConfig	= GetProcAddress(hWinTab,"WTConfig");

                WinTabInvoker.WTOpen = (WTOpenDelegate)Marshal.GetDelegateForFunctionPointer(fpWTOpen,typeof(WTOpenDelegate));
                WinTabInvoker.WTClose = (WTCloseDelegate)Marshal.GetDelegateForFunctionPointer(fpWTClose,typeof(WTCloseDelegate));
                WinTabInvoker.WTInfo = (WTInfoDelegate)Marshal.GetDelegateForFunctionPointer(fpWTInfo,typeof(WTInfoDelegate));
                WinTabInvoker.WTPacket = (WTPacketDelegate)Marshal.GetDelegateForFunctionPointer(fpWTPacket,typeof(WTPacketDelegate));
                WinTabInvoker.WTEnable = (WTEnableDelegate)Marshal.GetDelegateForFunctionPointer(fpWTEnable,typeof(WTEnableDelegate));
                WinTabInvoker.WTOverlap = (WTOverlapDelegate)Marshal.GetDelegateForFunctionPointer(fpWTOverlap,typeof(WTOverlapDelegate));
                WinTabInvoker.WTConfig = (WTConfigDelegate)Marshal.GetDelegateForFunctionPointer(fpWTConfig,typeof(WTConfigDelegate));
            } catch (Exception) { return false; }

            m_Valid = true;
            return true;
        }
    }
#endif

    static unsafe public partial class WinTab {
#if WINTAB_DYNAMIC
        static public bool LoadWinTab() {
            return WinTabInvoker.LoadWinTab();
        }

        /// <summary>
        /// ペンタブレットがインストールされている(WinTab32.dllが存在している)かどうか
        /// </summary>
        static public bool Valid {
            get { return WinTabInvoker.Valid; }
        }

        /// <summary>
        /// WinTab32.dllが読み込まれていなければ、例外を投げます。
        /// </summary>
        static public void Validate() {
            if (!Valid) { throw new WinTabException("WinTab32.dllが読み込まれる前に、関数が呼び出されました。"); }
        }

        static public HCTX _WTOpen(IntPtr hWnd,/*LOGCONTEXT*/void* lpLogCtx,BOOL fEnable) {
            Validate();
            return WinTabInvoker.WTOpen(hWnd,lpLogCtx,fEnable);
        }

        static public int _WTClose(HCTX hCTX) {
            Validate();
            return WinTabInvoker.WTClose(hCTX);
        }

        static public uint _WTInfo(uint nCategory,uint nIndex,void* pBuf) {
            Validate();
            return WinTabInvoker.WTInfo((uint)nCategory,nIndex,pBuf);
        }

        static public int _WTPacket(HCTX hCTX,uint wSerial,void* pBuf) {
            Validate();
            return WinTabInvoker.WTPacket(hCTX,wSerial,pBuf);
        }

        static public int _WTEnable(HCTX hCTX,BOOL fEnable) {
            Validate();
            return WinTabInvoker.WTEnable(hCTX,fEnable);
        }

        static public int _WTOverlap(HCTX hCTX,BOOL fEnable) {
            Validate();
            return WinTabInvoker.WTOverlap(hCTX,fEnable);
        }

        static public int _WTConfig(HCTX hCTX,IntPtr hWnd) {
            Validate();
            return WinTabInvoker.WTConfig(hCTX,hWnd);
        }
#else
        static public bool LoadWinTab() {
            return true;
        }

        [DllImport("WinTab32.dll",EntryPoint="WTOpenW")]
        static public extern IntPtr _WTOpen(IntPtr hWnd,/*LOGCONTEXT*/void* lpLogCtx,BOOL fEnable);
        [DllImport("WinTab32.dll",EntryPoint="WTClose")]
        static public extern int _WTClose(HCTX hCTX);
        [DllImport("WinTab32.dll",EntryPoint="WTInfoW")]
        static public extern uint _WTInfo(uint nCategory,uint nIndex,void* pBuf);
        [DllImport("WinTab32.dll",EntryPoint="WTPacket")]
        static public extern int _WTPacket(HCTX hCTX,uint wSerial,void* pBuf);
        [DllImport("WinTab32.dll",EntryPoint="WTEnable")]
        static public extern int _WTEnable(HCTX hCTX,BOOL fEnable);
        [DllImport("WinTab32.dll",EntryPoint="WTOverlap")]
        static public extern int _WTOverlap(HCTX hCTX,BOOL fEnable);
        [DllImport("WinTab32.dll",EntryPoint="WTConfig")]
        static public extern int _WTConfig(HCTX hCTX,IntPtr hWnd);
#endif
        /// <summary>
        /// 新しいコンテキストを開きます。
        /// </summary>
        /// <param name="hWnd">メッセージを受信するハンドル</param>
        /// <param name="lpLogCtx">コンテキストのパラメータをあらわす LOGCONTEXT 構造体へのポインタ</param>
        /// <param name="fEnable">コンテキストを初期化時から有効にするか</param>
        /// <exception cref="WinTabDotnet.WinTabException"></exception>
        /// <returns>コンテキストのハンドルが返ります。</returns>
        static public HCTX WTOpen(IntPtr hWnd,/*LOGCONTEXT*/void* lpLogCtx,BOOL fEnable) {
            HCTX  ret = _WTOpen(hWnd,lpLogCtx,fEnable);
            if (ret==IntPtr.Zero) { throw new WinTabException("WTOpen failed."); }
            return ret;
        }

        /// <summary>
        /// 存在するコンテキストを閉じます。
        /// </summary>
        /// <param name="hCTX">閉じるコンテキスト</param>
        /// <exception cref="WinTabDotnet.WinTabException"></exception>
        static public void WTClose(HCTX hCTX) {
            if (_WTClose(hCTX)==0) { throw new WinTabException("WTClose failed."); }
        }
        
        /// <summary>
        /// ペンタブレットに関する情報を得ます。
        /// </summary>
        /// <param name="nCategory">取得するカテゴリ</param>
        /// <param name="nIndex">取得するインデックス (WinTab.InfoIndex.INTERFACE,STATUS...)</param>
        /// <param name="pBuf">情報を代入するバッファ</param>
        /// <returns>バッファへ書き込んだ量を返します。</returns>
        /// <exception cref="WinTabDotnet.WinTabException"></exception>
        static public uint WTInfo(InfoCategory nCategory,uint nIndex,void* pBuf) {
            uint  ret = _WTInfo((uint)nCategory,nIndex,pBuf);
            if (ret==0) { throw new WinTabException("WTInfo failed."); }
            return ret;
        }

        /// <summary>
        /// パケットが到着している場合、パケットを取得します。
        /// </summary>
        /// <param name="hCTX">Context which packet arrived (WParam)</param>
        /// <param name="wSerial">Tablet event serial number (LParam)</param>
        /// <param name="pBuf">Buffer which returns packet values.</param>
        /// <exception cref="WinTabDotnet.WinTabException"></exception>
        static public void WTPacket(HCTX hCTX,uint wSerial,void* pBuf) {
            if (_WTPacket(hCTX,wSerial,pBuf)==0) { throw new WinTabException("WTPacket failed. (Packet queue is empty?)"); }
        }

        /// <summary>
        /// Enables or disables a tablet context.
        /// </summary>
        /// <param name="hCTX">Context which to be enabled or disabled</param>
        /// <param name="fEnable">Enable/disable flag</param>
        /// <exception cref="WinTabDotnet.WinTabException"></exception>
        static public void WTEnable(HCTX hCTX,BOOL fEnable) {
            if (_WTEnable(hCTX,fEnable)==0) { throw new WinTabException("WTEnable failed."); }
        }

        /// <summary>
        /// WTOverlap.
        /// </summary>
        /// <param name="hCTX"></param>
        /// <param name="fEnable"></param>
        static public void WTOverlap(HCTX hCTX,BOOL fEnable) {
            if (_WTOverlap(hCTX,fEnable)==0) { throw new WinTabException("WTOverlap failed."); }
        }

        /// <summary>
        /// Show context configuration dialogbox if device supports.
        /// </summary>
        /// <param name="hCTX">Context handle</param>
        /// <param name="hWnd">Dialogbox parent window</param>
        /// <returns>Returns true if the tablet context was changed, false otherwise.</returns>
        static public bool WTConfig(HCTX hCTX,IntPtr hWnd) {
            return _WTConfig(hCTX,hWnd)!=0;
        }

        [DllImport("User32.dll")]
        static public extern int GetSystemMetrics(int nIndex);
    };

    // --------------------------------------------------------------------------
    // Packet

    [StructLayout(LayoutKind.Sequential)]
    public struct ALLPACKET {
        public HCTX     pkContext;
        public uint     pkStatus;
        public int      pkTime;
        public WTPKT    pkChanged;
        public uint     pkSerialNumber;
        public uint     pkCursor;
        public uint     pkButtons;
        public int      pkX,pkY,pkZ;
        public int      pkNormalPressure;
        public int      pkTangendPressure;
    };

    // --------------------------------------------------------------------------
    // Main

    static public partial class WinTab {
        // --------------------------------------------------------------------------
        // Auxiliary functions
        static private unsafe uint GetInfoUInt(InfoCategory cat,uint idx) {
            uint  p;
            WTInfo(cat,idx,&p);
            return p;
        }

        static private unsafe uint[] GetInfoUInts(InfoCategory cat,uint idx,uint count) {
            uint[] r = new uint[count];
            fixed (uint* p = r) {
                WTInfo(cat,idx,p);
            }
            return r;
        }

        static private unsafe ushort GetInfoWord(InfoCategory cat,uint idx) {
            ushort p;
            WTInfo(cat,idx,&p);
            return p;
        }

        static private unsafe byte GetInfoByte(InfoCategory cat,uint idx) {
            byte p;
            WTInfo(cat,idx,&p);
            return p;
        }

        static private unsafe byte[] GetInfoBytes(InfoCategory cat,uint idx,uint bytes) {
            byte[] r = new byte[bytes];
            fixed (byte* p = r) {
                WTInfo(cat,idx,p);
            }
            return r;
        }

        static private unsafe string GetInfoString(InfoCategory cat,uint idx) {
            char* buf = stackalloc char[128];
            WTInfo(cat,idx,buf);
            return new string(buf);
        }

        static private unsafe string[] GetInfoStrings(InfoCategory cat,uint idx) {
            char[] buf = new char[256];
            char[] cbuf = new char[256];
            ArrayList  r = new ArrayList();
            
            fixed (char* p = buf) {
                uint f = WTInfo(cat,idx,p);
                int lp=0;
                for (int i=0;i<buf.Length-1;i++) {
                    if (p[i]=='\0') {
                        cbuf.Initialize();
                        Array.Copy(buf,lp,cbuf,0,i-lp);
                        r.Insert(0,new string(cbuf));
                        lp=0;

                        if (p[i+1]=='\0') { break; }
                    } else { lp++; }
                }
            }
            return (string[])r.ToArray(typeof(string[]));
        }

        static private unsafe AXIS GetInfoAxis(InfoCategory cat,uint idx) {
            AXIS  p;
            WTInfo(cat,idx,&p);
            return p;
        }

        static private unsafe AXIS[] GetInfoAxisA(InfoCategory cat,uint idx,int c) {
            AXIS[]  buf = new AXIS[3];
            fixed (AXIS* p = buf) { WTInfo(cat,idx,p); }
            return buf;
        }

        // --------------------------------------------------------------------------
        // Interface Properties

        /// <summary>
        /// タブレットハードウェアを識別する文字列。
        /// Tablet hardware identification string.
        /// </summary>
        static public string WinTabID {
            get { return GetInfoString(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.WINTABID); }
        }

        /// <summary>
        /// 詳細バージョン情報。上位バイト：メジャーバージョン、下位バイト：マイナーバージョン。
        /// Specification version. High/low-order byte contains the major/minor version.
        /// </summary>
        static public ushort SpecVersion {
            get { return GetInfoWord(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.SPECVERSION); }
        }

        /// <summary>
        /// 内部バージョン情報。上位バイト：メジャーバージョン、下位バイト：マイナーバージョン。
        /// Implementation version. High/low-order byte contains the major/minor version.
        /// </summary>
        static public ushort ImplVersion {
            get { return GetInfoWord(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.IMPLVERSION); }
        }

        /// <summary>
        /// サポートされているデバイス数。
        /// Number of devices supported.
        /// </summary>
        static public uint SupportedDevices {
            get { return GetInfoUInt(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.NDEVICES); }
        }

        /// <summary>
        /// サポートされているカーソル形状(ペン、消しゴム等)の数。
        /// Total number of cursor types supported.
        /// </summary>
        static public uint SupportedCursors {
            get { return GetInfoUInt(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.NCURSORS); }
        }

        /// <summary>
        /// サポートされているコンテクストの数。
        /// Number of contexts supported.
        /// </summary>
        static public uint SupportedContexts {
            get { return GetInfoUInt(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.NCONTEXTS); }
        }

        /// <summary>
        /// サポートされたコンテクストオプションによって示されたフラグ。
        /// Flags indicating which context options are supported.
        /// </summary>
        static public uint ContextOptions {
            get { return GetInfoUInt(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.CTXOPTIONS); }
        }

        /// <summary>
        /// 保存される情報のサイズ。
        /// Size of the save information.
        /// </summary>
        static public uint ContextSaveSize {
            get { return GetInfoUInt(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.CTXSAVESIZE); }
        }

        /// <summary>
        /// サポートされた拡張データアイテムの数。
        /// Number of extension data items supported.
        /// </summary>
        static public uint Extensions {
            get { return GetInfoUInt(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.NEXTENSIONS); }
        }

        /// <summary>
        /// サポートされたマネージャーハンドルの数。
        /// Number of manager handles supported.
        /// </summary>
        static public uint Managers {
            get { return GetInfoUInt(InfoCategory.INTERFACE,(uint)InfoIndex.INTERFACE.NMANAGERS); }
        }

        // --------------------------------------------------------------------------
        // Device Properties

        /// <summary>
        /// 表示可能なデバイス、製造主、リビジョンラベルを表す文字列。
        /// Displayable string describing the device, manufacturer, and revision level.
        /// </summary>
        static public string DeviceName {
            get { return GetInfoString(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.NAME); }
        }

        /// <summary>
        /// ハードウェアとデバイス能力を表現するフラグ。
        /// Hardware and driver capabilities.
        /// </summary>
        static public HardwareCapabilities DeviceCapabilities {
            get { return (HardwareCapabilities)GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.HARDWARE); }
        }

        /// <summary>
        /// サポートされているカーソル形状(ペン、消しゴム等)の数。
        /// Number of supported cursor types.
        /// </summary>
        static public uint DeviceCursorTypes {
            get { return GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.NCSRTYPES); }
        }

        /// <summary>
        /// 初期カーソル形状。
        /// First cursor type number for the device.
        /// </summary>
        static public uint DeviceFirstCursor {
            get { return GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.FIRSTCSR); }
        }

        /// <summary>
        /// パケットレポートレートの最大値(ヘルツ単位)。
        /// Maximum packet report rate in Hertz.
        /// </summary>
        static public uint DevicePacketRate {
            get { return GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.PKTRATE); }
        }

        /// <summary>
        /// パケットデータアイテムが常に表示状態であることを示すビットマスク。
        /// Bit mask indicating which packet data items are always available.
        /// </summary>
        static public WTPKT DevicePacketData {
            get { return (WTPKT)GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.PKTDATA); }
        }

        /// <summary>
        /// パケットデータアイテムが物理的に相対値であることを示すビットマスク。
        /// Bit mask indicating which packet data items are physically relative.
        /// </summary>
        static public WTPKT DevicePacketMode {
            get { return (WTPKT)GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.PKTMODE); }
        }

        /// <summary>
        /// 確実にカーソルが接続されている場合のみ表示されているパケットデータアイテムのビットマスク.
        /// Bit mask indicating which packet data items are only available when certain cursors are connected.
        /// </summary>
        static public WTPKT DeviceCursorData {
            get { return (WTPKT)GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.CSRDATA); }
        }

        /// <summary>
        /// タブレットのネイティブ座標における、タブレットコンテクストのマージンサイズ。
        /// Size of tablet context margins in tablet native coordinates.
        /// </summary>
        static public uint DeviceXMargin {
            get { return GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.XMARGIN); }
        }

        /// <summary>
        /// タブレットのネイティブ座標における、タブレットコンテクストのマージンサイズ。
        /// Size of tablet context margins in tablet native coordinates.
        /// </summary>
        static public uint DeviceYMargin {
            get { return GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.YMARGIN); }
        }

        /// <summary>
        /// タブレットのネイティブ座標における、タブレットコンテクストのマージンサイズ。
        /// Size of tablet context margins in tablet native coordinates.
        /// </summary>
        static public uint DeviceZMargin {
            get { return GetInfoUInt(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.ZMARGIN); }
        }

        /// <summary>
        /// タブレットの範囲及び分解能。
        /// Tablet's range and resolution capabilities.
        /// </summary>
        static public AXIS DeviceX {
            get { return GetInfoAxis(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.X); }
        }

        /// <summary>
        /// タブレットの範囲及び分解能。
        /// Tablet's range and resolution capabilities.
        /// </summary>
        static public AXIS DeviceY {
            get { return GetInfoAxis(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.Y); }
        }

        /// <summary>
        /// タブレットの範囲及び分解能。
        /// Tablet's range and resolution capabilities.
        /// </summary>
        static public AXIS DeviceZ {
            get { return GetInfoAxis(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.Z); }
        }

        /// <summary>
        /// 通常の場合のタブレット筆圧の範囲と分解能。
        /// Tablet's range and resolution capabilities, for the normal pressure inputs.
        /// </summary>
        static public AXIS DeviceNPressure {
            get { return GetInfoAxis(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.NPRESSURE); }
        }

        /// <summary>
        /// 傾きを持った場合のタブレット筆圧の範囲と分解能。
        /// Tablet's range and resolution capabilities, for the tangential pressure inputs.
        /// </summary>
        static public AXIS DeviceTPressure {
            get { return GetInfoAxis(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.TPRESSURE); }
        }

        /// <summary>
        /// タブレットの適応範囲と分解能。AXISは3つの配列。
        /// 3-element array describing the tablet's orientation range and resolution capabilities.
        /// </summary>
        static public AXIS[] DeviceOrientation {
            get { return GetInfoAxisA(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.ORIENTATION,3); }
        }

        /// <summary>
        /// タブレットの角度範囲と分解能。AXISは3つの配列。
        /// 3-element array describing the tablet's rotation range and resolution capabilities.
        /// </summary>
        static public AXIS[] DeviceRotation {
            get { return GetInfoAxisA(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.ROTATION,3); }
        }

        /// <summary>
        /// デバイスの Plug and Play ID。
        /// Device's Plug and Play ID.
        /// </summary>
        static public string DevicePnPID {
            get { return GetInfoString(InfoCategory.DEVICES,(uint)InfoIndex.DEVICE.PNPID); }
        }

        // --------------------------------------------------------------------------
        // Cursor Properties

        /// <summary>
        /// 表示可能なカーソル名の文字列。
        /// Displayable string containing the name of the cursor.
        /// </summary>
        static public string CursorName {
            get { return GetInfoString(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.NAME); }
        }

        /// <summary>
        /// カーソルが現在接続されているかどうか。
        /// Returns whether the cursor is currently connected.
        /// </summary>
        static public bool CursorIsActive {
            get { return GetInfoUInt(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.ACTIVE)!=0; }
        }

        /// <summary>
        /// カーソルが接続されているときにサポートされるパケットデータアイテムを示すビットマスク。
        /// Bit mask indicating the packet data items supported when this cursor is connected.
        /// </summary>
        static public WTPKT CursorPacketData {
            get { return (WTPKT)GetInfoUInt(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.PKTDATA); }
        }

        /// <summary>
        /// カーソル上に存在するボタンの数。
        /// Number of buttons on this cursor.
        /// </summary>
        static public byte CursorButtons {
            get { return GetInfoByte(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.BUTTONS); }
        }

        /// <summary>
        /// ハードウェアから返された未加工のボタンデータのビット数。
        /// Number of bits of raw button data returned by the hardware.
        /// </summary>
        static public byte CursorRawButton {
            get { return GetInfoByte(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.BUTTONBITS); }
        }

        /// <summary>
        /// カーソルが持つボタンの名前を含んだ文字列配列。
        /// String array containing the names of the cursor's buttons.
        /// </summary>
        static public string[] CursorButtonNames {
            get { return GetInfoStrings(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.BTNNAMES); }
        }

        // start from CSR_BUTTONMAP.... to CSR_CAPABILITIES

        /// <summary>
        /// 個々の物理的ボタンに割り振られた、ボタン番号を意味する32バイトの配列。
        /// 32 byte array of logical button numbers, one for each physical button.
        /// </summary>
        static public byte[] CursorButtonNumberMap {
            get { return GetInfoBytes(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.BUTTONMAP,32); }
        }

        /// <summary>
        /// 個々の物理的ボタンに割り振られた、ボタンアクションコードを意味する32バイトの配列。
        /// 32 byte array of logical button numbers, one for each physical button.
        /// </summary>
        static public byte[] CursorSystemActionCodeMap {
            get { return GetInfoBytes(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.SYSBTNMAP,32); }
        }

        /// <summary>
        /// 標準筆圧によってコントロールされるボタンの物理的ボタン番号。
        /// Returns the physical button number of the button that is controlled by normal pressure.
        /// </summary>
        static public byte CursorNPressureButton {
            get { return GetInfoByte(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.NPBUTTON); }
        }

        /// <summary>
        /// 標準筆圧におけるボタンマークを表す2つのuintの配列。1=リリースマーク / 2=プレスマーク
        /// Array of two uints, specifying the button marks for the normal pressure button. The first/second uint contains the release/press mark.
        /// </summary>
        static public uint[] CursorNPressureButtonMark {
            get { return GetInfoUInts(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.NPBUTTON,2); }
        }

        /// <summary>
        /// 標準筆圧における筆圧の応答曲線をあらわすuintの配列。
        /// Array of uints describing the pressure response curve for normal pressure.
        /// </summary>
        static public uint[] CursorNPressureResponce {
            get { return GetInfoUInts(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.NPRESPONSE,512); }
        }

        /// <summary>
        /// 接面筆圧によってコントロールされるボタンの物理的ボタン番号。
        /// Returns the physical button number of the button that is controlled by tangential pressure.
        /// </summary>
        static public byte CursorTPressureButton {
            get { return GetInfoByte(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.NPBUTTON); }
        }

        /// <summary>
        /// 接面筆圧におけるボタンマークを表す2つのuintの配列。1=リリースマーク / 2=プレスマーク
        /// Array of two uints, specifying the button marks for the tangential pressure button. The first/second uint contains the release/press mark.
        /// </summary>
        static public uint[] CursorTPressureButtonMark {
            get { return GetInfoUInts(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.NPBUTTON,2); }
        }

        /// <summary>
        /// 接面における筆圧の応答曲線をあらわすuintの配列。
        /// Array of uints describing the pressure response curve for tangential pressure.
        /// </summary>
        static public uint[] CursorTPressureResponce {
            get { return GetInfoUInts(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.NPRESPONSE,512); }
        }

        /// <summary>
        /// 製品の物理的なカーソルの識別子。
        /// Manufacturer-specific physical identifier for the cursor
        /// </summary>
        static public uint CursorPhysicalID {
            get { return GetInfoUInt(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.PHYSID); }
        }

        /// <summary>
        /// カーソルタイプにCursorCapabilities.MULTIMODEが含まれる場合、カーソルモードのカーソル形状番号。
        /// Cursor mode number of this cursor type, if this cursor type has the CursorCapabilities.MULTIMODE.
        /// </summary>
        static public uint CursorMode {
            get { return GetInfoUInt(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.MODE); }
        }

        /// <summary>
        /// カーソルタイプにCursorCapabilities.AGGREGATEが含まれる場合、物理的なカーソルのデータの最小セット。
        /// Minimum set of data available from a physical cursor in this cursor type, if this cursor type has the CursorCapabilities.AGGREGATE.
        /// </summary>
        static public uint CursorMinPacket {
            get { return GetInfoUInt(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.MINPKTDATA); }
        }

        /// <summary>
        /// カーソルタイプにCursorCapabilities.AGGREGATEが含まれる場合、物理的なカーソルのボタンの最小数量。
        /// Minimum number of buttons of physical cursors in the cursor type, if this cursor type has the CursorCapabilities.AGGREGATE.
        /// </summary>
        static public uint CursorMinButton {
            get { return GetInfoUInt(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.MINBUTTONS); }
        }

        /// <summary>
        /// カーソルの能力を示すフラグ。
        /// Flags indicating cursor capabilities.
        /// </summary>
        static public CursorCapabilities CursorCapabilities {
            get { return (CursorCapabilities)GetInfoUInt(InfoCategory.CURSORS,(uint)InfoIndex.CURSOR.CAPABILITIES); }
        }
    }

    // --------------------------------------------------------------------------
    // Delegates/Events
    public struct PacketEventArgs { public ALLPACKET pkts; }
    public delegate void NewPacketsHandler(PacketEventArgs e);
    public delegate void ButtonUpHandler(PacketEventArgs e);
    public delegate void ButtonDownHandler(PacketEventArgs e);
    public delegate void CursorMoveHandler(PacketEventArgs e);
    public delegate void NPressureChangeHandler(PacketEventArgs e);
    public delegate void TPressureChangeHandler(PacketEventArgs e);
    public delegate void CursorChangeHandler(PacketEventArgs e);
    public delegate void OrientationChangeHandler(PacketEventArgs e);
    public delegate void RotationChangeHandler(PacketEventArgs e);

    public class WinTabMessenger {
        /// <summary>新しいパケットが到着した際に発生</summary>
        public event NewPacketsHandler        NewPackets;
        /// <summary>ボタンが押下された際に発生</summary>
        public event ButtonDownHandler        ButtonDown;
        /// <summary>ボタンが上げられた際に発生</summary>
        public event ButtonUpHandler          ButtonUp;
        /// <summary>カーソル移動時に発生</summary>
        public event CursorMoveHandler        CursorMove;
        /// <summary>筆圧が変更された際に発生</summary>
        public event NPressureChangeHandler   NPressureChange;
        /// <summary>接面筆圧が変更された際に発生</summary>
        public event TPressureChangeHandler   TPressureChange;
        /// <summary>カーソルが変更された際に発生</summary>
        public event CursorChangeHandler      CursorChange;
        /// <summary>カーソルの方向が変更された際に発生</summary>
        public event OrientationChangeHandler OrientationChange;
        /// <summary>カーソルの角度が変更された際に発生</summary>
        public event RotationChangeHandler    RotationChange;

        /// <summary>
        /// これをWinTabContext.Open時に指定したウィンドウのWndProcで呼び出して下さい。
        /// </summary>
        /// <param name="m">呼ぶ出すWndProcのMessage</param>
        /// <returns>パケットが処理されたかどうかのbool値。</returns>
        public unsafe bool WndProc(ref Message m) {
            switch (m.Msg) {
                case WinTab.WT_PACKET: {
                    PacketEventArgs  e;
                    WinTab.WTPacket(m.LParam,(uint)m.WParam,&(e.pkts));

                    if (NewPackets!=null) { NewPackets(e); }

                    switch ((RelativeButton)e.pkts.pkButtons) {
                        case RelativeButton.UP: if (ButtonUp!=null) { ButtonUp(e); }  break;
                        case RelativeButton.DOWN: if (ButtonDown!=null) { ButtonDown(e); }  break;
                    }

                    if (((e.pkts.pkChanged & WTPKT.X) |
                         (e.pkts.pkChanged & WTPKT.Y) |
                         (e.pkts.pkChanged & WTPKT.Z))!=0)
                    { if (CursorMove!=null) { CursorMove(e); } }

                    if ((e.pkts.pkChanged & WTPKT.NORMAL_PRESSURE)!=0)
                    { if (NPressureChange!=null) { NPressureChange(e); } }
                    if ((e.pkts.pkChanged & WTPKT.TANGENT_PRESSURE)!=0)
                    { if (TPressureChange!=null) { TPressureChange(e); } }
                    if ((e.pkts.pkChanged & WTPKT.CURSOR)!=0)
                    { if (CursorChange!=null) { CursorChange(e); } }
                    if ((e.pkts.pkChanged & WTPKT.ORIENTATION)!=0)
                    { if (OrientationChange!=null) { OrientationChange(e); } }
                    if ((e.pkts.pkChanged & WTPKT.ROTATION)!=0)
                    { if (RotationChange!=null) { RotationChange(e); }  }

                    return true;
                }
                    // case WinTab
            }
            return false;
        }
    };

    /// <summary>
    /// 相対モード指定用。
    /// </summary>
    public enum RelativeField {
        None,
        Time,
        Position,
        Button,
        NPressure,
        TPressure
    };

    /// <summary>
    /// コンテキストラッパークラス。
    /// </summary>
    public class WinTabContext {
        private HCTX        m_hCTX;         // ＾＾
        private bool        m_bEnabled;     // ローカルで保持ー。
        private LOGCONTEXT  m_srLogContext;

        public WinTabContext()          { m_hCTX=IntPtr.Zero; }
        public WinTabContext(HCTX i)    { m_hCTX=i; }
               ~WinTabContext()         { if (IsValid) { Close(); } }
        public static explicit operator WinTabContext(HCTX x) { return new WinTabContext(x); }
        public static explicit operator HCTX(WinTabContext x) { return x.m_hCTX; }

        /// <summary>
        /// 指定したオプションで、コンテキストを開きます。
        /// </summary>
        /// <param name="hWnd">ウィンドウメッセージを受信するハンドル</param>
        /// <param name="bEnable">初期状態で有効にするか</param>
        /// <param name="xorg">出力エリアの基点：X座標</param>
        /// <param name="yorg">出力エリアの基点：Y座標</param>
        /// <param name="vx">出力エリアのXサイズ</param>
        /// <param name="vy">出力エリアのYサイズ</param>
        /// <param name="cxo">コンテキストのオプション</param>
        /// <param name="rf">相対モードにする項目</param>
        /// <returns></returns>
        public unsafe WinTabContext Open(IntPtr hWnd,bool bEnable,int xorg,int yorg,int vx,int vy,ContextOption cxo,RelativeField rf) {
            fixed (LOGCONTEXT* p = &m_srLogContext) {
                WinTab.WTInfo(InfoCategory.DEFSYSCTX,0,p);

                m_srLogContext.lcMsgBase    = WinTab.WT_DEFBASE;
                m_srLogContext.lcOptions   |= ContextOption.MESSAGES;
                m_srLogContext.lcPktData    = WinTab.WTPKT_ALL;
                m_srLogContext.lcPktMode    = 0;

                if (cxo!=ContextOption.DEFAULT) {
                    if ((cxo|ContextOption.OFFMODE)!=0) {
                        m_srLogContext.lcOptions   -= (cxo-ContextOption.OFFMODE);
                    } else {
                        m_srLogContext.lcOptions    = cxo;
                    }
                }
                m_srLogContext.lcOptions   |= ContextOption.MESSAGES | ContextOption.CSRMESSAGES;

                if ((rf & RelativeField.Button)!=0)    { m_srLogContext.lcPktMode |= WTPKT.BUTTONS; }
                if ((rf & RelativeField.Position)!=0)  { m_srLogContext.lcPktMode |= WTPKT.X | WTPKT.Y | WTPKT.Z; }
                if ((rf & RelativeField.Button)!=0)    { m_srLogContext.lcPktMode |= WTPKT.BUTTONS; }
                if ((rf & RelativeField.NPressure)!=0) { m_srLogContext.lcPktMode |= WTPKT.NORMAL_PRESSURE; }
                if ((rf & RelativeField.TPressure)!=0) { m_srLogContext.lcPktMode |= WTPKT.TANGENT_PRESSURE; }

                m_srLogContext.lcMoveMask   = WinTab.WTPKT_ALL;
                m_srLogContext.lcBtnUpMask  = m_srLogContext.lcBtnDnMask;

                m_srLogContext.lcOutOrgX    = m_srLogContext.lcOutOrgY = xorg;
                m_srLogContext.lcSysOrgX    = m_srLogContext.lcSysOrgY = yorg;
                m_srLogContext.lcOutExtX    = m_srLogContext.lcSysExtX = vx;
                m_srLogContext.lcOutExtY    = -vy;
                m_srLogContext.lcSysExtY    = vy;

                m_hCTX = WinTab.WTOpen(hWnd,p,Convert.ToInt32(bEnable));
            }
            m_bEnabled = bEnable;
            return this;
        }

        /// <summary>
        /// 指定したオプションで、コンテキストを開きます。
        /// </summary>
        /// <param name="hWnd">ウィンドウメッセージを受信するハンドル</param>
        /// <param name="bEnable">初期状態で有効にするか</param>
        /// <param name="cxo">コンテキストのオプション</param>
        /// <param name="rf">相対モードにする項目</param>
        public WinTabContext Open(IntPtr hWnd,bool bEnable,ContextOption cxo,RelativeField rf) {
            return Open(hWnd,bEnable,0,0,WinTab.GetSystemMetrics(78),WinTab.GetSystemMetrics(79),cxo,rf);
        }

        /// <summary>
        /// 指定したオプションで、コンテキストを開きます。
        /// </summary>
        /// <param name="hWnd">ウィンドウメッセージを受信するハンドル</param>
        /// <param name="bEnable">初期状態で有効にするか</param>
        /// <param name="rf">相対モードにする項目</param>
        public WinTabContext Open(IntPtr hWnd,bool bEnable,RelativeField rf) {
            return Open(hWnd,bEnable,0,0,WinTab.GetSystemMetrics(78),WinTab.GetSystemMetrics(79),ContextOption.DEFAULT,rf);
        }

        /// <summary>
        /// 指定したオプションで、コンテキストを開きます。
        /// </summary>
        /// <param name="hWnd">ウィンドウメッセージを受信するハンドル</param>
        /// <param name="bEnable">初期状態で有効にするか</param>
        public WinTabContext Open(IntPtr hWnd,bool bEnable) {
            return Open(hWnd,bEnable,0,0,WinTab.GetSystemMetrics(78),WinTab.GetSystemMetrics(79),ContextOption.DEFAULT,RelativeField.None);
        }

        /// <summary>
        /// コンテキストを閉じます。
        /// </summary>
        /// <exception cref="WinTabDotnet.WinTabException">コンテキストが初期化されていない場合に発生</exception>
        public void Close() {
            if (m_hCTX!=IntPtr.Zero) {
                WinTab.WTClose(m_hCTX);
                m_hCTX=IntPtr.Zero;
            } else {
                throw new WinTabException("Context is not initialized.");
            }
        }

        /// <summary>
        /// コンテキストの設定ダイアログを表示 (サポートされている場合)
        /// </summary>
        /// <param name="hWnd">親ウィンドウのハンドル</param>
        /// <returns>コンテキストが変更されたかのbool値 (ダイアログが表示できなくても、例外は発生しません)</returns>
        public bool Config(IntPtr hWnd) {
            return WinTab.WTConfig(m_hCTX,hWnd);
        }

        /// <summary>
        /// コンテキストの有効及び無効を取得または設定します。
        /// </summary>
        public bool Enabled {
            get { return m_bEnabled; }
            set {
                if (m_bEnabled==value) { return; }
                WinTab.WTEnable(m_hCTX,Convert.ToInt32(value));
            }
        }

        public void Overlap(bool bOverlap) {
            WinTab.WTOverlap(m_hCTX,Convert.ToInt16(bOverlap));
        }

        /// <summary>
        /// コンテキストが初期化されているかを取得します。
        /// </summary>
        public bool IsValid {
            get { return m_hCTX!=IntPtr.Zero; }
        }
    };
}