/*  NDKmol - Molecular Viewer on Android NDK

     (C) Copyright 2011, biochem_fan

     This file is part of NDKmol.

     NDKmol is free software: you can redistribute it and/or modify
     it under the terms of the GNU Lesser General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU Lesser General Public License for more details.

     You should have received a copy of the GNU Lesser General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>. */

package jp.sfjp.webglmol.NDKmol;

import java.io.File;
import java.io.FileWriter;
import java.io.InputStream;

import jp.xii.relog.customlibrary.app.FileListDialog;
import jp.xii.relog.customlibrary.app.FileListDialog.onFileListDialogListener;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.res.Resources;
import android.opengl.GLSurfaceView;
import android.os.Bundle;
import android.os.Environment;
import android.text.InputFilter;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.widget.EditText;

public class NDKmolActivity extends Activity {
	private float startX, startY, startDistance;
	private float currentX, currentY, currentZ;
	private float currentCameraZ;
	private boolean isDragging;
	private Quaternion currentQ;
	private int mode = 2;

	static {
		System.loadLibrary("Ndkmol");
	}

	public GLSurfaceView glSV;
	public NdkView view; 

	void initializeResource() {
		String targetPath = getDataDir() + "/sample.pdb";
		Log.d("NDKmol", "Initializing sample data " + targetPath);
		File target = new File(targetPath);
		if (target.exists()) return;

		try {
			FileWriter out= new FileWriter(target);
			out.write(readResource(R.raw.initial));
			out.close();
		} catch (Exception e) {
			Log.d("initializeResource", "failed: " + e.toString());
		}
	}

	String readResource(int resourceId) {
		String ret = ""; 

		Resources res = this.getResources();
		InputStream st = null;
		try {
			st = res.openRawResource(resourceId);
			byte[] buffer = new byte[st.available()];
			while((st.read(buffer)) != -1) {}
			st.close();
			ret = new String(buffer);
		} catch (Exception e) {
			Log.d("ResourceOpen", e.toString());
		} finally{
		}
		return ret;
	}

	/** Called when the activity is first created. */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		initializeResource();

		glSV = new GLSurfaceView(this);
		view = new NdkView();
		glSV.setRenderer(view);

		glSV.setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
		setContentView(glSV);
		view.setProtein(getDataDir() + "/sample.pdb");
		glSV.requestRender();
	}

	
	public void open() {
		FileListDialog dialog = new FileListDialog(this);
		dialog.setDirectorySelect(false);
		dialog.setOnFileListDialogListener(new onFileListDialogListener() {
			public void onClickFileList(File file) {
				if(file == null){
				}else{
					try {
						view.setProtein(file.getCanonicalPath());
						view.prepareScene(true);
						glSV.requestRender();
					} catch (Exception e) {
						Log.d("Open", "Failed to open" + file.getAbsolutePath());
					}
				}
			}
		});

		dialog.show(getDataDir(), "Load PDB file");
	}
	
	public String getDataDir() { // FIXME: not working?
		String dataDir = "";
		if (!Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED)) {
			dataDir = this.getFilesDir().getAbsolutePath();
		} else {
			File sdfolder = new File(Environment.getExternalStorageDirectory().getAbsoluteFile() + "/PDB");
			sdfolder.mkdirs();
			dataDir = sdfolder.getAbsolutePath();
		}
		return dataDir;
	}
	
	public void alert(String msg) {
		AlertDialog.Builder builder = new AlertDialog.Builder(this);
		builder.setMessage(msg)
		.setCancelable(false)
		.setPositiveButton("OK", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int id) {
			} 
		});
		AlertDialog alert = builder.create();
		alert.show();
	}
	
	public void download() {
		final EditText editView = new EditText(this);
		final NDKmolActivity parent = this;
		InputFilter[] inputFilter = new InputFilter[1];
		inputFilter[0] = new InputFilter.LengthFilter(4);
		editView.setFilters(inputFilter);
		new AlertDialog.Builder(this)
		.setIcon(android.R.drawable.ic_dialog_info)
		.setTitle("Input PDB ID to download")
		.setView(editView)
		.setPositiveButton("Download", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int whichButton) {
				String urlPrefix = "http://www.rcsb.org/pdb/files/";
				String urlSuffix = ".pdb";
				String id = editView.getText().toString().toUpperCase();
				if (id.matches("^[a-zA-Z0-9]{4}$")) {
					new Downloader(parent, urlPrefix + id + urlSuffix, getDataDir() + "/" +  id + urlSuffix);
				} else {
					alert("PDB ID is not correct.");
				}
			}
		})
		.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
			public void onClick(DialogInterface dialog, int whichButton) {
			}
		})
		.show();
	}


	public boolean onCreateOptionsMenu(Menu menu){
		super.onCreateOptionsMenu(menu);
		getMenuInflater().inflate(R.menu.mainmenu, menu);

		return true;
	}

	public boolean onPrepareOptionsMenu(Menu menu) {
		if (view != null) {
			menu.findItem(R.id.Sidechain).setChecked(view.showSidechain);
			menu.findItem(R.id.Cell).setChecked(view.showUnitcell);

			switch (view.proteinMode) {
			case 0:
				menu.findItem(R.id.Ribbon).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.CA_trace).setChecked(true);
			}

			switch (view.hetatmMode) {
			case 0:
				menu.findItem(R.id.Sphere).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.Stick).setChecked(true);
				break;
			case 2:
				menu.findItem(R.id.Line).setChecked(true);;
			}

			switch (view.symmetryMode) {
			case 0:
				menu.findItem(R.id.Monomer).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.Biomt).setChecked(true);
				break;
			case 2:
				menu.findItem(R.id.Packing).setChecked(true);
			}

			switch (view.colorMode) {
			case 0:
				menu.findItem(R.id.Chainbow).setChecked(true);
				break;
			case 1:
				menu.findItem(R.id.Chain).setChecked(true);
				break;
			case 2:
				menu.findItem(R.id.Structure).setChecked(true);
			}

		}

		return super.onPrepareOptionsMenu(menu);
	}

	public boolean onOptionsItemSelected(MenuItem item) {
		boolean ret = true;
		switch (item.getItemId()) {
		default:
			ret = super.onOptionsItemSelected(item);
			break;
		case R.id.move:
			mode = 0;
			break;
		case R.id.zoom:
			mode = 1;
			break;
		case R.id.rotate:
			mode = 2;
			break;
		case R.id.open:
			open();
			break;
		case R.id.downloadPDB:
			download();
			break;

		case R.id.CA_trace:
			view.proteinMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Ribbon:
			view.proteinMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Monomer:
			view.symmetryMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Biomt:
			view.symmetryMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Packing:
			view.symmetryMode = 2; 
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Chainbow:
			view.colorMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Chain:
			view.colorMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Structure:
			view.colorMode = 2;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Sphere:
			view.hetatmMode = 0;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Stick:
			view.hetatmMode = 1;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;
		case R.id.Line:
			view.hetatmMode = 2;
			item.setChecked(true);
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Sidechain:
			item.setChecked(!item.isChecked());
			view.showSidechain = !view.showSidechain;
			view.prepareScene(false);
			glSV.requestRender();
			break;

		case R.id.Cell:
			item.setChecked(!item.isChecked());
			view.showUnitcell = !view.showUnitcell;
			view.prepareScene(false);
			glSV.requestRender();
			break;
		}
		return ret;
	}

	public boolean onTouchEvent(MotionEvent e) {
		float x = e.getX();
		float y = e.getY();
		int pointerCount = 1;
		float distance = -1; 

		switch (e.getAction()) {
		case MotionEvent.ACTION_DOWN:
			//        	Log.d("event", "down");
			isDragging = true;
			view.isMoving = true;
			startX = x;
			startY = y;

			currentX = view.objX;
			currentY = view.objY;
			currentZ = view.objZ;
			currentCameraZ = view.cameraZ;
			currentQ = view.rotationQ;
			startDistance = distance;

			break;
		case MotionEvent.ACTION_UP:
			view.isMoving = false;
			isDragging = false;
			glSV.requestRender();
			break; 
		case MotionEvent.ACTION_MOVE:
			if (isDragging) {
				if (pointerCount > 1 || mode == 0) { // translation
					Vector3 vec = new Vector3(- (startX - x) * 0.15f, (startY - y) * 0.15f, 0);
					Vector3 translation = view.rotationQ.rotateVector(vec);
					view.objX = currentX + translation.x;
					view.objY = currentY + translation.y;
					view.objZ = currentZ + translation.z;
				}
				if (mode == 1) { // zoom
					view.cameraZ = currentCameraZ + (startY - y) * 0.5f;
				}
				if (pointerCount > 1 && startDistance > 50) {
					//				view.cameraZ = currentCameraZ * distance / startDistance;
				}
				if (mode == 2) { // rotate
					float dx = (x - startX) / (float)view.width, dy = (y - startY) / (float)view.height;
					float r = (float)Math.sqrt(dx * dx + dy * dy);
					if (r == 0) return true;

					float rs = (float)Math.sin(r * Math.PI) / r;
					Quaternion dq = new Quaternion(rs * dy, rs * dx, 0,  (float)Math.cos(r * Math.PI)); 
					view.rotationQ = Quaternion.multiply(dq, currentQ);
				}
				glSV.requestRender();
			}
			break;
		}

		return true;
	}
}