// --------------------------------------------------------------------
// wm3d - A Flash Molecular Viewer
//
// Copyright (c) 2011-2013, tamanegi (tamanegi@users.sourceforge.jp)
// All rights reserved.
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.
// --------------------------------------------------------------------

import flash.display.Stage;

import flash.text.TextField;
import flash.text.TextFieldAutoSize;
import flash.text.TextFormat;
import flash.text.TextFormatAlign;

/**
  Display Text or Image
**/

class WMDisplayStatus {
  // parent Watermelon
  private var wm:Watermelon;
  private var stage:Stage;

  // private text fields
  private var tfCenter:TextField;
  private var tfTopLeft:TextField;
  private var tfTopRight:TextField;
  private var tfBottomLeft:TextField;
  private var tfBottomRight:TextField;

  // #########################################################################
  /**
    Constructor.
  **/
  public function new( w:Watermelon ) {
    wm = w;
    stage = wm.stage;

    tfCenter = new TextField();
    tfCenter.selectable = false;
    tfCenter.autoSize = TextFieldAutoSize.CENTER;

    tfTopLeft = new TextField();
    tfTopLeft.selectable = false;
    tfTopLeft.autoSize = TextFieldAutoSize.LEFT;

    tfTopRight = new TextField();
    tfTopRight.selectable = false;
    tfTopRight.autoSize = TextFieldAutoSize.RIGHT;

    tfBottomLeft = new TextField();
    tfBottomLeft.selectable = false;
    tfBottomLeft.autoSize = TextFieldAutoSize.LEFT;

    tfBottomRight = new TextField();
    tfBottomRight.selectable = false;
    tfBottomRight.autoSize = TextFieldAutoSize.RIGHT;
  }

  private function getMyTF( tf:TextFormat):TextFormat {
    return( WMBase.clone( tf, Type.getInstanceFields( TextFormat ) ) );
  }

  /**
    set TextFormat for all textfields
  **/
  public function setTextFormat( tf:TextFormat ) {
    setTextFormatAll( tfCenter, getMyTF(tf), TextFormatAlign.CENTER );
    setTextFormatAll( tfTopLeft, getMyTF(tf), TextFormatAlign.LEFT );
    setTextFormatAll( tfTopRight, getMyTF(tf), TextFormatAlign.RIGHT );
    setTextFormatAll( tfBottomLeft, getMyTF(tf), TextFormatAlign.LEFT );
    setTextFormatAll( tfBottomRight, getMyTF(tf), TextFormatAlign.RIGHT );
  }

  // f might be a member of this class
  private function setTextFormatAll( f:TextField,
                                     tf:TextFormat,
                                     align:TextFormatAlign ) {
    tf.align = align;
    f.setTextFormat( tf );
    f.defaultTextFormat = tf;
  }

  // show message
  private function showMessage( tf:TextField,
                                message:String,
                                px:Float,
                                py:Float ) {
    if ( tf == null ) return;
    removeMessage( tf ); // delete first
    tf.text = message;
    stage.addChild( tf );
    tf.x = px;
    tf.y = py;
  }

  /**
    show given `message` on the center of the window
  **/
  public function showMessageCenter( message:String ) {
    tfCenter.text = message;
    var px:Float = stage.stageWidth / 2 - tfCenter.width / 2;
    var py:Float = stage.stageHeight / 2 - tfCenter.height;
    showMessage( tfCenter, message, px, py );
  }

  /**
    show given `message` on the top-left of the window
  **/
  public function showMessageTopLeft( message:String ) {
    showMessage( tfTopLeft, message, 0, 0 );
  }

  /**
    show given `message` on the top-right of the window
  **/
  public function showMessageTopRight( message:String ) {
    tfTopRight.text = message;
    var px:Float = stage.stageWidth - tfTopRight.width;
    var py:Float = 0;
    showMessage( tfTopRight, message, px, py );
  }

  /**
    show given `message` on the bottom-left of the window
  **/
  public function showMessageBottomLeft( message:String ) {
    tfBottomLeft.text = message;
    var px:Float = 0;
    var py:Float = stage.stageHeight - tfBottomLeft.height - wm.getControllerHeight();
    showMessage( tfBottomLeft, message, px, py );
  }

  /**
    show given `message` on the bottom-right of the window
  **/
  public function showMessageBottomRight( message:String ) {
    tfBottomRight.text = message;
    var px:Float = stage.stageWidth - tfTopRight.width;
    var py:Float = stage.stageHeight - tfBottomRight.height - wm.getControllerHeight();
    showMessage( tfBottomRight, message, px, py );
  }

  // remove message
  private function removeMessage( tf:TextField ) {
    if ( tf == null ) return;
    try {
      var i = stage.getChildIndex( tf );
    } catch( e:flash.errors.ArgumentError ) {
      // do nothing
      return;
    }
    stage.removeChild( tf );
    tf.text = ""; // remove only text
  }

  /**
    remove all messages
  **/
  public function removeAllMessages() {
    removeMessageCenter();
    removeMessageTopLeft();
    removeMessageTopRight();
    removeMessageBottomLeft();
    removeMessageBottomRight();
  }
  /**
    remove the message shown in the center of the window
  **/
  public function removeMessageCenter() { removeMessage( tfCenter ); }
  /**
    remove the message shown in the top-left of the window
  **/
  public function removeMessageTopLeft() { removeMessage( tfTopLeft ); }
  /**
    remove the message shown in the top-right of the window
  **/
  public function removeMessageTopRight() { removeMessage( tfTopRight ); }
  /**
    remove the message shown in the bottom-left of the window
  **/
  public function removeMessageBottomLeft() { removeMessage( tfBottomLeft ); }
  /**
    remove the message shown in the bottom-right of the window
  **/
  public function removeMessageBottomRight() { removeMessage( tfBottomRight ); }
}
