import xpath.XPath;
import xpath.xml.XPathXml;
import xpath.xml.XPathHxXml;

import flash.geom.Matrix3D;
import flash.geom.Vector3D;
import flash.display3D.Context3D;

import flash.xml.XML;
import flash.xml.XMLList;
import flash.xml.XMLParser;

class WMSystem {
  private var atoms( null, null ):Array< WMAtom >;
  private var bonds( null, null ):Array< WMBond >;
  private var chains( null, null ):Array< WMChain >;
  private var shapes( null, null ):Array< WMShape >;
  private var labels( null, null ):Array< WMLabel >;
  private var obj3ds( null, null ):Array< WMObject3D >;

  private var objects( null, null ):Array< Dynamic >;
  private var objnum( null, null ):Int;

  private var autoScale( null, null ):Bool;
  private var scaleFactor( null, null ):Float;
  private var scaleFactorManual( null, null ):Float;

  private var origin( null, null ):Point3D;

  public function new() {
    autoScale = true;
    scaleFactor = 0.3;
    scaleFactorManual = 10.0;
    origin = null;
  }

  public function clear():Void {
    atoms = [];
    bonds = [];
    chains = [];
    shapes = [];
    labels = [];
    obj3ds = [];
    prepareObjects();
    origin = null;
  }

  // something experimental
  private function prepareObjects():Void {
    objects = [];
    objects[0] = atoms;
    objects[1] = bonds;
    objects[2] = chains;
    objects[3] = shapes;
    objects[4] = labels;
    objects[5] = obj3ds;
    objnum = objects.length;
  }

  public function draw( c:Context3D,
                        mpos:Matrix3D,
                        proj:Matrix3D,
                        voffset:Vector3D,
                        light:Vector3D,
                        cpos:Vector3D ):Void {
    for ( i in 0 ... objnum ) {
      var num:Int = objects[i].length;
      for ( j in 0 ... num ) {
        objects[i][j].draw( c, mpos, proj, voffset, light, cpos );
      }
    }
  }

  public function gen( ?wm:Watermelon = null,
                       ?c:Context3D = null,
                       ?myxml:Xml = null ):Void {
    if ( wm == null || c == null ) return;
    loadXML( wm, myxml );
    initializeCrd();
    for ( i in 0 ... objnum ) {
      var num:Int = objects[i].length;
      for ( j in 0 ... num ) {
        objects[i][j].gen( c );
      }
    }
  }

  public function initializeCrd():Void {
    if ( empty() ) return;
    if ( origin == null ) origin = geometricCenter();
    translate( Point3D.getMultiply( origin, -1.0 ) );
    rescaleCoord();
  }

  public function geometricCenter():Point3D {
    var ret:Point3D = new Point3D( 0.0, 0.0, 0.0 );
    var totnum = 0;
    for ( i in 0 ... objnum ) {
      var num:Int = objects[i].length;
      for ( j in 0 ... num ) {
        totnum += objects[i][j].num();
        ret.add( objects[i][j].sumPos() );
      }
    }
    ret.multiply( 1.0 / totnum );
    return( ret );
  }

  public function translate( p:Point3D ):Void {
    for ( i in 0 ... objnum ) {
      var num:Int = objects[i].length;
      for ( j in 0 ... num ) {
        objects[i][j].translate( p );
      }
    }
  }

  public function rescaleCoord():Void {
    var swidth = flash.Lib.current.stage.stageWidth * scaleFactor;
    var sheight = flash.Lib.current.stage.stageHeight * scaleFactor;
    if ( !autoScale ) {
      scaleCoord( scaleFactorManual );
      return;
    }
    // origin is assumed to be processed
    var xmax:Float = 0.0;
    var ymax:Float = 0.0;
    for ( i in 0 ... objnum ) {
      var num:Int = objects[i].length;
      for ( j in 0 ... num ) {
        var pmax:Point3D = objects[i][j].absmax();
        xmax = Math.max( xmax, Math.max( pmax.x, pmax.z ) );
        ymax = Math.max( xmax, pmax.y );
      }
    }
    xmax = Math.max( 1.0, xmax );
    ymax = Math.max( 1.0, ymax );
    // scale coordinate
    scaleCoord( Math.min( swidth / xmax, sheight / ymax ) );
  }

  public function scaleCoord( scale:Float ):Void {
    for ( i in 0 ... objnum ) {
      var num:Int = objects[i].length;
      for ( j in 0 ... num ) {
        objects[i][j].scaleCoord( scale );
      }
    }
  }

  public function loadXML( ?wm:Watermelon = null,
                           ?myxml:Xml = null ):Void {
    clear();
    if ( wm == null || myxml == null ) return;

    // read settings
    var xml_setting:XPath = new XPath( "SETTING" );
    __loadXMLSettings( wm, xml_setting.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read chains and their elements
    var xml_chain:XPath = new XPath( "CHAIN" );
    __loadXMLChains( wm, xml_chain.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read labels and their elements
    var xml_label:XPath = new XPath( "LABEL" );
    __loadXMLLabels( wm, xml_label.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read shapes and their elements
    var xml_shape:XPath = new XPath( "SHAPE" );
    __loadXMLShapes( wm, xml_shape.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read 3d-objects
    var xml_obj3d:XPath = new XPath( "OBJ3D" );
    __loadXMLObj3Ds( wm, xml_obj3d.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read ATOMs
    var xml_atom:XPath = new XPath( "ATOM" );
    __loadXMLAtoms( wm, xml_atom.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read BONDs
    var xml_bond:XPath = new XPath( "BOND" );
    __loadXMLBonds( wm, xml_bond.selectNodes( XPathHxXml.wrapNode( myxml ) ) );
  }

  private function __parseSettingAttributes( wm:Watermelon,
                                             nd:Xml ):Void {
    if ( nd.exists( "light" ) ) { // <LIGHT> in former versions
      wm.setLightDirection( Point3D.fromString( nd.get( "light" ) ) );
    }
    if ( nd.exists( "arrate" ) ) { // <AR> in former versions
      wm.setAutoRotationRate( Std.parseFloat( nd.get( "arrate" ) ) );
    }
    if ( nd.exists( "framerate" ) ) { // <FRAMERATE> in former versions
      wm.setFrameRate( Std.parseInt( nd.get( "framerate" ) ) );
    }
    if ( nd.exists( "origin" ) ) {
      origin = Point3D.fromString( nd.get( "origin" ) );
    }
    var strs = [ "bg", "bgcolor", "background", "backgroundcolor" ];
    for ( str in strs ) {
      if ( nd.exists( str ) ) {
        wm.bgcolor = Std.parseInt( nd.get( str ) );
      }
    }
  }

  private function __loadXMLSettings( wm:Watermelon,
                                      ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      __parseSettingAttributes( wm, cast( nd, XPathHxXml ).getWrappedXml() );
      var ndio:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      for ( ndc in ndio.elements() ) {
        var nn:String = ndc.nodeName;
        switch( nn ) {
          case "ATOM":
            WMAtom.defaults.loadFromXml( ndc );
          case "BOND":
            WMBond.defaults.loadFromXml( ndc );
            var strs = [ "rounded", "round" ];
            for ( s in strs ) {
              if ( ndc.exists( s ) ) {
                WMBond.def_round = ( Std.parseInt( ndc.get( s ) ) > 0 );
              }
            }
            strs = [ "exc", "exclude" ];
            for ( s in strs ) {
              if ( ndc.exists( s ) ) {
                WMBond.def_exclude = ( Std.parseInt( ndc.get( s ) ) > 0 );
              }
            }
            strs = [ "dash", "dashed" ];
            for ( s in strs ) {
              if ( ndc.exists( s ) ) {
                WMBond.def_dashed = Std.parseInt( ndc.get( s ) );
              }
            }
          case "RIBBON":
            WMRibbon.defaultsRibbon.loadFromXml( ndc );
          case "COIL":
            WMRibbon.defaultsCoil.loadFromXml( ndc );
          case "SHAPE":
            WMShape.defaults.loadFromXml( ndc );
          case "OBJ3D":
            WMObject3D.defaults.loadFromXml( ndc );
            if ( ndc.exists( "type" ) ) WMObject3D.def_type = ndc.get( "type" );
          case "LABEL":
            WMLabel.defaults.loadFromXml( ndc );
            if ( ndc.exists( "font" ) ) WMLabel.def_font = ndc.get( "font" );
          case "CAMERA":
            if ( ndc.exists( "z" ) ) {
              wm.setCameraPosZ( Std.parseFloat( ndc.get( "z" ) ) );
            }
          case "PROTECT":
            wm.setProtectData();
          case "RADIUS":
            if ( ndc.exists( "method" ) ) {
              var met:String = ndc.get( "method" );
              if ( met.toLowerCase() == "absolute" ) {
                WMBase.useRelative = false;
              } else if ( met.toLowerCase() == "relative" ) {
                WMBase.useRelative = true;
              }
            }
            if ( ndc.exists( "scale" ) ) {
              WMBase.characteristicSize = Std.parseFloat( ndc.get( "scale" ) );
            }
          case "WHEEL":
            if ( ndc.exists( "scale" ) ) {
              wm.setScaleWheel( Std.parseFloat( ndc.get( "scale" ) ) );
            }
            if ( ndc.exists( "depth" ) ) {
              wm.setScaleWheel( Std.parseFloat( ndc.get( "depth" ) ) );
            }
          case "MATRIX":
            if ( ndc.exists( "data" ) ) {
              var m:Matrix4 = Matrix4.fromString( ndc.get( "data" ) );
              wm.setInitialView( m );
            }
          case "AUTOSCALE":
            if ( ndc.exists( "manual" ) ) {
              var n:Int = Std.parseInt( ndc.get( "manual" ) );
              if ( n > 0 ) autoScale = false;
            }
            if ( ndc.exists( "autoscale" ) ) {
              scaleFactor = Std.parseFloat( ndc.get( "autoscale" ) );
            }
            if ( ndc.exists( "manualscale" ) ) {
              scaleFactorManual = Std.parseFloat( ndc.get( "manualscale" ) );
            }
        }
      }
    }
  }

  private function __loadXMLChains( wm:Watermelon,
                                    ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ni:Int = 1;
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      if ( ndi.exists( "N" ) ) ni = Std.parseInt( ndi.get( "N" ) );
      // get child elements describing control points
      var children:Array< Dynamic > = new Array< Dynamic >();
      for ( child in ndi.elementsNamed( "POINT" ) ) {
        var anon = { pos:child.get( "pos" ), index:Std.parseInt( child.get( "index" ) ), dir:null, n:-1 };
        if ( child.exists( "N" ) ) anon.n = Std.parseInt( child.get( "N" ) );
        if ( child.exists( "dir" ) ) anon.dir = child.get( "dir" );
        children.push( anon );
      }
      // ignore too short chains, since catmull-rom interpolation is not
      // available for such short chains
      if ( children.length < 4 ) {
        trace( "__loadXMLChains: ignoring very short chain... " );
        trace( "__loadXMLChains: A <CHAIN> must contain at least 4 <POINT>s" );
        continue;
      }
      children.sort( __hasSmallIndex );
      var chain:WMChain = new WMChain();
      chain.setPositions( children, ni );
      // read ribbons and coils
      for ( child in ndi.elementsNamed( "RIBBON" ) ) {
        var rib:WMRibbon = new WMRibbon( true );
        rib.loadFromXml( child );
        chain.register( rib );
      }
      for ( child in ndi.elementsNamed( "COIL" ) ) {
        var coi:WMRibbon = new WMRibbon( false );
        coi.loadFromXml( child );
        chain.register( coi );
      }
      chains.push( chain );
    }
  }

  private function __hasSmallIndex( o0:Dynamic,
                                    o1:Dynamic ):Int {
    if ( o0.index == o1.index ) return(0);
    if ( o0.index < o1.index ) return(-1);
    return(1);
  }

  private function __loadXMLLabels( wm:Watermelon,
                                    ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var lb:WMLabel = new WMLabel();
      lb.loadFromXml( ndi );
      labels.push( lb );
    }
  }

  private function __loadXMLShapes( wm:Watermelon,
                                    ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var shape:WMShape = new WMShape();
      shape.loadFromXml( ndi );
      shapes.push( shape );
    }
  }

  private function __loadXMLObj3Ds( wm:Watermelon,
                                    ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var ob:WMObject3D = new WMObject3D();
      ob.loadFromXml( ndi );
      obj3ds.push( ob );
    }
  }

  private function __loadXMLAtoms( wm:Watermelon,
                                   ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var at:WMAtom = new WMAtom();
      at.loadFromXml( ndi );
      atoms.push( at );
    }
  }

  private function __loadXMLBonds( wm:Watermelon,
                                   ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var bd:WMBond = new WMBond();
      bd.loadFromXml( ndi );
      bonds.push( bd );
    }
  }

  public function empty():Bool {
    if ( numAtoms() + numBonds() + numChains() + numShapes() == 0 ) return( true );
    return( false );
  }
  public function numAtoms():Int { return( atoms.length ); }
  public function numBonds():Int { return( bonds.length ); }
  public function numChains():Int { return( chains.length ); }
  public function numShapes():Int { return( shapes.length ); }
}
