#ifndef __TAMANEGI_PARSE_EQUATION__
#define __TAMANEGI_PARSE_EQUATION__

#include <iostream>
#include <cmath>
#include <ctype.h>
#include <cstdlib>
#include <map>
#include <vector>
#include <string>
#include <fstream>
#include <sstream>
#include <algorithm>

//// Parse Equation class
// available operators: + - * / ! ^
// available function: "sqrt", "cos", "sin", "tan", "log", "ln"
//                     "exp", "abs"
// available special variables: pi(), kb() (boltzmann constant)
//
// variables: $(index) is replaced by correspnding values
//            example, $1, $11

namespace parseeq {
  enum { // operator & value types
    // operator types
    __NOT_OPERATOR__,       // ???, case when it should be operator but not
    __OPERATOR_EQUAL__,       // =
    __OPERATOR_PLUS__,        // +
    __OPERATOR_PLUS_EQUAL__,  // +=
    __OPERATOR_MINUS__,       // -
    __OPERATOR_MINUS_EQUAL__, // -=
    __OPERATOR_MULTIPLY__,    // *
    __OPERATOR_DIVIDE__,      // /
    __OPERATOR_POWER__,       // ^
    __OPERATOR_FACTORIAL__,   // !
    __OPERATOR_OPEN_PAR__,    // (
    __OPERATOR_CLOSE_PAR__,   // )
    // value types
    __VALTYPE_NONE__,
    __VALTYPE_STATIC__,
    __VALTYPE_STATIC_INT__,
    __VALTYPE_VARIABLE__,
    __VALTYPE_PROCESSED__
  };

  // synonym of pair
  template< typename T1, typename T2 > class eqContainer
                                       : public std::pair< T1, T2 > {
    private:
      bool f_scalar, f_vector;
    public:
      eqContainer() {
        clear();
      }
      eqContainer( const T1 &v1,
                   const T2 &v2,
                   const bool &isv = false ) : std::pair< T1, T2 >( v1, v2 ) {
        if ( isv ) {
          f_vector = true;
        } else {
          f_scalar = true;
        }
      }
      eqContainer( const T1 &d ) {
        this->setScalar( d );
      }
      eqContainer( const T2 &d ) {
        this->setVector( d );
      }
      eqContainer( const eqContainer &eq ) {
        clear();
        f_vector = eq.isVector();
        f_scalar = eq.isScalar();
        this->first = eq.first;
        this->second = eq.second;
      }
      void clear() {
        f_scalar = f_vector = false;
      }
      bool isScalar() const { return( f_scalar ); }
      bool isVector() const { return( f_vector ); }
      void setFlagScalar() {
        f_scalar = true;
        f_vector = false;
      }
      void setFlagVector() {
        f_scalar = false;
        f_vector = true;
      }
      bool checkType() const { return( f_scalar || f_vector ); }
      void setScalar( const T1 &d ) {
        this->first = d;
        setFlagScalar();
      }
      void setVector( const T2 &d ) {
        this->second = d;
        setFlagVector();
      }
      eqContainer< T1, T2 > operator + ( const eqContainer< T1, T2 > &eq ) const {
        if ( !checkType() || !eq.checkType() ) {
          std::cerr << "eqContainer()::operator+ - unknown type of container is given." << std::endl;
          exit(1);
        }
        eqContainer< T1, T2 > ret;
        if ( isScalar() && eq.isScalar() ) {        // S + S
          ret.setScalar( this->first + eq.first );
        } else if ( isVector() && eq.isVector() ) { // V + V
          ret.setVector( this->second + eq.second );
        } else if ( isVector() ) {                  // V + S
          ret.setVector( this->second + eq.first );
        } else {                                    // S + V
          ret.setVector( eq.second + this->first );
        }
        return( ret );
      }
      eqContainer< T1, T2 > operator - ( const eqContainer< T1, T2 > &eq ) const {
        if ( !checkType() || !eq.checkType() ) {
          std::cerr << "eqContainer()::operator- - unknown type of container is given." << std::endl;
          exit(1);
        }
        eqContainer< T1, T2 > ret;
        if ( isScalar() && eq.isScalar() ) {        // S - S
          ret.setScalar( this->first - eq.first );
        } else if ( isVector() && eq.isVector() ) { // V - V
          ret.setVector( this->second - eq.second );
        } else if ( isVector() ) {                  // V - S
          ret.setVector( this->second - eq.first );
        } else {                                    // S - V
          ret.setVector( eq.second - this->first );
        }
        return( ret );
      }
      eqContainer< T1, T2 > operator * ( const eqContainer< T1, T2 > &eq ) const {
        if ( !checkType() || !eq.checkType() ) {
          std::cerr << "eqContainer()::operator* - unknown type of container is given." << std::endl;
          exit(1);
        }
        eqContainer< T1, T2 > ret;
        if ( isScalar() && eq.isScalar() ) {        // S * S
          ret.setScalar( this->first * eq.first );
        } else if ( isVector() && eq.isVector() ) { // V * V
          ret.setScalar( this->second * eq.second );
        } else if ( isVector() ) {                  // V * S
          ret.setVector( this->second * eq.first );
        } else {                                    // S * V
          ret.setVector( eq.second * this->first );
        }
        return( ret );
      }
      eqContainer< T1, T2 > operator / ( const eqContainer< T1, T2 > &eq ) const {
        if ( !checkType() || !eq.checkType() ) {
          std::cerr << "eqContainer()::operator/ - unknown type of container is given." << std::endl;
          exit(1);
        }
        eqContainer< T1, T2 > ret;
        if ( isScalar() && eq.isScalar() ) {        // S / S
          ret.setScalar( this->first / eq.first );
        } else if ( isVector() && eq.isVector() ) { // V / V
          // i do not what to do this
          //ret.setVector();
          //ret.second = this->second / eq.second;
          std::cerr << "ParseEq::operator/() - / for vectors is not defined." << std::endl;
          exit(1);
        } else if ( isVector() ) {                  // V / S
          ret.setVector( this->second / eq.first );
        } else {                                    // S / V
          ret.setVector( eq.second / this->first );
        }
        return( ret );
      }
  };

  double callFunction( const std::string &funcname,
                       const double &value ) {
    if ( funcname == "sqrt" ) {
      return( sqrt( value ) );
    } else if ( funcname == "cos" ) {
      return( cos( value ) );
    } else if ( funcname == "sin" ) {
      return( sin( value ) );
    } else if ( funcname == "tan" ) {
      return( tan( value ) );
    } else if ( funcname == "log" ) {
      return( log10( value ) );
    } else if ( funcname == "ln" ) {
      return( log( value ) );
    } else if ( funcname == "exp" ) {
      return( exp( value ) );
    } else if ( funcname == "abs" ) {
      return( fabs( value ) );
    }
    // unknown function found
    std::cerr << "ParseEq::callFunction() - unknown function, " << funcname << ", is called." << std::endl;
    std::cerr << "ParseEq::callFunction() - available functions are: sqrt, cos, sin, tan, log, ln, exp, abs" << std::endl;
    exit(1);
  }

  // general function(s)
  int isOperator( const char &ch ) {
    if ( ch == '=' ) {        // equal
      return(__OPERATOR_EQUAL__);
    } else if ( ch == '+' ) { // plus
      return(__OPERATOR_PLUS__);
    } else if ( ch == '-' ) { // minus
      return(__OPERATOR_MINUS__);
    } else if ( ch == '*' ) { // multiply
      return(__OPERATOR_MULTIPLY__);
    } else if ( ch == '/' ) { // divide
      return(__OPERATOR_DIVIDE__);
    } else if ( ch == '^' ) { // power
      return(__OPERATOR_POWER__);
    } else if ( ch == '!' ) { // factorial
      return(__OPERATOR_FACTORIAL__);
    } else if ( ch == '(' ) { // open parenthesis
      return(__OPERATOR_OPEN_PAR__);
    } else if ( ch == ')' ) { // close parenthesis
      return(__OPERATOR_CLOSE_PAR__);
    }
    return(__NOT_OPERATOR__); // not opertator
  }

  int factorial( const int &to ) {
    int ret = 1;
    for ( int i = 2; i <= to; ++i ) {
      ret *= i;
    }
    return( ret );
  }

  int getMyOperatorIndex( const int &curpos,
                          const std::vector< int > &operators ) {
    int ret = curpos;
    while( operators[ret] == __NOT_OPERATOR__ &&
           ret < (int)operators.size() - 1 ) {
      ++ret;
    }
    return( ret );
  }

  template< typename T >
      eqContainer< double, T > calc( const std::vector< int > &operators,
                                     const std::vector< eqContainer< double, T > > &values ) {
    // copy given variables
    std::vector< eqContainer< double, T > > vals = values;
    std::vector< int > ops = operators;
    ops.push_back( __NOT_OPERATOR__ ); // similar purpose to null in C char*
    if ( ops.size() != vals.size() ) {
      std::cerr << "ParseEq::calc() - invalid number of values and operators." << std::endl;
      std::cerr << "ParseEq::calc() - please fix the program." << std::endl;
      std::cerr << "ParseEq::calc() - # of operators, values are " << ops.size() << ", " << vals.size() << std::endl;
      exit(1);
    }
    /*for ( int i = 0; i < (int)ops.size(); ++i ) {
      std::cerr << ' ' << vals[i].first << ' ' << vals[i].second << ' ' << ops[i];
    }
    std::cerr << std::endl;*/
    //// calculation order: ^,
    ////                    * and /,
    ////                    + and -
    ////                    factorial(!) is already processed
    // calculate power, '^'
    calcPower( vals, ops );
    // calculate * and /
    calcMultiplyDivide( vals, ops );
    // calculate + and -
    calcAddSubtract( vals, ops );
    return( vals[0] );
  }

  // calculate power, this and the following calc??? functions are part of calc
  template< typename T > 
      void calcPower( std::vector< eqContainer< double, T > > &values,
                      std::vector< int > &operators ) {
    int counter = (int)operators.size() - 2;
    while( counter >= 0 ) {
      // note for calculation order: 5^3^2 = 5^9
      int myoperator = getMyOperatorIndex( counter, operators );
      int target = myoperator + 1;
      if ( operators[myoperator] == __OPERATOR_POWER__ ) {
        values[counter].first = pow( values[counter].first, values[target].first );
        operators[myoperator] = __NOT_OPERATOR__;
      }
      counter--;
    }
  }

  template< typename T >
      void calcMultiplyDivide( std::vector< eqContainer< double, T > > &values,
                               std::vector< int > &operators ) {
    int counter = 0;
    while( counter < (int)operators.size() ) {
      while(1) {
        int myoperator = getMyOperatorIndex( counter, operators );
        int target = myoperator + 1;
        if ( operators[myoperator] == __OPERATOR_DIVIDE__ ) {
          if ( values[target].first == 0.0 ) {
            std::cerr << "ParseEq::calc() - division by 0 is not allowed" << std::endl;
            std::cerr << "ParseEq::calc() - error at " << myoperator << "th operator." << std::endl;
            exit(1);
          }
          values[counter] = values[counter] / values[target];
        } else if ( operators[myoperator] == __OPERATOR_MULTIPLY__ ) {
          values[counter] = values[counter] * values[target];
        } else {
          counter = myoperator;
          break;
        }
        operators[myoperator] = __NOT_OPERATOR__;
      }
      ++counter;
    }
  }

  template < typename T >
      void calcAddSubtract( std::vector< eqContainer< double, T > > &values,
                            std::vector< int > &operators ) {
    int counter = 0;
    while( counter < (int)operators.size() ) {
      while(1) {
        int myoperator = getMyOperatorIndex( counter, operators );
        int target = myoperator + 1;
        if ( operators[myoperator] == __OPERATOR_PLUS__ ) {
          values[counter] = values[counter] + values[target];
        } else if ( operators[myoperator] == __OPERATOR_MINUS__ ) {
          values[counter] = values[counter] - values[target];
        } else {
          counter = myoperator;
          break;
        }
        operators[myoperator] = __NOT_OPERATOR__;
      }
      ++counter;
    }
  }

  // frontend of equation parser
  template< typename T > class Equation {
    private:
      // data storage
      std::map< std::string, eqContainer< double*, T* > > store;

      // return value: first  -> type of value
      //               second -> value
      std::pair< int, eqContainer< double, T > > whatsThis( const std::string &str,
                                                            const std::vector< eqContainer< double, T > > &tmp_values = std::vector< eqContainer< double, T > >() ) {
        bool isVector = false;
        bool hasDecimal = false;
        double special;
        if ( ( special = isSpecialValue( str ) ) != 0.0 ) {
          return( std::make_pair( (int)__VALTYPE_STATIC__, eqContainer< double, T >( special ) ) );
        }
        std::string curstr;
        for ( int i = 0, j = 0; i < (int)str.size(); ++i ) {
          // ignore blank spaces
          if ( isblank( str[i] ) ) continue;
          if ( str[i] == '.' ) { // may be double value
            if ( hasDecimal ) {  // strange value having more than two '.'s
              std::cerr << "ParseEq::whatsThis() - given string, " << str
                        << ", has more than two decimal points" << std::endl;
              exit(1);
            }
            hasDecimal = true;
          } else if ( str[i] == '$' ) { // variable
            // only allowed format is $??? or V$???
            if ( j != 0 ) {
              std::cerr << "ParseEq::whatsThis() - invalid format of variable." << std::endl;
              std::cerr << "ParseEq::whatsThis() - variable must be $(number) such as $1, $10." << std::endl;
              exit(1);
            }
            if ( str.size() < i + 1 ) {
              std::cerr << "ParseEq::whatsThis() - string ended with \'$\' ????." << std::endl;
              exit(1);
            }
            eqContainer< double*, T* > var = getRefVariable( str.substr( i + 1 ) );
            eqContainer< double, T > ret;
            if ( var.isScalar() ) {
              ret.setScalar( *(var.first) );
            } else if ( var.isVector() ) {
              ret.setVector( *(var.second) );
            }
            return( std::make_pair( (int)__VALTYPE_VARIABLE__, ret ) );
          } else if ( str[i] == '@' ) { // it must be already processed
            int index = atoi( str.substr( i+1, str.size() - 2 ).c_str() );
            if ( j == 0 ) {
              if ( index >= tmp_values.size() ) {
                std::cerr << "ParseEq:: unexpected error found in the equation." << std::endl;
                std::cerr << "ParseEq:: do not use \'@\' in the equation." << std::endl;
                std::cerr << "ParseEq:: if not, this error is caused by a program internal error." << std::endl;
                exit(1);
              }
              return( std::make_pair( (int)__VALTYPE_PROCESSED__, tmp_values[index] ) );
            } else {
              // there maybe function name before @
              double hoge = tmp_values[index].first;
              return( std::make_pair( (int)__VALTYPE_PROCESSED__, eqContainer< double, T >( callFunction( curstr, hoge ) ) ) );
            }
          }
          curstr += str[i];
          ++j;
        }
        // the value may be a constant value
        int rettyp = __VALTYPE_STATIC__;
        // may be this is neccessary only for factorial
        if ( !hasDecimal ) rettyp = __VALTYPE_STATIC_INT__;
        return( std::make_pair( rettyp, eqContainer< double, T >( atof( str.c_str() ) ) ) );
      }

      // special values such as pi(), gas constant, boltzmann constant
      double isSpecialValue( const std::string &str ) const {
        if ( str == "KB" ) {
          return( 1.3806503e-23 ); // m^2 kg s^-2 K^-1
        } else if ( str == "RG" ) {
          return( 8.314472 ); // m^2 kg s^-2 K^-1 mol^-1
        } else if ( str == "NA" ) {
          return( 6.0221415e+23 );
        } else if ( str == "PI" ) {
          return( M_PI );
        }
        return( 0.0 );
      }

      //eqContainer< double, T > getValueFromString( const std::string &str,
      //                                             const std::vector< eqContainer< double, T > > &tmp_values ) {
      //  eqContainer< double, T > ret;
      //  // check for special variable
      //  double special;
      //  if ( ( special = isSpecialValue( str ) ) != 0.0 ) return( special );
      //  std::pair< int, eqContainer< double, T > > type = whatsThis( str, tmp_values );
      //  if ( type.first == __VALTYPE_STATIC__ ||
      //       type.first == __VALTYPE_STATIC_INT__ ) {
      //    ret.setScalar( atof( str.c_str() ) );
      //  } else if ( type.first == __VALTYPE_PROCESSED__ ) {
      //    if ( type.second == 0 ) {
      //      ret = tmp_values[atoi(str.substr(1,str.size()-2).c_str())];
      //    } else { // there may be function
      //      std::string funcname = str.substr( 0, type.second );
      //      double hoge = tmp_values[atoi(str.substr(type.second+1,str.size()-2).c_str())].first;
      //      ret.setScalar( callFucntion( funcname, hoge ) );
      //    }
      //  } else if ( type.first == __VALTYPE_VARIABLE__ ) {
      //    ret.setScalar( getVariable( type.second ) );
      //  } else if ( type.first == __VALTYPE_VECTOR_VARIABLE__ ) {
      //    ret.setVector( getVectorVariable( type.second ) );
      //  }
      //  return( ret );
      //}
    public:
      Equation() {}
      void clear() {
        store.empty();
      }
      bool empty() const { return( store.empty() ); }
      // U should not be pointer
      template< typename U >
          void registerValue( const std::string &name,
                              U &v ) {
        store.insert( std::make_pair( name, eqContainer< double*, T* >( &v ) ) );
      }
      eqContainer< double, T > eval( const std::string &origstring ) {
        std::string str = origstring;
        eqContainer< double, T > ret;
        std::vector< eqContainer< double, T > > tmp_values;
        std::vector< eqContainer< double, T > > values;
        std::vector< bool > isVector;
        std::vector< int > operators;
        while(1) {
          // find open and close parentheses in the equation
          int openpar = str.find_first_of( '(' );
          int closepar = str.find_last_of( ')' );
          if ( openpar != std::string::npos &&
               closepar != std::string::npos ) {
            // there may be ()
            if ( closepar - openpar < 1 ) {
              std::cerr << "ParseEq::eval() - parentheses mismatch" << std::endl;
              std::cerr << "ParseEq::eval() - debug: openpar, closepar = " << openpar << ", " << closepar << std::endl;
              exit(1);
            }
            tmp_values.push_back( eval( str.substr( openpar + 1, closepar - openpar - 1 ) ) );
            std::string replace_string = "@";
            std::stringstream sstr;
            sstr << (int)tmp_values.size() - 1;
            replace_string += sstr.str();
            replace_string += "@";
            str.replace( openpar, closepar - openpar + 1, replace_string );
          } else if ( openpar == std::string::npos ^
                      closepar == std::string::npos ) {
            // in case one of openpar and closepar is std::string::npos
            std::cerr << "ParseEq::eval() - parentheses mismatch" << std::endl;
            std::cerr << "ParseEq::eval() - debug: openpar, closepar = " << openpar << ", " << closepar << std::endl;
            exit(1);
          } else { // no parentheses found
            break;
          }
        }
        // if locked, what comes next must be operator
        // otherwise, end of equation
        bool isLocked = false;
        int curpos = 0;
        std::string curstr, prevstr;
        while( curpos < str.size() ) {
          if ( isblank( str[curpos] ) ) {
            ++curpos;
            continue;
          }
          // is this operator?
          int oper = isOperator( str[curpos] );
          if ( oper != __NOT_OPERATOR__ ) {
            if ( curstr.empty() ) { // suspicious situation
              if ( oper == __OPERATOR_EQUAL__ ) {
                if ( operators.empty() ) {
                  std::cerr << "ParseEq::eval() - invalid equation." << std::endl;
                  std::cerr << "ParseEq::eval() - equation has unparsable \'=\'." << std::endl;
                  exit(1);
                }
                if ( prevstr[0] != '$' ) {
                  std::cerr << "ParseEq::eval() - += can be applicable only to a variable." << std::endl;
                  exit(1);
                }
                eqContainer< double*, T* > var = getRefVariable( prevstr.substr(1) );
                eqContainer< double, T > var0, var1;
                eqContainer< double, T > var2 = eval( str.substr( curpos + 1 ) );
                if ( var.isScalar() ) var0.setScalar( *(var.first) );
                if ( var.isVector() ) var0.setVector( *(var.second) );
                if ( operators.back() == __OPERATOR_PLUS__ ) {         // +=
                  var1 = var0 + var2;
                } else if ( operators.back() == __OPERATOR_MINUS__ ) { // -=
                  var1 = var0 - var2;
                }
                if ( var1.isScalar() ) {
                  *(var.first) = var1.first;
                  var.setFlagScalar();
                } else if ( var1.isVector() ) {
                  *(var.second) = var1.second;
                  var.setFlagVector();
                }
                return( var1 );
              } else if ( oper == __OPERATOR_PLUS__ ) {
                // meaningless '+', unary operator
                ++curpos;
                continue;
              } else if ( oper == __OPERATOR_MINUS__ ) {
                // standard unary operator
                curstr += str[curpos];
              }
            } else if ( oper == __OPERATOR_FACTORIAL__ ) {
              int g = evalFactorial( curstr );
              std::stringstream sstr;
              sstr << g;
              curstr = sstr.str();
              isLocked = true;
            } else if ( oper == __OPERATOR_EQUAL__ ) {
              if ( curstr[0] != '$' ) {
                std::cerr << "ParseEq::eval() - += can be applicable only to a variable." << std::endl;
                exit(1);
              }
              eqContainer< double*, T* > var = getRefVariable( curstr.substr(1) );
              eqContainer< double, T > var0;
              eqContainer< double, T > var2 = eval( str.substr( curpos + 1 ) );
              if ( var.isScalar() ) var0.setScalar( *(var.first) );
              if ( var.isVector() ) var0.setVector( *(var.second) );
              if ( var2.isScalar() ) {
                *(var.first) = var2.first;
                var.setFlagScalar();
              } else if ( var2.isVector() ) {
                *(var.second) = var2.second;
                var.setFlagVector();
              }
              return( var2 );
            } else {
              if ( curstr[curstr.size()-1] == '-' ) {
                // strange situatio like 1 --- 3
                // note: 1 -- 3 will be accepted (understood as 1+3)
                std::cerr << "ParseEq::eval() - cannot understand equation." << std::endl;
                std::cerr << "ParseEq::eval() - current string is: " << curstr << std::endl;
                exit(1);
              }
              // add 0 to values ended with '.'
              if ( curstr[curstr.size()-1] == '.' ) curstr += '0';
              values.push_back( whatsThis( curstr, tmp_values ).second );
              operators.push_back( oper );
              // clear temporary variables
              prevstr = curstr;
              curstr.clear();
              isLocked = false;
            }
          } else { // not an operator
            if ( isLocked ) {
              // example, 3!3
              std::cerr << "ParseEq::eval() - unparsable string." << std::endl;
              exit(1);
            }
            curstr += str[curpos];
          }
          ++curpos;
        }
        if ( curstr.empty() ) {
          std::cerr << "ParseEq::eval() - equation ended with an operator????" << std::endl;
          exit(1);
        }
        values.push_back( whatsThis( curstr, tmp_values ).second );
        return( calc< T >( operators, values ) );
      }
      int evalFactorial( const std::string &str ) {
        // ! is assumed to be 1!
        std::string curstr = str;
        if ( curstr.empty() ) curstr = "1";
        std::pair< int, eqContainer< double, T > > type = whatsThis( curstr );
        int f = type.second.first;
        if ( type.first != __VALTYPE_VARIABLE__ &&
             type.first != __VALTYPE_STATIC_INT__ ) {
          std::cerr << "ParseEq::eval() - factorial is applicable only on integer value or variable." << std::endl;
          std::cerr << "ParseEq::eval() - given value, " << curstr << ", seems not to be integer value." << std::endl;
          exit(1);
        }
        return( factorial( f ) );
      }
      double* getPointerVariable( const int &index ) {
        //checkIndex( index );
        return( store[index].first );
      }
      eqContainer< double*, T* > getRefVariable( const std::string &str ) {
        typename std::map< std::string, eqContainer< double*, T* > >::iterator it = store.find( str );
        if ( it == store.end() ) {
          std::cerr << "ParseEq::getRefVariable() - variable named " << str << " not found." << std::endl;
          exit(1);
        }
        return( it->second );
      }
  };

  template< typename T1, typename T2 >
      std::ostream& operator << ( std::ostream &out,
                                  const eqContainer< T1, T2 > &d ) {
    if ( d.isScalar() ) {
      out << d.first;
    } else if ( d.isVector() ) {
      out << d.second;
    }
    return( out );
  }
};

#endif /* __TAMANEGI_PARSE_EQUATION__ */
