#include <iostream>
#include <cmath>
#include <vector>
#include <algorithm>
#include "general.hpp"
using namespace std;

Vector2D::Vector2D() {
  clear();
}

Vector2D::Vector2D( const double &rx,
                    const double &ry ) {
  x = rx;
  y = ry;
}

Vector2D::Vector2D( const Vector2D &v ) {
  x = v.x;
  y = v.y;
}

void Vector2D::clear() {
  x = y = 0.0;
}

Vector2D Vector2D::operator + ( const Vector2D &v ) const {
  Vector2D ret;

  ret.x = x + v.x;
  ret.y = y + v.y;

  return( ret );
}

Vector2D Vector2D::operator + ( const double &v ) const {
  Vector2D ret;

  ret.x = x + v;
  ret.y = y + v;

  return( ret );
}

Vector2D Vector2D::operator - ( const Vector2D &v ) const {
  Vector2D ret;

  ret.x = x - v.x;
  ret.y = y - v.y;

  return( ret );
}

Vector2D Vector2D::operator - ( const double &v ) const {
  Vector2D ret;

  ret.x = x - v;
  ret.y = y - v;

  return( ret );
}

Vector2D Vector2D::operator * ( const double &v ) const {
  Vector2D ret;

  ret.x = v * x;
  ret.y = v * y;

  return( ret );
}

double Vector2D::operator * ( const Vector2D &v ) const {
  return( x * v.x + y * v.y );
}

Vector2D Vector2D::operator += ( const Vector2D &v ) {
  x += v.x;
  y += v.y;

  return( *this );
}

Vector2D Vector2D::operator -= ( const Vector2D &v ) {
  x -= v.x;
  y -= v.y;

  return( *this );
}

Vector2D Vector2D::operator -= ( const double &v ) {
  x -= v;
  y -= v;

  return( *this );
}

Vector2D Vector2D::operator *= ( const double &v ) {
  x *= v;
  y *= v;

  return( *this );
}

Vector2D Vector2D::operator / ( const double &v ) const {
  Vector2D ret;

  ret.x = x / v;
  ret.y = y / v;

  return( ret );
}

ostream& operator << ( ostream &out,
                       const Vector2D &v ) {
  out << v.x << ' ' << v.y;
  return( out );
}

istream& operator >> ( istream &in,
                       Vector2D &v ) {
  in >> v.x >> v.y;
  return( in );
}

double Vector2D::volume() const {
  return( area() );
}

double Vector2D::sum() const {
  return( x + y );
}

double Vector2D::area() const {
  return( x * y );
}

double Vector2D::sqabs() const {
  return( x * x + y * y );
}

double Vector2D::sqabs( const Vector2D &box ) {
  unroll( box );
  return( sqabs() );
}

void Vector2D::normalize() {
  double div = sqrt( sqabs() );
  x /= div;
  y /= div;
}

double Vector2D::minabsval() const {
  return( min( abs(x), abs(y) ) );
}

double Vector2D::maxabsval() const {
  return( max( abs(x), abs(y) ) );
}

void Vector2D::unroll( const Vector2D &box ) {
  x -= box.x * rint( x / box.x );
  y -= box.y * rint( y / box.y );
}

void Vector2D::unrollf( const Vector2D &box ) {
  x -= box.x * floor( x / box.x );
  y -= box.y * floor( y / box.y );
}

cellIndex Vector2D::numCell( const double &cutoff,
                             const double &margin ) const {
  return( numCell( Vector2D( cutoff, cutoff ), margin) );
}

cellIndex Vector2D::numCell( const Vector2D &cutoff,
                             const double &margin ) const {
  cellIndex ret;

  ret.i = 0; // 2 次元なので
  ret.j = (int)( floor( x / cutoff.x ) );
  ret.k = (int)( floor( y / cutoff.y ) );

  if ( min( ret.j, ret.k ) == 0 ) return( cellIndex( 0, 0, 0 ) );

  // あまりに余裕がない場合はちょっとごまかす
  if ( ( x / (double)ret.j - cutoff.x ) < margin ) ret.j--;
  if ( ( y / (double)ret.k - cutoff.y ) < margin ) ret.k--;

  return( ret );
}

cellIndex Vector2D::whichCell( const Vector2D &box,
                               const cellIndex &syssize ) const {
  Vector2D tmp;
  cellIndex ret;

  tmp.x = x - box.x * floor( x / box.x );
  tmp.y = y - box.y * floor( y / box.y );

  ret.i = 0;
  ret.j = (int)( (double)syssize.j * tmp.x / box.x );
  ret.k = (int)( (double)syssize.k * tmp.y / box.y );

  return( ret.within( syssize ) );
}

void Vector2D::assignRand() {
  x = drand48();
  y = drand48();
}

bool Vector2D::is_zero() const {
  if ( x == 0.0 && y == 0.0 ) return( true );
  return( false );
}

//---------------------------------------------------

Vector3D::Vector3D() {
  clear();
}

Vector3D::Vector3D( const double &rx,
                    const double &ry,
                    const double &rz ) {
  x = rx;
  y = ry;
  z = rz;
}

Vector3D::Vector3D( const Vector3D &v ) {
  x = v.x;
  y = v.y;
  z = v.z;
}

void Vector3D::clear() {
  x = y = z = 0.0;
}

Vector3D Vector3D::operator + ( const Vector3D &v ) const {
  Vector3D ret;

  ret.x = x + v.x;
  ret.y = y + v.y;
  ret.z = z + v.z;

  return( ret );
}

Vector3D Vector3D::operator + ( const double &v ) const {
  Vector3D ret;

  ret.x = x + v;
  ret.y = y + v;
  ret.z = z + v;

  return( ret );
}

Vector3D Vector3D::operator - ( const Vector3D &v ) const {
  Vector3D ret;

  ret.x = x - v.x;
  ret.y = y - v.y;
  ret.z = z - v.z;

  return( ret );
}

Vector3D Vector3D::operator - ( const double &v ) const {
  Vector3D ret;

  ret.x = x - v;
  ret.y = y - v;
  ret.z = z - v;

  return( ret );
}

Vector3D Vector3D::operator * ( const double &v ) const {
  Vector3D ret;

  ret.x = v * x;
  ret.y = v * y;
  ret.z = v * z;

  return( ret );
}

double Vector3D::operator * ( const Vector3D &v ) const {
  return( dotProduct( v ) );
}

Vector3D Vector3D::operator += ( const Vector3D &v ) {
  x += v.x;
  y += v.y;
  z += v.z;

  return( *this );
}

Vector3D Vector3D::operator -= ( const Vector3D &v ) {
  x -= v.x;
  y -= v.y;
  z -= v.z;

  return( *this );
}

Vector3D Vector3D::operator -= ( const double &v ) {
  x -= v;
  y -= v;
  z -= v;

  return( *this );
}

Vector3D Vector3D::operator *= ( const double &v ) {
  x *= v;
  y *= v;
  z *= v;

  return( *this );
}

Vector3D Vector3D::operator / ( const double &v ) const {
  Vector3D ret;

  ret.x = x / v;
  ret.y = y / v;
  ret.z = z / v;

  return( ret );
}

ostream& operator << ( ostream &out,
                       const Vector3D &v ) {
  out << v.x << ' ' << v.y << ' ' << v.z;
  return( out );
}

istream& operator >> ( istream &in,
                       Vector3D &v ) {
  in >> v.x >> v.y >> v.z;
  return( in );
}

double Vector3D::dotProduct( const Vector3D &v ) const {
  return( x * v.x + y * v.y + z * v.z );
}

Vector3D Vector3D::vectorProduct( const Vector3D &v ) const {
  Vector3D ret;

  ret.x = y * v.z - z * v.y;
  ret.y = z * v.x - x * v.z;
  ret.z = x * v.y - y * v.x;

  return( ret );
}

double Vector3D::volume() const {
  return( x * y * z );
}

double Vector3D::sum() const {
  return( x + y + z );
}

double Vector3D::sqabs() const {
  return( x * x + y * y + z * z );
}

double Vector3D::sqabs( const Vector3D &box ) {
  unroll( box );
  return( sqabs() );
}

void Vector3D::normalize() {
  double div = sqrt( sqabs() );
  x /= div;
  y /= div;
  z /= div;
}

double Vector3D::minabsval() const {
  return( min( min( abs(x), abs(y) ), abs(z) ) );
}

double Vector3D::maxabsval() const {
  return( max( max( abs(x), abs(y) ), abs(z) ) );
}

void Vector3D::unroll( const Vector3D &box ) {
  x -= box.x * rint( x / box.x );
  y -= box.y * rint( y / box.y );
  z -= box.z * rint( z / box.z );
}

void Vector3D::unrollf( const Vector3D &box ) {
  x -= box.x * floor( x / box.x );
  y -= box.y * floor( y / box.y );
  z -= box.z * floor( z / box.z );
}

cellIndex Vector3D::numCell( const double &cutoff,
                             const double &margin ) const {
  return( numCell( Vector3D( cutoff, cutoff, cutoff ), margin ) );
}

cellIndex Vector3D::numCell( const Vector3D &cutoff,
                             const double &margin ) const {
  cellIndex ret;

  ret.i = (int)( floor( x / cutoff.x ) );
  ret.j = (int)( floor( y / cutoff.y ) );
  ret.k = (int)( floor( z / cutoff.z ) );

  if ( min( min( ret.i, ret.j ), ret.k ) <= 1 ) {
    return( cellIndex( 0, 0, 0 ) ); // セルじゃないことをアピール
  }

  // あまりに余裕がない場合はちょっとごまかす
  if ( ( x / (double)ret.i - cutoff.x ) < margin ) ret.i--;
  if ( ( y / (double)ret.j - cutoff.y ) < margin ) ret.j--;
  if ( ( z / (double)ret.k - cutoff.z ) < margin ) ret.k--;

  return( ret );
}

cellIndex Vector3D::whichCell( const Vector3D &box,
                               const cellIndex &syssize ) const {
  Vector3D tmp;
  cellIndex ret;

  tmp.x = x - box.x * floor( x / box.x );
  tmp.y = y - box.y * floor( y / box.y );
  tmp.z = z - box.z * floor( z / box.z );

  ret.i = (int)( (double)syssize.i * tmp.x / box.x );
  ret.j = (int)( (double)syssize.j * tmp.y / box.y );
  ret.k = (int)( (double)syssize.k * tmp.z / box.z );

  return( ret.within( syssize ) );
}

void Vector3D::assignRand() {
  x = drand48();
  y = drand48();
  z = drand48();
}

bool Vector3D::is_zero() const {
  if ( x == 0.0 && y == 0.0 && z == 0.0 ) return( true );
  return( false );
}

//--------------------------------------------

cellIndex::cellIndex() {
  clear();
}

cellIndex::cellIndex( const cellIndex &c ) {
  i = c.i;
  j = c.j;
  k = c.k;
}

cellIndex::cellIndex( const int &ci,
                      const int &cj,
                      const int &ck ) {
  i = ci;
  j = cj;
  k = ck;
}

void cellIndex::clear() {
  i = j = k = 0;
}

cellIndex cellIndex::operator + ( const cellIndex &c ) const {
  cellIndex ret;

  ret.i = i + c.i;
  ret.j = j + c.j;
  ret.k = k + c.k;

  return( ret );
}

cellIndex cellIndex::operator - ( const cellIndex &c ) const {
  cellIndex ret;

  ret.i = i - c.i;
  ret.j = j - c.j;
  ret.k = k - c.k;

  return( ret );
}

cellIndex cellIndex::operator += ( const cellIndex &c ) {
  i += c.i;
  j += c.j;
  k += c.k;

  return( *this );
}

cellIndex cellIndex::operator -= ( const cellIndex &c ) {
  i -= c.i;
  j -= c.j;
  k -= c.k;

  return( *this );
}

cellIndex cellIndex::within( const cellIndex &c ) const {
  cellIndex ret;
  ret.i = min( max( c.i - 1, 0 ), i );
  ret.j = min( max( c.j - 1, 0 ), j );
  ret.k = min( max( c.k - 1, 0 ), k );
  return( ret );
}

bool cellIndex::isAdjacent( const cellIndex &c,
                            const cellIndex &size ) const {
  cellIndex diff = *this - c; // 距離ベクトル
  diff.unroll( size );
  if ( max( max( diff.i, diff.j ), diff.k ) == 1 ) return( true );
  return( false ); // 自セルの場合(0)も false
}

void cellIndex::unroll( const cellIndex &size ) {
  if ( size.i > 0 ) {
    i -= size.i * (int)floor( (double)i / (double)size.i );
  }
  if ( size.j > 0 ) {
    j -= size.j * (int)floor( (double)j / (double)size.j );
  }
  if ( size.k > 0 ) {
    k -= size.k * (int)floor( (double)k / (double)size.k );
  }
}

bool cellIndex::operator < ( const cellIndex &c ) const {
  if ( i < c.i ) return( true );
  else if ( i > c.i ) return( false );

  // x 一緒の場合
  if ( j < c.j ) return( true );
  else if ( j > c.j ) return( false );

  // y も一緒の場合
  if ( k < c.k ) return( true );
  return( false );
}

bool cellIndex::operator > ( const cellIndex &c ) const {
  if ( i > c.i ) return( true );
  else if ( i < c.i ) return( false );

  // x 一緒の場合
  if ( j > c.j ) return( true );
  else if ( j < c.j ) return( false );

  // y も一緒の場合
  if ( k > c.k ) return( true );
  return( false );
}

bool cellIndex::operator != ( const cellIndex &c ) const {
  if ( i == c.i && j == c.j && k == c.k ) return( false );
  return( true );
}

bool cellIndex::operator == ( const cellIndex &c ) const {
  if ( i == c.i && j == c.j && k == c.k ) return( true );
  return( false );
}

cellPair::cellPair() {
  clear();
}

void cellPair::clear() {
  ci.clear();
  cj.clear();
}

cellPair::cellPair( const cellIndex &pci,
                    const cellIndex &pcj ) {
  set( pci, pcj );
}

void cellPair::set( const cellIndex &pci,
                    const cellIndex &pcj ) {
  ci = pci;
  cj = pcj;
}

void cellPair::sort() { // ci は cj より小さい
  if ( ci > cj ) {
    cellIndex tmp = ci;
    ci = cj;
    cj = tmp;
  }
}

bool cellPair::operator < ( const cellPair &pc ) const {
  if ( ci < pc.ci ) return( true );
  else if ( ci > pc.ci ) return( false );

  if ( cj < pc.cj ) return( true );
  return( false );
}

bool cellPair::operator > ( const cellPair &pc ) const {
  if ( ci > pc.ci ) return( true );
  else if ( ci < pc.ci ) return( false );

  if ( cj > pc.cj ) return( true );
  return( false );
}

bool cellPair::operator == ( const cellPair &pc ) const {
  if ( ci == pc.ci && cj == pc.cj ) return( true );
  return( false );
}

ostream& operator << ( ostream &out,
                       const cellIndex &c ) {
  out << c.i << ' ' << c.j << ' ' << c.k;
  return( out );
}

istream& operator >> ( istream &in,
                       cellIndex &c ) {
  in >> c.i >> c.j >> c.k;
  return( in );
}
