import xpath.XPath;
import xpath.xml.XPathXml;
import xpath.xml.XPathHxXml;

import flash.geom.Matrix3D;
import flash.geom.Vector3D;
import flash.display3D.Context3D;

import flash.xml.XML;
import flash.xml.XMLList;
import flash.xml.XMLParser;

class WMSystem {
  public var myString( __getMyString, __setMyString ):String;
  private var atoms( null, null ):Array< WMAtom >;
  private var bonds( null, null ):Array< WMBond >;
  private var chains( null, null ):Array< WMChain >;

  public function new() {
    // read XML data from resource file attached in compilation phase.
    // -resource (filename):structure
    myString = haxe.Resource.getString( "structure" );
    // if you modify myString after the call, please call gen() to
    // reconstruct the scene.
  }

  public function clear() {
    atoms = [];
    bonds = [];
    chains = [];
  }

  public function draw( c:Context3D,
                        mpos:Matrix3D,
                        proj:Matrix3D,
                        voffset:Vector3D,
                        light:Vector3D,
                        cpos:Vector3D ):Void {
    for ( atom in atoms ) atom.draw( c, mpos, proj, voffset, light, cpos );
    for ( bond in bonds ) bond.draw( c, mpos, proj, voffset, light, cpos );
    for ( chain in chains ) chain.draw( c, mpos, proj, voffset, light, cpos );
  }

  public function gen( ?wm:Watermelon = null,
                       ?c:Context3D = null ):Void {
    if ( wm == null || c == null ) return;
    loadXML( wm );
    initializeCrd();
    for ( atom in atoms ) atom.gen( c );
    for ( bond in bonds ) bond.gen( c );
    for ( chain in chains ) chain.gen( c );
  }

  public function initializeCrd():Void {
    if ( empty() ) return;
    var origin = geometricCenter();
    translate( Point3D.getMultiply( origin, -1.0 ) );
    rescaleCoord();
  }

  public function geometricCenter():Point3D {
    var ret:Point3D = new Point3D( 0.0, 0.0, 0.0 );
    for ( atom in atoms ) ret.add( atom.pos );
    for ( bond in bonds ) {
      ret.add( bond.pos0 );
      ret.add( bond.pos1 );
    }
    var totnum = atoms.length + bonds.length * 2;
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( p in pos ) ret.add( p );
      totnum += pos.length;
    }
    ret.multiply( 1.0 / totnum );
    return( ret );
  }

  public function translate( p:Point3D ):Void {
    for ( atom in atoms ) atom.pos.add( p );
    for ( bond in bonds ) {
      bond.pos0.add( p );
      bond.pos1.add( p );
    }
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( cp in pos ) cp.add( p );
    }
  }

  public function rescaleCoord():Void {
    var scale:Float = 0.3;
    var swidth = flash.Lib.current.stage.stageWidth * scale;
    var sheight = flash.Lib.current.stage.stageHeight * scale;

    // origin is assumed to be ( 0, 0, 0 )
    var xmax:Float = 0.0;
    var ymax:Float = 0.0;
    for ( atom in atoms ) {
      var p:Point3D = atom.pos;
      xmax = Math.max( xmax, Math.max( Math.abs( p.x ), Math.abs( p.z ) ) );
      ymax = Math.max( ymax, Math.abs( p.y ) );
    }
    for ( bond in bonds ) {
      var p0:Point3D = bond.pos0;
      var p1:Point3D = bond.pos1;
      xmax = Math.max( xmax, Math.max( Math.abs( p0.x ), Math.abs( p0.z ) ) );
      xmax = Math.max( xmax, Math.max( Math.abs( p1.x ), Math.abs( p1.z ) ) );
      ymax = Math.max( ymax, Math.max( Math.abs( p0.y ), Math.abs( p1.y ) ) );
    }
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( p in pos ) {
        xmax = Math.max( xmax, Math.max( Math.abs( p.x ), Math.abs( p.z ) ) );
        ymax = Math.max( ymax, Math.abs( p.y ) );
      }
    }
    xmax = Math.max( 1.0, xmax );
    ymax = Math.max( 1.0, ymax );
    // scale coordinate
    scaleCoord( Math.min( swidth / xmax, sheight / ymax ) );
  }

  public function scaleCoord( scale:Float ):Void {
    for ( atom in atoms ) atom.pos.multiply( scale );
    for ( bond in bonds ) {
      bond.pos0.multiply( scale );
      bond.pos1.multiply( scale );
    }
    for ( chain in chains ) {
      var pos:Array< Point3D > = chain.getPositions();
      for ( p in pos ) p.multiply( scale );
    }
  }

  public function loadXML( ?wm:Watermelon = null ):Void {
    clear();
    if ( wm == null ) return;
    var myxml:Xml = Xml.parse( myString );

    // read settings
    var xml_setting:XPath = new XPath( "WMXML/SETTING" );
    __loadXMLSettings( wm, xml_setting.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read chains and their elements
    var xml_chain:XPath = new XPath( "WMXML/CHAIN" );
    __loadXMLChains( wm, xml_chain.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read ATOMs
    var xml_atom:XPath = new XPath( "WMXML/ATOM" );
    __loadXMLAtoms( wm, xml_atom.selectNodes( XPathHxXml.wrapNode( myxml ) ) );

    // read BONDs
    var xml_bond:XPath = new XPath( "WMXML/BOND" );
    __loadXMLBonds( wm, xml_bond.selectNodes( XPathHxXml.wrapNode( myxml ) ) );
  }

  private function __loadXMLSettings( wm:Watermelon,
                                      ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndio:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      for ( ndc in ndio.elements() ) {
        var nn:String = ndc.nodeName;
        switch( nn ) {
          case "ATOM":
            WMAtom.defaults.loadFromXml( ndc );
          case "BOND":
            WMBond.defaults.loadFromXml( ndc );
            if ( ndc.exists( "rounded" ) ) {
              WMBond.def_round = ( Std.parseInt( ndc.get( "rounded" ) ) > 0 );
            }
            if ( ndc.exists( "round" ) ) {
              WMBond.def_round = ( Std.parseInt( ndc.get( "round" ) ) > 0 );
            }
            if ( ndc.exists( "exclude" ) ) {
              WMBond.def_exclude = ( Std.parseInt( ndc.get( "exclude" ) ) > 0 );
            }
          case "RIBBON":
            WMRibbon.defaultsRibbon.loadFromXml( ndc );
          case "COIL":
            WMRibbon.defaultsCoil.loadFromXml( ndc );
          case "CAMERA":
            if ( ndc.exists( "z" ) ) {
              wm.setCameraPosZ( Std.parseFloat( ndc.get( "z" ) ) );
            }
          case "LIGHT":
            if ( ndc.exists( "dir" ) ) {
              wm.setLightDirection( Point3D.fromString( ndc.get( "pos" ) ) );
            }
          case "AR":
            if ( ndc.exists( "rate" ) ) {
              wm.setAutoRotationRate( Std.parseFloat( ndc.get( "rate" ) ) );
            }
          case "PROTECT":
            wm.setProtectData();
          case "FRAMERATE":
            wm.setFrameRate( Std.parseInt( "rate" ) );
          case "WHEEL":
            if ( ndc.exists( "scale" ) ) {
              wm.setScaleWheel( Std.parseFloat( ndc.get( "scale" ) ) );
            }
            if ( ndc.exists( "depth" ) ) {
              wm.setScaleWheel( Std.parseFloat( ndc.get( "depth" ) ) );
            }
          case "MATRIX":
            if ( ndc.exists( "data" ) ) {
              var m:Matrix4 = Matrix4.fromString( ndc.get( "data" ) );
              wm.setInitialView( m );
            }
        }
      }
    }
  }

  private function __loadXMLChains( wm:Watermelon,
                                    ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ni:Int = 1;
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      if ( ndi.exists( "N" ) ) ni = Std.parseInt( ndi.get( "N" ) );
      var chain:WMChain = new WMChain();
      // get child elements describing control points
      var children:Array< Dynamic > = new Array< Dynamic >();
      for ( child in ndi.elementsNamed( "POINT" ) ) {
        var anon = { pos:child.get( "pos" ), index:Std.parseInt( child.get( "index" ) ), n:-1 };
        if ( child.exists( "N" ) ) anon.n = Std.parseInt( child.get( "N" ) );
        children.push( anon );
      }
      children.sort( __hasSmallIndex );
      chain.setPositions( children, ni );
      // read ribbons and coils
      for ( child in ndi.elementsNamed( "RIBBON" ) ) {
        var rib:WMRibbon = new WMRibbon( true );
        rib.loadFromXml( child );
        chain.register( rib );
      }
      for ( child in ndi.elementsNamed( "COIL" ) ) {
        var coi:WMRibbon = new WMRibbon( false );
        coi.loadFromXml( child );
        chain.register( coi );
      }
      chains.push( chain );
    }
  }

  private function __hasSmallIndex( o0:Dynamic,
                                    o1:Dynamic ):Int {
    if ( o0.index == o1.index ) return(0);
    if ( o0.index < o1.index ) return(-1);
    return(1);
  }

  private function __loadXMLAtoms( wm:Watermelon,
                                   ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var at:WMAtom = new WMAtom();
      at.loadFromXml( ndi );
      atoms.push( at );
    }
  }

  private function __loadXMLBonds( wm:Watermelon,
                                   ?nodes:Iterable< XPathXml > = null ):Void {
    for ( nd in nodes ) {
      var ndi:Xml = cast( nd, XPathHxXml ).getWrappedXml();
      var bd:WMBond = new WMBond();
      bd.loadFromXml( ndi );
      bonds.push( bd );
    }
  }

  public function empty():Bool {
    if ( numAtoms() + numBonds() + numChains() == 0 ) return( true );
    return( false );
  }
  public function numAtoms():Int { return( atoms.length ); }
  public function numBonds():Int { return( bonds.length ); }
  public function numChains():Int { return( chains.length ); }
  public function __getMyString():String { return( myString ); }
  public function __setMyString( s:String ):String {
    myString = s;
    return( myString );
  }
}
