class SmoothChain {
  // number of points to be inserted between two adjacent control points
  // NOTE: Points are interpolated evenly in terms of t.
  //       The actual distance is usually not even due to higher order
  //       terms. Only if the coefficients of higher terms (t^2, t^3) are
  //       zero and the coefficient of linear term is non-zero,
  //       interpolated points are evenly separated.
  public var n_interp( __getInterpolateNum, __setInterpolateNum ):Int;
  public var curves( __getCurves, null ):Array< CubicCurve >;
  public var controls( null, null ):Array< WMCPoint >;

  public function new( ?ops:Array< Dynamic > = null,
                       ?ni:Int = 2 ) {
    n_interp = ni;
    controls = new Array< WMCPoint >();
    curves = new Array< CubicCurve >();

    initialize( ops );
  }

  public function clear():Void {
    curves = [];
    controls = [];
  }

  public function initialize( ?ops:Array< Dynamic > = null ):Void {
    if ( ops == null ) return; // nothing to do
    if ( ops.length < 4 ) {    // unable to build chain
      trace( "WMChain::initialize: at least 4 control points are necessary to build a chain" );
      return;
    }
    genCurves( ops );
    genPoints( ops );
    genFaceDirections();
  }

  private function genCurves( pos:Array< Dynamic > ):Void {
    curves = [];
    var num:Int = pos.length;
    // generate first curve
    curves.push( new CubicCurve( null, pos[0].p, pos[1].p, pos[2].p ) );
    for ( i in 0 ... num - 3 ) {
      curves.push( new CubicCurve( pos[i].p, pos[i+1].p, pos[i+2].p, pos[i+3].p ) );
    }
    // last one
    curves.push( new CubicCurve( pos[num-3].p, pos[num-2].p, pos[num-1].p ) );
  }

  private function genPoints( pos:Array< Dynamic > ):Void {
    controls = [];
    var num:Int = curves.length;
    controls.push( new WMCPoint( curves[0].getVal( 0.0 ) ) );
    for ( i in 0 ... num ) {
      var ni:Int = n_interp;
      if ( pos[i].n != null && pos[i].n > 0 ) ni = pos[i].n;
      var fn:Float = 1.0 / ni;
      for ( j in 1 ... ni + 1 ) {
        controls.push( new WMCPoint( curves[i].getVal( j * fn ) ) );
      }
    }
  }

  private function genFaceDirections():Void {
    var num:Int = controls.length;
    var prev_face:Point3D = null;
    for ( i in 1 ... num - 1 ) {
      var v0:Point3D = Point3D.getSub( controls[i].pos, controls[i-1].pos );
      var v1:Point3D = Point3D.getSub( controls[i].pos, controls[i+1].pos );
      var vc0:Point3D = Point3D.getCross( v0, v1 );
      vc0.normalize();
      if ( prev_face != null ) {
        if ( v0.getAngle( v1 ) > 0.945 * Math.PI ) {
          controls[i].facedir = prev_face.clone();
        } else {
          if ( vc0.getAngle( prev_face ) > 0.5 * Math.PI ) vc0.multiply( -1.0 );
          controls[i].facedir = vc0.clone();
          prev_face = vc0.clone();
        }
      } else {
        controls[i].facedir = vc0.clone();
        prev_face = vc0.clone();
      }
    }
    controls[0].facedir = controls[1].facedir.clone();
    controls[num-1].facedir = controls[num-2].facedir.clone();
  }

  public function genCoil( width:Float,
                           quality:Int,
                           from:Int,
                           to:Int ):Tube3D {
    return( new Tube3D( getPartOfControls( from, to ), width, quality ) );
  }

  public function genRibbon( width:Float,
                             from:Int,
                             to:Int ):Ribbon3D {
    var _p0:Array< Point3D > = new Array< Point3D >();
    var _p1:Array< Point3D > = new Array< Point3D >();
    for ( i in from ... to ) {
      if ( i >= controls.length ) break;
      var work:Point3D = Point3D.getMultiply( controls[i].facedir, width );
      _p0.push( Point3D.getAdd( controls[i].pos, work ) );
      _p1.push( Point3D.getSub( controls[i].pos, work ) );
    }
    return( new Ribbon3D( _p0, _p1 ) );
  }

  private function getPartOfControls( from:Int,
                                      to:Int ):Array< Point3D > {
    var ret:Array< Point3D > = new Array< Point3D >();
    var rangeF:Int = Std.int( Math.max( from, 0 ) );
    var rangeT:Int = Std.int( Math.min( to, controls.length ) );
    for ( i in rangeF ... rangeT ) ret.push( controls[i].pos );
    return( ret );
  }

  public function getPositions():Array< Point3D > {
    var ret:Array< Point3D > = new Array< Point3D >();
    for ( cont in controls ) ret.push( cont.pos );
    return( ret );
  }

  public function getCurve( i:Int ):CubicCurve {
    return( curves[i] );
  }

  public function __getInterpolateNum():Int { return( n_interp ); }
  public function __getCurves():Array< CubicCurve > { return( curves ); }
  public function __setInterpolateNum( n:Int ):Int {
    n_interp = n;
    return( n );
  }
}
