/*
    Video maid
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "general.h"
#if GTK_CHECK_VERSION(2,8,0)
# include "icons.h"
#endif /* GTK_CHECK_VERSION(2,8,0) */
#include "sigfile.h"
#include "sigmain.h"
#include "thread.h"
#include "misc/fileio.h"
#include "orz/orzhistory.h"
#include "orz/orzmdi.h"
#include <time.h>
#if defined (TM_IN_SYS_TIME) && defined (HAVE_SYS_TIME_H)
# include <sys/time.h>
#endif /* defined (TM_IN_SYS_TIME) && defined (HAVE_SYS_TIME_H) */
#ifdef G_OS_WIN32
# include <windows.h>
# include <tchar.h>
# include <time.h>
#endif /* G_OS_WIN32 */


#define THREAD_CMD_WINDOW 0
#define THREAD_CMD_HISTORY 1
#define THREAD_CMD_INSERT 2
#define THREAD_CMD_MODIFY 3
#define THREAD_CMD_DELETE 4
#define THREAD_CMD_BREAK 5


typedef struct _ThreadCommand
{
  gint cmd;             /* ja:0:挿入,1:削除,2:変更,3:中断,4:終了 */
  /* ja:ファイル/履歴/挿入 */
  gchar *file;          /* ja:ファイル名 */
  /* ja:ファイル */
  VmaidWindow *vmaid;
  /* ja:挿入 */
  gint mode;            /* ja:0:終了,1:開く,2:保存,3:フレーム数,4:反転 */
  time_t time;          /* ja:時刻 */
  gboolean userbreak;   /* ja:TRUE:継続,FALSE:中断要求 */
  gint percent;         /* ja:進行状況(0...100) */
  /* ja:中断 */
  gboolean all;
#ifdef USE_THREAD
  GThread *id;
#else /* not USE_THREAD */
  gint id;
#endif /* not USE_THREAD */
} ThreadCommand;
typedef struct _ThreadList
{
  gboolean userbreak;   /* ja:TRUE:継続,FALSE:中断要求 */
  gint mode;            /* ja:0:終了,1:開く,2:保存,3:フレーム数,4:反転 */
  gint percent;         /* ja:進行状況(0...100) */
#ifdef USE_THREAD
  GThread *id;
#else /* not USE_THREAD */
  gint id;
#endif /* not USE_THREAD */
} ThreadList;


static GList *glist_cmd = NULL;
static GList *glist_thread = NULL;
#ifdef USE_THREAD
G_LOCK_DEFINE_STATIC (critical);
static volatile gboolean critical = FALSE;
#endif /* USE_THREAD */


/******************************************************************************
*                                                                             *
* ja:スレッド描画関数群                                                       *
*                                                                             *
******************************************************************************/
#ifndef USE_THREAD
/*  ja:メニューを設定する
    task,TRUE:通常,FALSE:終了のみ                                           */
static void
thread_set_menu_bar (gboolean task)
{
  gint i;
  GList *glist;
  GtkWidget *menu_shell;

  menu_shell = misc_find_menu (menu_entries, "/file/");
  glist = gtk_container_get_children (GTK_CONTAINER (menu_shell));
  for (i = g_list_length (glist) - 2; i >= 0; i--)
    gtk_widget_set_sensitive (GTK_WIDGET (g_list_nth_data (glist, i)), task);
  g_list_free (glist);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/edit"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/view"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/option"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/window"), task);
  gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/help"), task);
  for (i = 0; toolbar_entries[i].path; i++)
    if (toolbar_entries[i].tool_item)
      gtk_widget_set_sensitive (GTK_WIDGET (toolbar_entries[i].tool_item),
                                                                        task);
  if (task)
    set_menu_bar ((VmaidWindow *)orz_mdi_get_data (ORZ_MDI (mdi), -1));
}
#endif /* not USE_THREAD */


/*  ja:スレッドのアイドル処理
    data,データ
     RET,TRUE:継続,FALSE:終了                                               */
#ifdef USE_THREAD
gboolean
#else /* not USE_THREAD */
static gboolean
#endif /* not USE_THREAD */
thread_idle (gpointer data)
{
  guint length;
  GList *glist;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
  gdk_threads_enter ();
#endif /* USE_THREAD */
  length = g_list_length (glist_thread);
  /* ja:中断された終了を探す */
  for (glist = g_list_first (glist_thread); glist; glist = g_list_next (glist))
    {
      ThreadList *tl;

      tl = glist->data;
      if (tl->mode == THREAD_MODE_EXIT && !tl->userbreak)
        {
          ThreadCommand *tc;

          tc = g_malloc0 (sizeof (ThreadCommand));
          tc->cmd = THREAD_CMD_DELETE;
          tc->id = tl->id;
          glist_cmd = g_list_append (glist_cmd, tc);
        }
    }
  while (glist_cmd)
    {
      ThreadCommand *tc;

      glist = g_list_first (glist_cmd);
      tc = glist->data;
      glist_cmd = g_list_remove (glist_cmd, tc);
      switch (tc->cmd)
        {
          case THREAD_CMD_WINDOW:
            {
              gchar *filename;
              gint page_num;
              GtkWidget *xbox, *vbox;
              const static gchar *rot47 = "\0"
        "%9:D AC@8C2> 4@>6D H:E9 }~ (p##p}%*[ E@ E96 6IE6?E A6C>:EE65 3J =2H]";

              if (tc->file)
                {
                  filename = fileio_get_full_path (tc->file);
                }
              else
                {
                  static gint fcount = 0; /* ja:新規ファイルの数 */

                  /* ja:新規 */
                  filename = g_strdup_printf ("new%04d.avi", fcount++ % 1000);
                }
              /* ja:表示エリア */
              tc->vmaid->drawing = gtk_drawing_area_new ();
              g_signal_connect (G_OBJECT (tc->vmaid->drawing),
                                                        "focus-in-event",
                            G_CALLBACK (signal_focus_in), tc->vmaid);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing),
                                                        "focus-out-event",
                            G_CALLBACK (signal_focus_out), tc->vmaid);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing), "realize",
                            G_CALLBACK (signal_realize), NULL);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing),
                                                        "configure-event",
                            G_CALLBACK (signal_config), tc->vmaid);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing), "expose-event",
                            G_CALLBACK (signal_expose), tc->vmaid);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing),
                                                        "button-press-event",
                            G_CALLBACK (signal_button_press_draw), tc->vmaid);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing),
                                                        "motion-notify-event",
                            G_CALLBACK (signal_motion_notify), tc->vmaid);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing),
                                                        "button-release-event",
                            G_CALLBACK (signal_button_release), tc->vmaid);
              g_signal_connect (G_OBJECT (tc->vmaid->drawing), "scroll-event",
                            G_CALLBACK (signal_scroll), tc->vmaid);
              g_signal_connect_after (G_OBJECT (tc->vmaid->drawing),
                                                        "key-press-event",
                            G_CALLBACK (signal_key_press), tc->vmaid);
              g_signal_connect (GTK_OBJECT (tc->vmaid->drawing), "destroy",
                            G_CALLBACK (signal_destroy_draw), tc->vmaid);
              gtk_widget_add_events (tc->vmaid->drawing,
                                GDK_POINTER_MOTION_MASK | GDK_BUTTON_PRESS_MASK
                                | GDK_BUTTON_RELEASE_MASK | GDK_KEY_PRESS_MASK
                                | GDK_KEY_RELEASE_MASK | GDK_FOCUS_CHANGE_MASK
                                | GDK_SCROLL_MASK);
              gtk_widget_set_size_request (tc->vmaid->drawing, tc->vmaid->width
                        + (tc->vmaid->selfrm ? tc->vmaid->width * 2 + 4 : 2),
                    system_font_height * 2 + ((tc->vmaid->avi_edit[0] ? 1 : 0)
                    + (tc->vmaid->avi_edit[1] ? 1 : 0)) * tc->vmaid->height);
              GTK_WIDGET_SET_FLAGS (tc->vmaid->drawing, GTK_CAN_FOCUS);
              /* ja:スクロールバー */
              tc->vmaid->hscroll = gtk_hscrollbar_new
                    (GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 0, 1, 1, 1)));
              /* ja:ボックス */
              xbox = gtk_hbox_new (FALSE, 0);
              vbox = gtk_vbox_new (FALSE, 0);
              gtk_box_pack_start (GTK_BOX (vbox),
                                        tc->vmaid->drawing, FALSE, FALSE, 0);
              gtk_box_pack_start (GTK_BOX (vbox),
                                        tc->vmaid->hscroll, FALSE, FALSE, 0);
              gtk_box_pack_start (GTK_BOX (xbox), vbox, *rot47, *rot47, 0);
              /* ja:表示 */
              page_num = orz_mdi_append_page (ORZ_MDI (mdi),
                                xbox, filename, tc->file != NULL, tc->vmaid);
              gtk_widget_show_all (xbox);
              gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page_num);
              gtk_widget_grab_focus (tc->vmaid->drawing);
              g_free (filename);
            }
            break;
          case THREAD_CMD_HISTORY:
            orz_history_add_file (ORZ_HISTORY (history), tc->file);
            break;
          case THREAD_CMD_INSERT:
            {
              const gchar *fmt = "%c";
              gchar str[512], *text_mode, *text_file, *text_time, *text_status;
              GtkListStore *store;
              GtkStockItem stock_item;
              GtkTreeIter iter;
              ThreadList *tl;
#if GTK_CHECK_VERSION(2,6,0)
              gint percent;
#endif /* GTK_CHECK_VERSION(2,6,0) */
#if GTK_CHECK_VERSION(2,8,0)
              const gchar *stock_id;
              GdkPixbuf *pixbuf_mode, *pixbuf_file;
#endif /* GTK_CHECK_VERSION(2,8,0) */

              tl = g_malloc (sizeof (ThreadList));
              tl->mode = tc->mode;
              tl->percent = tc->percent;
              tl->userbreak = tc->userbreak;
              tl->id = tc->id;
              /* ja:モード */
              switch (tc->mode)
                {
                  case THREAD_MODE_EXIT:
                    gtk_stock_lookup (GTK_STOCK_QUIT, &stock_item);
#if GTK_CHECK_VERSION(2,8,0)
                    pixbuf_mode = NULL;
                    stock_id = stock_item.stock_id;
#endif /* GTK_CHECK_VERSION(2,8,0) */
                    text_mode = misc_mnemonic_to_text (stock_item.label);
                    break;
                  case THREAD_MODE_OPEN:
                    gtk_stock_lookup (GTK_STOCK_OPEN, &stock_item);
#if GTK_CHECK_VERSION(2,8,0)
                    pixbuf_mode = NULL;
                    stock_id = stock_item.stock_id;
#endif /* GTK_CHECK_VERSION(2,8,0) */
                    text_mode = misc_mnemonic_to_text (stock_item.label);
                    break;
                  case THREAD_MODE_SAVE:
                    gtk_stock_lookup (GTK_STOCK_SAVE_AS, &stock_item);
#if GTK_CHECK_VERSION(2,8,0)
                    pixbuf_mode = NULL;
                    stock_id = stock_item.stock_id;
#endif /* GTK_CHECK_VERSION(2,8,0) */
                    text_mode = misc_mnemonic_to_text (stock_item.label);
                    break;
                  case THREAD_MODE_CHANGE:
#if GTK_CHECK_VERSION(2,8,0)
                    pixbuf_mode = gdk_pixbuf_new_from_xpm_data (change16_xpm);
                    stock_id = NULL;
#endif /* GTK_CHECK_VERSION(2,8,0) */
                    text_mode = g_strdup (_("Change Frames"));
                    break;
                  case THREAD_MODE_REVERSE:
#if GTK_CHECK_VERSION(2,8,0)
                    pixbuf_mode = gdk_pixbuf_new_from_xpm_data (reverse16_xpm);
                    stock_id = NULL;
#endif /* GTK_CHECK_VERSION(2,8,0) */
                    text_mode = g_strdup (_("Reverse"));
                    break;
                  default:
#if GTK_CHECK_VERSION(2,8,0)
                    pixbuf_mode = NULL;
                    stock_id = NULL;
#endif /* GTK_CHECK_VERSION(2,8,0) */
                    text_mode = NULL;
                }
              /* ja:ファイル名 */
#if GTK_CHECK_VERSION(2,8,0)
              pixbuf_file = fileio_extract_icon (tc->file, GTK_ICON_SIZE_MENU);
#endif /* GTK_CHECK_VERSION(2,8,0) */
#if GLIB_CHECK_VERSION(2,6,0)
              text_file = tc->file ? g_filename_display_name (tc->file) : NULL;
#else /* not GLIB_CHECK_VERSION(2,6,0) */
              text_file = tc->file ? g_filename_to_utf8 (tc->file, -1,
                                                    NULL, NULL, NULL) : NULL;
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
              /* ja:時刻 */
              if (strftime (str, sizeof (str), fmt, localtime (&tc->time)) > 0)
                {
                  text_time = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
                }
              else
                {
                  gsize leng;

                  text_time = g_locale_to_utf8 (ctime (&tc->time), -1,
                                                            NULL, NULL, NULL);
                  leng = g_strlen (text_time);
                  if (text_time[leng - 1] == '\n')
                    text_time[leng - 1] = '\0';
                }
              /* ja:状態 */
#if GTK_CHECK_VERSION(2,6,0)
              if (!tl->userbreak)
                {
                  text_status = g_strdup ("*");
                  percent = 0;
                }
              else if (0 <= tl->percent && tl->percent <= 100)
                {
                  text_status = NULL;
                  percent = tl->percent;
                }
              else
                {
                  text_status = g_strdup (_("Error"));
                  percent = 0;
                }
#else /* not GTK_CHECK_VERSION(2,6,0) */
              text_status = tl->userbreak
                                    ? 0 <= tl->percent && tl->percent <= 100
                                    ? g_strdup_printf ("%d", tl->percent)
                                    : g_strdup (_("Error")) : g_strdup ("*");
#endif /* not GTK_CHECK_VERSION(2,6,0) */
              /* ja:設定 */
              store = GTK_LIST_STORE
                            (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
              gtk_list_store_append (store, &iter);
              gtk_list_store_set (store, &iter,
#if GTK_CHECK_VERSION(2,8,0)
                                        THREAD_TVIEW_MODE_PIXBUF, pixbuf_mode,
                                        THREAD_TVIEW_MODE_STOCK,  stock_id,
#endif /* GTK_CHECK_VERSION(2,8,0) */
                                        THREAD_TVIEW_MODE_TEXT,   text_mode,
#if GTK_CHECK_VERSION(2,8,0)
                                        THREAD_TVIEW_FILE_PIXBUF, pixbuf_file,
#endif /* GTK_CHECK_VERSION(2,8,0) */
                                        THREAD_TVIEW_FILE_TEXT,   text_file,
                                        THREAD_TVIEW_TIME,        text_time,
#if GTK_CHECK_VERSION(2,6,0)
                                        THREAD_TVIEW_STATUS_PROG, percent,
#endif /* GTK_CHECK_VERSION(2,6,0) */
                                        THREAD_TVIEW_STATUS_TEXT, text_status,
                                        THREAD_TVIEW_POINTER,     tl, -1);
              gtk_tree_view_columns_autosize (GTK_TREE_VIEW (tview));
#if GTK_CHECK_VERSION(2,8,0)
              if (pixbuf_mode)
                g_object_unref (pixbuf_mode);
              if (pixbuf_file)
                g_object_unref (pixbuf_file);
#endif /* GTK_CHECK_VERSION(2,8,0) */
              g_free (text_mode);
              g_free (text_file);
              g_free (text_time);
              g_free (text_status);
              glist_thread = g_list_append (glist_thread, tl);
            }
            break;
          case THREAD_CMD_DELETE:
            {
              gint i;
              GtkListStore *store;

              /* ja:IDに一致する項目を探す */
              store = GTK_LIST_STORE
                            (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
              for (i = gtk_tree_model_iter_n_children
                            (GTK_TREE_MODEL (store), NULL) - 1; i >= 0; i--)
                {
                  GtkTreeIter iter;
                  ThreadList *tl;

                  gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store),
                                                            &iter, NULL, i);
                  gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
                  if (tl->id == tc->id)
                    {
                      gtk_list_store_remove (store, &iter);
                      glist_thread = g_list_remove (glist_thread, tl);
                      g_free (tl);
                    }
                }
            }
            break;
          case THREAD_CMD_MODIFY:
            {
              gint i;
              GtkListStore *store;
              GtkTreeIter iter;

              store = GTK_LIST_STORE
                            (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
              /* ja:IDに一致する項目を探す */
              for (i = 0; gtk_tree_model_iter_nth_child
                                (GTK_TREE_MODEL (store), &iter, NULL, i); i++)
                {
                  ThreadList *tl;

                  gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
                  if (tl->id == tc->id)
                    {
#if GTK_CHECK_VERSION(2,6,0)
                      if (!tl->userbreak)
                        gtk_list_store_set (store, &iter,
                                    THREAD_TVIEW_STATUS_TEXT, "*",
                                    THREAD_TVIEW_STATUS_PROG, 0, -1);
                      else if (0 <= tl->percent && tl->percent <= 100)
                        gtk_list_store_set (store, &iter,
                                    THREAD_TVIEW_STATUS_TEXT, NULL,
                                    THREAD_TVIEW_STATUS_PROG, tl->percent, -1);
                      else
                        gtk_list_store_set (store, &iter,
                                    THREAD_TVIEW_STATUS_TEXT, _("Error"),
                                    THREAD_TVIEW_STATUS_PROG, 0, -1);
#else /* not GTK_CHECK_VERSION(2,6,0) */
                      gchar *text;

                      text = tl->userbreak
                                    ? 0 <= tl->percent && tl->percent <= 100
                                    ? g_strdup_printf ("%d", tl->percent)
                                    : g_strdup (_("Error")) : g_strdup ("*");
                      gtk_list_store_set (store, &iter,
                                        THREAD_TVIEW_STATUS_TEXT, text, -1);
                      g_free (text);
#endif /* not GTK_CHECK_VERSION(2,6,0) */
                      break;
                    }
                }
            }
            break;
          case THREAD_CMD_BREAK:
            {
              gint i;
              GtkListStore *store;
              GtkTreeIter iter;
              GtkTreeSelection *select;

              store = GTK_LIST_STORE
                            (gtk_tree_view_get_model (GTK_TREE_VIEW (tview)));
              select = gtk_tree_view_get_selection (GTK_TREE_VIEW (tview));
              for (i = 0; gtk_tree_model_iter_nth_child
                                (GTK_TREE_MODEL (store), &iter, NULL, i); i++)
                if (tc->all
                        || gtk_tree_selection_iter_is_selected (select, &iter))
                  {
                    ThreadList *tl;

                    gtk_tree_model_get (GTK_TREE_MODEL (store), &iter,
                                                THREAD_TVIEW_POINTER, &tl, -1);
                    if (tl->userbreak)
                      {
                        ThreadCommand *tc_new;

                        tl->userbreak = FALSE;
                        tl->percent = -1;
                        tc_new = g_malloc0 (sizeof (ThreadCommand));
                        tc_new->cmd = THREAD_CMD_MODIFY;
                        tc_new->id = tl->id;
                        glist_cmd = g_list_append (glist_cmd, tc_new);
                      }
                  }
            }
        }
      g_free (tc->file);
      g_free (tc);
    }
  /* ja:終了を判定 */
  if (g_list_length (glist_thread) == 1)
    {
      ThreadList *tl;

      tl = glist_thread->data;
      if (tl->mode == THREAD_MODE_EXIT
                                && orz_mdi_get_n_pages (ORZ_MDI (mdi)) <= 0)
        {
          gtk_list_store_clear (GTK_LIST_STORE
                            (gtk_tree_view_get_model (GTK_TREE_VIEW (tview))));
          g_list_free (glist_thread);
          glist_thread = NULL;
          g_free (tl);
          gtk_widget_destroy (window);
        }
    }
#ifdef USE_THREAD
  gdk_threads_leave ();
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  if (length <= 0 && g_list_length (glist_thread) > 0)
    {
      thread_set_menu_bar (FALSE);
      gtk_widget_set_sensitive (mdi, FALSE);
    }
  else if (length > 0 && g_list_length (glist_thread) <= 0)
    {
      thread_set_menu_bar (TRUE);
      gtk_widget_set_sensitive (mdi, TRUE);
    }
#endif /* not USE_THREAD */
  return TRUE;
}


/******************************************************************************
*                                                                             *
* ja:スレッド制御関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:ウインドウを開く
    vmaid,ウインドウ情報
     file,ファイル名
      RET,TRUE:正常終了,FALSE:エラー                                        */
gboolean
thread_window (VmaidWindow *vmaid,
               const gchar *file)
{
  ThreadCommand *tc;

  tc = g_malloc0 (sizeof (ThreadCommand));
  tc->cmd = THREAD_CMD_WINDOW;
  tc->file = file ? g_strdup (file) : NULL;
  tc->vmaid = vmaid;
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  glist_cmd = g_list_append (glist_cmd, tc);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  thread_idle (NULL);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:履歴に追加する
    file,ファイル名
     RET,TRUE:正常終了,FALSE:エラー                                         */
gboolean
thread_history (const gchar *file)
{
  ThreadCommand *tc;

  tc = g_malloc0 (sizeof (ThreadCommand));
  tc->cmd = THREAD_CMD_HISTORY;
  tc->file = file ? g_strdup (file) : NULL;
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  glist_cmd = g_list_append (glist_cmd, tc);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  thread_idle (NULL);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:リストに加える
      id,スレッドID
    mode,モード(0:終了,1:開く,2:保存,3:フレーム数,4:反転)
    file,ファイル名
     RET,TRUE:正常終了,FALSE:エラー                                         */
#ifdef USE_THREAD
gboolean
thread_insert (GThread     *id,
               const gint   mode,
               const gchar *file)
#else /* not USE_THREAD */
gboolean
thread_insert (gint         id,
               const gint   mode,
               const gchar *file)
#endif /* not USE_THREAD */
{
  ThreadCommand *tc;

  tc = g_malloc0 (sizeof (ThreadCommand));
  tc->cmd = THREAD_CMD_INSERT;
  tc->mode = mode;
  tc->file = file ? g_strdup (file) : NULL;
  tc->time = time (NULL);
  tc->userbreak = TRUE;
  tc->percent = 0;
  tc->id = id;
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  glist_cmd = g_list_append (glist_cmd, tc);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  thread_idle (NULL);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:リストから削除する
     id,スレッドID
    RET,TRUE:正常終了,FALSE:エラー                                          */
#ifdef USE_THREAD
gboolean
thread_delete (GThread *id)
#else /* not USE_THREAD */
gboolean
thread_delete (gint id)
#endif /* not USE_THREAD */
{
  ThreadCommand *tc;

  tc = g_malloc0 (sizeof (ThreadCommand));
  tc->cmd = THREAD_CMD_DELETE;
  tc->id = id;
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  glist_cmd = g_list_append (glist_cmd, tc);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  thread_idle (NULL);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:リストを更新する
         id,スレッドID
    percent,進行状況
        RET,TRUE:継続,FALSE:中断                                            */
#ifdef USE_THREAD
gboolean
thread_idling (GThread    *id,
               const gint  percent)
#else /* not USE_THREAD */
gboolean
thread_idling (gint       id,
               const gint percent)
#endif /* not USE_THREAD */
{
  gboolean result = FALSE;
  GList *glist;

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  /* ja:項目を探す */
  for (glist = g_list_first (glist_cmd); glist; glist = g_list_next (glist))
    {
      ThreadCommand *tc;

      tc = glist->data;
      if (tc->cmd == THREAD_CMD_INSERT && tc->id == id)
        {
          if (tc->userbreak)
            tc->percent = percent;
          result = tc->userbreak;
          break;
        }
    }
  for (glist = g_list_first (glist_thread); glist; glist = g_list_next (glist))
    {
      ThreadList *tl;

      tl = glist->data;
      if (tl->id == id)
        {
          if (tl->userbreak && tl->percent != percent)
            {
              /* ja:中断ではなくかつ進行状況が異なるとき */
              ThreadCommand *tc;

              tl->percent = percent;
              tc = g_malloc0 (sizeof (ThreadCommand));
              tc->cmd = THREAD_CMD_MODIFY;
              tc->id = id;
              glist_cmd = g_list_append (glist_cmd, tc);
            }
          result = tl->userbreak;
          break;
        }
    }
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  thread_idle (NULL);
  while (gtk_events_pending ())
    gtk_main_iteration ();
#endif /* not USE_THREAD */
  return result;
}


/*  ja:スレッドを停止させる
    all,TRUE:すべてのスレッド,FALSE:選択されたスレッドのみ
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
thread_break (const gboolean all)
{
  ThreadCommand *tc;

  tc = g_malloc0 (sizeof (ThreadCommand));
  tc->cmd = THREAD_CMD_BREAK;
  tc->all = all;
#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  glist_cmd = g_list_append (glist_cmd, tc);
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#else /* not USE_THREAD */
  thread_idle (NULL);
#endif /* not USE_THREAD */
  return TRUE;
}


/*  ja:プログラムを終了させる
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
thread_exit (void)
{
  gboolean result = TRUE;
  GList *glist;
#ifdef USE_THREAD
  GThread *id = NULL;
#else /* not USE_THREAD */
  gint id = THREAD_ID_CLOSE;
#endif /* not USE_THREAD */

#ifdef USE_THREAD
  G_LOCK (critical);
  critical = TRUE;
#endif /* USE_THREAD */
  /* ja:項目を探す */
  for (glist = g_list_first (glist_cmd); glist; glist = g_list_next (glist))
    {
      ThreadCommand *tc;

      tc = glist->data;
      if (tc->cmd == THREAD_CMD_INSERT
                                    && (tc->percent < 0 || 100 < tc->percent))
        /* ja:既にエラーになっているスレッドを終了させる */
        tc->userbreak = FALSE;
      else if (tc->cmd == THREAD_CMD_INSERT && tc->mode == THREAD_MODE_EXIT)
        /* ja:既に終了スレッドがあるならば、新規に登録しない */
        result = FALSE;
    }
  for (glist = g_list_first (glist_thread); glist; glist = g_list_next (glist))
    {
      ThreadList *tl;

      tl = glist->data;
      if ((tl->percent < 0 || 100 < tl->percent) && tl->userbreak)
        {
          /* ja:既にエラーになっているスレッドを終了させる */
          ThreadCommand *tc;

          tl->userbreak = FALSE;
          tc = g_malloc0 (sizeof (ThreadCommand));
          tc->cmd = THREAD_CMD_MODIFY;
          tc->id = tl->id;
          glist_cmd = g_list_append (glist_cmd, tc);
        }
      else if (tl->mode == THREAD_MODE_EXIT)
        {
          /* ja:既に終了スレッドがあるならば、新規に登録しない */
          result = FALSE;
        }
    }
#ifdef USE_THREAD
  critical = FALSE;
  G_UNLOCK (critical);
#endif /* USE_THREAD */
  if (result)
    thread_insert (id, THREAD_MODE_EXIT, NULL);
#ifndef USE_THREAD
  thread_idle (NULL);
#endif /* not USE_THREAD */
  return result;
}
