/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "avibase.h"


/******************************************************************************
*                                                                             *
* ja:初期化関数                                                               *
*                                                                             *
******************************************************************************/
/*  ja:AVIを初期化する                                                      */
gboolean
avi_init (void)
{
  return icm_init () && acm_init ();
}


/*  ja:AVIを終了する                                                        */
gboolean
avi_exit (void)
{
  if (avi_lang)
    {
      gint i;

      for (i = 0; avi_lang[i].name; i++)
        g_free ((gpointer)avi_lang[i].name);
      g_free (avi_lang);
      avi_lang = NULL;
    }
  return icm_exit () & acm_exit ();
}


/******************************************************************************
*                                                                             *
* ja:フォーマット関数                                                         *
*                                                                             *
******************************************************************************/
/*  ja:AVIのRateを取得する
    avi_edit,AVI編集ハンドル
         RET,Rate                                                           */
guint32
avi_edit_get_rate (AviEdit *avi_edit)
{
  return avi_edit ? avi_edit->rate : 0;
}


/*  ja:AVIのScaleを取得する
    avi_edit,AVI編集ハンドル
         RET,Scale                                                          */
guint32
avi_edit_get_scale (AviEdit *avi_edit)
{
  return avi_edit ? avi_edit->scale : 0;
}


/*  ja:AVIの優先度を取得する
    avi_edit,AVI編集ハンドル
         RET,優先度                                                         */
guint16
avi_edit_get_priority (AviEdit *avi_edit)
{
  return avi_edit ? avi_edit->priority : 0;
}


/*  ja:AVIの言語を取得する
    avi_edit,AVI編集ハンドル
         RET,言語                                                           */
guint16
avi_edit_get_language (AviEdit *avi_edit)
{
  return avi_edit ? avi_edit->language : 0;
}


/*  ja:AVIの名前を取得する
    avi_edit,AVI編集ハンドル
         RET,名前                                                           */
const gchar *
avi_edit_get_name (AviEdit *avi_edit)
{
  return avi_edit ? avi_edit->name : NULL;
}


/*  ja:AVIの幅を取得する
    avi_edit,AVI編集ハンドル
         RET,幅,0:エラー                                                    */
gint32
avi_edit_get_width (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->bmih ? bmih_get_width (avi_edit->bmih) : 0;
}


/*  ja:AVIの高さを取得する
    avi_edit,AVI編集ハンドル
         RET,高さ,0:エラー                                                  */
gint32
avi_edit_get_height (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->bmih ? bmih_get_height (avi_edit->bmih) : 0;
}


/*  ja:AVIの色数を取得する
    avi_edit,AVI編集ハンドル
         RET,色数,0:エラー                                                  */
guint16
avi_edit_get_bit_count (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->bmih ? bmih_get_bit_count (avi_edit->bmih) : 0;
}


/*  ja:AVIのX解像度を取得する
    avi_edit,AVI編集ハンドル
         RET,X解像度,-1:エラー                                              */
gint32
avi_edit_get_x_pixels_per_meter (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->bmih
                        ? bmih_get_x_pixels_per_meter (avi_edit->bmih) : -1;
}


/*  ja:AVIのY解像度を取得する
    avi_edit,AVI編集ハンドル
         RET,Y解像度,-1:エラー                                              */
gint32
avi_edit_get_y_pixels_per_meter (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->bmih
                        ? bmih_get_y_pixels_per_meter (avi_edit->bmih) : -1;
}


/*  ja:AVIのレートを取得する
    avi_edit,AVI編集ハンドル
         RET,レート,0:エラー                                                */
guint32
avi_edit_get_samples_per_sec (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->wfx
                                ? wfx_get_samples_per_sec (avi_edit->wfx) : 0;
}


/*  ja:AVIのチャンネルを取得する
    avi_edit,AVI編集ハンドル
         RET,チャンネル,0:エラー                                            */
guint16
avi_edit_get_channels (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->wfx ? wfx_get_channels (avi_edit->wfx) : 0;
}


/*  ja:AVIのビットを取得する
    avi_edit,AVI編集ハンドル
         RET,ビット,0:エラー                                                */
guint16
avi_edit_get_bits_per_sample (AviEdit *avi_edit)
{
  return avi_edit && avi_edit->wfx
                                ? wfx_get_bits_per_sample (avi_edit->wfx) : 0;
}


/*  ja:AVIのRateを設定する
    avi_edit,AVI編集ハンドル
        rate,Rate                                                           */
void
avi_edit_set_rate (AviEdit       *avi_edit,
                   const guint32  rate)
{
  if (avi_edit && rate > 0)
    avi_edit->rate = rate;
}


/*  ja:AVIのScaleを設定する
    avi_edit,AVI編集ハンドル
       scale,Scale                                                          */
void
avi_edit_set_scale (AviEdit       *avi_edit,
                    const guint32  scale)
{
  if (avi_edit && scale > 0)
    avi_edit->scale = scale;
}


/*  ja:AVIの優先度を設定する
    avi_edit,AVI編集ハンドル
    priority,優先度                                                         */
void
avi_edit_set_priority (AviEdit       *avi_edit,
                       const guint16  priority)
{
  if (avi_edit)
    avi_edit->priority = priority;
}


/*  ja:AVIの言語を設定する
    avi_edit,AVI編集ハンドル
    language,言語                                                           */
void
avi_edit_set_language (AviEdit       *avi_edit,
                       const guint16  language)
{
  if (avi_edit)
    avi_edit->language = language;
}


/*  ja:AVIの名前を設定する
    avi_edit,AVI編集ハンドル
        name,名前                                                           */
void
avi_edit_set_name (AviEdit     *avi_edit,
                   const gchar *name)
{
  if (avi_edit)
    {
      g_free (avi_edit->name);
      avi_edit->name = name && name[0] != '\0' ? g_strdup (name) : NULL;
    }
}


/*  ja:AVIの幅を設定する
    avi_edit,AVI編集ハンドル
       width,幅                                                             */
void
avi_edit_set_width (AviEdit      *avi_edit,
                    const gint32  width)
{
  if (avi_edit && avi_edit->bmih)
    {
      bmih_set_width (avi_edit->bmih, width);
      bmih_set_size_image (avi_edit->bmih, bm_image_bytes (avi_edit->bmih));
    }
}


/*  ja:AVIの高さを設定する
    avi_edit,AVI編集ハンドル
      height,高さ                                                           */
void
avi_edit_set_height (AviEdit      *avi_edit,
                     const gint32  height)
{
  if (avi_edit && avi_edit->bmih)
    {
      bmih_set_height (avi_edit->bmih, height);
      bmih_set_size_image (avi_edit->bmih, bm_image_bytes (avi_edit->bmih));
    }
}


/*  ja:AVIの色数を設定する
     avi_edit,AVI編集ハンドル
    bit_count,色数                                                          */
void
avi_edit_set_bit_count (AviEdit       *avi_edit,
                        const guint16  bit_count)
{
  if (avi_edit && avi_edit->bmih)
    {
      bmih_set_bit_count (avi_edit->bmih, bit_count);
      avi_edit->bmih = g_realloc (avi_edit->bmih,
                                            bm_header_bytes (avi_edit->bmih));
    }
}


/*  ja:AVIのX解像度を設定する
              avi_edit,AVI編集ハンドル
    x_pixels_per_meter,X解像度                                              */
void
avi_edit_set_x_pixels_per_meter (AviEdit      *avi_edit,
                                 const gint32  x_pixels_per_meter)
{
  if (avi_edit && avi_edit->bmih)
    bmih_set_x_pixels_per_meter (avi_edit->bmih, x_pixels_per_meter);
}


/*  ja:AVIのY解像度を設定する
              avi_edit,AVI編集ハンドル
    y_pixels_per_meter,Y解像度                                              */
void
avi_edit_set_y_pixels_per_meter (AviEdit      *avi_edit,
                                 const gint32  y_pixels_per_meter)
{
  if (avi_edit && avi_edit->bmih)
    bmih_set_y_pixels_per_meter (avi_edit->bmih, y_pixels_per_meter);
}


/*  ja:AVIのレートを設定する
           avi_edit,AVI編集ハンドル
    samples_per_sec,レート                                                  */
void
avi_edit_set_samples_per_sec (AviEdit       *avi_edit,
                              const guint32  samples_per_sec)
{
  if (avi_edit && avi_edit->wfx)
    {
      wfx_set_samples_per_sec (avi_edit->wfx, samples_per_sec);
      wfx_set_average_bytes_per_sec (avi_edit->wfx, samples_per_sec
                            * (wfx_get_channels (avi_edit->wfx)
                            * wfx_get_bits_per_sample (avi_edit->wfx) / 8));
      avi_edit_set_rate (avi_edit, samples_per_sec);
      avi_edit_set_scale (avi_edit, 1);
    }
}


/*  ja:AVIのチャンネルを設定する
    avi_edit,AVI編集ハンドル
    channels,チャンネル                                                     */
void
avi_edit_set_channels (AviEdit       *avi_edit,
                       const guint16  channels)
{
  if (avi_edit && avi_edit->wfx)
    {
      guint16 block_align;

      wfx_set_channels (avi_edit->wfx, channels);
      block_align = channels * wfx_get_bits_per_sample (avi_edit->wfx) / 8;
      wfx_set_average_bytes_per_sec (avi_edit->wfx,
                        wfx_get_samples_per_sec (avi_edit->wfx) * block_align);
      wfx_set_block_align (avi_edit->wfx, block_align);
    }
}


/*  ja:AVIのビットを設定する
           avi_edit,AVI編集ハンドル
    bits_per_sample,ビット                                                  */
void
avi_edit_set_bits_per_sample (AviEdit       *avi_edit,
                              const guint16  bits_per_sample)
{
  if (avi_edit && avi_edit->wfx)
    {
      guint16 block_align;

      wfx_set_bits_per_sample (avi_edit->wfx, bits_per_sample);
      block_align = wfx_get_channels (avi_edit->wfx) * bits_per_sample / 8;
      wfx_set_average_bytes_per_sec (avi_edit->wfx,
                        wfx_get_samples_per_sec (avi_edit->wfx) * block_align);
      wfx_set_block_align (avi_edit->wfx, block_align);
    }
}


/*  ja:AVIのタイプを求める
    avi_edit,AVI編集ハンドル
         RET,AVI_TYPE_VIDEO/AVI_TYPE_AUDIO/AVI_TYPE_UNKNOWN                 */
gint
avi_edit_type (AviEdit *avi_edit)
{
  return avi_edit ? avi_edit->type : AVI_TYPE_UNKNOWN;
}


/*  ja:AVIの長さを求める
    avi_edit,AVI編集ハンドル
         RET,サンプル数                                                     */
gint
avi_edit_length (AviEdit *avi_edit)
{
  return avi_edit ? avi_edit->length : 0;
}


/******************************************************************************
*                                                                             *
* ja:時間関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:AVIのサンプル番号から時間を求める
    avi_edit,AVI編集ハンドル
      sample,サンプル番号
         RET,時間(ミリ秒)                                                   */
gint
avi_time_from_sample (AviEdit    *avi_edit,
                      const gint  sample)
{
  switch (avi_edit_type (avi_edit))
    {
      case AVI_TYPE_VIDEO:
        return (glonglong)sample * avi_edit_get_scale (avi_edit) * 1000
                                                / avi_edit_get_rate (avi_edit);
      case AVI_TYPE_AUDIO:
        return (glonglong)sample * 1000 / AVI_PCM_SAMPLES_PER_SEC;
    }
  return 0;
}


/*  ja:AVIの時間からサンプル番号を求める
    avi_edit,AVI編集ハンドル
        time,時間(ミリ秒)
         RET,サンプル番号                                                   */
gint
avi_time_to_sample (AviEdit    *avi_edit,
                    const gint  time)
{
  switch (avi_edit_type (avi_edit))
    {
      case AVI_TYPE_VIDEO:
        return (glonglong)time * avi_edit_get_rate (avi_edit)
                        / ((glonglong)avi_edit_get_scale (avi_edit) * 1000);
      case AVI_TYPE_AUDIO:
        return (glonglong)time * AVI_PCM_SAMPLES_PER_SEC / 1000;
    }
  return 0;
}


/*  ja:AVIの長さを時間で求める
    avi_edit,AVI編集ハンドル
         RET,時間(ミリ秒)                                                   */
gint
avi_time_length (AviEdit *avi_edit)
{
  return avi_time_from_sample (avi_edit, avi_edit_length (avi_edit));
}


/*  ja:AVIのサンプル番号から対応するサンプル番号を求める
    avi_edit0,対応元のAVI編集ハンドル
       sample,対応元のサンプル番号
    avi_edit1,対応先のAVI編集ハンドル
          RET,対応先のサンプル番号                                          */
gint
avi_time_sync_sample (AviEdit    *avi_edit0,
                      const gint  sample,
                      AviEdit    *avi_edit1)
{
  switch (avi_edit_type (avi_edit0))
    {
      case AVI_TYPE_VIDEO:
        switch (avi_edit_type (avi_edit1))
          {
            case AVI_TYPE_VIDEO:
              return (glonglong)sample
                        * avi_edit_get_scale (avi_edit0)
                        * avi_edit_get_rate (avi_edit1)
                        / ((glonglong)avi_edit_get_rate (avi_edit0)
                                            * avi_edit_get_scale (avi_edit1));
            case AVI_TYPE_AUDIO:
              return (glonglong)sample * avi_edit_get_scale (avi_edit0)
                    * AVI_PCM_SAMPLES_PER_SEC / avi_edit_get_rate (avi_edit0);
          }
        break;
      case AVI_TYPE_AUDIO:
        switch (avi_edit_type (avi_edit1))
          {
            case AVI_TYPE_VIDEO:
              return (glonglong)sample * avi_edit_get_rate (avi_edit1)
                                / ((glonglong)avi_edit_get_scale (avi_edit1)
                                                    * AVI_PCM_SAMPLES_PER_SEC);
            case AVI_TYPE_AUDIO:
              return sample;
          }
    }
  return 0;
}
