/*
    Video maid
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "file.h"
#include "sigfile.h"
#include "sigmain.h"
#include "size.h"
#include "thread.h"
#include "misc/fileio.h"
#include "orz/orzhistory.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* ja:ファイル入力関数群                                                       *
*                                                                             *
******************************************************************************/
/*  ja:AVIファイルを開く
     avi,AVI編集構造体
    file,ファイル名                                                         */
static void
file_open_avi (AviEdit     *avi_edit[],
               const gchar *file)
{
  AviEdit **avi_edit_tmp;

  avi_edit[0] = avi_edit[1] = NULL;
  avi_edit_tmp = avi_edit_open (file);
  if (avi_edit_tmp)
    {
      gint i;

      for (i = 0; avi_edit_tmp[i]; i++)
        switch (avi_edit_type (avi_edit_tmp[i]))
          {
            case AVI_TYPE_VIDEO:
              if (!avi_edit[0])
                avi_edit[0] = avi_edit_tmp[i];
              else
                avi_edit_close (avi_edit_tmp[i]);
              break;
            case AVI_TYPE_AUDIO:
              if (!avi_edit[1])
                avi_edit[1] = avi_edit_tmp[i];
              else
                avi_edit_close (avi_edit_tmp[i]);
              break;
            default:
              avi_edit_close (avi_edit_tmp[i]);
          }
      g_free (avi_edit_tmp);
    }
}


/*  ja:ウインドウを開く
    vmaid,ウインドウ情報
     file,ファイル名
      RET,TRUE:正常終了,FALSE:エラー                                        */
gboolean
file_open_window (VmaidWindow *vmaid,
                  const gchar *file)
{
  gchar *filename;
  gint page_num;
  GtkWidget *xbox, *vbox;
  const static gchar *rot47 = "\0"
        "%9:D AC@8C2> 4@>6D H:E9 }~ (p##p}%*[ E@ E96 6IE6?E A6C>:EE65 3J =2H]";

  if (file)
    {
      filename = fileio_get_full_path (file);
    }
  else
    {
      static gint fcount = 0; /* ja:新規ファイルの数 */

      /* ja:新規 */
      filename = g_strdup_printf ("new%04d.avi", fcount++ % 1000);
    }
  /* ja:表示エリア */
  vmaid->drawing = gtk_drawing_area_new ();
  g_signal_connect (G_OBJECT (vmaid->drawing), "focus-in-event",
                                G_CALLBACK (signal_focus_in), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "focus-out-event",
                                G_CALLBACK (signal_focus_out), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "realize",
                                G_CALLBACK (signal_realize), NULL);
  g_signal_connect (G_OBJECT (vmaid->drawing), "configure-event",
                                G_CALLBACK (signal_config), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "expose-event",
                                G_CALLBACK (signal_expose), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "button-press-event",
                                G_CALLBACK (signal_button_press_draw), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "motion-notify-event",
                                G_CALLBACK (signal_motion_notify), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "button-release-event",
                                G_CALLBACK (signal_button_release), vmaid);
  g_signal_connect (G_OBJECT (vmaid->drawing), "scroll-event",
                                G_CALLBACK (signal_scroll), vmaid);
  g_signal_connect_after (G_OBJECT (vmaid->drawing), "key-press-event",
                                G_CALLBACK (signal_key_press), vmaid);
  g_signal_connect (GTK_OBJECT (vmaid->drawing), "destroy",
                                G_CALLBACK (signal_destroy_draw), vmaid);
  gtk_widget_add_events (vmaid->drawing, GDK_POINTER_MOTION_MASK
                            | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
                            | GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
                            | GDK_FOCUS_CHANGE_MASK);
  gtk_widget_set_size_request (vmaid->drawing,
                    vmaid->width + (vmaid->selfrm ? vmaid->width * 2 + 4 : 2),
                        system_font_height * 2 + ((vmaid->avi_edit[0] ? 1 : 0)
                            + (vmaid->avi_edit[1] ? 1 : 0)) * vmaid->height);
  GTK_WIDGET_SET_FLAGS (vmaid->drawing, GTK_CAN_FOCUS);
  /* ja:スクロールバー */
  vmaid->hscroll = gtk_hscrollbar_new
                    (GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 0, 1, 1, 1)));
  /* ja:ボックス */
  xbox = gtk_hbox_new (FALSE, 0);
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), vmaid->drawing, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), vmaid->hscroll, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (xbox), vbox, *rot47, *rot47, 0);
  /* ja:表示 */
  page_num = orz_mdi_append_page (ORZ_MDI (mdi),
                                        xbox, filename, file != NULL, vmaid);
  gtk_widget_show_all (xbox);
  gtk_notebook_set_page (GTK_NOTEBOOK (mdi), page_num);
  gtk_widget_grab_focus (vmaid->drawing);
  g_free (filename);
  return TRUE;
}


/*  ja:ファイルを開く(独立スレッド)
    file_open,Open情報
     RET,ウインドウ情報                                                     */
VmaidWindow *
file_open_edit (FileOpen *file_open)
{
  gint i;
#ifdef USE_THREAD
  GThread *id;
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */
  VmaidWindow *vmaid;

#ifdef USE_THREAD
  id = g_thread_self ();
#endif /* USE_THREAD */
  thread_insert (id, THREAD_MODE_OPEN, file_open ? file_open->file : NULL);

  vmaid = g_malloc0 (sizeof (VmaidWindow));
  vmaid->selfrm = default_selfrm;
  vmaid->select.stream = -1;
  vmaid->select.frame = -1;
  if (file_open && file_open->file)
    {
      gchar *format = NULL;
      gint j, value, max = 0;

      file_open_avi (vmaid->avi_edit, file_open->file);
      if ((vmaid->avi_edit[0] || vmaid->avi_edit[1]) && file_open->merge)
        {
          gchar *name;

          /* ja:連番 */
          name = g_path_get_basename (file_open->file);
          for (i = g_strlen (name) - 1; i >= 0; i--)
            if ('0' <= name[i] && name[i] <= '9')
              break;
          if (i >= 0)
            {
              gchar *dir, *head, *foot;

              dir = g_path_get_dirname (file_open->file);
              head = g_strdup (name);
              foot = name + i + 1;
              for (j = i - 1; j >= 0; j--)
                if (head[j] < '0' || '9' < head[j])
                  break;
              misc_str_to_val (&value, head + j + 1, 10, FALSE);
              head[j + 1] = '\0';
              format = g_strdup_printf ("%s/%s%%0%dd%s",
                                                    dir, head, i - j, foot);
              g_free (dir);
              g_free (head);
              max = 1;
              while (j < i)
                {
                  max *= 10;
                  j++;
                }
            }
          g_free (name);
        }
      if (format)
        {
          gboolean result = TRUE;
          gchar *file;
          AviEdit *avi_edit_old[2], *avi_edit_tmp[2];

          /* ja:連番 */
          for (i = value - 1; i >= 0; i--)
            {
              /* ja:連番前 */
              file = g_strdup_printf (format, i);
              file_open_avi (avi_edit_tmp, file);
              g_free (file);
              if (!avi_edit_tmp[0] && !avi_edit_tmp[1])
                break;
              for (j = 0; j < 2; j++)
                if (vmaid->avi_edit[j])
                  {
                    avi_edit_old[j] = avi_edit_dup (vmaid->avi_edit[j]);
                    if (!avi_edit_paste (vmaid->avi_edit[j],
                                                        0, avi_edit_tmp[j]))
                      result = FALSE;
                    }
                  else
                    {
                      avi_edit_old[j] = NULL;
                      vmaid->avi_edit[j] = avi_edit_tmp[j];
                    }
              if (result)
                {
                  for (j = 0; j < 2; j++)
                    if (avi_edit_old[j])
                      avi_edit_close (avi_edit_old[j]);
                }
              else
                {
                  for (j = 0; j < 2; j++)
                    {
                      if (vmaid->avi_edit[j])
                        avi_edit_close (vmaid->avi_edit[j]);
                      vmaid->avi_edit[j] = avi_edit_old[j];
                    }
                  break;
                }
            }
          for (i = value + 1, result = TRUE; i < max && result; i++)
            {
              /* ja:連番後 */
              file = g_strdup_printf (format, i);
              file_open_avi (avi_edit_tmp, file);
              g_free (file);
              if (!avi_edit_tmp[0] && !avi_edit_tmp[1])
                break;
              for (j = 0; j < 2; j++)
                if (vmaid->avi_edit[j])
                  {
                    avi_edit_old[j] = avi_edit_dup (vmaid->avi_edit[j]);
                    if (!avi_edit_paste (vmaid->avi_edit[j],
                        avi_edit_length (vmaid->avi_edit[j]), avi_edit_tmp[j]))
                      result = FALSE;
                  }
                else
                  {
                    avi_edit_old[j] = NULL;
                    vmaid->avi_edit[j] = avi_edit_tmp[j];
                  }
              if (result)
                {
                  for (j = 0; j < 2; j++)
                    if (avi_edit_old[j])
                      avi_edit_close (avi_edit_old[j]);
                }
              else
                {
                  for (j = 0; j < 2; j++)
                    {
                      if (vmaid->avi_edit[j])
                        avi_edit_close (vmaid->avi_edit[j]);
                      vmaid->avi_edit[j] = avi_edit_old[j];
                    }
                }
            }
        }
    }

  if (!thread_idling (id, 100))
    {
      g_free (file_open->file);
      g_free (file_open);
      for (i = 0; i < 2; i++)
        if (vmaid->avi_edit[i])
          avi_edit_close (vmaid->avi_edit[i]);
      g_free (vmaid);
      return NULL;
    }

  if (vmaid->avi_edit[0])
    {
      size_set_scale (default_view, &vmaid->width, &vmaid->height,
                                    avi_edit_get_width (vmaid->avi_edit[0]),
                                    avi_edit_get_height (vmaid->avi_edit[0]));
      vmaid->rate = avi_edit_get_rate (vmaid->avi_edit[0]);
      vmaid->scale = avi_edit_get_scale (vmaid->avi_edit[0]);
    }
  else
    {
      size_set_scale (default_view, &vmaid->width, &vmaid->height, -1, -1);
      vmaid->rate = default_rate;
      vmaid->scale = 1;
      if (vmaid->avi_edit[1])
        vmaid->cursor.stream = 1;
    }

#ifdef USE_THREAD
  gdk_threads_enter ();
#endif /* USE_THREAD */
  file_open_window (vmaid, file_open ? file_open->file : NULL);
  /* ja:新規作成ではなく、ファイルが存在する時には履歴に加える */
  if (file_open && g_file_test (file_open->file, G_FILE_TEST_EXISTS))
    orz_history_add_file (ORZ_HISTORY (history), file_open->file);
#ifdef USE_THREAD
  gdk_threads_leave ();
#endif /* USE_THREAD */

  if (file_open)
    {
      g_free (file_open->file);
      g_free (file_open);
    }
  thread_delete (id);

  return vmaid;
}


/******************************************************************************
*                                                                             *
* ja:ファイル出力関数群                                                       *
*                                                                             *
******************************************************************************/
static gboolean
file_save_edit_callback (gint percent, gpointer user_data)
{
#ifdef USE_THREAD
  return thread_idling (g_thread_self (), percent);
#else /* not USE_THREAD */
  return thread_idling (THREAD_ID_STANDARD, percent);
#endif /* not USE_THREAD */
}


/*  ja:AVIファイルを保存する(独立スレッド)
    file_save,Save情報
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
file_save_edit (FileSave *file_save)
{
  gboolean result;
  gint i;
#ifdef USE_THREAD
  GThread *id;
  id = g_thread_self ();
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */

  thread_insert (id, THREAD_MODE_SAVE, file_save->file);

  result = avi_save_with_options (file_save->avi_edit, file_save->file,
                        file_save->avi_save, file_save_edit_callback, NULL);
  g_free (file_save->file);
  for (i = 0; i < 2; i++)
    if (file_save->avi_edit[i])
      avi_edit_close (file_save->avi_edit[i]);
  avi_save_free (file_save->avi_save);
  g_free (file_save);

  if (!result)
    while (thread_idling (id, -1));

  thread_delete (id);
  return result;
}
