/*
    avicore
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#ifndef __AVICORE_H__
#define __AVICORE_H__


#include "common.h"
#include "bitmap.h"
#include "wave.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/******************************************************************************
*                                                                             *
* ja:編集構造体                                                               *
*                                                                             *
******************************************************************************/
#define AVI_TYPE_UNKNOWN    0
#define AVI_TYPE_VIDEO      1
#define AVI_TYPE_AUDIO      2
#define AVI_FILE_UNKNOWN    0
#define AVI_FILE_AVI        1
#define AVI_FILE_GSR        2
#define AVI_FILE_BITMAP     3
#define AVI_FILE_WAVE       4
#define AVI_FILE_SCENARIO   5


typedef struct _AviEdit AviEdit;
typedef struct _AviFrame AviFrame;
typedef struct _AviPcm AviPcm;
typedef struct _AviSave AviSave;
typedef struct _AviLanguage
{
  const gchar *name;
  guint16 id;
} AviLanguage;


/******************************************************************************
*                                                                             *
* ja:初期化関数                                                               *
*                                                                             *
******************************************************************************/
/*  ja:AVIを初期化する                                                      */
gboolean
avi_init (void);


/*  ja:AVIを終了する                                                        */
gboolean
avi_exit (void);


/******************************************************************************
*                                                                             *
* ja:フォーマット関数                                                         *
*                                                                             *
******************************************************************************/
/*  ja:AVIのRateを取得する
    avi_edit,AVI編集ハンドル
         RET,Rate                                                           */
guint32
avi_edit_get_rate (AviEdit *avi_edit);


/*  ja:AVIのScaleを取得する
    avi_edit,AVI編集ハンドル
         RET,Scale                                                          */
guint32
avi_edit_get_scale (AviEdit *avi_edit);


/*  ja:AVIの優先度を取得する
    avi_edit,AVI編集ハンドル
         RET,優先度                                                         */
guint16
avi_edit_get_priority (AviEdit *avi_edit);


/*  ja:AVIの言語を取得する
    avi_edit,AVI編集ハンドル
         RET,言語                                                           */
guint16
avi_edit_get_language (AviEdit *avi_edit);


/*  ja:AVIの名前を取得する
    avi_edit,AVI編集ハンドル
         RET,名前                                                           */
const gchar *
avi_edit_get_name (AviEdit *avi_edit);


/*  ja:AVIの幅を取得する
    avi_edit,AVI編集ハンドル
         RET,幅,0:エラー                                                    */
gint32
avi_edit_get_width (AviEdit *avi_edit);


/*  ja:AVIの高さを取得する
    avi_edit,AVI編集ハンドル
         RET,高さ,0:エラー                                                  */
gint32
avi_edit_get_height (AviEdit *avi_edit);


/*  ja:AVIの色数を取得する
    avi_edit,AVI編集ハンドル
         RET,色数,0:エラー                                                  */
guint16
avi_edit_get_bit_count (AviEdit *avi_edit);


/*  ja:AVIのX解像度を取得する
    avi_edit,AVI編集ハンドル
         RET,X解像度,-1:エラー                                              */
gint32
avi_edit_get_x_pixels_per_meter (AviEdit *avi_edit);


/*  ja:AVIのY解像度を取得する
    avi_edit,AVI編集ハンドル
         RET,Y解像度,-1:エラー                                              */
gint32
avi_edit_get_y_pixels_per_meter (AviEdit *avi_edit);


/*  ja:AVIのレートを取得する
    avi_edit,AVI編集ハンドル
         RET,レート,0:エラー                                                */
guint32
avi_edit_get_samples_per_sec (AviEdit *avi_edit);


/*  ja:AVIのチャンネルを取得する
    avi_edit,AVI編集ハンドル
         RET,チャンネル,0:エラー                                            */
guint16
avi_edit_get_channels (AviEdit *avi_edit);


/*  ja:AVIのビットを取得する
    avi_edit,AVI編集ハンドル
         RET,ビット,0:エラー                                                */
guint16
avi_edit_get_bits_per_sample (AviEdit *avi_edit);


/*  ja:AVIのRateを設定する
    avi_edit,AVI編集ハンドル
        rate,Rate                                                           */
void
avi_edit_set_rate (AviEdit *avi_edit,
                   guint32  rate);


/*  ja:AVIのScaleを設定する
    avi_edit,AVI編集ハンドル
       scale,Scale                                                          */
void
avi_edit_set_scale (AviEdit *avi_edit,
                    guint32  scale);


/*  ja:AVIの優先度を設定する
    avi_edit,AVI編集ハンドル
    priority,優先度                                                         */
void
avi_edit_set_priority (AviEdit *avi_edit,
                       guint16  priority);


/*  ja:AVIの言語を設定する
    avi_edit,AVI編集ハンドル
    language,言語                                                           */
void
avi_edit_set_language (AviEdit *avi_edit,
                       guint16  language);


/*  ja:AVIの名前を設定する
    avi_edit,AVI編集ハンドル
        name,名前                                                           */
void
avi_edit_set_name (AviEdit     *avi_edit,
                   const gchar *name);


/*  ja:AVIの幅を設定する
    avi_edit,AVI編集ハンドル
       width,幅                                                             */
void
avi_edit_set_width (AviEdit *avi_edit,
                    gint32   width);


/*  ja:AVIの高さを設定する
    avi_edit,AVI編集ハンドル
      height,高さ                                                           */
void
avi_edit_set_height (AviEdit *avi_edit,
                     gint32   height);


/*  ja:AVIの色数を設定する
     avi_edit,AVI編集ハンドル
    bit_count,色数                                                          */
void
avi_edit_set_bit_count (AviEdit *avi_edit,
                        guint16  bit_count);


/*  ja:AVIのX解像度を設定する
              avi_edit,AVI編集ハンドル
    x_pixels_per_meter,X解像度                                              */
void
avi_edit_set_x_pixels_per_meter (AviEdit *avi_edit,
                                 gint32   x_pixels_per_meter);


/*  ja:AVIのY解像度を設定する
              avi_edit,AVI編集ハンドル
    y_pixels_per_meter,Y解像度                                              */
void
avi_edit_set_y_pixels_per_meter (AviEdit *avi_edit,
                                 gint32   y_pixels_per_meter);


/*  ja:AVIのレートを設定する
           avi_edit,AVI編集ハンドル
    samples_per_sec,レート                                                  */
void
avi_edit_set_samples_per_sec (AviEdit *avi_edit,
                              guint32  samples_per_sec);


/*  ja:AVIのチャンネルを設定する
    avi_edit,AVI編集ハンドル
    channels,チャンネル                                                     */
void
avi_edit_set_channels (AviEdit *avi_edit,
                       guint16  channels);


/*  ja:AVIのビットを設定する
           avi_edit,AVI編集ハンドル
    bits_per_sample,ビット                                                  */
void
avi_edit_set_bits_per_sample (AviEdit *avi_edit,
                              guint16  bits_per_sample);


/*  ja:AVIのタイプを求める
    avi_edit,AVI編集ハンドル
         RET,AVI_TYPE_VIDEO/AVI_TYPE_AUDIO/AVI_TYPE_UNKNOWN                 */
gint
avi_edit_type (AviEdit *avi_edit);


/*  ja:AVIの長さを求める
    avi_edit,AVI編集ハンドル
         RET,サンプル数                                                     */
gint
avi_edit_length (AviEdit *avi_edit);


/******************************************************************************
*                                                                             *
* ja:時間関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:AVIのサンプル番号から時間を求める
    avi_edit,AVI編集ハンドル
      sample,サンプル番号
         RET,時間(ミリ秒)                                                   */
gint
avi_time_from_sample (AviEdit    *avi_edit,
                      const gint  sample);


/*  ja:AVIの時間からサンプル番号を求める
    avi_edit,AVI編集ハンドル
        time,時間(ミリ秒)
         RET,サンプル番号                                                   */
gint
avi_time_to_sample (AviEdit    *avi_edit,
                    const gint  time);


/*  ja:AVIの長さを時間で求める
    avi_edit,AVI編集ハンドル
         RET,時間(ミリ秒)                                                   */
gint
avi_time_length (AviEdit *avi_edit);


/*  ja:AVIのサンプル番号から対応するサンプル番号を求める
    avi_edit0,対応元のAVI編集ハンドル
       sample,対応元のサンプル番号
    avi_edit1,対応先のAVI編集ハンドル
          RET,対応先のサンプル番号                                          */
gint
avi_time_sync_sample (AviEdit    *avi_edit0,
                      const gint  sample,
                      AviEdit    *avi_edit1);


/******************************************************************************
*                                                                             *
* ja:AVI言語関数                                                              *
*                                                                             *
******************************************************************************/
/*  ja:言語IDと名前を求める
    RET,言語IDと名前                                                        */
const AviLanguage *
avi_lang_get_struct (void);


/*  ja:言語IDから名前を求める
     ID,言語ID
    RET,名前                                                                */
const gchar *
avi_lang_get_name (const guint16 id);


/******************************************************************************
*                                                                             *
* ja:AVI入力関数                                                              *
*                                                                             *
******************************************************************************/
/*  ja:ファイルを判別する
    file,ファイル名
     RET,AVI_TYPE_*                                                         */
gint
avi_main_type (const gchar *file);


/*  ja:ストリームの数を取得する
    file,ファイル名
     RET,ストリームの数,0:エラー                                            */
gint
avi_main_streams (const gchar *file);


/*  ja:AVIをファイルから開く
    file,ファイル名
     RET,AVI編集ハンドル,NULL:エラー                                        */
AviEdit **
avi_edit_open (const gchar *file);


/*  ja:AVIを閉じる
    avi_edit,AVI編集ハンドル
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_edit_close (AviEdit *avi_edit);


/*  ja:AVIを複製する
    avi_edit,AVI編集ハンドル
         RET,AVI編集ハンドル                                                */
#define avi_edit_dup(avi_edit)                                              \
                            avi_edit_copy(avi_edit,0,avi_edit_length(avi_edit))


/******************************************************************************
*                                                                             *
* ja:フレーム関数                                                             *
*                                                                             *
******************************************************************************/
/*  ja:DIBに展開されたフレームを開く
    avi_edit,AVI編集ハンドル
         RET,AVIフレーム構造体,NULL:エラー                                  */
AviFrame *
avi_frame_open (AviEdit *avi_edit);


/*  ja:DIBに展開されたフレームを閉じる
    avi_frame,AVIフレーム構造体
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_frame_close (AviFrame *avi_frame);


/*  ja:DIBに展開されたフレームを取得する
    avi_frame,AVIフレーム構造体
          pos,取得するフレーム(サンプル番号)
        width,幅
       height,高さ
    bit_count,色数
          RET,パックDIBへのポインタ,NULL:エラー                             */
BitmapInfoHeader *
avi_frame_get_bitmap (AviFrame   *avi_frame,
                      const gint  pos,
                      const gint  width,
                      const gint  height,
                      const gint  bit_count);


/*  ja:RAWに展開されたフレームを取得する
    avi_frame,AVIフレーム構造体
          pos,取得するフレーム(サンプル番号)
        width,幅
       height,高さ
          RET,RAWへのポインタ,NULL:エラー                               */
guint8 *
avi_frame_get_raw (AviFrame   *avi_frame,
                   const gint  pos,
                   const gint  width,
                   const gint  height);


/*  ja:フレームがキーフレームか判定する
    avi_edit,AVI編集ハンドル
         pos,取得するフレーム(サンプル番号)
         RET,TRUE:キーフレーム,FALSE:非キーフレーム                         */
gboolean
avi_frame_is_key (AviEdit    *avi_edit,
                  const gint  pos);


/******************************************************************************
*                                                                             *
* ja:PCM関数                                                             *
*                                                                             *
******************************************************************************/
/*  ja:PCMに展開されたオーディオを開く
    avi_edit,AVI編集ハンドル
         RET,AVIPCM構造体,NULL:エラー                                       */
AviPcm *
avi_pcm_open (AviEdit *avi_edit);


/*  ja:PCMに展開されたオーディオを閉じる
    avi_pcm,AVIPCM構造体
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
avi_pcm_close (AviPcm *avi_pcm);


/*  ja:PCMに展開されたオーディオを取得する
        avi_pcm,AVIPCM構造体
          start,取得する仮想サンプル番号
        samples,仮想サンプル数
        out_buf,PCMデータ
    out_samples,PCMデータの実サンプル数
            RET,TRUE:正常終了,FALSE:エラー                                  */
gboolean
avi_pcm_get_raw (AviPcm      *avi_pcm,
                 const gint   start,
                 const gsize  samples,
                 gpointer    *out_buf,
                 gsize       *out_samples);


/******************************************************************************
*                                                                             *
* ja:編集関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:AVIを削除する
    avi_edit,AVI編集ハンドル
       start,削除をはじめるサンプル番号
     samples,サンプル数
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_edit_delete (AviEdit    *avi_edit,
                 const gint  start,
                 const gint  samples);


/*  ja:AVIをコピーする
    avi_edit,AVI編集ハンドル
       start,コピーをはじめるサンプル番号
     samples,サンプル数
         RET,AVI編集ハンドル,NULL:エラー                                    */
AviEdit *
avi_edit_copy (AviEdit    *avi_edit,
               const gint  start,
               const gint  samples);


/*  ja:AVIを貼り付ける
    avi_dst,貼り付けられるAVI編集ハンドル
        pos,貼り付けるサンプル番号
    avi_src,貼り付けるAVI編集ハンドル
        RET,TRUE:正常終了,FALSE:エラー                                      */
gboolean
avi_edit_paste (AviEdit    *avi_dst,
                const gint  pos,
                AviEdit    *avi_src);


/******************************************************************************
*                                                                             *
* ja:シナリオオブジェクト関数                                                 *
*                                                                             *
******************************************************************************/
/*  ja:AVIをシナリオオブジェクトに変換する
      avi_edit,AVI編集ハンドルへのポインタ
    interleave,TRUE:インターリーブする,FALSE:インターリーブしない
           RET,シナリオオブジェクト,NULL:エラー                             */
gchar *
avi_edit_to_scenario (AviEdit        **avi_edit,
                      const gboolean   interleave);


/*  ja:シナリオオブジェクトからAVIを作る
    scenario,シナリオオブジェクト
         RET,AVI編集ハンドルへのポインタ,NULL:エラー                        */
AviEdit **
avi_edit_from_scenario (const gchar *scenario);


/******************************************************************************
*                                                                             *
* ja:保存関数                                                                 *
*                                                                             *
******************************************************************************/
/*  ja:保存用のオプションを解放する
    avi_save,オプション
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
avi_save_free (AviSave *avi_save);


/*  ja:保存されるファイルのタイプを返す
    avi_save,オプション
         RET,ファイルタイプ,-1:エラー                                       */
gint
avi_save_type (AviSave *avi_save);


/*  ja:すべて保存されるかチェックする
    avi_edit,AVI編集ハンドル
    avi_save,オプション
         RET,TRUE:すべて保存される,FALSE:すべて保存されない                 */
gboolean
avi_save_is_all (AviEdit **avi_edit,
                 AviSave  *avi_save);


/*  ja:ビットマップファイルに保存する
     avi_edit,AVI編集ハンドル
         file,ファイル名
        start,最初のフレーム
          end,最後のフレーム
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_bitmap (AviEdit      *avi_edit,
                 const gchar  *file,
                 const gint    start,
                 const gint    end,
                 gboolean    (*func)(gint, gpointer),
                 gpointer      user_data);


/*  ja:WAVEファイルに保存する
      avi_edit,AVI編集ハンドル
          file,ファイル名
           wfx,WaveFormatEx構造体へのポインタ(NULL:PCM)
    recompress,TRUE:再圧縮,FALSE:差分圧縮(wfxがNULLならば無視)
          func,コールバック関数
     user_data,データ
           RET,TRUE:正常終了,FALSE:エラー                                   */
gboolean
avi_save_wave (AviEdit             *avi_edit,
               const gchar         *file,
               const WaveFormatEx  *wfx,
               const gboolean       recompress,
               gboolean           (*func)(gint, gpointer),
               gpointer             user_data);


/*  ja:ファイルに保存する
     avi_edit,AVI編集ハンドルへのポインタ
         file,ファイル名
     avi_save,オプション
         func,コールバック関数
    user_data,データ
          RET,TRUE:正常終了,FALSE:エラー                                    */
gboolean
avi_save_with_options (AviEdit     **avi_edit,
                       const gchar  *file,
                       AviSave      *avi_save,
                       gboolean    (*func)(gint, gpointer),
                       gpointer      user_data);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __AVICORE_H__ */
