/*
    avicore
    copyright (c) 1998-2004 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef __CHUNK_H__
#define __CHUNK_H__


#include "common.h"
#include "misc/fileio.h"


#ifdef __cplusplus
extern "C" {
#endif


typedef struct ChunkList_Tag
{
  goffset offset;               /* ja:オフセット */
  guint32 id;                   /* ja:ID */
  guint32 size;                 /* ja:サイズ */
  guint32 form;                 /* ja:フォームタイプ */
} ChunkList;
typedef struct Chunk_Tag
{
  gint num;                     /* ja:リストの数 */
  ChunkList *list;
} Chunk;


/******************************************************************************
*                                                                             *
* ja:Chunk                                                                    *
*                                                                             *
******************************************************************************/
/*  ja:2つの文字からIDを作る
     c0,1文字目
     c1,2文字目
    RET,ID                                                                  */
#define make2cc(c0,c1)                                                      \
    (                                                                       \
        (guint16)(guint8)(c0)|((guint16)(guint8)(c1)<<8)                    \
    )


/*  ja:4つの文字からIDを作る
     c0,1文字目
     c1,2文字目
     c2,3文字目
     c3,4文字目
    RET,ID                                                                  */
#define make4cc(c0,c1,c2,c3)                                                \
    (                                                                       \
        (guint32)(guint8)(c0)|((guint32)(guint8)(c1)<<8)                    \
        |((guint32)(guint8)(c2)<<16)|((guint32)(guint8)(c3)<<24)            \
    )


/*  ja:4文字コードの文字を取得する
    fcc,4文字コード
      n,文字位置(0...3)
    RET,文字                                                                */
#define get4cc(fcc,n) ((guint8)(fcc)>>((n)*8))


/*  ja:4文字コードを小文字にする
    fcc,4文字コード
    RET,4文字コード                                                         */
#define tolower4cc(fcc)                                                     \
    make4cc(                                                                \
        g_ascii_tolower(get4cc(fcc,0)),                                     \
        g_ascii_tolower(get4cc(fcc,1)),                                     \
        g_ascii_tolower(get4cc(fcc,2)),                                     \
        g_ascii_tolower(get4cc(fcc,3))                                      \
    )


#define CK_DEF_RIFF             make4cc('R','I','F','F')
#define CK_DEF_LIST             make4cc('L','I','S','T')
#define CK_FORM_AVI             make4cc('A','V','I',' ')
#define CK_FORM_AVIX            make4cc('A','V','I','X')
#define CK_FORM_WAVE            make4cc('W','A','V','E')
#define CK_LIST_AVIHEADER       make4cc('h','d','r','l')
#define CK_LIST_STREAMHEADER    make4cc('s','t','r','l')
#define CK_LIST_AVIMOVIE        make4cc('m','o','v','i')
#define CK_LIST_OPENDML         make4cc('o','d','m','l')
#define CK_ID_AVIMAINHDR        make4cc('a','v','i','h')
#define CK_ID_STREAMHEADER      make4cc('s','t','r','h')
#define CK_ID_STREAMFORMAT      make4cc('s','t','r','f')
#define CK_ID_STREAMHANDLERDATA	make4cc('s','t','r','d')
#define CK_ID_STREAMNAME        make4cc('s','t','r','n')
#define CK_ID_AVINEWINDEX       make4cc('i','d','x','1')
#define CK_ID_AVIPADDING        make4cc('J','U','N','K')
#define CK_ID_WAVEFORMAT        make4cc('f','m','t',' ')
#define CK_ID_WAVEDATA          make4cc('d','a','t','a')
#define CK_ID_AVISUPERINDEX     make4cc('i','n','d','x')
#define CK_ID_OPENDML           make4cc('d','m','l','h')


/*  ja:ファイルディスクリターからチャンク構造体を作る
    fio,ファイルポインタ
    RET,チャンク構造体,NULL:エラー                                          */
#define chunk_open(fio) (fileio_seek(fio,0,FILEIO_SEEK_SET)!=-1             \
                                                ?g_malloc0(sizeof(Chunk)):NULL)


/*  ja:チャンク構造体を解放する
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
chunk_free (Chunk *ck);


/*  ja:チャンクの中に入る
    fio,ファイルポインタ
     ck,チャンク構造体
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
chunk_in (FileIO *fio,
          Chunk  *ck);


/*  ja:チャンクから出る
    fio,ファイルポインタ
     ck,チャンク構造体
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
chunk_out (FileIO *fio,
           Chunk  *ck);


/*  ja:次のチャンチャンクに移動する
    fio,ファイルポインタ
     ck,チャンク構造体
    RET,TRUE:正常終了,FALSE:エラー                                          */
gboolean
chunk_next (FileIO *fio,
            Chunk  *ck);


/*  ja:チャンクのIDを取得する
    fio,ファイルポインタ
    RET,ID,-1:エラー                                                        */
guint32
chunk_id (FileIO *fio);


/*  ja:チャンクのサイズを取得する
    fio,ファイルポインタ
    RET,サイズ,-1:エラー                                                    */
guint32
chunk_size (FileIO *fio);


/*  ja:チャンクのフォームを取得する
    fio,ファイルポインタ
    RET,フォーム,-1:エラー                                                  */
guint32
chunk_form (FileIO *fio);


#ifdef __cplusplus
}
#endif


#endif
