require 'generator/xml'

module Generator
	class Html < Xml
		def initialize(callbacks=nil)
			@callbacks = [callbacks].flatten.compact
			@escapes = {'<'=>'&lt;','>'=>'&gt;','&'=>'&amp;','"'=>'&quot;'}
			super()
		end
		def generate(node, outobj='')
			case node
			when nil then
				return outobj
			when NodeList then
				list = node
				0.upto(list.length-1) do |i|
					generate(list.item(i), outobj)
				end
				return outobj
			end
			name = node.nodeName.downcase.gsub(/[^a-zA-Z0-9]/, '_')
			func = "#{name}_onhtml"
			@callbacks.each do |callback|
				if callback.respond_to?(func) then
					res = callback.method(func).call(node, outobj)
					return outobj if res
				end
			end
			if self.respond_to?(func) then
				self.method(func).call(node, outobj)
			else
				generate(node.childNodes, outobj)
			end
			return outobj
		end
		def _default_element_onhtml(node, outobj='')
			outobj << begin_tag(node)
			generate(node.childNodes, outobj)
			outobj << end_tag(node)
			return outobj
		end
		def _default_empty_onhtml(node, outobj='')
			outobj << empty_tag(node)
			return outobj
		end
		def _document_onhtml(node, outobj='')
			generate(node.documentElement, outobj)
		end
		def _document_fragment_onhtml(node, outobj='')
			generate(node.childNodes, outobj)
		end
		def _text_onhtml(node, outobj='')
			outobj << escape(node.nodeValue) if node.nodeValue
			return outobj
		end
		def _comment_onhtml(node, outobj='')
			outobj << "<!--"+escape(node.nodeValue)+"-->"
			return outobj
		end
	private
		def begin_tag(node, attr={})
			case node
			when String then
				name = node
			else
				name = node.nodeName
				attr = Hash::new
				map = node.attributes
				if map then
					0.upto(map.length-1) do |i|
						attrnode = map.item(i)
						next unless /^[a-zA-Z]+$/ === attrnode.nodeName
						attr[attrnode.nodeName] = attrnode.nodeValue
					end
				end
			end
			tag = "<#{name}"
			attr.keys.sort.each do |param|
				tag << %Q[ #{param}="#{escape(attr[param].to_s)}"]
			end
			tag << '>'
			return tag
		end
		def end_tag(node)
			case node
			when String then
				name = node
			else
				name = node.nodeName
			end
			tag = "</#{name}>"
			return tag
		end
		def empty_tag(node, attr={})
			res = begin_tag(node, attr).sub(/>$/, ' />')
			return res
		end
		#
		# Alias each tags
		#
		def Html::define_default(tags, default)
			tags.each do |tag|
				alias_method("#{tag.strip}_onhtml", "_default_#{default}_onhtml")
			end
		end
		#
		# HTML tags definitions
		#
		Html::define_default(<<TAGS, "element")
a
address
applet
area
b
base
basefont
bgsound
big
blink
blockquote
body
br
button
caption
center
code
colgroup
dd
del
div
dl
dt
em
embed
fieldset
font
form
frame
frameset
h1
h2
h3
h4
h5
h6
head
hr
html
i
iframe
img
input
ins
kbd
label
legend
li
link
map
marquee
meta
nobr
noembed
noframes
object
ol
option
p
param
pre
rp
rt
ruby
s
script
select
small
span
strike
strong
style
sub
sup
table
tbody
td
textarea
tfoot
th
thead
title
tr
tt
u
ul
var
TAGS
		#
		# HTML empty tags definitions
		#
		Html::define_default(<<TAGS, "empty")
base
br
col
hr
img
input
link
meta
param
TAGS
	end
end
