=begin
	DOM mini is under DOM Level 1.
=end

unless defined?(DOMImplementation) then

require 'singleton'

class DOMImplementation
	include Singleton
end

class Node
	# NodeType
	ELEMENT_NODE       = 1
	ATTRIBUTE_NODE     = 2
	TEXT_NODE          = 3
	CDATA_SECTION_NODE = 4
	ENTITY_REFERENCE_NODE = 5
	ENTITY_NODE        = 6
	PROCESSING_INSTRUCTION_NODE = 7
	COMMENT_NODE       = 8
	DOCUMENT_NODE      = 9
	DOCUMENT_TYPE_NODE = 10
	DOCUMENT_FRAGMENT_NODE = 11
	NOTATION_NODE      = 12
	attr_reader \
		:nodeName,
		:nodeValue,
		:nodeType,
		:parentNode,
		:childNodes,
		:firstChild,
		:lastChild,
		:previousSibling,
		:nextSibling,
		:attributes,
		:ownerDocument
	def initialize(doc, name)
		@nodeName = name
		@nodeValue = nil
		@nodeType = 0
		@parentNode = nil
		@childNodes = NodeList::new
		@firstChild = nil
		@lastChild = nil
		@previousSibling = nil
		@nextSibling = nil
		@attributes = nil
		@ownerDocument = doc
	end
	def hasChildNodes
		@childNodes.length > 0
	end
	def appendChild(newNode)
		newNode.parentNode = self
		if DocumentFragment === newNode then
			append(newNode.childNodes)
		else
			append(newNode)
		end
		@childNodes.append(newNode)
		@firstChild = newNode unless @firstChild
		@lastChild = newNode
		return newNode
	end
	def removeChild(oldNode)
		@childNodes.remove(oldNode)
		remove(oldNode)
		return oldNode
	end
protected
	attr_writer \
		:nodeValue,
		:parentNode,
		:previousSibling,
		:nextSibling
private
	def append(newNode)
		case newNode
		when DocumentFragment then
			0.upto(@childNodes.length-1) do |i|
				append(@childNodes.item(i))
			end
		else
			if @lastNode then
				newNode.previousSibling = @lastNode
				@lastNode.nextSibling = newNode
			end
		end
		return newNode
	end
	def remove(oldNode)
		oldPrevious = oldNode.previousSibling
		oldNext = oldNode.nextSibling
		oldNode.previousSibling = oldNode.nextSibling = nil
		oldPrevious.nextSibling = oldNext if oldPrevious
		oldNext.previousSibling = oldPrevious if oldNext
		return oldNode
	end
end

class Document < Node
	attr_reader :implementation
	def initialize(implementation)
		super(self, '#document')
		@implementation = implementation
	end
	def documentElement
		0.upto(@childNodes.length-1) do |i|
			item = @childNodes.item(i)
			return item if Element === item
		end
		return nil
	end
	def createDocumentFragment
		DocumentFragment::new(self)
	end
	def createElement(name)
		Element::new(self, name)
	end
	def createAttribute(name)
		Attr::new(self, name)
	end
	def createTextNode(text)
		Text::new(self, text)
	end
	def createComment(comment)
		Comment::new(self, comment)
	end
	def getElementsByTagName(name)
		documentElement.getElementsByTagName(name)
	end
end

class DocumentFragment < Node
	def initialize(doc)
		super(doc, '#document-fragment')
	end
end

class Element < Node
	alias tagName nodeName
	def initialize(doc, name)
		super(doc, name)
		@nodeName = name
		@attributes = NamedNodeMap::new
	end
	def getAttribute(name)
		attr = @attributes.getNamedItem(name)
		attr ? attr.nodeValue : nil
	end
	def setAttribute(name, value)
		newAttr = ownerDocument.createAttribute(name)
		newAttr.value = value
		@attributes.setNamedItem(newAttr)
	end
	def getElementsByTagName(name)
		list = NodeList::new
		0.upto(@childNodes.length-1) do |i|
			item = @childNodes.item(i)
			list.append(item) if name == '*' or item.nodeName == name
			list.append(item.getElementsByTagName(name)) if Element === item
		end
		return list
	end
end

class Attr < Node
	def initialize(doc, name)
		super(doc, name)
	end
	alias name nodeName
	alias value nodeValue
	def value=(val)
		@nodeValue = val
	end
end

class CharacterData < Node
	def initialize(doc, name, data)
		super(doc, name)
		@nodeValue = data
	end
	alias data nodeValue
	def data=(val)
		@nodeValue = val
	end
end

class Text < CharacterData
	def initialize(doc, data)
		super(doc, '#text', data)
	end
end

class Comment < CharacterData
	def initialize(doc, data)
		super(doc, '#comment', data)
	end
end

class NodeList
	def initialize
		@nodeList = Array::new
	end
	def item(index)
		@nodeList[index]
	end
	def length
		@nodeList.length
	end
	def append(newNode)
		if NodeList === newNode then
			@nodeList += newNode.nodeList
		else
			@nodeList << newNode
		end
	end
	def remove(oldNode)
		idx = @nodeList.index(oldNode)
		raise "#{oldNode.nodeName} not found" unless idx
		@nodeList.delete_at(idx)
		return oldNode
	end
protected
	attr_reader :nodeList
end

class NamedNodeMap
	def initialize
		@namedNodeMap = Hash::new
	end
	def getNamedItem(name)
		@namedNodeMap[name]
	end
	def setNamedItem(arg)
		@namedNodeMap[arg.nodeName] = arg
	end
	def item(index)
		key = @namedNodeMap.keys.sort[index]
		@namedNodeMap[key]
	end
	def length
		@namedNodeMap.length
	end
end

end
