package net.osdn.util.jersey;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Writer;
import java.lang.reflect.Constructor;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ResourceInfo;
import javax.ws.rs.core.Context;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;

public abstract class Resource {

	private static final String SOURCE_PATH = "/WEB-INF/src";
	private static final Pattern COMMENT = Pattern.compile("/\\*{1,}(.*?)\\*{1,}/", Pattern.DOTALL);
	private static final ObjectMapper mapper;
	
	static {
		mapper = new JacksonConfig().getContext(null);
	}
	
	/** 指定したオブジェクトをJSON文字列に変換します。
	 * 
	 * @param value オブジェクト
	 * @return JSON文字列
	 * @throws IOException オブジェクトをJSON文字列に変換できなかった場合
	 */
	public static String toJsonString(Object value) throws IOException {
		return mapper.writeValueAsString(value);
	}
	
	/** 指定したJSON文字列を指定したクラスのオブジェクト
	 * 
	 * @param <T> 変換先の型
	 * @param json JSON文字列
	 * @param cls 変換先のクラス
	 * @return 変換されたオブジェクト
	 * @throws IOException JSON文字列をオブジェクトに変換できなかった場合
	 */
	public static <T> T fromJsonString(Class<T> cls, String json) throws IOException {
		if(json == null || json.equals("null")) {
			return null;
		}
		return mapper.readValue(json, cls);
	}
	
	@Context
	private ContainerRequestContext requestContext;

	@Context
	private ResourceInfo resourceInfo;
	
	@Context
	private HttpServletRequest httpServletRequest;
	
	@Context
	protected ServletContext context;
	
	private PrintWriter logWriter;
	
	private String[] source;
	private Map<Integer, String> comments = new HashMap<Integer, String>();
	
	public ContainerRequestContext getRequestContext() {
		return requestContext;
	}
	
	public ResourceInfo getResourceInfo() {
		return resourceInfo;
	}
	
	public HttpServletRequest getHttpServletRequest() {
		return httpServletRequest;
	}
	
	public synchronized PrintWriter getLogWriter() {
		if(logWriter == null) {
			try {
				@SuppressWarnings("unchecked")
				Class<? extends PrintWriter> logWriterClass = (Class<? extends PrintWriter>)Class.forName("net.osdn.util.jersey.log.LogWriter");
				Constructor<? extends PrintWriter> constructor = logWriterClass.getConstructor(int.class);
				logWriter = constructor.newInstance(getRequestContext().hashCode());
			} catch(Exception e) {
				e.printStackTrace();
				//EmptyPrintWriter
				return new PrintWriter(new Writer() {
					@Override
					public void write(char[] cbuf, int off, int len) throws IOException {
					}
					@Override
					public void flush() throws IOException {
					}
					@Override
					public void close() throws IOException {
					}
				});
			}
		}
		return logWriter;
	}
	
	public void print(boolean b) {
		System.out.print(b);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.print(b);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(char c) {
		System.out.print(c);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.print(c);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(int i) {
		System.out.print(i);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.print(i);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(long l) {
		System.out.print(l);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.print(l);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(float f) {
		System.out.print(f);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.print(f);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(double d) {
		System.out.print(d);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.print(d);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(char[] s) {
		if(s != null) {
			System.out.print(s);
		} else {
			System.out.print("null");
		}
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				if(s != null) {
					w.print(s);
				} else {
					w.print("null");
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(String s) {
		if(s != null) {
			System.out.print(s);
		} else {
			System.out.print("null");
		}
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				if(s != null) {
					w.print(s);
				} else {
					w.print("null");
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void print(Object obj) {
		if(obj != null) {
			System.out.print(obj);
		} else {
			System.out.print("null");
		}
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				if(obj != null) {
					w.print(obj);
				} else {
					w.print("null");
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println() {
		System.out.println();
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.println();
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(boolean b) {
		System.out.println(b);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.println(b);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(char c) {
		System.out.println(c);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.println(c);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(int i) {
		System.out.println(i);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.println(i);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(long l) {
		System.out.println(l);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.println(l);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(float f) {
		System.out.println(f);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.println(f);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(double d) {
		System.out.println(d);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.println(d);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(char[] s) {
		if(s != null) {
			System.out.println(s);
		} else {
			System.out.println("null");
		}
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				if(s != null) {
					w.println(s);
				} else {
					w.println("null");
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(String s) {
		if(s != null) {
			System.out.println(s);
		} else {
			System.out.println("null");
		}
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				if(s != null) {
					w.println(s);
				} else {
					w.println("null");
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void println(Object obj) {
		if(obj != null) {
			System.out.println(obj);
		} else {
			System.out.println("null");
		}
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				if(obj != null) {
					w.println(obj);
				} else {
					w.println("null");
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}
	
	public void printf(String format, Object... args) {
		System.out.printf(format, args);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.printf(format, args);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}

	public void printf(Locale l, String format, Object... args) {
		System.out.printf(l, format, args);
		PrintWriter w = getLogWriter();
		if(w != null) {
			try {
				w.printf(l, format, args);
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
	}

	protected String[] getSource() throws MissingResourceException, IOException {
		if(source == null) {
			String name = getClass().getName();
			int i = name.indexOf('$');
			if(i > 0) {
				name = name.substring(0, i);
			}
			String path = SOURCE_PATH + "/" + name.replace(".", "/") + ".java";
			
			InputStream in = context.getResourceAsStream(path);
			if(in == null) {
				throw new MissingResourceException(path, name, null);
			} else {
				BufferedReader r = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8));
				List<String> lines = new ArrayList<String>(); 
				String line;
				while((line = r.readLine()) != null) {
					lines.add(line);
				}
				source = lines.toArray(new String[]{});
			}
		}
		return source;
	}
	
	protected String below() throws MissingResourceException {
		int lineNumber = -1;
		
		StackTraceElement[] trace = Thread.currentThread().getStackTrace();
		if(trace.length <= 2) {
			return null;
		}
		lineNumber = trace[2].getLineNumber();
		String comment = comments.get(lineNumber);
		if(comment != null) {
			return comment;
		}
		
		StringBuilder sb = new StringBuilder();
		String[] source;
		try {
			source = getSource();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
		for(int i = lineNumber; i < source.length; i++) {
			sb.append(source[i].trim());
			sb.append("\n");
		}
		
		Matcher m = COMMENT.matcher(sb);
		if(m.find()) {
			comment = m.group(1);
			comment = comment.substring(comment.indexOf('\n') + 1, comment.lastIndexOf('\n'));
		} else {
			comment = "";
		}
		comments.put(lineNumber, comment);
		
		return comment;
	}
	
	protected class Problem extends ApplicationProblem  {
		private static final long serialVersionUID = 1L;

		private int code;
		
		public Problem(String title) {
			this(0, title, null);
		}
		
		public Problem(String title, String detail) {
			this(0, title, detail);
		}
		
		public Problem(int code, String title, String detail) {
			setCode(code);
			setTitle(title != null ? title : "");
			setDetail(detail != null ? detail : "");
			setInstance(getHttpServletRequest().getRequestURI());
		}
		
		@JsonProperty("code")
		public int getCode() {
			return code;
		}
		
		@JsonProperty("code")
		public Problem setCode(int code) {
			this.code = code;
			return this;
		}
	}
}
