use strict;
use warnings qw(all);
use utf8;

use threads;
use threads::shared;

use Test;

use POSIX qw(ceil floor);

BEGIN {
	plan(
		tests	=> 34,
	);
}

package Swatchdog::Actions::Perl::Refreshable::Test;
use strict;
use warnings qw(all);
use utf8;

use threads;
use threads::shared;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);
use Time::HiRes;

use Swatchdog::Actions::Perl::Refreshable;

use vars qw(
	@ISA
	$OutputDetail
	$Task
	$TaskStartAt
	$TaskRefreshedAt
	$TaskEndAt
);

@ISA = qw(Swatchdog::Actions::Perl::Refreshable);

$OutputDetail = 0;
$Task = undef;
$TaskStartAt = undef;
$TaskRefreshedAt = undef;
$TaskEndAt = undef;

share($OutputDetail);
share($Task);
share($TaskStartAt);
share($TaskRefreshedAt);
share($TaskEndAt);

sub Dump($) {
	my	$data = shift;
	local	$Data::Dumper::Indent = 0;
	local	$Data::Dumper::Terse = 1;

	Dumper($data);
}

sub new($@) {
	my $class = shift;
	my $self  = Swatchdog::Actions::Perl::Refreshable->new(
		period => 8.0,
		@_
	);

	bless $self, $class;

	return $self;
}

sub period($;$) {
	return $_[0]->attribute('period', @_[1 .. $#_]);
}

sub phase($;$) {
	return $_[0]->attribute('phase', @_[1 .. $#_]);
}

sub log($$$@) {
	my	$self = shift;
	my	$level = shift;
	my	($format, @args) = @_;

	if ($OutputDetail) {
		printf STDERR "%s %d.%d %s $format\n",
			strftime('%T', localtime),
			$$, threads->tid(),
			defined $self->{property}->{phase}
				? $self->{property}->{phase} : '-'
				,
			@args
			;
	}
}

sub mustEventBeDealtWith($) {
	my	$self = shift;
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	$self->logTrace('END   %s', (caller 0)[3]);
	return $self->{property}->{mustDealWith};
}

sub doTask($) {
	my	$self = shift;
	my	$caught;
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	$Task = $self->task();
	$self->logDebug('task: %s', Dump($Task));
	$TaskStartAt = Time::HiRes::time();
	do {
		$caught = undef;
		eval {
			local	$SIG{$Task->{signal}} = sub {
				$caught = $_[0];
				$TaskRefreshedAt = Time::HiRes::time();
				$self->logInfo('Caught SIG%s', $caught);
				die "Caught SIG$caught\n";
			};

			$self->sleep($self->period, $Task->{useThread});
		};
		if ($@ && !defined($caught)) {
			croak($@);
		}
	} while (defined $caught);
	$TaskEndAt = Time::HiRes::time();

	$self->logTrace('END   %s', (caller 0)[3]);
}
package main;

sub sleepTime($$) {
	my	$min = shift;
	my	$unit = shift;
	my	$value = $min;

	croak()	unless ($unit > 0);
	for (my $i = 0; $value <= $min || $value <= 0; $i++) {
		$value = ceil($min + $i*$unit);
	}

	return $value;
}

################
# Test case 1
# Load test for Swatchdog::Actions::Perl
################
eval { require Swatchdog::Actions::Perl::Refreshable; return 1;};
ok($@, '');
croak() unless ($@ eq '');

my $pa = Swatchdog::Actions::Perl::Refreshable::Test->new();
my $sleepTime;
################
# Test case 2-5
# sleep
################
$pa->logInfo('**** TEST PHASE 1 ****');
{
	my	($startAt, $endAt);

	$pa->useThread(0);
	$startAt = Time::HiRes::time();
	$pa->sleep($pa->sleepUnit/2.0);
	$endAt = Time::HiRes::time();

	ok($endAt - $startAt >= $pa->sleepUnit/2.0);
	ok($endAt - $startAt < $pa->sleepUnit);

	$pa->useThread(1);
	$startAt = Time::HiRes::time();
	$pa->sleep($pa->sleepUnit/2.0);
	$endAt = Time::HiRes::time();

	ok($endAt - $startAt >= $pa->sleepUnit);
	ok($endAt - $startAt < 2.0*$pa->sleepUnit);
}

################
# Test case 6-11
# no refresh
################
$pa->logInfo('**** TEST PHASE 2 ****');

$Swatchdog::Actions::Perl::Refreshable::Test::Task = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt = undef;

$pa->onEventOccurred(mustDealWith => 1, phase => 'first');

$sleepTime = sleepTime($pa->period, $pa->sleepUnit);
$pa->log('debug', '>>>> %d', $sleepTime);
threads->yield();
Time::HiRes::sleep $sleepTime;
$pa->logDebug('<<<<');

ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::Task);
ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt);
ok(!defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt);
ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt);
ok(
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt
		>=
	$pa->period
);
ok(
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt
		<=
	$pa->sleepUnit + $pa->period
);

$pa->logDebug('start at   : %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt
);
$pa->logDebug('end   at   : %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
);
$pa->logDebug('end - start: %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt
);
$pa->logDebug('unit       : %f', $pa->sleepUnit);
$pa->logDebug('period     : %d', $pa->period);

################
# Test case 12-13
# clean 1
################
$pa->logInfo('**** TEST PHASE 3 ****');

ok(exists
	$Swatchdog::Actions::Perl::Refreshable::Package2TaskHash{
		'Swatchdog::Actions::Perl::Refreshable::Test'
	}->{$pa->taskKey()}
);

$pa->cleanTaskHash();

ok(!exists
	$Swatchdog::Actions::Perl::Refreshable::Package2TaskHash{
		'Swatchdog::Actions::Perl::Refreshable::Test'
	}->{$pa->taskKey()}
);

################
# Test case 14-22
# refresh
################
$pa->logInfo('**** TEST PHASE 4 ****');

$Swatchdog::Actions::Perl::Refreshable::Test::Task = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt = undef;

$pa->onEventOccurred(mustDealWith => 1, phase => 'first');

$sleepTime = sleepTime($pa->period/2, $pa->sleepUnit);
$pa->logDebug('>>>> %d', $sleepTime);
threads->yield();
Time::HiRes::sleep $sleepTime;
$pa->logDebug('<<<<');

ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::Task);
ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt);
ok(!defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt);
ok(!defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt);

$pa->onEventOccurred(mustDealWith => 1, phase => 'second');

$sleepTime = sleepTime(0, $pa->sleepUnit);
$pa->logDebug('>>>> %d', $sleepTime);
threads->yield();
Time::HiRes::sleep $sleepTime;
$pa->logDebug('<<<<');

ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt);

$sleepTime = sleepTime($pa->period/2 - $sleepTime, $pa->sleepUnit);
$pa->logDebug('>>>> %d', $sleepTime);
threads->yield();
Time::HiRes::sleep $sleepTime;
$pa->logDebug('<<<<');

ok(!defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt);

$sleepTime = sleepTime($pa->period - $sleepTime, $pa->sleepUnit);
$pa->logDebug('>>>> %d', $sleepTime);
threads->yield();
Time::HiRes::sleep $sleepTime;
$pa->logDebug('<<<<');

ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt);
ok(
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
		>=
	$pa->period
);
ok(
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
		<=
	$pa->sleepUnit + $pa->period
);

$pa->logDebug('refreshed at   : %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
);
$pa->logDebug('end       at   : %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
);
$pa->logDebug('end - refreshed: %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
);
$pa->logDebug('unit           : %f', $pa->sleepUnit);
$pa->logDebug('period         : %d', $pa->period);

################
# Test case 23-24
# clean 2
################
$pa->logInfo('**** TEST PHASE 5 ****');

ok(exists
	$Swatchdog::Actions::Perl::Refreshable::Package2TaskHash{
		'Swatchdog::Actions::Perl::Refreshable::Test'
	}->{$Swatchdog::Actions::Perl::Refreshable::Test::Task->{taskKey}}
);

Swatchdog::Actions::Perl::Refreshable::CleanTaskHash($pa->taskHash());

ok(!exists
	$Swatchdog::Actions::Perl::Refreshable::Package2TaskHash{
		'Swatchdog::Actions::Perl::Refreshable::Test'
	}->{$Swatchdog::Actions::Perl::Refreshable::Test::Task->{taskKey}}
);

################
# Test case 25-28
# another signal
################
$pa->logInfo('**** TEST PHASE 6 ****');

$Swatchdog::Actions::Perl::Refreshable::Test::Task = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskStartAt = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt = undef;
$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt = undef;

$pa->refreshSignal('USR1');
$pa->onEventOccurred(mustDealWith => 1, phase => 'first');

$sleepTime = sleepTime($pa->period/2, $pa->sleepUnit);
threads->yield();
Time::HiRes::sleep $sleepTime;

$pa->onEventOccurred(mustDealWith => 1, phase => 'second');

$sleepTime = sleepTime($pa->period, $pa->sleepUnit);
threads->yield();
Time::HiRes::sleep $sleepTime;

ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt);
ok(defined $Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt);
ok(
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
		>=
	$pa->period
);
ok(
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
		<=
	$pa->sleepUnit + $pa->period
);

$pa->logDebug('refreshed at   : %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
);
$pa->logDebug('end       at   : %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
);
$pa->logDebug('end - refreshed: %f',
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskEndAt
		-
	$Swatchdog::Actions::Perl::Refreshable::Test::TaskRefreshedAt
);
$pa->logDebug('unit           : %f', $pa->sleepUnit);
$pa->logDebug('period         : %d', $pa->period);

################
# Test case 29-32
# use process
################
$pa->logInfo('**** TEST PHASE 7 ****');

$pa->refreshSignal($Swatchdog::Actions::Perl::Refreshable::RefreshSignal);
$pa->useThread(0);

ok(!$pa->asynchronous);

$pa->asynchronous(1);

ok($pa->useThread);

$pa->useThread(0);
$pa->onEventOccurred(mustDealWith => 1, phase => 'first');

Time::HiRes::sleep $pa->period/2;

$Swatchdog::Actions::Perl::Refreshable::Test::Task = $pa->task;
$pa->onEventOccurred(mustDealWith => 1, phase => 'second');

Time::HiRes::sleep $pa->period/2 + 0.5;

ok(kill(0, $Swatchdog::Actions::Perl::Refreshable::Test::Task->{taskId}));

Time::HiRes::sleep $pa->period/2 + 0.5;

ok(!kill(0, $Swatchdog::Actions::Perl::Refreshable::Test::Task->{taskId}));


################
# Test case 33 - 34
# clean 3
################
$pa->logInfo('**** TEST PHASE 8 ****');

#$pa->logDebug(Swatchdog::Actions::Perl::Refreshable::Test::Dump(\%Swatchdog::Actions::Perl::Refreshable::Package2TaskHash));
ok(exists
	$Swatchdog::Actions::Perl::Refreshable::Package2TaskHash{
		'Swatchdog::Actions::Perl::Refreshable::Test'
	}->{$Swatchdog::Actions::Perl::Refreshable::Test::Task->{taskKey}}
);
Swatchdog::Actions::Perl::Refreshable::CleanAllTaskHash();
ok(!exists
	$Swatchdog::Actions::Perl::Refreshable::Package2TaskHash{
		'Swatchdog::Actions::Perl::Refreshable::Test'
	}->{$Swatchdog::Actions::Perl::Refreshable::Test::Task->{taskKey}}
);
