package Swatchdog::Actions;

use strict;
use POSIX ":sys_wait_h";

################################################################
# Add or Remove SIGCHLD handler
#
# setSigchldHandler($name, $code, $prio) -- add new handler
# 	setSigchldHandler adds new SIGCHLD handler named $name.
# 	The body of the handler is $code, which is a subroutine.
# 	It is called with two arguments. The first one is process id
# 	of the child process which terminates. The second one is
#   exit code of the child process.
# 	$prio is priority. It must be integer. If it is omitted,
#   0 is used as its value. A handler which has smaller priority
#   will be execute first.
# setSigchldHandler($name) -- remove handlers named $name 
# 	setSigchldHandler remove SIGCHLD handlers named $name.
# setSigchldHandler(undef) -- remove handlers which has no name. 
# 	setSigchldHandler remove SIGCHLD handlers which has no name.
################################################################
use vars qw(
	@SigchldHandler
);

@SigchldHandler = ();

sub _CmpSigchldHandler {
	my	($a, $b) = @_;
	my	$value;

	$value = $a->{prio} <=> $b->{prio};
	if ($value == 0) {
		if (defined $a->{name}) {
			if (defined $b->{name}) {
				$value = $a->{name} cmp $b->{name};
			}
			else {
				$value = 1;
			}
		}
		else {
			if (defined $b->{name}) {
				$value = -1
			}
			else {
				$value = 0;
			}
		}
	}

	return $value;
}

sub setSigchldHandler($;$$) {
	my	$handler = { name => $_[0], code => $_[1], prio => $_[2] };
	my	$pos;
	my	@target;

	if (defined $handler->{code}) {
		$handler->{prio} = 0	unless (defined $handler->{prio});

		for ($pos = 0; $pos < scalar @SigchldHandler; $pos++) {
			last	if (
				_CmpSigchldHandler($handler, $SigchldHandler[$pos]) < 0
			);
		}
		@SigchldHandler = (
			@SigchldHandler[0 .. $pos-1],
			$handler,
			@SigchldHandler[$pos .. $#SigchldHandler]
		);
		push @target, $handler;
	}
	else {
		my	$tester = defined $handler->{name}
			? sub { defined($_->{name}) && $_->{name} eq $handler->{name} }
			: sub { !defined($_->{name}) }
			;
		my	@notarget;

		foreach (@SigchldHandler) {
			if (&{$tester}($_)) {
				push @target, $_;
			}
			else {
				push @notarget, $_;
			}
		}
		@SigchldHandler = @notarget;
	}

	return @target;
}

$SIG{CHLD} = sub {
	local	($!, $?);

	while ((my $pid = waitpid(-1, WNOHANG)) > 0) {
		foreach my $handler (@SigchldHandler) {
			&{$handler->{code}}($pid, $?);
		}
	}
};

################################################################
# "exec" Action
#
# exec_command(args) -- fork and execute a command
################################################################

sub exec_command {
	my %args = (@_);
	my $exec_pid;
	my $command;

	if (exists $args{'COMMAND'}) {
		$command = $args{'COMMAND'};
	}
	else {
		warn "$0: No command was specified in exec action.\n";
		return 1;
	}

	return 0
		if (
			exists($args{'WHEN'})
				and
			not inside_time_window($args{'WHEN'})
		);

	return
		if (
			exists($args{'THRESHOLDING'})
				and
			$args{'THRESHOLDING'} eq 'on'
				and
			not &Swatchdog::Threshold::threshold(%args)
		);

	EXECFORK: {
		if ($exec_pid = fork) {
			return 0;
		}
		elsif (defined $exec_pid) {
			exec($command);
		}
		elsif ($! =~ /No more processes/) {
			# EAGAIN, supposedly recoverable fork error
			sleep 5;
			redo EXECFORK;
		} else {
			warn "$0: Can't fork to exec $command: $!\n";
			return 1;
		}
	}

	return 0;
}
