package Swatchdog::Actions::Perl::Refreshable;

use strict;
use warnings qw(all);
use utf8;

use threads;
use threads::shared;

use Carp;
use Time::HiRes;

use Swatchdog::Actions::Perl;

use vars qw(
	$VERSION
	@ISA

	$RefreshSignal
	$SleepUnit
	%Package2TaskHash
);

BEGIN {
	$VERSION = '0.1';
}

@ISA = qw(Swatchdog::Actions::Perl);

$RefreshSignal = 'HUP';
$SleepUnit = 0.23;
%Package2TaskHash = ();
share(%Package2TaskHash);

sub _CleanTaskHash($$) {
	my	$taskHash = shift;
	my	$taskKey = shift;

	if (exists $taskHash->{$taskKey}) {
		if ($taskHash->{$taskKey}->{useThread}) {
			my	$thread = threads->object($taskHash->{$taskKey}->{taskId});

			if (defined $thread and $thread->is_joinable()) {
				$thread->join();
				$thread = undef;
			}
			unless (defined $thread) {
				delete $taskHash->{$taskKey};
			}
		}
		else {
			unless (kill 0, $taskHash->{$taskKey}->{taskId}) {
				delete $taskHash->{$taskKey};
			}
		}
	}
}

sub CleanTaskHash($) {
	my	$taskHash = shift;
	lock($taskHash);

	foreach my $taskKey (keys %{$taskHash}) {
		_CleanTaskHash($taskHash, $taskKey);
	}
}

sub CleanAllTaskHash() {
	lock(%Package2TaskHash);
	while (my ($pack, $hash) = each %Package2TaskHash) {
		CleanTaskHash($hash);
	}
}

sub new($@) {
	my	$class = shift;
	my	$self = Swatchdog::Actions::Perl->new(
		refreshSignal	=> $RefreshSignal,
		sleepUnit		=> $SleepUnit,
		@_
	);

	bless $self, $class;

	return $self;
}

sub useThread($;$) {
	my	$self = shift;
	my	$oldValue = $self->{attribute}->{useThread};

	if (scalar(@_) > 0) {
		$self->{attribute}->{useThread} = $_[0];
		if (!$self->{attribute}->{useThread}) {
			$self->{attribute}->{asynchronous} = 0;
		}
	}

	return $oldValue;
}

sub asynchronous($;$) {
	my	$self = shift;
	my	$oldValue = $self->{attribute}->{asynchronous};

	if (scalar(@_) > 0) {
		$self->{attribute}->{asynchronous} = $_[0];
		if ($self->{attribute}->{asynchronous}) {
			$self->{attribute}->{useThread} = 1;
		}
	}

	return $oldValue;
}

sub refreshSignal($;$) {
	return $_[0]->attribute('refreshSignal', @_[1 .. $#_]);
}

sub sleepUnit($;$) {
	return $_[0]->attribute('sleepUnit', @_[1 .. $#_]);
}

sub taskKey($) {
	my	$self = shift;
	my	$taskKey = '';

	return $taskKey;
}

sub taskHash($) {
	lock(%Package2TaskHash);
	my	$self = shift;
	my	$class = ref($self);

	unless (exists $Package2TaskHash{$class}) {
		$Package2TaskHash{$class} = shared_clone({});
	}

	return $Package2TaskHash{$class};
}

sub cleanTaskHash($) {
	my	$self = shift;
	my	$taskHash = $self->taskHash;
	lock($taskHash);
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	_CleanTaskHash($taskHash, $self->taskKey);

	$self->logTrace('END   %s', (caller 0)[3]);
}

sub task($) {
	my	$self = shift;
	my	$taskHash = $self->taskHash;
	lock($taskHash);

	$self->cleanTaskHash();

	return $taskHash->{$self->taskKey};
}

sub doTask($) {
	my	$self = shift;
	$self->logTrace('BEGIN %s', (caller 0)[3]);

	$self->logWarn('You must implement in a sub class.');

	$self->logTrace('END   %s', (caller 0)[3]);
}

sub dealWithEvent($) {
	my	$self = shift;
	my	$taskHash = $self->taskHash;
	lock($taskHash);
	my	$task;
	$self->logDebug('BEGIN %s', (caller 0)[3]);

	$task = $self->task;
	if (defined $task) {
		$self->logInfo(
			'Refresh the task #%d with SIG%s.',
			$task->{taskId}, $task->{signal}
		);
		if ($task->{useThread}) {
			threads->object($task->{taskId})->kill('SIG'.$task->{signal});
		}
		else {
			kill $task->{signal}, $task->{taskId};
		}
	}
	else {
		$task = shared_clone({
			taskKey	=> $self->taskKey,
			taskId	=> undef,
			signal	=> $self->refreshSignal,
			useThread	=> $self->useThread
		});
		if ($task->{useThread}) {
			my	$thread = threads->create(sub {
				my	$self = shift;

				threads->yield();
				$self->doTask();
			}, $self);

			if (defined $thread) {
				$task->{taskId} = $thread->tid();
			}
		}
		else {
			$task->{taskId} = fork;
			if (defined $task->{taskId}) {
				if ($task->{taskId} == 0) {
					$task->{taskId} = $$;
					$taskHash->{$task->{taskKey}} = $task;
					$self->doTask();
					exit;
				}
			}
		}

		if (defined $task->{taskId}) {
			$taskHash->{$task->{taskKey}} = $task;
			$self->logInfo('New task #%d is created.', $task->{taskId});
		}
		else {
			$self->logFatal("Can't create new task: %s", $!);
		}
	}

	$self->logDebug('END   %s', (caller 0)[3]);
}

sub sleep($$;$) {
	my	$self = shift;
	my	$length = shift;
	my	$useThread = scalar(@_) > 0 ? $_[0] : $self->useThread;

	if ($useThread) {
		my	$until = Time::HiRes::time() + $length;
		my	$unit = $self->sleepUnit;
	
		threads->yield();
		while (Time::HiRes::time() < $until) {
			Time::HiRes::sleep($unit);
			threads->yield();
		}
	}
	elsif (0 < $length) {
		Time::HiRes::sleep($length);
	}
}

1;

=head1 NAME

Swatchdog::Actions::Perl::Refreshable - Refreshable Long Task

=head1 SYNOPSIS

In your perl module file:

    package Swatchdog::Actions::Perl::DelayAction;
    use Swatchdog::Actions::Perl::Refreshable;

    use vars qw(
        @ISA
        $Delay
    );
    
    @ISA = qw(Swatchdog::Actions::Perl::Refreshable);
    $Delay = 30*60;
    
    sub new($@) {
        my $class = shift;
        my $self = Swatchdog::Actions::Perl::Refreshable->new(
            delay => $Delay,
            @_
        );
    
        bless $self, $class;
    
        return $self;
    }
    
    sub delay($;$) {
        return $_[0]->attribute('delay', @_[1 .. $#_]);
    }

    sub mustEventBeDealtWith($) {
        my  $self = shift;
        $self->logTrace('BEGIN %s', (caller 0)[3]);
    
        $self->logTrace('END   %s', (caller 0)[3]);
        return 1;
    }
    
    sub taskKey($) {
        my  $self = shift;
        my  $taskKey;
        $self->logTrace('BEGIN %s', (caller 0)[3]);
    
        $taskKey = $self->{property}->{user};
    
        $self->logTrace('END   %s', (caller 0)[3]);
        return $taskKey;
    }
    
    sub dotTask($) {
        my $self = shift;
        my $task = $self->task;
        $self->logInfo('BEGIN %s', (caller 0)[3]);
    
        local $SIG{$task->{signal}} = sub {
            my  ($sig) = @_;
    
            $this->logInfo('Caught SIG%s. But ignore it.', $sig);
        };
        
        WAIT_AND_DO: eval {
            local $SIG{$task->{signal}} = sub {
                my  ($sig) = @_;
    
                $this->logInfo('Caught SIG%s.', $sig);
                die "Caught a signal\n";
            };
    
            $self->sleep($self->delay(), $task->{useThread});
            DO_YOUR_ACTUAL_ACTION();
        };
        if ($@) {
            if ($@ =~ m/^Caught a signal\b/) {
                goto WAIT_AND_DO;
            }
            else {
                $self->logError('%s', $@);
            }
        }
    
        $self->logInfo('END   %s', (caller 0)[3]);
    }

In the configuration file of Swatchdog:

    perlcode 0 use Swatchdog::Actions::Perl::DelayAction;

    perlcode 0 my $DelayAction = Swatchdog::Actions::Perl::DelayAction->new();
    watchfor /^\w{3}\s+\d{1-2} \d{2}:\d{2}:\d{2} \S+ sshd\[\d+\]: Accepted publickey for (\w+) from: (\S+) port /
        perlcode 2 $DelayAction->onEventOccurred(
        perlcode 2     user => $1, from => $2,
        perlcode 2     secure => 1
        perlcode 2 );

=head1 DESCRIPTION

This module is a base class, which provides long task that is begun by a log message. It is a derived class of L<Swatchdog::Actions::Perl>.

The task is identified by a key which is return value of the method taskKey().
While the task is running, if a new event of the same key occurs,
the signal SIGHUP will be sent to the running task.

=head1 METHODS

First, please refer to L<the manual of Swatchdog::Actions::Perl|Swatchdog::Actions::Perl>.

=head2 Methods that must be overridden

B<IMPORTANT>:
Don't inadvertently override the method dealWithEvent().
It has been overridden in this module.
Override the method dotTask() in substitution for dealWithEvent().

=over 4

=item $action->taskKey()

This method returns a task key.
Each task is identified by a task key.
If you need only one task, you may not override it.

=item $action->doTask()

This method is the task itself.

While this method is running, if a new event of the same key occurs,
the signal SIGHUP will be sent. For this reason, you must define this method so that it can receive a SIGHUP signal.

This method runs on a new thread or a new process, even if the attribute S<asynchronous> is set to a false value.

=back

=head2 Attributes

=over 4

=item $action->refreshSignal()

=item $action->refreshSignal($signalName)

This is a name of the signal which is sent to refresh running task.

Default: C<$RefreshSignal>

=item $action->sleepUnit()

=item $action->sleepUnit($float)

Roughness of the time step for the method sleep().
Its unit is seconds.

Default: C<$SleepUnit>

=item $action->useThread()

=item $action->useThread($bool)

This is the same as $action->SUPER::useThread with the exception of the following.

When given I<$bool> is a false value, the attribute S<asynchronous> will be set to a false value.

=item $action->asynchronous()

=item $action->asynchronous($bool)

This is the same as $action->SUPER::asynchronous with the exception of the following.

When given I<$bool> is a true value, the attribute S<useThread> will be set to a true value.

=back

=head2 Other Methods

=over 4

=item $action->taskHash()

This method returns a hash reference in the class of I<$action>.
A key of the hash is a task key, and its value is a task.

This hash is shared to threads.

=item $action->cleanTaskHash()

When the task of I<$action> has terminated, this method remove it from the task hash.

=item $action->task()

This method returns a task.
It is identified by the method taskKey().
This method is almost the same as C<$action-E<gt>taskHash-E<gt>{$action-E<gt>taskKey}>.

A task is a hash reference which is shared to threads.
It has following keys.

=over 8

=item taskKey

The value returned by the method taskKey() when the task is created.

=item useThread

The value returned by the attribute S<useThread> when the task is created.

=item signal

The value returned by the attribute S<refreshSignal> when the task is created.

=item taskId

If the value of the key S<useThread> is true, this is a thread ID.
Otherwise, it is a process ID.

=back

=item $action->sleep($float)

This method is similar to Time::HiRes::sleep.
However, when the attribute S<useThread> is true, it is slightly different.

In a thread of perl, signals which are sent to a sleeping thread are blocked.
Therefore, a sleep of a thread must be a repeat of small sleep of the system.

When the attribute S<useThread> is true, this method repeats small sleep of the system.
The length of small sleep is the value of the attribute S<sleepUnit>.

=back

=head1 FUNCTIONS

=over 4

=item CleanAllTaskHash()

This function removes entries that are no longer required from task hashes of all derived classes.

=item CleanTaskHash($taskHash)

This function removes entries that are no longer required from task hash I<$taskHash>.

=back

=head1 VARIABLES

=over 4

=item $RefreshSignal

Default value of the attribute S<refreshSignal>.

=item $SleepUnit

Default value of the attribute S<sleepUnit>.

=item %Package2TaskHash

This maps package name (class name) onto its task hash.
This is shared to threads.

=back

=head1 VERSION

    0.01

=head1 AUTHOR

Yuji Okamura <okamura[at]users.osdn.me>

=head1 COPYRIGHT

Copyright (C) 2015 Yuji OKamura. All Rights Reserved.

This module is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

